#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.

from fenrirscreenreader.core.i18n import _
from fenrirscreenreader.utils import line_utils
from fenrirscreenreader.utils import word_utils


class command:
    def __init__(self):
        self.lastIdent = -1

    def initialize(self, environment):
        self.env = environment

    def shutdown(self):
        pass

    def get_description(self):
        return ""

    def run(self):
        if not self.env["runtime"]["SettingsManager"].get_setting_as_bool(
            "focus", "cursor"
        ):
            return
        if self.env["runtime"]["ScreenManager"].is_screen_change():
            self.lastIdent = 0
            return
        # this leads to problems in vim -> status line change -> no
        # announcement, so we do check the lengh as hack
        if self.env["runtime"]["ScreenManager"].is_delta():
            return

        # is a vertical change?
        if not self.env["runtime"]["CursorManager"].is_cursor_vertical_move():
            return

        x, y, curr_line = line_utils.get_current_line(
            self.env["screen"]["new_cursor"]["x"],
            self.env["screen"]["new_cursor"]["y"],
            self.env["screen"]["new_content_text"],
        )
        if curr_line.isspace():
            self.env["runtime"]["OutputManager"].present_text(
                _("blank"), sound_icon="EmptyLine", interrupt=True, flush=False
            )
        else:
            # ident
            curr_ident = len(curr_line) - len(curr_line.lstrip())
            if self.lastIdent == -1:
                self.lastIdent = curr_ident
            do_interrupt = True
            if self.env["runtime"]["SettingsManager"].get_setting_as_bool(
                "general", "autoPresentIndent"
            ):
                if self.env["runtime"]["SettingsManager"].get_setting_as_int(
                    "general", "autoPresentIndentMode"
                ) in [0, 1]:
                    self.env["runtime"]["OutputManager"].play_frequence(
                        curr_ident * 50, 0.1, interrupt=do_interrupt
                    )
                if self.env["runtime"]["SettingsManager"].get_setting_as_int(
                    "general", "autoPresentIndentMode"
                ) in [0, 2]:
                    if self.lastIdent != curr_ident:
                        self.env["runtime"]["OutputManager"].present_text(
                            _("indented ") + str(curr_ident) + " ",
                            interrupt=do_interrupt,
                            flush=False,
                        )
                        do_interrupt = False
            # barrier
            say_line = curr_line
            if self.env["runtime"]["SettingsManager"].get_setting_as_bool(
                "barrier", "enabled"
            ):
                is_barrier, barrierLine = self.env["runtime"][
                    "BarrierManager"
                ].handle_line_barrier(
                    self.env["screen"]["new_content_text"].split("\n"),
                    self.env["screen"]["new_cursor"]["x"],
                    self.env["screen"]["new_cursor"]["y"],
                )
                if is_barrier:
                    say_line = barrierLine
            # output
            self.env["runtime"]["OutputManager"].present_text(
                say_line, interrupt=do_interrupt, flush=False
            )
            self.lastIdent = curr_ident

    def set_callback(self, callback):
        pass
