/*
 * Copyright (c) 2015-2017, Intel Corporation
 * Copyright (c) 2020, 2021, VectorCamp PC
 * Copyright (c) 2023, Arm Limited
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * Neither the name of Intel Corporation nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/** \file
 * \brief Truffle: character class acceleration.
 */

#include "truffle.h"
#include "ue2common.h"
#include "util/arch.h"
#include "util/bitutils.h"

#include "truffle_simd.hpp"
#ifdef CAN_USE_WIDE_TRUFFLE
#ifdef HAVE_SVE
const u8 *truffleExecWide(m256 mask, const u8 *buf,
                      const u8 *buf_end) {
    if (svcntb() == 16) {
        return truffleExecSVE<true, true>(mask, buf, buf_end);
    } else {
        return truffleExecSVE<true, false>(mask, buf, buf_end);
    }
}

const u8 *rtruffleExecWide(m256 mask, const u8 *buf,
                       const u8 *buf_end) {
    if (svcntb() == 16) {
        return rtruffleExecSVE<true, true>(mask, buf, buf_end);
    } else {
        return rtruffleExecSVE<true, false>(mask, buf, buf_end);
    }
}
#else // HAVE_SVE
#error "Wide truffle enabled for the target architecture but no implementation found"
#endif // HAVE_SVE
#endif // CAN_USE_WIDE_TRUFFLE


#ifdef HAVE_SVE
const u8 *truffleExec(m128 mask_lo, m128 mask_hi, const u8 *buf,
                      const u8 *buf_end) {
    if (svcntb() == 16) {
        return truffleExecSVE<false, true>({.lo = mask_lo, .hi = mask_hi}, buf, buf_end);
    } else {
        return truffleExecSVE<false, false>({.lo = mask_lo, .hi = mask_hi}, buf, buf_end);
    }
}

const u8 *rtruffleExec(m128 mask_lo, m128 mask_hi, const u8 *buf,
                       const u8 *buf_end) {
    if (svcntb() == 16) {
        return rtruffleExecSVE<false, true>({.lo = mask_lo, .hi = mask_hi}, buf, buf_end);
    } else {
        return rtruffleExecSVE<false, false>({.lo = mask_lo, .hi = mask_hi}, buf, buf_end);
    }
}
#else
const u8 *truffleExec(m128 mask_lo, m128 mask_hi, const u8 *buf,
                      const u8 *buf_end) {
    return truffleExecReal<VECTORSIZE>(mask_lo, mask_hi, buf, buf_end);
}

const u8 *rtruffleExec(m128 mask_lo, m128 mask_hi, const u8 *buf,
                       const u8 *buf_end) {
    return rtruffleExecReal<VECTORSIZE>(mask_lo, mask_hi, buf, buf_end);
}
#endif //HAVE_SVE
