package net.minidev.json.parser;

import java.io.InputStream;
import java.io.Reader;
import net.minidev.json.JSONValue;
import net.minidev.json.writer.JsonReaderI;

/**
 * json-smart will parse multiple json separated by blank or line break character.
 *
 * <p>multiple json example: <br>
 * {"json1": "value1"} {"json2": "value2"} <br>
 * or <br>
 * [{"json1-key1": "value1"}] [{"json2": "value2"}]
 */
public class MultipleJsonParser {

  private final JSONParserBase jsonParser;

  public MultipleJsonParser(byte[] in, int permissiveMode) {
    this.jsonParser = new JSONParserByteArray(in, permissiveMode);
  }

  public MultipleJsonParser(InputStream in, int permissiveMode) {
    this.jsonParser = new JSONParserInputStream(in, permissiveMode);
  }

  public MultipleJsonParser(Reader in, int permissiveMode) {
    this.jsonParser = new JSONParserReader(in, permissiveMode);
  }

  public MultipleJsonParser(String in, int permissiveMode) {
    this.jsonParser = new JSONParserString(in, permissiveMode);
  }

  /**
   * Parse next json with defaultReader <br>
   * use to return Primitive Type, or String, Or JsonObject or JsonArray generated by a
   * ContainerFactory
   */
  public Object parseNext() throws ParseException {
    return jsonParser.parseNext(JSONValue.defaultReader.DEFAULT);
  }

  /**
   * Parse next json with target JsonReaderI <br>
   * use to return Primitive Type, or String, Or JsonObject or JsonArray generated by a
   * ContainerFactory
   */
  public <T> T parseNext(JsonReaderI<T> mapper) throws ParseException {
    return this.jsonParser.parseNext(mapper);
  }

  /** Parse next json with target Class */
  public <T> T parseNext(Class<T> mapTo) throws ParseException {
    return this.jsonParser.parseNext(JSONValue.defaultReader.getMapper(mapTo));
  }

  /**
   * Checks if there is another JSON value available in the input.
   *
   * @return true if another JSON value exists, false otherwise
   */
  public boolean hasNext() {
    return this.jsonParser.hasNext();
  }
}
