/* Copyright © Triad National Security, LLC, and others. */

#define _GNU_SOURCE
#include "config.h"

#include <fnmatch.h>
#include <string.h>
#include <unistd.h>

#include "all.h"


/** Macros **/

/* FNM_EXTMATCH is a GNU extension to support extended globs in fnmatch(3).
   If not available, define as 0 to ignore this flag. */
#ifndef HAVE_FNM_EXTMATCH
#define FNM_EXTMATCH 0
#endif


/** Function prototypes (private) **/


/** Functions **/

/* Read the file listing environment variables at path, with records separated
   by delim, and return a corresponding list of struct env_var. Reads the
   entire file one time without seeking. If there is a problem reading the
   file, or with any individual variable, exit with error.

   The purpose of delim is to allow both newline- and zero-delimited files. We
   did consider using a heuristic to choose the file’s delimiter, but there
   seemed to be two problems. First, every heuristic we considered had flaws.
   Second, use of a heuristic would require reading the file twice or seeking.
   We don’t want to demand non-seekable files (e.g., pipes), and if we read
   the file into a buffer before parsing, we’d need our own getdelim(3). See
   issue #1124 for further discussion. */
struct env_var *env_file_read(const char *path, int delim)
{
   struct env_var *vars;
   FILE *fp;

   Tfe (fp = fopen(path, "r"), "can't open: %s", path);

   vars = list_new(0, sizeof(struct env_var));
   for (size_t line_no = 1; true; line_no++)
   {
      struct env_var var;
      char *line;
      errno = 0;
      line = getdelim_ch(fp, delim);
      if (line == NULL) // EOF
         break;
      if (line[0] == '\0') // skip blank lines
         continue;
      var = env_parse(line, path, line_no);
      list_append((void **)&vars, &var, sizeof(var));
   }

   Zfe (fclose(fp), "can't close: %s", path);
   return vars;
}

/* Return the value of environment variable name if set; otherwise, return
   value_default instead. */
char *env_get(const char *name, char *value_default)
{
#undef getenv
   char *ret = getenv(name);
#define getenv FN_BLOCKED
   return ret ? ret : value_default;
}

/** Unset environment variable @p name. */
void env_unset(const char *name)
{
#undef unsetenv
   Z__ (unsetenv(name));
#define unsetenv FN_BLOCKED
}

/* Parse the environment variable in line and return it as a struct env_var.
   Exit with error on syntax error; if path is non-NULL, attribute the problem
   to that path at line_no. Note: Trailing whitespace such as newline is
   *included* in the value. */
struct env_var env_parse(const char *line, const char *path, size_t lineno)
{
   char *name, *value, *where;

   if (path == NULL)
      where = strdup_ch(line);
   else
      where = asprintf_ch("%s:%zu", path, lineno);

   // Split line into variable name and value.
   split(&name, &value, line, '=');
   Tf_ (value != NULL, "can't parse variable: no delimiter: %s", where);
   Tf_ (name[0] != 0, "can't parse variable: empty name: %s", where);

   // Strip leading and trailing single quotes from value, if both present.
   if (   strlen(value) >= 2
       && value[0] == '\''
       && value[strlen(value) - 1] == '\'') {
      value[strlen(value) - 1] = 0;
      value++;
   }

   return (struct env_var){ name, value };
}

/* Set environment variable name to value. If expand, then further expand
   variables in value marked with "$" as described in the man page. */
void env_set(const char *name, const char *value, const bool expand)
{
   // Walk through value fragments separated by colon and expand variables.
   if (expand) {
      char *vwk;               // modifiable copy of value
      char *vwk_cur;           // current location in vwk
      char *vout = "";         // output (expanded) string
      bool first_out = false;  // true after 1st output element written
      vwk = strdup_ch(value);
      vwk_cur = vwk;
      while (true) {                            // loop executes ≥ once
         char *elem = strsep(&vwk_cur, ":");    // NULL -> no more elements
         if (elem == NULL)
            break;
         if (elem[0] == '$' && elem[1] != 0) {  // looks like $VARIABLE
#undef getenv
            elem = getenv(elem + 1);            // NULL if unset
#define getenv FN_BLOCKED
            if (elem != NULL && elem[0] == 0)   // set but empty
               elem = NULL;                     // convert to unset
         }
         if (elem != NULL) {   // empty -> omit from output list
            vout = cats(3, vout, first_out ? ":" : "", elem);
            first_out = true;
         }
      }
      value = vout;
   }

   // Save results.
   DEBUG("environment: %s=%s", name, value);
#undef setenv
   Z__ (setenv(name, value, 1));
#define setenv FN_BLOCKED
}

/* Set the environment variables listed in d. */
void envs_hook_set(struct container *c, void *d)
{
   struct env_var *vars = d;
   envs_set(vars, c->env_expand);
}

/* Set the environment variables specified in file d. */
void envs_hook_set_file(struct container *c, void *d)
{
   struct env_file *ef = d;
   envs_set(env_file_read(ef->path, ef->delim), c->env_expand);
}

/* Unset the environment variables matching glob d. */
void envs_hook_unset(struct container *c, void *d)
{
   envs_unset((char *)d);
}

/** Set zero or more environment variables

    @param vars    List of variables to set.

    @param expand  If true, expand variable references in values. */
void envs_set(const struct env_var *vars, const bool expand)
{
   for (size_t i = 0; vars[i].name != NULL; i++)
      env_set(vars[i].name, vars[i].value, expand);
}

/* Remove variables matching glob from the environment. This is tricky,
   because there is no standard library function to iterate through the
   environment, and the environ global array can be re-ordered after
   unsetenv(3) [1]. Thus, the only safe way without additional storage is an
   O(n^2) search until no matches remain.

   Our approach is O(n): we build up a copy of environ, skipping variables
   that match the glob, and then assign environ to the copy. This is a valid
   thing to do [2].

   [1]: https://unix.stackexchange.com/a/302987
   [2]: http://man7.org/linux/man-pages/man3/exec.3p.html */
void envs_unset(const char *glob)
{
   char **new_environ = list_new(0, sizeof(char *));
   for (size_t i = 0; environ[i] != NULL; i++) {
      char *name, *value;
      int matchp;
      split(&name, &value, environ[i], '=');
      T__ (value != NULL);  // environ entries must always have equals
      matchp = fnmatch(glob, name, FNM_EXTMATCH);  // extglobs if available
      if (matchp == 0) {
         DEBUG("environment: unset %s", name);
      } else {
         T__ (matchp == FNM_NOMATCH);
         *(value - 1) = '=';  // rejoin line
         list_append((void **)&new_environ, &name, sizeof(name));
      }
   }
   environ = new_environ;
}