// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.openqa.selenium.remote;

import static org.openqa.selenium.remote.CapabilityType.BROWSER_NAME;

import com.google.auto.service.AutoService;
import java.util.Optional;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.ImmutableCapabilities;
import org.openqa.selenium.SessionNotCreatedException;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.remote.http.ClientConfig;

@AutoService(WebDriverInfo.class)
public class FakeWebDriverInfo implements WebDriverInfo {

  static final String FAKE_BROWSER = "selenium-test";

  @Override
  public String getDisplayName() {
    return FAKE_BROWSER;
  }

  @Override
  public Capabilities getCanonicalCapabilities() {
    return new ImmutableCapabilities(BROWSER_NAME, FAKE_BROWSER);
  }

  @Override
  public boolean isSupporting(Capabilities capabilities) {
    return FAKE_BROWSER.equals(capabilities.getCapability(BROWSER_NAME));
  }

  @Override
  public boolean isSupportingCdp() {
    return true;
  }

  @Override
  public boolean isSupportingBiDi() {
    return false;
  }

  @Override
  public boolean isAvailable() {
    return true;
  }

  @Override
  public boolean isPresent() {
    return false;
  }

  @Override
  public int getMaximumSimultaneousSessions() {
    return Runtime.getRuntime().availableProcessors();
  }

  @Override
  public Optional<WebDriver> createDriver(Capabilities capabilities, ClientConfig clientConfig)
      throws SessionNotCreatedException {

    return Optional.of(new FakeWebDriver(capabilities, clientConfig));
  }

  public static class FakeWebDriver extends RemoteWebDriver {
    // For cglib
    protected FakeWebDriver() {}

    public FakeWebDriver(Capabilities capabilities, ClientConfig clientConfig) {
      super(command -> null, capabilities, clientConfig);
    }

    @Override
    protected void startSession(Capabilities capabilities) {
      // no-op
    }

    @Override
    public void quit() {
      // no-op
    }
  }
}
