/* XDRBG with SHAKE-512 support functions
 *
 * Copyright (C) 2024 - 2025, Stephan Mueller <smueller@chronox.de>
 *
 * License: see LICENSE file in root directory
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, ALL OF
 * WHICH ARE HEREBY DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF NOT ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */

#include "alignment.h"
#include "compare.h"
#include "fips_mode.h"
#include "lc_xdrbg.h"
#include "visibility.h"
#include "xdrbg_internal.h"

/********************************** Selftest **********************************/

static void xdrbg512_drng_selftest(void)
{
	LC_FIPS_RODATA_SECTION
	static const uint8_t seed[] = {
		FIPS140_MOD(0x00),
		0x01,
		0x02,
		0x03,
		0x04,
		0x05,
		0x06,
		0x07,
		0x08,
	};
	LC_FIPS_RODATA_SECTION
	static const uint8_t exp[] = {
		0x18, 0x95, 0x64, 0xdd, 0x84, 0x9e, 0x4d, 0x24,
		0x68, 0x9f, 0xec, 0x3c, 0xd5, 0x4c, 0x0c, 0x76,
		0xbf, 0x99, 0xc9, 0xda, 0x84, 0xbc, 0x69, 0xd1,
		0xaf, 0x5b, 0xba, 0xfb, 0x10, 0x4b, 0x0f, 0x73,
		0x56, 0x41, 0x7a, 0x16, 0x2f, 0x34, 0x09, 0xc9,
		0x46, 0x79, 0x93, 0xa6, 0x0e, 0xc9, 0x6e, 0xba,
		0x0d, 0x20, 0x3a, 0x8a, 0xfd, 0x5c, 0x03, 0x94,
		0x30, 0x2c, 0xf5, 0x1e, 0x67, 0x25, 0xed, 0x55,
		0xdf, 0xf3, 0xdc, 0x3e, 0xe8, 0xf9, 0x2c, 0xd7,
		0x1c, 0x95, 0x0c, 0xda, 0xc9, 0x45, 0xdf, 0xe5,
		0xb5, 0x30, 0x76, 0x74, 0x1e, 0x48, 0xf5, 0x72,
		0x5d, 0x12, 0xa9, 0xd2, 0xfb, 0xd1, 0xd9, 0x43,
		0xc9, 0xc0, 0x80, 0x8f, 0x9b, 0x8c, 0x05, 0x83,
		0x62, 0xb0, 0xa6, 0xd3, 0x3b, 0xb6, 0x4e, 0xb0,
		0x41, 0x54, 0x77, 0xee, 0x26, 0xe6, 0x23, 0x8f,
		0xf3, 0x5b, 0x5f, 0x06, 0x2f, 0x49, 0x3a, 0x18,
		0x9c, 0xf3, 0x63, 0x7b, 0x5b, 0xc0, 0xce, 0xa0,
		0xe7, 0xb5, 0xc9, 0x33, 0x61, 0x95, 0xad, 0xb3,
		0x71, 0xeb, 0x7a, 0xe1, 0xba, 0x80, 0x62, 0x6a,
		0x90, 0x04, 0x07, 0x4d, 0xc4, 0x27, 0x79, 0x3a,
		0x19, 0xed, 0x1c, 0xbf, 0x50, 0x0a, 0x73, 0x56,
		0x44, 0xfb, 0x84, 0xe2, 0x26, 0xc5, 0x09, 0xb3,
		0x2a, 0x64, 0xfc, 0x66, 0x17, 0x54, 0x40, 0x9b,
		0xfc, 0x53, 0xd3, 0xfd, 0xa1, 0x7a, 0x28, 0x22,
		0x6b, 0xbf, 0xdb, 0x1d, 0x0d, 0xa7, 0x00, 0xb6,
		0xdc, 0x00, 0x23, 0x89, 0xcf, 0x81, 0x06, 0x9f,
		0x3f, 0xf6, 0x9b, 0x05, 0xf3, 0x87, 0x65, 0x79,
		0xbf, 0x16, 0x58, 0x99, 0x70, 0xfc, 0xc1, 0xa0,
		0x72, 0x38, 0x95, 0xfa, 0x12, 0xaf, 0x4b, 0xcc,
		0x08, 0xe5, 0x75, 0xd4, 0x4f, 0x76, 0x82, 0xf0,
		0x5c, 0x67, 0x62, 0x93, 0x18, 0x24, 0xee, 0x40,
		0x3b, 0x24, 0x56, 0x6e, 0x47, 0xda, 0x49, 0xcf,
		0x2c, 0x99, 0x25, 0xc4, 0x29, 0xcc, 0x5e, 0x57,
		0x88, 0xca, 0x33, 0x32, 0x2f, 0xa8, 0xd1, 0x59,
		0xc7, 0x07, 0x70, 0xc5, 0x88, 0xfc, 0x6d, 0x5a,
		0xeb, 0xf2, 0xf1, 0x7b, 0xfd, 0x70, 0x4c, 0x99,
		0x5a, 0xb1, 0x5e, 0x6b, 0x03, 0x0c, 0x7e, 0xe1,
		0x69, 0x74, 0x88, 0xa3, 0xe1, 0x77, 0xad, 0x0a,
		0x59, 0xd8, 0x2c, 0x28, 0x03, 0x0a, 0x9b, 0xda,
		0x33, 0x16, 0xb2, 0x0b, 0x2c, 0x49, 0xed, 0x9b,
		0xe6, 0x80, 0x6a, 0x08, 0x87, 0xe2, 0x4b, 0x11,
		0xd8, 0x6a, 0xcd, 0x3d, 0xd0, 0x49, 0xb8, 0xc4,
		0xe7, 0x9c, 0x4e, 0x20, 0x3e, 0x56, 0xb6, 0x38,
		0xe2,
	};
	uint8_t act[sizeof(exp)] __align(sizeof(uint32_t));

	LC_SELFTEST_RUN(lc_xdrbg512_drng->algorithm_type);

	LC_XDRBG512_DRNG_CTX_ON_STACK(shake_ctx);

	if (lc_xdrbg_drng_seed_nocheck(shake_ctx->rng_state, seed, sizeof(seed),
				       NULL, 0))
		goto out;
	lc_rng_generate(shake_ctx, NULL, 0, act, sizeof(act));

out:
	lc_compare_selftest(lc_xdrbg512_drng->algorithm_type, act, exp,
			    sizeof(exp), "SHAKE-512 XDRBG");
	lc_rng_zero(shake_ctx);
}

static int lc_xdrbg512_drng_seed(void *_state, const uint8_t *seed,
				 size_t seedlen, const uint8_t *alpha,
				 size_t alphalen)
{
	xdrbg512_drng_selftest();
	LC_SELFTEST_COMPLETED(lc_xdrbg512_drng->algorithm_type);

	return lc_xdrbg_drng_seed_nocheck(_state, seed, seedlen, alpha,
					  alphalen);
}

LC_INTERFACE_FUNCTION(int, lc_xdrbg512_drng_alloc, struct lc_rng_ctx **state)
{
	struct lc_rng_ctx *out_state = NULL;
	int ret;

	if (!state)
		return -EINVAL;

	ret = lc_alloc_aligned_secure((void *)&out_state,
				      LC_HASH_COMMON_ALIGNMENT,
				      LC_XDRBG512_DRNG_CTX_SIZE);
	if (ret)
		return -ret;

	LC_XDRBG512_RNG_CTX(out_state);

	*state = out_state;

	return 0;
}

static const struct lc_rng _lc_xdrbg512_drng = {
	.generate = lc_xdrbg_drng_generate,
	.seed = lc_xdrbg512_drng_seed,
	.zero = lc_xdrbg_drng_zero,
	.algorithm_type = LC_ALG_STATUS_XDRBG512,
};
LC_INTERFACE_SYMBOL(const struct lc_rng *,
		    lc_xdrbg512_drng) = &_lc_xdrbg512_drng;
