// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief  e+ e- -> pi+ pi- J/psi
  class BESIII_2017_I1604892 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2017_I1604892);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projections
      declare(FinalState(), "FS");
      size_t ih = 0;
      for (double eVal : allowedEnergies()) {

        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;

        book(_h[en+"mass1"], 1, 1+ih, 1);
        book(_h[en+"mass2"], 1, 1+ih, 2);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
      book(_h["cTheta"], 2, 1, 1);
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      Particles fs = apply<FinalState>(event, "FS").particles();
      Particles Jpsi,other;
      for (const Particle& p : fs) {
        Particle parent=p;
        while (!parent.parents().empty()) {
          parent=parent.parents()[0];
          if (parent.abspid()==PID::JPSI) break;
        }
        if (parent.abspid()!=PID::JPSI) {
          other.push_back(p);
          continue;
        }
        bool found=false;
        for (const auto& psi : Jpsi) {
          // J/psi already in list
          if (fuzzyEquals(psi.mom(),parent.mom())) {
            found=true;
            break;
          }
        }
        if (!found) Jpsi.push_back(parent);
      }
      if (Jpsi.size()!=1 || other.size()!=2) vetoEvent;
      if (other[0].pid() != -other[1].pid() || other[0].abspid()!=PID::PIPLUS) vetoEvent;
      _h[_sqs+"mass1"]->fill((other[0].mom()+other[1].mom()).mass());
      for (size_t ix=0; ix<2; ++ix) {
        FourMomentum pZ = Jpsi[0].mom()+other[ix].mom();
        double mZ = pZ.mass()/GeV;
        _h[_sqs+"mass2"]->fill(mZ);
        if (mZ>3.86 && mZ<3.92) {
          LorentzTransform boost = LorentzTransform::mkFrameTransformFromBeta(pZ.betaVec());
          Vector3 axis1 = pZ.p3().unit();
          Vector3 axis2 = boost.transform(other[ix].mom()).p3().unit();
          _h["cTheta"]->fill(abs(axis1.dot(axis2)));
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      normalize(_h, 1.0, false);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2017_I1604892);

}
