/***************************************************************************
                          dchubsearch.cpp  -  description
                             -------------------
    begin                : Fri Mar 15 2002
    copyright            : (C) 2002 by Mathias Kster
    email                : mathen@ketelhot.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qapp.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qradiobutton.h>
#include <qlistview.h>
#include <qcheckbox.h>
#include <qlcdnumber.h>
#include <qcombobox.h>
#include <qregexp.h>
#include <qspinbox.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qstringlist.h>
#include <qtabwidget.h>
#include <qbuttongroup.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qfiledialog.h>
#include <qclipboard.h>

#ifndef WIN32
#include <stdlib.h>
#endif

#include <dcconfig.h>
#include <dcserverlist.h>
#include <dcfiletool.h>
#include <dctransferview.h>
#include <dcmenuhandler.h>
#include <dcwidget.h>

#include "dchubsearch.h"

DCHubSearch * pHubSearch;

/** */
DCHubSearch::DCHubSearch(QWidget* parent, const char *name, int wflags) : DCDialogHubSearch(parent, name, wflags)
{
	int idx;

	pWorkspace = (QWorkspace*)parent;

	// set the ColumnWidthMode to manual, we will take care of this task
	for( idx = 0; idx < ListView_SEARCHRESULT->columns(); idx++ )
	{
		ListView_SEARCHRESULT->setColumnWidthMode( idx, QListView::Manual );
	}

	ehSearchType = ehstNONE;

	pMessageList = new CList<CObject>();

	pHubSearchClientList = new CList<CObject>();

	pLogStringList = new QStringList();

	pHubServerList = 0;

	InitDocument();
}

/** */
DCHubSearch::~DCHubSearch()
{
	ehSearchType = ehstNONE;

	Disconnect();

	Timer.stop();

	// clear the client list
	HubSearchClientListThread.Lock();

	sHubSearchClient * HubSearchClient;

	if ( pHubSearchClientList != 0 )
	{
		while ( ( (HubSearchClient = (sHubSearchClient *)pHubSearchClientList->Next(0)) != NULL) )
		{
			HubSearchClient->Client->SetCallBackFunction(0);
			HubSearchClient->Client->Disconnect(TRUE);
			delete HubSearchClient->Client;
			HubSearchClient->Client = 0;
			pHubSearchClientList->Remove(HubSearchClient);
			delete HubSearchClient;
		}

		delete pHubSearchClientList;
		pHubSearchClientList = 0;
	}

	HubSearchClientListThread.UnLock();

	// clear the messagelist
	SocketCallbackThread.Lock();

	if ( pMessageList != 0 )
	{
		delete pMessageList;
		pMessageList = 0;
	}

	SocketCallbackThread.UnLock();

	// clear the logstring list
	LogThread.Lock();

	if ( pLogStringList != 0 )
	{
		delete pLogStringList;
		pLogStringList = 0;
	}

	LogThread.UnLock();

	// clear hubserverlist
	if ( pHubServerList != 0 )
	{
		delete pHubServerList;
		pHubServerList = 0;
	}
}

/** */
void DCHubSearch::InitDocument()
{
	connect( LineEdit_SEARCH, SIGNAL(returnPressed()), this, SLOT(slotSearchReturnPressed()) );
	connect( PushButton_SEARCH, SIGNAL(clicked()), this, SLOT(slotSearchReturnPressed()) );
	connect( PushButton_STOP, SIGNAL(clicked()), this, SLOT(slotClickedStop()) );
	connect( ListView_SEARCHRESULT,SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slotDoubleClickedSearchResult(QListViewItem*)) );
	connect( TabWidget_HUBSEARCH,SIGNAL(currentChanged(QWidget*)), this, SLOT(slotTabWidgetCurrentChange(QWidget*)) );
	connect( ListView_SEARCHRESULT,SIGNAL(rightButtonClicked( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedSearchResult(QListViewItem*, const QPoint &, int )) );

	connect( &Timer, SIGNAL(timeout()), this, SLOT(timerDone()) );
}

/** current tab widget change slot */
void DCHubSearch::slotTabWidgetCurrentChange(QWidget*)
{
	ResizeListViewColumn();
}

/** overridden so that the columns are resized on show() */
void DCHubSearch::show()
{
	QWidget::show();

	ResizeListViewColumn();
}

/** resize event handler */
void DCHubSearch::resizeEvent( QResizeEvent * )
{
	ResizeListViewColumn();
}

/** resize the ListView columns */
void DCHubSearch::ResizeListViewColumn()
{
	int width;

	if ( ListView_SEARCHRESULT->isVisible() )
	{
		width = ListView_SEARCHRESULT->width();

		ListView_SEARCHRESULT->setColumnWidth( 0, ((width*7)/28) );
		ListView_SEARCHRESULT->setColumnWidth( 1, ((width*3)/28) );
		ListView_SEARCHRESULT->setColumnWidth( 2, ((width*4)/28) );
		ListView_SEARCHRESULT->setColumnWidth( 3, ((width*1)/28) );
		ListView_SEARCHRESULT->setColumnWidth( 4, ((width*4)/28) );
		ListView_SEARCHRESULT->setColumnWidth( 5, ((width*6)/28) );
		ListView_SEARCHRESULT->setColumnWidth( 6, ((width*3)/28) );
	}
}

/** */
void DCHubSearch::SetSearchView( bool enabled )
{
	PushButton_SEARCH->setEnabled(enabled);
	LineEdit_SEARCH->setEnabled(enabled);
	ButtonGroup_HUBS->setEnabled(enabled);
	GroupBox_FILTER->setEnabled(enabled);
	SpinBox_MAXTHREADS->setEnabled(enabled);

	PushButton_STOP->setEnabled(!enabled);
}

/** */
int DCHubSearch::DC_CallBack( CObject * Object )
{
	SocketCallbackThread.Lock();

	int err = -1;

	if ( ehSearchType != ehstNONE )
	{
		if ( (Object != 0) && (pMessageList != 0) )
		{
			pMessageList->Add(Object);
			err = 0;
		}
        }

	SocketCallbackThread.UnLock();

	return err;
}

/** */
void DCHubSearch::timerDone()
{
	long l;

	if ( (ehSearchType == ehstGLOBAL) ||
	     (ehSearchType == ehstWAITTIMEOUT) ||
	     (ehSearchType == ehstSTOP) )
	{
		CheckClient();

		if ( ehSearchType == ehstGLOBAL )
		{
			NewClient();
		}
	}

	// search is running
	if ( ehSearchType != ehstNONE )
	{
		ShowLog();
		qApp->processEvents();
	}

	if ( ehSearchType != ehstNONE )
	{
		ShowResults(FALSE);
		qApp->processEvents();
	}

	// check all clients removed
	if ( ehSearchType == ehstSTOP )
	{
		HubSearchClientListThread.Lock();

		l = 0;

		if ( pHubSearchClientList != 0 )
		{
			l = pHubSearchClientList->Count();
		}

		HubSearchClientListThread.UnLock();

		if ( l == 0 )
		{
			// update view
			ehSearchType = ehstNONE;

			SetSearchView( TRUE );

			qApp->processEvents();
		}
	}

	if ( ehSearchType != ehstNONE )
	{
		// restart timer
		Timer.start( 500, TRUE );
	}
}

/** */
void DCHubSearch::ShowLog()
{
	QString s;
	unsigned int i;
	bool bscroll;

	for(;;)
	{
		// update log
		if ( LogThread.TryLock() == FALSE )
		{
			break;
		}

		i = 0;

		if (pLogStringList)
		{
			i = pLogStringList->count();
			if ( i > 0 )
			{
				s = *pLogStringList->at(0);
				pLogStringList->remove(pLogStringList->at(0));
			}
		}

		LogThread.UnLock();

		if ( i == 0 )
		{
			return;
		}

		if ( TextEdit_LOG->verticalScrollBar()->maxValue() == TextEdit_LOG->verticalScrollBar()->value() )
		{
			bscroll = TRUE;
		}
		else
		{
			bscroll = FALSE;
		}

		TextEdit_LOG->append(s);

		if ( bscroll )
		{
			TextEdit_LOG->scrollToBottom();
			TextEdit_LOG->moveCursor( QTextEdit::MoveEnd, FALSE );
		}
	}
}

/** */
void DCHubSearch::CheckClient()
{
	sHubSearchClient * HubSearchClient,* OldHubSearchClient;

	HubSearchClientListThread.Lock();

	if ( pHubSearchClientList != 0 )
	{
		HubSearchClient = OldHubSearchClient = 0;

		while ( ( (HubSearchClient = (sHubSearchClient *)pHubSearchClientList->Next(HubSearchClient)) != NULL) )
		{
			if ( (HubSearchClient->bRemove == TRUE) || ((time(0)-HubSearchClient->timeHubTimeout) >= 60) )
			{
				HubSearchClient->Client->SetCallBackFunction(0);

				if ( HubSearchClient->Client->GetConnectionState() != estNONE )
				{
					if ( HubSearchClient->Client->GetConnectionState() != estDISCONNECTING )
					{
						HubSearchClient->Client->Disconnect(TRUE);
					}
					OldHubSearchClient = HubSearchClient;
				}
				else
				{
					delete HubSearchClient->Client;
					HubSearchClient->Client = 0;
					pHubSearchClientList->Remove(HubSearchClient);
					delete HubSearchClient;
					HubSearchClient = OldHubSearchClient;
				}
			}
        	        else
			{
				// remove all clients
				if ( ehSearchType == ehstSTOP )
				{
					HubSearchClient->Client->SetCallBackFunction(0);
					HubSearchClient->bRemove = TRUE;
				}

				OldHubSearchClient = HubSearchClient;
			}
		}
	}

	HubSearchClientListThread.UnLock();
}

/** */
void DCHubSearch::ShowResults( bool bClearList )
{
	CDCMessage *DCMsg;
	CObject * Object;
	long i;
	bool b;

	if ( bClearList == TRUE )
	{
		ListView_SEARCHRESULT->clear();
	}

	if ( SocketCallbackThread.TryLock() == FALSE )
	{
		return;
	}

	i = 0;
	b = FALSE;

	if ( pMessageList != 0 )
	{
		for(i=0;i<pMessageList->Count();i++)
		{
	 		Object = pMessageList->Next(0);

			if( Object == 0 )
			{
				continue;
			}

			DCMsg = (CDCMessage*) Object;

			switch ( DCMsg->eType )
			{
				case DC_MESSAGE_SEARCHRESULT:
				{
					b = DC_SearchResult( (CMessageSearchResult *) Object );
					if( b == TRUE )
						ListView_SEARCHRESULT->setUpdatesEnabled(FALSE);
					break;
				}

				default:
				{
					break;
				}
			}

			pMessageList->Del(Object);
		}
	}

	SocketCallbackThread.UnLock();

	if ( b == TRUE )
	{
		ListView_SEARCHRESULT->setUpdatesEnabled(TRUE);
		ListView_SEARCHRESULT->triggerUpdate();
	}
}

/** */
void DCHubSearch::NewClient()
{
	int l;
	int i,port;
	QString server,addr,s;
	sHubSearchClient * HubSearchClient;

	if ( pHubServerList == 0 )
	{
		return;
	}

	// done ?
	if ( pHubServerList->count() == iCurrentHubServer )
	{
		AddLog(tr("Serverlist done\n"));

		ehSearchType = ehstWAITTIMEOUT;

		return;
	}

	HubSearchClientListThread.Lock();

	l = pHubSearchClientList->Count();

	HubSearchClientListThread.UnLock();

	for(;(l<iMaxThreads)&&(iCurrentHubServer<pHubServerList->count());l++)
	{
		CClient * Client = new CClient();

		HubSearchClient = new sHubSearchClient;

		HubSearchClient->Client = Client;
		HubSearchClient->bRemove = FALSE;
		HubSearchClient->timeHubTimeout = time(0);

		Client->SetNick(Config->GetNick());
		Client->SetComment(Config->GetDescription());
		Client->SetConnectionType(Config->GetSpeed());
		Client->SetEMail(Config->GetEMail());
		Client->SetVersion(VERSION);
		Client->SetShareSize(CString().setNum(Config->GetShareSize()));
		Client->SetMode(Config->GetMode());
		Client->SetHandleUserList(FALSE);
		Client->SetCallBackFunction(DC_ClientCallBack);

		server = *pHubServerList->at(iCurrentHubServer);
		iCurrentHubServer++;

		AddLog( tr("Search on: [") + QString().setNum(iCurrentHubServer) + "] " + server + "\n" );

		HubSearchClientListThread.Lock();

		pHubSearchClientList->Add((CObject*)HubSearchClient);

		if ( (i=server.find(':')) != -1 )
		{
			addr = server.left(i);
			s = server.mid(i+1,server.length()-i-1);
			if ( s != "" )
			{
				port = atoi(s);
				Client->Connect(addr.ascii(),port);
			}
			else
			{
				Client->Connect(addr.ascii());
			}
		}
		else
		{
			Client->Connect(server.ascii());
		}

		HubSearchClientListThread.UnLock();
	}
}

/** search result */
bool DCHubSearch::DC_SearchResult( CMessageSearchResult * MessageSearchResult )
{
	bool res = FALSE;
	int i;
	QString s,spath,sname;
	DCFileTool FileTool;
	ulonglong size;

	if ( SpinBox_MAXRESULT->value() > 0 )
	{
		if ( ListView_SEARCHRESULT->childCount() >= SpinBox_MAXRESULT->value() )
		{
			return res;
		}
	}

	if ( MessageSearchResult->iFreeSlot < SpinBox_FREESLOTS->value() )
	{
		return res;
	}

	if ( ComboBox_SEARCHLIMIT->currentItem() == 2 )
	{
		size = SpinBox_SEARCHSIZE->value();

		if ( ComboBox_SEARCHUNIT->currentItem() == 1 )
		{
			size *= 1024;
		}
		else if ( ComboBox_SEARCHUNIT->currentItem() == 2 )
		{
			size *= 1024*1024;
		}
		else if ( ComboBox_SEARCHUNIT->currentItem() == 3 )
		{
			size *= 1024*1024*1024;
		}

		if ( MessageSearchResult->lSize != size )
		{
			return res;
		}
	}

	// split filename and path
	s = MessageSearchResult->sFile.Data();

	if ( (i = s.findRev('\\')) == -1 )
	{
		spath = "";
		sname = s;
	}
	else
	{
		spath = s.left(i+1);
		sname = s.mid(i+1,s.length()-i-1);
	}

//	if ( 0 )
//	{
/*
		ListView_SEARCHRESULT->setRootIsDecorated(TRUE);

		item = ListView_SEARCHRESULT->findItem( sname, 0 );

		if ( item == 0 )
		{
			item = new QListViewItem(ListView_SEARCHRESULT, \
				sname, \
				MessageSearchResult->sSize.Data() );
		}
		else if ( CString(item->text(1).ascii()).asULL() != MessageSearchResult->lSize )
		{
			item = new QListViewItem(ListView_SEARCHRESULT, \
				sname, \
				MessageSearchResult->sSize.Data() );
		}

		new QListViewItem(item, \
			sname, \
			MessageSearchResult->sSize.Data(), \
			MessageSearchResult->sNick.Data(), \
			MessageSearchResult->sSlot.Data(), \
			MessageSearchResult->sHubName.Data(), \
			spath,
			MessageSearchResult->sHubHost.Data() );

		item->setText(2,QString().setNum(item->childCount()));
*/
//	}
//	else
	{
//		ListView_SEARCHRESULT->setRootIsDecorated(FALSE);

		s = FileTool.GetSizeString(MessageSearchResult->lSize).Data();
		DC_QListViewItem *item = new DC_QListViewItem(ListView_SEARCHRESULT);
		item->myvalue = MessageSearchResult->lSize;
		item->mycol = 1;

		item->setText(0,sname);
		item->setText(1,s);
		item->setText(2,MessageSearchResult->sNick.Data());
		item->setText(3,QString().setNum(MessageSearchResult->iFreeSlot)+'/'+QString().setNum(MessageSearchResult->iMaxSlot));
		item->setText(4,MessageSearchResult->sHubName.Data());
		item->setText(5,spath);
		item->setText(6,MessageSearchResult->sHubHost.Data());

		res = TRUE;
	}

	LCDNumber_RESULTS->display(ListView_SEARCHRESULT->childCount());

	return res;
}

/** */
QString DCHubSearch::GetSearchString()
{
	ulonglong size;
	QString s,l;

	l = LineEdit_SEARCH->text();

	if ( l == "" )
	{
		return "";
	}

	// replace spaces with $
	l = l.replace( QRegExp(" "), "$" );

	s = "$Search ";

	QString h = Config->GetUDPHostString().Data();

	if ( h == "" )
	{
		return "";
	}

	s += h;
	s += " ";

	// for exact size searches set size to 0
	if ( ComboBox_SEARCHLIMIT->currentItem() == 2 )
	{
		size = 0;
	}
	else
	{
		size = SpinBox_SEARCHSIZE->value();
	}

	if ( size == 0 )
	{
		s += "F?";
	}
	else
	{
		s += "T?";
	}

	if ( ComboBox_SEARCHUNIT->currentItem() == 1 )
	{
		size *= 1024;
	}
	else if ( ComboBox_SEARCHUNIT->currentItem() == 2 )
	{
		size *= 1024*1024;
	}
	else if ( ComboBox_SEARCHUNIT->currentItem() == 3 )
	{
		size *= 1024*1024*1024;
	}

	// F is default
	if ( (ComboBox_SEARCHLIMIT->currentItem() == 0) || (ComboBox_SEARCHLIMIT->currentItem() == 2) || (size==0) )
	{
		s += "F?";
	}
	else
	{
		s += "T?";
	}

	s += CString().setNum(size).Data() + QString("?");

	s += QString().setNum(ComboBox_SEARCHTYPE->currentItem()+1) + "?";

	s += l + "|";

	return s;
}

/** */
void DCHubSearch::slotClickedStop()
{
	ehSearchType = ehstSTOP;

	PushButton_STOP->setEnabled(FALSE);

	Disconnect();
}

/** */
void DCHubSearch::slotSearchReturnPressed()
{
	long i;
	sHubSearchClient * HubSearchClient;

	ehSearchType = ehstNONE;

	// disconnect udp socket
	Disconnect();

	// clear client list
	HubSearchClientListThread.Lock();

	while ( ( (HubSearchClient = (sHubSearchClient *)pHubSearchClientList->Next(0)) != NULL) )
	{
		pHubSearchClientList->Remove(HubSearchClient);

		HubSearchClient->Client->SetCallBackFunction(0);
		HubSearchClient->Client->Disconnect();
		delete HubSearchClient->Client;
		HubSearchClient->Client = 0;
		delete HubSearchClient;
	}

	HubSearchClientListThread.UnLock();

	// clear messagelist
	SocketCallbackThread.Lock();

	for(i=0;i<pMessageList->Count();i++)
	{
 		CObject *Object = pMessageList->Next(0);
		pMessageList->Del(Object);
	}

	SocketCallbackThread.UnLock();

	// clear searchresults & log
	ListView_SEARCHRESULT->clear();
	TextEdit_LOG->clear();

	// reset counter
	LCDNumber_RESULTS->display(0);

	if ( Config->GetMode() == ecmPassive )
	{
		QMessageBox::critical( this, tr("Hub Search Error"),
			QString(tr("Hubsearch is only available in active mode.")));
		return;
	}

	if ( LineEdit_SEARCH->text() == "" )
	{
		QMessageBox::critical( this, tr("Hub Search Error"),
			QString(tr("You must enter a search word.")));
		return;
	}

	if ( Connect( "", Config->GetUDPListenPort(), estUDP ) != 0 )
	{
		QMessageBox::critical( this, tr("Hub Search Error"),
			QString(tr("Connect to local udp port failed. (") + GetSocketError().Data() + ")"));
		return;
	}

	// reset values
	iCurrentHubServer = 0;

	if ( pHubServerList != 0 )
	{
		delete pHubServerList;
		pHubServerList = 0;
	}

	sSearchString = GetSearchString();

	iMaxThreads = SpinBox_MAXTHREADS->value();

	if ( RadioButton_CONNECTEDHUBS->isChecked() == TRUE )
	{
		ehSearchType = ehstLOCAL;

		if ( pServerList->SendStringToConnectedServers(sSearchString.ascii()) == 0 )
		{
			ehSearchType = ehstNONE;

			QMessageBox::critical( this, tr("Hub Search Error"),
				QString(tr("No connected hubs found.")));

			return;
		}

		ehSearchType = ehstWAITTIMEOUT;
	}
	else
	{
		if ( RadioButton_AVAILABLEHUBS->isChecked() == TRUE )
		{
			// get public hub server list
			pHubServerList = pServerList->GetPublicHubServerList();
		}
		else
		{
			// get bookmark hub server list
			pHubServerList = pServerList->GetBookmarkHubServerList();
		}

		if ( pHubServerList->count() <= 0 )
		{
			QMessageBox::critical( this, tr("Hub Search Error"),
				QString(tr("No hubs found.")));
			return;
		}

		ehSearchType = ehstGLOBAL;

		NewClient();
	}

	// update view
	SetSearchView( FALSE );

	Timer.start( 500, TRUE );

	return;
}

/** */
void DCHubSearch::slotDoubleClickedSearchResult( QListViewItem * )
{
}

/** */
void DCHubSearch::slotRightButtonClickedSearchResult( QListViewItem * item, const QPoint &, int column )
{
	DCFileTool filetool;
	CString s;
	int id;
	ulonglong pos;
	ulonglong size;

	if ( item == 0 )
	{
		return;
	}

	size = ((DC_QListViewItem *)item)->myvalue;

	QPopupMenu *m;

	m = new QPopupMenu(this);

	DCMenuHandler::InsertMenuItem( m, emiDOWNLOAD );
	DCMenuHandler::InsertMenuItem( m, emiDOWNLOAD_TO, TRUE );
	DCMenuHandler::InsertMenuItem( m, emiDOWNLOAD_AS, TRUE );
	DCMenuHandler::InsertMenuItem( m, emiSEPARATOR );
	DCMenuHandler::InsertMenuItem( m, emiBROWSE_USER_FILES );
	DCMenuHandler::InsertMenuItem( m, emiSEPARATOR );
	DCMenuHandler::InsertMenuItem( m, emiCOPY_COLUMN_TO_CLIPBOARD );
	DCMenuHandler::InsertMenuItem( m, emiCOPY_ROW_TO_CLIPBOARD );
	DCMenuHandler::InsertMenuItem( m, emiSEPARATOR );
	DCMenuHandler::InsertMenuItem( m, emiCONNECT_TO_HUB );

	id = m->exec(QCursor::pos());

	delete m;

	if ( id == -1 )
	{
		return;
	}

	if ( (id == emiDOWNLOAD) || (id == emiDOWNLOAD_AS) || (id == emiDOWNLOAD_TO) )
	{
		QString localrootpath = "";
		QString localname = "";

		localname = item->text(0);

		// select downloadfolder
		if ( id == emiDOWNLOAD_TO )
		{
			localrootpath = QFileDialog::getExistingDirectory( "", this, "bdf", tr("Select download folder"), TRUE );

			if ( localrootpath == "" )
				return;
		}
		else if ( id == emiDOWNLOAD_AS )
		{
			localrootpath = QFileDialog::getSaveFileName( "", "", this, "bdf", tr("Select file") );

			if ( localrootpath == "" )
				return;

			QFileInfo fi(localrootpath);
			localrootpath = fi.dirPath();
			localname     = fi.fileName();

			if ( (localrootpath=="") || (localname=="") )
				return;
		}

		if ( filetool.CheckFile( localname, size, pos, localrootpath ) == FALSE )
		{
			return;
		}

		// add transfer to the waitlist
		pTransferView->AddWaitTransfer( item->text(2).ascii(), item->text(4).ascii(), item->text(6).ascii(),
						(item->text(5)+item->text(0)).ascii(), localname.ascii(), "", localrootpath.ascii(), eltFILE,
						size, pos, size );

	}
	else if ( id == emiBROWSE_USER_FILES)
	{
		/** add transfer to the waitlist */
		pTransferView->AddWaitTransfer( item->text(2).ascii(), item->text(4).ascii(), item->text(6).ascii(),
						"MyList.DcLst", "MyList.DcLst", "", "", eltBUFFER,
						0, 0, 0 );
	}
	else if ( id == emiCOPY_COLUMN_TO_CLIPBOARD )
	{
		QListViewItem *item1;
		QClipboard *cb = QApplication::clipboard();
		cb->setText("");

		if ( (item1=ListView_SEARCHRESULT->currentItem()) != 0 )
		{
			cb->setText( item->text(column) );
		}
	}
	else if ( id == emiCOPY_ROW_TO_CLIPBOARD )
	{
		int idx;
		QString s="";
		QListViewItem *item1;
		QClipboard *cb = QApplication::clipboard();
		cb->setText("");

		if ( (item1=ListView_SEARCHRESULT->currentItem()) != 0 )
		{
			for( idx = 0; idx < ListView_SEARCHRESULT->columns(); idx++ )
			{
				s += item1->text(idx) + "\n";
			}
			cb->setText(s);
		}
	}
	else if ( id == emiCONNECT_TO_HUB )
	{
		// connect to the hub
		pServerList->Connect( item->text(4), item->text(6) );
	}
}

/** */
void DCHubSearch::AddLog( QString line )
{
	LogThread.Lock();

	if ( (line != "") && (pLogStringList != 0) )
	{
		pLogStringList->append(line);
	}

	LogThread.UnLock();
}

/** */
void DCHubSearch::RemoveClient( CClient * Client )
{
	HubSearchClientListThread.Lock();

	sHubSearchClient * HubSearchClient;

	if ( pHubSearchClientList != 0 )
	{
		HubSearchClient = 0;

		while ( (HubSearchClient = (sHubSearchClient *)pHubSearchClientList->Next((CObject*)HubSearchClient)) != NULL )
		{
			if ( HubSearchClient->Client == Client )
			{
				HubSearchClient->Client->SetCallBackFunction(0);
				HubSearchClient->bRemove = TRUE;
				break;
			}
		}
	}

	HubSearchClientListThread.UnLock();
}

/** */
int DCHubSearch::DC_ClientCallBack( CClient * Client, CObject * Object )
{
	pHubSearch->ClientThread.Lock();

	CClient * c = Client;
	CObject * o = Object;

	if ( (c == 0) || (o == 0) )
	{
		pHubSearch->ClientThread.UnLock();
		return -1;
	}

	CDCMessage *DCMsg;

	DCMsg = (CDCMessage*) o;

	switch ( DCMsg->eType )
	{
		case DC_MESSAGE_CONNECTION_STATE:
		{
			CMessageConnectionState *msg = (CMessageConnectionState*) o;

			switch(msg->eState)
			{
				case estCONNECTED:
				{
					break;
				}

				case estSOCKETERROR:
				{
					pHubSearch->AddLog( QString(tr("SocketError on ")) + c->GetIP().Data() + QString(":") + QString().setNum(c->GetPort()) + "\n" );
					break;
				}

				case estDISCONNECTED:
				{
					pHubSearch->RemoveClient(c);
				}

				default:
				{
					break;
				}
			}

			break;
		}

		case DC_MESSAGE_VALIDATEDENIDE:
		{
			pHubSearch->AddLog( QString(tr("Validate denide ")) + c->GetIP().Data() + QString(":") + QString().setNum(c->GetPort()) + "\n" );
			Client->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_HUBISFULL:
		{
			pHubSearch->AddLog( QString(tr("Hub is full on ")) + c->GetIP().Data() + QString(":") + QString().setNum(c->GetPort()) + "\n" );
			Client->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_FORCEMOVE:
		{
			pHubSearch->AddLog( QString(tr("Force move on ")) + c->GetIP().Data() + QString(":") + QString().setNum(c->GetPort()) + "\n" );
			Client->Disconnect(TRUE);
			break;
		}

		case DC_MESSAGE_LOCK:
		case DC_MESSAGE_HUBNAME:
		case DC_MESSAGE_CHAT:
		case DC_MESSAGE_MYNICK:
		case DC_MESSAGE_PRIVATECHAT:
		case DC_MESSAGE_SEARCH:
		case DC_MESSAGE_HELLO:
		case DC_MESSAGE_QUIT:
		{
			break;
		}

		case DC_MESSAGE_MYINFO:
		{
			QString s;
			CMessageMyInfo * MessageMyInfo = (CMessageMyInfo *)o;

			if ( MessageMyInfo->sNick == Config->GetNick() )
			{
				s = pHubSearch->sSearchString;
				c->SendString( s.ascii() );
				pHubSearch->RemoveClient(c);
			}
			break;
		}

		default:
		{
			printf("callback: %d\n",DCMsg->eType);//fflush(stdout);
			break;
		}
	}

	delete o;

	pHubSearch->ClientThread.UnLock();

	return 0;
}
