"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLegendTooltipVisibleText = exports.getLegendTooltipVisibleData = exports.getLegendTooltipSize = exports.getLegendTooltipDataProps = exports.getCursorTooltipPoniterOrientation = exports.getCursorTooltipCenterOffset = void 0;
const tslib_1 = require("tslib");
/* eslint-disable camelcase */
const chart_color_black_500_1 = tslib_1.__importDefault(require("@patternfly/react-tokens/dist/js/chart_color_black_500"));
const victory_core_1 = require("victory-core");
const ChartTheme_1 = require("../ChartTheme");
const chart_legend_1 = require("./chart-legend");
/**
 * When using a cursor container, the tooltip can be offset from the cursor point. If offsetCursorDimensionX is true,
 * the tooltip will appear to the right the vertical cursor. If offsetCursorDimensionY is true, the tooltip will appear
 * above the vertical cursor.
 */
exports.getCursorTooltipCenterOffset = ({ offsetCursorDimensionX = false, offsetCursorDimensionY = false, theme }) => {
    const pointerLength = theme && theme.tooltip ? victory_core_1.Helpers.evaluateProp(theme.tooltip.pointerLength) : 10;
    const offsetX = ({ center, flyoutWidth, width }) => {
        const offset = flyoutWidth / 2 + pointerLength;
        return width > center.x + flyoutWidth + pointerLength ? offset : -offset;
    };
    const offsetY = ({ center, flyoutHeight, width }) => {
        const offset = flyoutHeight / 2 + pointerLength;
        return width > center.y + flyoutHeight + pointerLength ? -offset : offset;
    };
    return {
        x: offsetCursorDimensionX ? offsetX : 0,
        y: offsetCursorDimensionY ? offsetY : 0
    };
};
/**
 * When using a cursor container, the tooltip pointer orientation can be adjusted as the cursor approaches the edge of
 * the chart. If horizontal is true, the tooltip pointer will either be 'left' or 'right'. If horizontal is true, the
 * tooltip pointer will either be 'top' or 'bottom'.
 */
exports.getCursorTooltipPoniterOrientation = ({ horizontal = true, theme }) => {
    const pointerLength = theme && theme.tooltip ? victory_core_1.Helpers.evaluateProp(theme.tooltip.pointerLength) : 10;
    const orientationX = ({ center, flyoutWidth, width }) => width > center.x + flyoutWidth + pointerLength ? 'left' : 'right';
    const orientationY = ({ center, flyoutHeight, height }) => height > center.y + flyoutHeight + pointerLength ? 'top' : 'bottom';
    return horizontal ? orientationX : orientationY;
};
// Returns props associated with legend data
exports.getLegendTooltipDataProps = (defaultProps) => (Object.assign({ borderPadding: 0, gutter: 0, orientation: 'vertical', padding: 0, rowGutter: 0, style: {
        labels: {
            fill: ChartTheme_1.ChartLegendTooltipStyles.label.fill,
            lineHeight: 0.275,
            padding: 0
        },
        title: {
            fill: ChartTheme_1.ChartLegendTooltipStyles.label.fill,
            padding: 0
        }
    } }, defaultProps));
// Returns the legend height and width
exports.getLegendTooltipSize = ({ legendData, legendOrientation = 'vertical', legendProps, text = '', theme }) => {
    const textEvaluated = victory_core_1.Helpers.evaluateProp(text);
    const _text = Array.isArray(textEvaluated) ? textEvaluated : [textEvaluated];
    // Find max char lengths
    let maxDataLength = 0;
    let maxTextLength = 0;
    _text.map((name, index) => {
        if (name) {
            if (name.length > maxTextLength) {
                maxTextLength = name.length;
            }
            const hasData = legendData && legendData[index] && legendData[index].name;
            if (hasData) {
                if (legendData[index].name.length > maxDataLength) {
                    maxDataLength = legendData[index].name.length;
                }
            }
        }
    });
    // Set length to ensure minimum spacing between label and value
    let maxLength = maxDataLength + maxTextLength;
    if (maxDataLength < 20) {
        maxLength += 2;
    }
    // Get spacing to help align legend labels and text values
    const spacer = 'x';
    const getSpacing = (legendLabel, textLabel) => {
        let spacing = '';
        if (maxLength === 0) {
            return spacing;
        }
        const legendLabelChars = legendLabel ? legendLabel.length : 0;
        const textLabelChars = textLabel ? textLabel.length : 0;
        const maxChars = legendLabelChars + textLabelChars;
        // Add spacer
        while (spacing.length < maxLength - maxChars) {
            spacing += spacer;
        }
        return spacing;
    };
    // Format all text (similar to below) to help determine overall width.
    //
    // {name: "Cats   no data"}
    // {name: "Dogs         1"}
    // {name: "Birds        4"}
    // {name: "Mice         3"}
    const data = _text.map((label, index) => {
        const hasData = legendData && legendData[index] && legendData[index].name;
        const spacing = hasData ? getSpacing(legendData[index].name, label) : '';
        return {
            name: `${hasData ? legendData[index].name : ''}${spacing}${label}`
        };
    });
    // Replace whitespace with spacer char for consistency in width
    const formattedData = data.map(val => ({
        name: val.name.replace(/ /g, spacer)
    }));
    // This should include both legend data and text
    const widthDimensions = chart_legend_1.getLegendDimensions({
        legendData: formattedData,
        legendOrientation,
        legendProps,
        theme
    });
    // This should only use text. The row gutter changes when displaying all "no data" messages
    const heightDimensions = chart_legend_1.getLegendDimensions({
        legendData: _text.map((name) => ({ name })),
        legendOrientation,
        legendProps,
        theme
    });
    return {
        height: heightDimensions.height,
        width: widthDimensions.width > 0 ? widthDimensions.width : 0
    };
};
// Returns visible legend data, while syncing color scale. If textAsLegendData is true, the text prop is used as
// legend data so y values can be passed individually to the label component
exports.getLegendTooltipVisibleData = ({ activePoints, colorScale, legendData, text, textAsLegendData = false, theme }) => {
    const textEvaluated = victory_core_1.Helpers.evaluateProp(text);
    const _text = Array.isArray(textEvaluated) ? textEvaluated : [textEvaluated];
    const result = [];
    // Sync data with interactive legends
    if (legendData) {
        let index = -1;
        for (let i = 0; i < legendData.length; i++) {
            const data = legendData[i];
            const activePoint = activePoints ? activePoints.find(item => item.childName === data.childName) : '';
            if (!activePoint ||
                (data.symbol && data.symbol.type === 'eyeSlash' && data.symbol.fill === chart_color_black_500_1.default.value)) {
                continue; // Skip hidden data
            }
            if (index++ < _text.length - 1) {
                const themeColor = theme && theme.legend && theme.legend.colorScale
                    ? theme.legend.colorScale[i % theme.legend.colorScale.length]
                    : undefined;
                result.push({
                    name: textAsLegendData ? _text[index] : data.name,
                    symbol: Object.assign({ fill: colorScale ? colorScale[i % colorScale.length] : themeColor }, data.symbol)
                });
            }
        }
    }
    return result;
};
// Returns visible text for interactive legends
exports.getLegendTooltipVisibleText = ({ activePoints, legendData, text }) => {
    const textEvaluated = victory_core_1.Helpers.evaluateProp(text);
    const _text = Array.isArray(textEvaluated) ? textEvaluated : [textEvaluated];
    const result = [];
    if (legendData) {
        let index = -1;
        for (const data of legendData) {
            const activePoint = activePoints ? activePoints.find(item => item.childName === data.childName) : '';
            if (!activePoint ||
                (data.symbol && data.symbol.type === 'eyeSlash' && data.symbol.fill === chart_color_black_500_1.default.value)) {
                continue; // Skip hidden data
            }
            if (index++ < _text.length - 1) {
                result.push(_text[index]);
            }
        }
    }
    return result;
};
//# sourceMappingURL=chart-tooltip.js.map