package main

import (
	"bufio"
	"fmt"
	"io"
	"net/http"
	"net/http/httptest"
	"os"
	"sort"
	"strings"

	domainstats "kubevirt.io/kubevirt/pkg/monitoring/domainstats/prometheus" // import for prometheus metrics
	_ "kubevirt.io/kubevirt/pkg/virt-controller/watch"
)

// constant parts of the file
const (
	genFileComment = `<!--
	This is an auto-generated file.
	PLEASE DO NOT EDIT THIS FILE.
	See "Developing new metrics" below how to generate this file
-->`
	title      = "# KubeVirt metrics\n"
	background = "This document aims to help users that are not familiar with all metrics exposed by different KubeVirt components.\n" +
		"All metrics documented here are auto-generated by the utility tool `tools/doc-generator` and reflects exactly what is being exposed.\n\n"

	KVSpecificMetrics = "## KubeVirt Metrics List\n" +
		"### kubevirt_info\n" +
		"Version information.\n\n"

	opening = genFileComment + "\n\n" +
		title +
		background +
		KVSpecificMetrics

	// footer
	footerHeading = "## Developing new metrics\n"
	footerContent = "After developing new metrics or changing old ones, please run `make generate` to regenerate this document.\n\n" +
		"If you feel that the new metric doesn't follow these rules, please change `doc-generator` with your needs.\n"

	footer = footerHeading + footerContent
)

// These are the recording rules, and so they are not included in the response to /metric; we'll need to add them
// manually
const (
	vmiPhaseCountName = "kubevirt_vmi_phase_count"
	vmiPhaseCountDesc = "Sum of VMIs per phase and node.\n\n`phase` can be one of the following: [`Pending`, `Scheduling`, `Scheduled`, `Running`, `Succeeded`, `Failed`, `Unknown`]"

	vmiEvictionBlockerName = "kubevirt_vmi_non_evictable"
	vmiEvictionBlockerDesc = "Indication for a VirtualMachine that its eviction strategy is set to Live Migration but is not migratable."
)

func main() {
	handler := domainstats.Handler(1)
	domainstats.RegisterFakeCollector()

	req, err := http.NewRequest(http.MethodGet, "/metrics", nil)
	checkError(err)

	recorder := httptest.NewRecorder()

	handler.ServeHTTP(recorder, req)

	if status := recorder.Code; status == http.StatusOK {
		err := parseVirtMetrics(recorder.Body, &metrics)
		checkError(err)

	} else {
		panic(fmt.Errorf("got HTTP status code of %d from /metrics", recorder.Code))
	}
	writeToFile(metrics)
}

func writeToFile(metrics metricList) {
	newFile, err := os.Create("newmetrics.md")
	checkError(err)
	defer newFile.Close()

	fmt.Fprint(newFile, opening)
	metrics.writeToFile(newFile)

	fmt.Fprint(newFile, footer)

}

type metric struct {
	name        string
	description string
}

func (m metric) writeToFile(newFile io.WriteCloser) {
	fmt.Fprintln(newFile, "###", m.name)
	fmt.Fprintln(newFile, m.description)
	fmt.Fprintln(newFile)
}

type metricList []metric

// Len implements sort.Interface.Len
func (m metricList) Len() int {
	return len(m)
}

// Less implements sort.Interface.Less
func (m metricList) Less(i, j int) bool {
	return m[i].name < m[j].name
}

// Swap implements sort.Interface.Swap
func (m metricList) Swap(i, j int) {
	m[i], m[j] = m[j], m[i]
}

func (m *metricList) add(line string) {
	split := strings.Split(line, " ")
	name := split[2]
	split[3] = strings.Title(split[3])
	description := strings.Join(split[3:], " ")
	*m = append(*m, metric{name: name, description: description})
}

func (m metricList) writeToFile(newFile io.WriteCloser) {
	for _, met := range m {
		met.writeToFile(newFile)
	}
}

var (
	metrics = metricList{
		{
			name:        vmiPhaseCountName,
			description: vmiPhaseCountDesc,
		},
		{
			name:        vmiEvictionBlockerName,
			description: vmiEvictionBlockerDesc,
		},
	}
)

const filter = "kubevirt_"

func parseVirtMetrics(r io.Reader, metrics *metricList) error {
	scan := bufio.NewScanner(r)
	for scan.Scan() {
		helpLine := scan.Text()
		if strings.HasPrefix(helpLine, "# HELP ") {
			if strings.Contains(helpLine, filter) {
				metrics.add(helpLine)
			}
		}
	}

	if scan.Err() != nil {
		return fmt.Errorf("failed to parse metrics from prometheus endpoint, %w", scan.Err())
	}

	sort.Sort(metrics)

	return nil
}

func checkError(err error) {
	if err != nil {
		panic(err)
	}
}
