// Copyright ©2015 The gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package testlapack

import (
	"fmt"
	"testing"

	"github.com/gonum/floats"
)

type dlasq4teststruct struct {
	z                                     []float64
	i0, n0, pp, n0in                      int
	dmin, dmin1, dmin2, dn, dn1, dn2, tau float64
	ttype                                 int
	g                                     float64

	zOut     []float64
	tauOut   float64
	ttypeOut int
	gOut     float64
}

func printDlasq4FortranInput(d dlasq4teststruct) {
	z := d.z
	printFortranArray(z, "z")
	fmt.Println("i0 =", d.i0)
	fmt.Println("n0 =", d.n0)
	fmt.Println("pp =", d.pp)
	fmt.Println("n0in =", d.n0in)

	fmt.Println("dmin =", fortran64(d.dmin))
	fmt.Println("dmin1 =", fortran64(d.dmin1))
	fmt.Println("dmin2 =", fortran64(d.dmin2))
	fmt.Println("dn =", fortran64(d.dn))
	fmt.Println("dn1 =", fortran64(d.dn1))
	fmt.Println("dn2 =", fortran64(d.dn2))
	fmt.Println("g =", fortran64(d.g))
	fmt.Println("tau =", fortran64(d.tau))
	fmt.Println("ttype =", d.ttype)
}

type Dlasq4er interface {
	Dlasq4(i0, n0 int, z []float64, pp int, n0in int, dmin, dmin1, dmin2, dn, dn1, dn2, tau float64, ttype int, g float64) (tauOut float64, ttypeOut int, gOut float64)
}

func Dlasq4Test(t *testing.T, impl Dlasq4er) {
	dTol := 1e-14
	// Tests computed from calling the netlib Dlasq
	for cas, test := range []dlasq4teststruct{
		{
			z:        []float64{0.1914365246180821E+01, 0.1564384297703890E+01, 0.2493389162143899E+00, 0.3499809484769305E+00, 0.1315996513131545E+01, 0.1363862112490627E+01, 0.9898466611970759E-01, 0.2014733168553078E+00, 0.6023973979587287E+00, 0.6465544792741794E+00, 0.2210033410638781E-02, 0.5482758480425683E-01, 0.9861857233678967E-01, 0.2428190810745492E-01, 0.4756321484454819E+00, 0.7654669763997353E-01, 0.2588748143677115E+00, 0.6127784069508770E+00, 0.1078611376690004E+00, 0.1217285558623164E+00, 0.6442896492255246E+00, 0.2293835804898155E+00, 0.6203230486639705E+00, 0.5227672064047094E+00, 0.3695660678607585E+00, 0.7645233184745865E+00, 0.5378838054252265E+00, 0.2253657980501426E+00, 0.3562533181264623E+00, 0.8820486722335483E+00, 0.2222132496436145E-01, 0.1208845131814035E-01, 0.1275094303021685E+01, 0.6548746852163357E+00, 0.1647324354821218E+00, 0.6424409427697111E+00, 0.1007530576543866E+01, 0.3269551736546701E+00, 0.3453881601783118E+00, 0.8453078383713172E+00, 0.2679391719153404E+00, 0.4116714838778281E+00, 0.7328677736683723E+00, 0.2016558482158241E+00, 0.8360828138307410E+00, 0.9737579452195326E+00, 0.4813660709592822E+00, 0.5951926422795808E+00, 0.6495370513676459E+00, 0.6761876248148171E+00, 0.2325475880222648E+00, 0.4547154975121112E+00, 0.1993624802893807E+00, 0.3321819367342255E+00, 0.3782318916911257E+00, 0.9972813157741996E-01, 0.9830449403503746E+00, 0.7561080996844842E+00, 0.4429733864040367E+00, 0.6051687323570161E+00, 0.1173279550602403E+01, 0.7195724480316686E+00, 0.5035524069144587E+00, 0.8966804889747714E+00, 0.3058980395058521E+00, 0.6588832353928662E+00, 0.3014634433415453E+00, 0.1505672110274446E+00, 0.1289422237567578E+01, 0.6124645310993601E+00, 0.7583364305799440E+00, 0.9784211498097629E+00, 0.4977814779461571E+00, 0.9993813577491869E+00, 0.2841468847862598E+00, 0.2567365507769143E+00, 0.9257539794205765E+00, 0.5509268385614666E+00, 0.5231355605450990E-04, 0.6589740256453697E+00, 0.2117869221381033E-04, 0.7349224826832024E-04, 0.0000000000000000E+00, 0.0000000000000000E+00},
			i0:       1,
			n0:       21,
			pp:       0,
			n0in:     21,
			dmin:     -0.0000000000000000,
			dmin1:    0.0000000000000000,
			dmin2:    0.0000000000000000,
			dn:       0.0000000000000000,
			dn1:      0.0000000000000000,
			dn2:      0.0000000000000000,
			tau:      0.0000000000000000,
			ttype:    0,
			g:        0.0000000000000000,
			zOut:     []float64{0.1914365246180821E+01, 0.1564384297703890E+01, 0.2493389162143899E+00, 0.3499809484769305E+00, 0.1315996513131545E+01, 0.1363862112490627E+01, 0.9898466611970759E-01, 0.2014733168553078E+00, 0.6023973979587287E+00, 0.6465544792741794E+00, 0.2210033410638781E-02, 0.5482758480425683E-01, 0.9861857233678967E-01, 0.2428190810745492E-01, 0.4756321484454819E+00, 0.7654669763997353E-01, 0.2588748143677115E+00, 0.6127784069508770E+00, 0.1078611376690004E+00, 0.1217285558623164E+00, 0.6442896492255246E+00, 0.2293835804898155E+00, 0.6203230486639705E+00, 0.5227672064047094E+00, 0.3695660678607585E+00, 0.7645233184745865E+00, 0.5378838054252265E+00, 0.2253657980501426E+00, 0.3562533181264623E+00, 0.8820486722335483E+00, 0.2222132496436145E-01, 0.1208845131814035E-01, 0.1275094303021685E+01, 0.6548746852163357E+00, 0.1647324354821218E+00, 0.6424409427697111E+00, 0.1007530576543866E+01, 0.3269551736546701E+00, 0.3453881601783118E+00, 0.8453078383713172E+00, 0.2679391719153404E+00, 0.4116714838778281E+00, 0.7328677736683723E+00, 0.2016558482158241E+00, 0.8360828138307410E+00, 0.9737579452195326E+00, 0.4813660709592822E+00, 0.5951926422795808E+00, 0.6495370513676459E+00, 0.6761876248148171E+00, 0.2325475880222648E+00, 0.4547154975121112E+00, 0.1993624802893807E+00, 0.3321819367342255E+00, 0.3782318916911257E+00, 0.9972813157741996E-01, 0.9830449403503746E+00, 0.7561080996844842E+00, 0.4429733864040367E+00, 0.6051687323570161E+00, 0.1173279550602403E+01, 0.7195724480316686E+00, 0.5035524069144587E+00, 0.8966804889747714E+00, 0.3058980395058521E+00, 0.6588832353928662E+00, 0.3014634433415453E+00, 0.1505672110274446E+00, 0.1289422237567578E+01, 0.6124645310993601E+00, 0.7583364305799440E+00, 0.9784211498097629E+00, 0.4977814779461571E+00, 0.9993813577491869E+00, 0.2841468847862598E+00, 0.2567365507769143E+00, 0.9257539794205765E+00, 0.5509268385614666E+00, 0.5231355605450990E-04, 0.6589740256453697E+00, 0.2117869221381033E-04, 0.7349224826832024E-04, 0.0000000000000000E+00, 0.0000000000000000E+00},
			tauOut:   0.0000000000000000,
			ttypeOut: -1,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1914365246180821E+01, 0.2163704162395211E+01, 0.2493389162143899E+00, 0.1516515751224039E+00, 0.1315996513131545E+01, 0.1263329604128848E+01, 0.9898466611970759E-01, 0.4719916727467415E-01, 0.6023973979587287E+00, 0.5574082640946934E+00, 0.2210033410638781E-02, 0.3910066531356214E-03, 0.9861857233678967E-01, 0.5738597141291359E+00, 0.4756321484454819E+00, 0.2145632131068746E+00, 0.2588748143677115E+00, 0.1521727389298373E+00, 0.1078611376690004E+00, 0.4566771620366771E+00, 0.6442896492255246E+00, 0.8079355358528180E+00, 0.6203230486639705E+00, 0.2837483186776231E+00, 0.3695660678607585E+00, 0.6237015546083620E+00, 0.5378838054252265E+00, 0.3072349091217998E+00, 0.3562533181264623E+00, 0.7123973396902394E-01, 0.2222132496436145E-01, 0.3977314805803597E+00, 0.1275094303021685E+01, 0.1042095257923447E+01, 0.1647324354821218E+00, 0.1592685164190333E+00, 0.1007530576543866E+01, 0.1193650220303144E+01, 0.3453881601783118E+00, 0.7752942700755104E-01, 0.2679391719153404E+00, 0.9232775185761617E+00, 0.7328677736683723E+00, 0.6636554427529671E+00, 0.8360828138307410E+00, 0.6537934420370561E+00, 0.4813660709592822E+00, 0.4782322339990674E+00, 0.6495370513676459E+00, 0.4038524053908432E+00, 0.2325475880222648E+00, 0.1147975431483785E+00, 0.1993624802893807E+00, 0.4627968288321279E+00, 0.3782318916911257E+00, 0.8034172324482011E+00, 0.9830449403503746E+00, 0.6226010943062101E+00, 0.4429733864040367E+00, 0.8347746582554776E+00, 0.1173279550602403E+01, 0.8420572992613844E+00, 0.5035524069144587E+00, 0.1829278057427913E+00, 0.3058980395058521E+00, 0.4244336771046062E+00, 0.3014634433415453E+00, 0.9158407747236312E+00, 0.1289422237567578E+01, 0.1131917893423890E+01, 0.7583364305799440E+00, 0.3334922359541972E+00, 0.4977814779461571E+00, 0.4484361267782198E+00, 0.2841468847862598E+00, 0.5865943745895725E+00, 0.9257539794205765E+00, 0.3392119183870583E+00, 0.5231355605450990E-04, 0.3266196269153995E-08, 0.2117869221381033E-04, 0.2117542601754118E-04, 0.0000000000000000E+00, 0.3910066531356214E-03, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			n0in:     21,
			dmin:     2.1175426017541180E-005,
			dmin1:    4.4311601260836921E-002,
			dmin2:    4.4311601260836921E-002,
			dn:       2.1175426017541180E-005,
			dn1:      0.33915960483100382,
			dn2:      0.16428924199195991,
			tau:      0.0000000000000000,
			ttype:    -1,
			g:        0.0000000000000000,
			zOut:     []float64{0.1914365246180821E+01, 0.2163704162395211E+01, 0.2493389162143899E+00, 0.1516515751224039E+00, 0.1315996513131545E+01, 0.1263329604128848E+01, 0.9898466611970759E-01, 0.4719916727467415E-01, 0.6023973979587287E+00, 0.5574082640946934E+00, 0.2210033410638781E-02, 0.3910066531356214E-03, 0.9861857233678967E-01, 0.5738597141291359E+00, 0.4756321484454819E+00, 0.2145632131068746E+00, 0.2588748143677115E+00, 0.1521727389298373E+00, 0.1078611376690004E+00, 0.4566771620366771E+00, 0.6442896492255246E+00, 0.8079355358528180E+00, 0.6203230486639705E+00, 0.2837483186776231E+00, 0.3695660678607585E+00, 0.6237015546083620E+00, 0.5378838054252265E+00, 0.3072349091217998E+00, 0.3562533181264623E+00, 0.7123973396902394E-01, 0.2222132496436145E-01, 0.3977314805803597E+00, 0.1275094303021685E+01, 0.1042095257923447E+01, 0.1647324354821218E+00, 0.1592685164190333E+00, 0.1007530576543866E+01, 0.1193650220303144E+01, 0.3453881601783118E+00, 0.7752942700755104E-01, 0.2679391719153404E+00, 0.9232775185761617E+00, 0.7328677736683723E+00, 0.6636554427529671E+00, 0.8360828138307410E+00, 0.6537934420370561E+00, 0.4813660709592822E+00, 0.4782322339990674E+00, 0.6495370513676459E+00, 0.4038524053908432E+00, 0.2325475880222648E+00, 0.1147975431483785E+00, 0.1993624802893807E+00, 0.4627968288321279E+00, 0.3782318916911257E+00, 0.8034172324482011E+00, 0.9830449403503746E+00, 0.6226010943062101E+00, 0.4429733864040367E+00, 0.8347746582554776E+00, 0.1173279550602403E+01, 0.8420572992613844E+00, 0.5035524069144587E+00, 0.1829278057427913E+00, 0.3058980395058521E+00, 0.4244336771046062E+00, 0.3014634433415453E+00, 0.9158407747236312E+00, 0.1289422237567578E+01, 0.1131917893423890E+01, 0.7583364305799440E+00, 0.3334922359541972E+00, 0.4977814779461571E+00, 0.4484361267782198E+00, 0.2841468847862598E+00, 0.5865943745895725E+00, 0.9257539794205765E+00, 0.3392119183870583E+00, 0.5231355605450990E-04, 0.3266196269153995E-08, 0.2117869221381033E-04, 0.2117542601754118E-04, 0.0000000000000000E+00, 0.3910066531356214E-03, 0.1037537856266618 - 321},
			tauOut:   0.0000000000000000,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2315355737517615E+01, 0.2163704162395211E+01, 0.8274578340618610E-01, 0.1516515751224039E+00, 0.1227782987997336E+01, 0.1263329604128848E+01, 0.2142822156235013E-01, 0.4719916727467415E-01, 0.5363710491854788E+00, 0.5574082640946934E+00, 0.4183353417969536E-03, 0.3910066531356214E-03, 0.7880045918942136E+00, 0.5738597141291359E+00, 0.4143462125464707E-01, 0.2145632131068746E+00, 0.5674152797118673E+00, 0.1521727389298373E+00, 0.6502569120260687E+00, 0.4566771620366771E+00, 0.4414269425043723E+00, 0.8079355358528180E+00, 0.4009140594652070E+00, 0.2837483186776231E+00, 0.5300224042649548E+00, 0.6237015546083620E+00, 0.4129510944388858E-01, 0.3072349091217998E+00, 0.4276761051054951E+00, 0.7123973396902394E-01, 0.9691308092544145E+00, 0.3977314805803597E+00, 0.2322329650880660E+00, 0.1042095257923447E+01, 0.8186215063776209E+00, 0.1592685164190333E+00, 0.4525581409330741E+00, 0.1193650220303144E+01, 0.1581701233715052E+00, 0.7752942700755104E-01, 0.1428762837957623E+01, 0.9232775185761617E+00, 0.3036848136842134E+00, 0.6636554427529671E+00, 0.8283408623519102E+00, 0.6537934420370561E+00, 0.2331591338951825E+00, 0.4782322339990674E+00, 0.2854908146440392E+00, 0.4038524053908432E+00, 0.1860933389154074E+00, 0.1147975431483785E+00, 0.1080120722364922E+01, 0.4627968288321279E+00, 0.4631042046962229E+00, 0.8034172324482011E+00, 0.9942715478654648E+00, 0.6226010943062101E+00, 0.7069779837626068E+00, 0.8347746582554776E+00, 0.3180071212415688E+00, 0.8420572992613844E+00, 0.2441477440283845E+00, 0.1829278057427913E+00, 0.1096126707799853E+01, 0.4244336771046062E+00, 0.9457451890006905E+00, 0.9158407747236312E+00, 0.5196649403773971E+00, 0.1131917893423890E+01, 0.2877815203259632E+00, 0.3334922359541972E+00, 0.7472489810418290E+00, 0.4484361267782198E+00, 0.2662831374385604E+00, 0.5865943745895725E+00, 0.7292878421469419E-01, 0.3392119183870583E+00, 0.9483648767903632E-12, 0.3266196269153995E-08, 0.2117542506917630E-04, 0.2117542601754118E-04, 0.4183353417969536E-03, 0.3910066531356214E-03},
			i0:       1,
			n0:       21,
			pp:       0,
			n0in:     21,
			dmin:     2.1175425069176302E-005,
			dmin1:    2.9944624525135358E-002,
			dmin2:    2.9944624525135358E-002,
			dn:       2.1175425069176302E-005,
			dn1:      7.2928780948497918E-002,
			dn2:      0.16065460645225654,
			tau:      0.0000000000000000,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2315355737517615E+01, 0.2163704162395211E+01, 0.8274578340618610E-01, 0.1516515751224039E+00, 0.1227782987997336E+01, 0.1263329604128848E+01, 0.2142822156235013E-01, 0.4719916727467415E-01, 0.5363710491854788E+00, 0.5574082640946934E+00, 0.4183353417969536E-03, 0.3910066531356214E-03, 0.7880045918942136E+00, 0.5738597141291359E+00, 0.4143462125464707E-01, 0.2145632131068746E+00, 0.5674152797118673E+00, 0.1521727389298373E+00, 0.6502569120260687E+00, 0.4566771620366771E+00, 0.4414269425043723E+00, 0.8079355358528180E+00, 0.4009140594652070E+00, 0.2837483186776231E+00, 0.5300224042649548E+00, 0.6237015546083620E+00, 0.4129510944388858E-01, 0.3072349091217998E+00, 0.4276761051054951E+00, 0.7123973396902394E-01, 0.9691308092544145E+00, 0.3977314805803597E+00, 0.2322329650880660E+00, 0.1042095257923447E+01, 0.8186215063776209E+00, 0.1592685164190333E+00, 0.4525581409330741E+00, 0.1193650220303144E+01, 0.1581701233715052E+00, 0.7752942700755104E-01, 0.1428762837957623E+01, 0.9232775185761617E+00, 0.3036848136842134E+00, 0.6636554427529671E+00, 0.8283408623519102E+00, 0.6537934420370561E+00, 0.2331591338951825E+00, 0.4782322339990674E+00, 0.2854908146440392E+00, 0.4038524053908432E+00, 0.1860933389154074E+00, 0.1147975431483785E+00, 0.1080120722364922E+01, 0.4627968288321279E+00, 0.4631042046962229E+00, 0.8034172324482011E+00, 0.9942715478654648E+00, 0.6226010943062101E+00, 0.7069779837626068E+00, 0.8347746582554776E+00, 0.3180071212415688E+00, 0.8420572992613844E+00, 0.2441477440283845E+00, 0.1829278057427913E+00, 0.1096126707799853E+01, 0.4244336771046062E+00, 0.9457451890006905E+00, 0.9158407747236312E+00, 0.5196649403773971E+00, 0.1131917893423890E+01, 0.2877815203259632E+00, 0.3334922359541972E+00, 0.7472489810418290E+00, 0.4484361267782198E+00, 0.2662831374385604E+00, 0.5865943745895725E+00, 0.7292878421469419E-01, 0.3392119183870583E+00, 0.9483648767903632E-12, 0.3266196269153995E-08, 0.2117542506917630E-04, 0.2117542601754118E-04, 0.4183353417969536E-03, 0.3910066531356214E-03},
			tauOut:   2.1175313795360271E-005,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2315355737517615E+01, 0.2398080345610006E+01, 0.8274578340618610E-01, 0.4236466279397526E-01, 0.1227782987997336E+01, 0.1206825371451915E+01, 0.2142822156235013E-01, 0.9523728911788614E-02, 0.5363710491854788E+00, 0.5272444803016919E+00, 0.4183353417969536E-03, 0.6252320936560726E-03, 0.7880045918942136E+00, 0.8287928057414093E+00, 0.4143462125464707E-01, 0.2836732781232222E-01, 0.5674152797118673E+00, 0.1189283688611819E+01, 0.6502569120260687E+00, 0.2413561400585997E+00, 0.4414269425043723E+00, 0.6009636865971842E+00, 0.4009140594652070E+00, 0.3535878097802652E+00, 0.5300224042649548E+00, 0.2177085286147829E+00, 0.4129510944388858E-01, 0.8112190955144877E-01, 0.4276761051054951E+00, 0.1315663829494665E+01, 0.9691308092544145E+00, 0.1710650671895379E+00, 0.2322329650880660E+00, 0.8797682289623537E+00, 0.8186215063776209E+00, 0.4211038940233675E+00, 0.4525581409330741E+00, 0.1896031949674164E+00, 0.1581701233715052E+00, 0.1191897606932286E+01, 0.1428762837957623E+01, 0.5405288693957555E+00, 0.3036848136842134E+00, 0.4653859482687157E+00, 0.8283408623519102E+00, 0.5960928726645816E+00, 0.2331591338951825E+00, 0.1116684901463164E+00, 0.2854908146440392E+00, 0.3598944880993349E+00, 0.1860933389154074E+00, 0.5585061130503639E+00, 0.1080120722364922E+01, 0.9846976386969850E+00, 0.4631042046962229E+00, 0.4676068229793028E+00, 0.9942715478654648E+00, 0.1233621533334973E+01, 0.7069779837626068E+00, 0.1822471700779458E+00, 0.3180071212415688E+00, 0.3798865198782122E+00, 0.2441477440283845E+00, 0.7044652781161848E+00, 0.1096126707799853E+01, 0.1337385443370563E+01, 0.9457451890006905E+00, 0.3674861422265960E+00, 0.5196649403773971E+00, 0.4399391431629689E+00, 0.2877815203259632E+00, 0.4888049885267526E+00, 0.7472489810418290E+00, 0.5247059546398414E+00, 0.2662831374385604E+00, 0.3701064434002514E-01, 0.7292878421469419E-01, 0.3589696456182207E-01, 0.9483648767903632E-12, 0.5594353069081231E-15, 0.2117542506917630E-04, 0.1112732565966979E-09, 0.4183353417969536E-03, 0.6252320936560726E-03, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			n0in:     21,
			dmin:     1.1127325659669794E-010,
			dmin1:    3.1433071595911154E-002,
			dmin2:    3.1433071595911154E-002,
			dn:       1.1127325659669794E-010,
			dn1:      3.5896964560873705E-002,
			dn2:      0.25842281720128102,
			tau:      2.1175313795360271E-005,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2315355737517615E+01, 0.2398080345610006E+01, 0.8274578340618610E-01, 0.4236466279397526E-01, 0.1227782987997336E+01, 0.1206825371451915E+01, 0.2142822156235013E-01, 0.9523728911788614E-02, 0.5363710491854788E+00, 0.5272444803016919E+00, 0.4183353417969536E-03, 0.6252320936560726E-03, 0.7880045918942136E+00, 0.8287928057414093E+00, 0.4143462125464707E-01, 0.2836732781232222E-01, 0.5674152797118673E+00, 0.1189283688611819E+01, 0.6502569120260687E+00, 0.2413561400585997E+00, 0.4414269425043723E+00, 0.6009636865971842E+00, 0.4009140594652070E+00, 0.3535878097802652E+00, 0.5300224042649548E+00, 0.2177085286147829E+00, 0.4129510944388858E-01, 0.8112190955144877E-01, 0.4276761051054951E+00, 0.1315663829494665E+01, 0.9691308092544145E+00, 0.1710650671895379E+00, 0.2322329650880660E+00, 0.8797682289623537E+00, 0.8186215063776209E+00, 0.4211038940233675E+00, 0.4525581409330741E+00, 0.1896031949674164E+00, 0.1581701233715052E+00, 0.1191897606932286E+01, 0.1428762837957623E+01, 0.5405288693957555E+00, 0.3036848136842134E+00, 0.4653859482687157E+00, 0.8283408623519102E+00, 0.5960928726645816E+00, 0.2331591338951825E+00, 0.1116684901463164E+00, 0.2854908146440392E+00, 0.3598944880993349E+00, 0.1860933389154074E+00, 0.5585061130503639E+00, 0.1080120722364922E+01, 0.9846976386969850E+00, 0.4631042046962229E+00, 0.4676068229793028E+00, 0.9942715478654648E+00, 0.1233621533334973E+01, 0.7069779837626068E+00, 0.1822471700779458E+00, 0.3180071212415688E+00, 0.3798865198782122E+00, 0.2441477440283845E+00, 0.7044652781161848E+00, 0.1096126707799853E+01, 0.1337385443370563E+01, 0.9457451890006905E+00, 0.3674861422265960E+00, 0.5196649403773971E+00, 0.4399391431629689E+00, 0.2877815203259632E+00, 0.4888049885267526E+00, 0.7472489810418290E+00, 0.5247059546398414E+00, 0.2662831374385604E+00, 0.3701064434002514E-01, 0.7292878421469419E-01, 0.3589696456182207E-01, 0.9483648767903632E-12, 0.5594353069081231E-15, 0.2117542506917630E-04, 0.1112732565966979E-09, 0.4183353417969536E-03, 0.6252320936560726E-03, 0.1037537856266618 - 321},
			tauOut:   2.1175313795360271E-005,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2440445008292708E+01, 0.2398080345610006E+01, 0.2094976520226600E-01, 0.4236466279397526E-01, 0.1195399335050165E+01, 0.1206825371451915E+01, 0.4200549016048655E-02, 0.9523728911788614E-02, 0.5236691632680260E+00, 0.5272444803016919E+00, 0.9895328911616120E-03, 0.6252320936560726E-03, 0.8561706005512968E+00, 0.8287928057414093E+00, 0.3940429656773515E-01, 0.2836732781232222E-01, 0.1391235531991410E+01, 0.1189283688611819E+01, 0.1042571673718422E+00, 0.2413561400585997E+00, 0.8502943288943339E+00, 0.6009636865971842E+00, 0.9053227710395735E-01, 0.3535878097802652E+00, 0.2082981609510011E+00, 0.2177085286147829E+00, 0.5123864833424303E+00, 0.8112190955144877E-01, 0.9743424132304999E+00, 0.1315663829494665E+01, 0.1544607000116935E+00, 0.1710650671895379E+00, 0.1146411422862754E+01, 0.8797682289623537E+00, 0.6964571542795012E-01, 0.4211038940233675E+00, 0.1311855086360479E+01, 0.1896031949674164E+00, 0.4911023119923957E+00, 0.1191897606932286E+01, 0.5148125055608023E+00, 0.5405288693957555E+00, 0.5388626806938843E+00, 0.4653859482687157E+00, 0.1688986820057405E+00, 0.5960928726645816E+00, 0.2379466412690434E+00, 0.1116684901463164E+00, 0.6804539597693821E+00, 0.3598944880993349E+00, 0.8082246312519304E+00, 0.5585061130503639E+00, 0.6440798303130841E+00, 0.9846976386969850E+00, 0.8956185534970393E+00, 0.4676068229793028E+00, 0.5202501498046066E+00, 0.1233621533334973E+01, 0.1330768347199243E+00, 0.1822471700779458E+00, 0.9512749631631994E+00, 0.3798865198782122E+00, 0.9903988276741268E+00, 0.7044652781161848E+00, 0.7144727578117591E+00, 0.1337385443370563E+01, 0.2262808998212762E+00, 0.3674861422265960E+00, 0.7024632317571722E+00, 0.4399391431629689E+00, 0.3651136124179467E+00, 0.4888049885267526E+00, 0.1966029864506465E+00, 0.5247059546398414E+00, 0.6757627705811050E-02, 0.3701064434002514E-01, 0.2913933674473832E-01, 0.3589696456182207E-01, 0.2136293938333395E-23, 0.5594353069081231E-15, 0.0000000000000000E+00, 0.1112732565966979E-09, 0.9895328911616120E-03, 0.6252320936560726E-03},
			i0:       1,
			n0:       21,
			pp:       0,
			n0in:     21,
			dmin:     0.0000000000000000,
			dmin1:    2.9139336744737766E-002,
			dmin2:    4.9426557292086552E-002,
			dn:       -2.0808762284537102E-024,
			dn1:      2.9139336744737766E-002,
			dn2:      0.15959234211062134,
			tau:      1.1127325659669789E-010,
			ttype:    -15,
			g:        0.0000000000000000,
			zOut:     []float64{0.2440445008292708E+01, 0.2398080345610006E+01, 0.2094976520226600E-01, 0.4236466279397526E-01, 0.1195399335050165E+01, 0.1206825371451915E+01, 0.4200549016048655E-02, 0.9523728911788614E-02, 0.5236691632680260E+00, 0.5272444803016919E+00, 0.9895328911616120E-03, 0.6252320936560726E-03, 0.8561706005512968E+00, 0.8287928057414093E+00, 0.3940429656773515E-01, 0.2836732781232222E-01, 0.1391235531991410E+01, 0.1189283688611819E+01, 0.1042571673718422E+00, 0.2413561400585997E+00, 0.8502943288943339E+00, 0.6009636865971842E+00, 0.9053227710395735E-01, 0.3535878097802652E+00, 0.2082981609510011E+00, 0.2177085286147829E+00, 0.5123864833424303E+00, 0.8112190955144877E-01, 0.9743424132304999E+00, 0.1315663829494665E+01, 0.1544607000116935E+00, 0.1710650671895379E+00, 0.1146411422862754E+01, 0.8797682289623537E+00, 0.6964571542795012E-01, 0.4211038940233675E+00, 0.1311855086360479E+01, 0.1896031949674164E+00, 0.4911023119923957E+00, 0.1191897606932286E+01, 0.5148125055608023E+00, 0.5405288693957555E+00, 0.5388626806938843E+00, 0.4653859482687157E+00, 0.1688986820057405E+00, 0.5960928726645816E+00, 0.2379466412690434E+00, 0.1116684901463164E+00, 0.6804539597693821E+00, 0.3598944880993349E+00, 0.8082246312519304E+00, 0.5585061130503639E+00, 0.6440798303130841E+00, 0.9846976386969850E+00, 0.8956185534970393E+00, 0.4676068229793028E+00, 0.5202501498046066E+00, 0.1233621533334973E+01, 0.1330768347199243E+00, 0.1822471700779458E+00, 0.9512749631631994E+00, 0.3798865198782122E+00, 0.9903988276741268E+00, 0.7044652781161848E+00, 0.7144727578117591E+00, 0.1337385443370563E+01, 0.2262808998212762E+00, 0.3674861422265960E+00, 0.7024632317571722E+00, 0.4399391431629689E+00, 0.3651136124179467E+00, 0.4888049885267526E+00, 0.1966029864506465E+00, 0.5247059546398414E+00, 0.6757627705811050E-02, 0.3701064434002514E-01, 0.2913933674473832E-01, 0.3589696456182207E-01, 0.2136293938333395E-23, 0.5594353069081231E-15, 0.0000000000000000E+00, 0.1112732565966979E-09, 0.9895328911616120E-03, 0.6252320936560726E-03},
			tauOut:   -0.0000000000000000,
			ttypeOut: -1,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2440445008292708E+01, 0.2461394773494974E+01, 0.2094976520226600E-01, 0.1017444891892999E-01, 0.1195399335050165E+01, 0.1189425435147283E+01, 0.4200549016048655E-02, 0.1849378635683999E-02, 0.5236691632680260E+00, 0.5228093175235037E+00, 0.9895328911616120E-03, 0.1620493249248586E-02, 0.8561706005512968E+00, 0.8939544038697832E+00, 0.3940429656773515E-01, 0.6132377362967349E-01, 0.1391235531991410E+01, 0.1434168925733579E+01, 0.1042571673718422E+00, 0.6181229879703373E-01, 0.8502943288943339E+00, 0.8790143072012576E+00, 0.9053227710395735E-01, 0.2145324219750511E-01, 0.2082981609510011E+00, 0.6992314020959263E+00, 0.5123864833424303E+00, 0.7139837844669097E+00, 0.9743424132304999E+00, 0.4148193287752837E+00, 0.1544607000116935E+00, 0.4268738185358478E+00, 0.1146411422862754E+01, 0.7891833197548568E+00, 0.6964571542795012E-01, 0.1157716892137957E+00, 0.1311855086360479E+01, 0.1687185709139079E+01, 0.4911023119923957E+00, 0.1498504938454686E+00, 0.5148125055608023E+00, 0.9038246924092180E+00, 0.5388626806938843E+00, 0.1006978425303630E+00, 0.1688986820057405E+00, 0.3061474807444209E+00, 0.2379466412690434E+00, 0.5288684194677825E+00, 0.6804539597693821E+00, 0.9598101715535300E+00, 0.8082246312519304E+00, 0.5423584775195998E+00, 0.6440798303130841E+00, 0.9973399062905237E+00, 0.8956185534970393E+00, 0.4671884516860899E+00, 0.5202501498046066E+00, 0.1861385328384410E+00, 0.1330768347199243E+00, 0.6800991665489665E+00, 0.9512749631631994E+00, 0.1261574624288360E+01, 0.9903988276741268E+00, 0.5608966509936130E+00, 0.7144727578117591E+00, 0.3798570066394223E+00, 0.2262808998212762E+00, 0.4184574968871406E+00, 0.7024632317571722E+00, 0.6491193472879784E+00, 0.3651136124179467E+00, 0.1105843276664904E+00, 0.1966029864506465E+00, 0.9277628648996712E-01, 0.6757627705811050E-02, 0.2122447413720272E-02, 0.2913933674473832E-01, 0.2701688933101806E-01, 0.2136293938333395E-23, 0.0000000000000000E+00, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			n0in:     21,
			dmin:     0.0000000000000000,
			dmin1:    2.7016889331018056E-002,
			dmin2:    5.3061698118516694E-002,
			dn:       0.0000000000000000,
			dn1:      2.7016889331018056E-002,
			dn2:      8.6018658784156071E-002,
			tau:      0.0000000000000000,
			ttype:    -1,
			g:        0.0000000000000000,
			zOut:     []float64{0.2440445008292708E+01, 0.2461394773494974E+01, 0.2094976520226600E-01, 0.1017444891892999E-01, 0.1195399335050165E+01, 0.1189425435147283E+01, 0.4200549016048655E-02, 0.1849378635683999E-02, 0.5236691632680260E+00, 0.5228093175235037E+00, 0.9895328911616120E-03, 0.1620493249248586E-02, 0.8561706005512968E+00, 0.8939544038697832E+00, 0.3940429656773515E-01, 0.6132377362967349E-01, 0.1391235531991410E+01, 0.1434168925733579E+01, 0.1042571673718422E+00, 0.6181229879703373E-01, 0.8502943288943339E+00, 0.8790143072012576E+00, 0.9053227710395735E-01, 0.2145324219750511E-01, 0.2082981609510011E+00, 0.6992314020959263E+00, 0.5123864833424303E+00, 0.7139837844669097E+00, 0.9743424132304999E+00, 0.4148193287752837E+00, 0.1544607000116935E+00, 0.4268738185358478E+00, 0.1146411422862754E+01, 0.7891833197548568E+00, 0.6964571542795012E-01, 0.1157716892137957E+00, 0.1311855086360479E+01, 0.1687185709139079E+01, 0.4911023119923957E+00, 0.1498504938454686E+00, 0.5148125055608023E+00, 0.9038246924092180E+00, 0.5388626806938843E+00, 0.1006978425303630E+00, 0.1688986820057405E+00, 0.3061474807444209E+00, 0.2379466412690434E+00, 0.5288684194677825E+00, 0.6804539597693821E+00, 0.9598101715535300E+00, 0.8082246312519304E+00, 0.5423584775195998E+00, 0.6440798303130841E+00, 0.9973399062905237E+00, 0.8956185534970393E+00, 0.4671884516860899E+00, 0.5202501498046066E+00, 0.1861385328384410E+00, 0.1330768347199243E+00, 0.6800991665489665E+00, 0.9512749631631994E+00, 0.1261574624288360E+01, 0.9903988276741268E+00, 0.5608966509936130E+00, 0.7144727578117591E+00, 0.3798570066394223E+00, 0.2262808998212762E+00, 0.4184574968871406E+00, 0.7024632317571722E+00, 0.6491193472879784E+00, 0.3651136124179467E+00, 0.1105843276664904E+00, 0.1966029864506465E+00, 0.9277628648996712E-01, 0.6757627705811050E-02, 0.2122447413720272E-02, 0.2913933674473832E-01, 0.2701688933101806E-01, 0.2136293938333395E-23, 0.0000000000000000E+00, 0.2117542506861687E-04},
			tauOut:   -0.0000000000000000,
			ttypeOut: -1,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2471569222413904E+01, 0.2461394773494974E+01, 0.4896382518051712E-02, 0.1017444891892999E-01, 0.1186378431264915E+01, 0.1189425435147283E+01, 0.8149780515932184E-03, 0.1849378635683999E-02, 0.5236148327211592E+00, 0.5228093175235037E+00, 0.2766627272719901E-02, 0.1620493249248586E-02, 0.9525115502267366E+00, 0.8939544038697832E+00, 0.9233342160256496E-01, 0.6132377362967349E-01, 0.1403647802928048E+01, 0.1434168925733579E+01, 0.3870906568602875E-01, 0.6181229879703373E-01, 0.8617584837127339E+00, 0.8790143072012576E+00, 0.1740717486950262E-01, 0.2145324219750511E-01, 0.1395808011693333E+01, 0.6992314020959263E+00, 0.2121884039551361E+00, 0.7139837844669097E+00, 0.6295047433559955E+00, 0.4148193287752837E+00, 0.5351535485381410E+00, 0.4268738185358478E+00, 0.3698014604305115E+00, 0.7891833197548568E+00, 0.5281978587564573E+00, 0.1157716892137957E+00, 0.1308838344228090E+01, 0.1687185709139079E+01, 0.1034799882693896E+00, 0.1498504938454686E+00, 0.9010425466701916E+00, 0.9038246924092180E+00, 0.3421413441684364E-01, 0.1006978425303630E+00, 0.8008017657953598E+00, 0.3061474807444209E+00, 0.6338813300623194E+00, 0.5288684194677825E+00, 0.8682873190108105E+00, 0.9598101715535300E+00, 0.6229686202966810E+00, 0.5423584775195998E+00, 0.8415597376799326E+00, 0.9973399062905237E+00, 0.1033340463692495E+00, 0.4671884516860899E+00, 0.7629036530181579E+00, 0.1861385328384410E+00, 0.1124645093942705E+01, 0.6800991665489665E+00, 0.6978261813392677E+00, 0.1261574624288360E+01, 0.3053203341720497E+00, 0.5608966509936130E+00, 0.4929941693545132E+00, 0.3798570066394223E+00, 0.5509778292160957E+00, 0.4184574968871406E+00, 0.2087258457383731E+00, 0.6491193472879784E+00, 0.4915348757406203E-01, 0.1105843276664904E+00, 0.4574524632962537E-01, 0.9277628648996712E-01, 0.1253505697055357E-02, 0.2122447413720272E-02, 0.2576338363396270E-01, 0.2701688933101806E-01, 0.8149780515932184E-03, 0.1620493249248586E-02},
			i0:       1,
			n0:       20,
			pp:       0,
			n0in:     20,
			dmin:     2.5763383633962696E-002,
			dmin1:    4.3622798915905092E-002,
			dmin2:    7.4536672467372611E-002,
			dn:       2.5763383633962696E-002,
			dn1:      4.3622798915905092E-002,
			dn2:      9.8141518071882677E-002,
			tau:      0.0000000000000000,
			ttype:    -1,
			g:        0.0000000000000000,
			zOut:     []float64{0.2471569222413904E+01, 0.2461394773494974E+01, 0.4896382518051712E-02, 0.1017444891892999E-01, 0.1186378431264915E+01, 0.1189425435147283E+01, 0.8149780515932184E-03, 0.1849378635683999E-02, 0.5236148327211592E+00, 0.5228093175235037E+00, 0.2766627272719901E-02, 0.1620493249248586E-02, 0.9525115502267366E+00, 0.8939544038697832E+00, 0.9233342160256496E-01, 0.6132377362967349E-01, 0.1403647802928048E+01, 0.1434168925733579E+01, 0.3870906568602875E-01, 0.6181229879703373E-01, 0.8617584837127339E+00, 0.8790143072012576E+00, 0.1740717486950262E-01, 0.2145324219750511E-01, 0.1395808011693333E+01, 0.6992314020959263E+00, 0.2121884039551361E+00, 0.7139837844669097E+00, 0.6295047433559955E+00, 0.4148193287752837E+00, 0.5351535485381410E+00, 0.4268738185358478E+00, 0.3698014604305115E+00, 0.7891833197548568E+00, 0.5281978587564573E+00, 0.1157716892137957E+00, 0.1308838344228090E+01, 0.1687185709139079E+01, 0.1034799882693896E+00, 0.1498504938454686E+00, 0.9010425466701916E+00, 0.9038246924092180E+00, 0.3421413441684364E-01, 0.1006978425303630E+00, 0.8008017657953598E+00, 0.3061474807444209E+00, 0.6338813300623194E+00, 0.5288684194677825E+00, 0.8682873190108105E+00, 0.9598101715535300E+00, 0.6229686202966810E+00, 0.5423584775195998E+00, 0.8415597376799326E+00, 0.9973399062905237E+00, 0.1033340463692495E+00, 0.4671884516860899E+00, 0.7629036530181579E+00, 0.1861385328384410E+00, 0.1124645093942705E+01, 0.6800991665489665E+00, 0.6978261813392677E+00, 0.1261574624288360E+01, 0.3053203341720497E+00, 0.5608966509936130E+00, 0.4929941693545132E+00, 0.3798570066394223E+00, 0.5509778292160957E+00, 0.4184574968871406E+00, 0.2087258457383731E+00, 0.6491193472879784E+00, 0.4915348757406203E-01, 0.1105843276664904E+00, 0.4574524632962537E-01, 0.9277628648996712E-01, 0.1253505697055357E-02, 0.2122447413720272E-02, 0.2576338363396270E-01, 0.2701688933101806E-01, 0.8149780515932184E-03, 0.1620493249248586E-02},
			tauOut:   2.0080554394878082E-002,
			ttypeOut: -3,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2471569222413904E+01, 0.2471445466333236E+01, 0.4896382518051712E-02, 0.2350431231346416E-02, 0.1186378431264915E+01, 0.1179822839486443E+01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.5236148327211592E+00, 0.5209996276036221E+00, 0.2766627272719901E-02, 0.5058054349403302E-02, 0.9525115502267366E+00, 0.1034766778881179E+01, 0.9233342160256496E-01, 0.1252490967185870E+00, 0.1403647802928048E+01, 0.1312087633296770E+01, 0.3870906568602875E-01, 0.2542350442532051E-01, 0.8617584837127339E+00, 0.8487220155581966E+00, 0.1740717486950262E-01, 0.2862783537884150E-01, 0.1395808011693333E+01, 0.1574348441670908E+01, 0.2121884039551361E+00, 0.8484373804386666E-01, 0.6295047433559955E+00, 0.1074794415251550E+01, 0.5351535485381410E+00, 0.1841287608083240E+00, 0.3698014604305115E+00, 0.7088504197799252E+00, 0.5281978587564573E+00, 0.9752771411128711E+00, 0.1308838344228090E+01, 0.4320210527858890E+00, 0.1034799882693896E+00, 0.2158225196628609E+00, 0.9010425466701916E+00, 0.7144140228254550E+00, 0.3421413441684364E-01, 0.3835134583138245E-01, 0.8008017657953598E+00, 0.1391311611427577E+01, 0.6338813300623194E+00, 0.3955915526975877E+00, 0.8682873190108105E+00, 0.1090644248011184E+01, 0.6229686202966810E+00, 0.4806932321292802E+00, 0.8415597376799326E+00, 0.4591804133211825E+00, 0.1033340463692495E+00, 0.1716839812178710E+00, 0.7629036530181579E+00, 0.1710844627144272E+01, 0.1124645093942705E+01, 0.4587247601659613E+00, 0.6978261813392677E+00, 0.5394016167466366E+00, 0.3053203341720497E+00, 0.2790520826393697E+00, 0.4929941693545132E+00, 0.7598997773325197E+00, 0.5509778292160957E+00, 0.1513401067044909E+00, 0.2087258457383731E+00, 0.1015190880092246E+00, 0.4915348757406203E-01, 0.2214892234681356E-01, 0.4574524632962537E-01, 0.1982969108114764E-01, 0.1253505697055357E-02, 0.1628595626045726E-02, 0.2576338363396270E-01, 0.1911464940919745E-01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			n0in:     20,
			dmin:     1.8576185384092288E-002,
			dmin1:    1.8576185384092288E-002,
			dmin2:    5.2365600435162571E-002,
			dn:       1.9114649409197451E-002,
			dn1:      1.8576185384092288E-002,
			dn2:      5.2365600435162571E-002,
			tau:      5.0201385987195205E-003,
			ttype:    -15,
			g:        0.0000000000000000,
			zOut:     []float64{0.2471569222413904E+01, 0.2471445466333236E+01, 0.4896382518051712E-02, 0.2350431231346416E-02, 0.1186378431264915E+01, 0.1179822839486443E+01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.5236148327211592E+00, 0.5209996276036221E+00, 0.2766627272719901E-02, 0.5058054349403302E-02, 0.9525115502267366E+00, 0.1034766778881179E+01, 0.9233342160256496E-01, 0.1252490967185870E+00, 0.1403647802928048E+01, 0.1312087633296770E+01, 0.3870906568602875E-01, 0.2542350442532051E-01, 0.8617584837127339E+00, 0.8487220155581966E+00, 0.1740717486950262E-01, 0.2862783537884150E-01, 0.1395808011693333E+01, 0.1574348441670908E+01, 0.2121884039551361E+00, 0.8484373804386666E-01, 0.6295047433559955E+00, 0.1074794415251550E+01, 0.5351535485381410E+00, 0.1841287608083240E+00, 0.3698014604305115E+00, 0.7088504197799252E+00, 0.5281978587564573E+00, 0.9752771411128711E+00, 0.1308838344228090E+01, 0.4320210527858890E+00, 0.1034799882693896E+00, 0.2158225196628609E+00, 0.9010425466701916E+00, 0.7144140228254550E+00, 0.3421413441684364E-01, 0.3835134583138245E-01, 0.8008017657953598E+00, 0.1391311611427577E+01, 0.6338813300623194E+00, 0.3955915526975877E+00, 0.8682873190108105E+00, 0.1090644248011184E+01, 0.6229686202966810E+00, 0.4806932321292802E+00, 0.8415597376799326E+00, 0.4591804133211825E+00, 0.1033340463692495E+00, 0.1716839812178710E+00, 0.7629036530181579E+00, 0.1710844627144272E+01, 0.1124645093942705E+01, 0.4587247601659613E+00, 0.6978261813392677E+00, 0.5394016167466366E+00, 0.3053203341720497E+00, 0.2790520826393697E+00, 0.4929941693545132E+00, 0.7598997773325197E+00, 0.5509778292160957E+00, 0.1513401067044909E+00, 0.2087258457383731E+00, 0.1015190880092246E+00, 0.4915348757406203E-01, 0.2214892234681356E-01, 0.4574524632962537E-01, 0.1982969108114764E-01, 0.1253505697055357E-02, 0.1628595626045726E-02, 0.2576338363396270E-01, 0.1911464940919745E-01, 0.8149780515932184E-03, 0.3616937915375072E-03, 0.2117542506861687E-04},
			tauOut:   5.4769133315280185E-003,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2468318984233055E+01, 0.2471445466333236E+01, 0.1123474100024551E-02, 0.2350431231346416E-02, 0.1173584145846428E+01, 0.1179822839486443E+01, 0.1605699355811189E-03, 0.3616937915375072E-03, 0.5204201986859162E+00, 0.5209996276036221E+00, 0.1005707814522541E-01, 0.5058054349403302E-02, 0.1144481884123012E+01, 0.1034766778881179E+01, 0.1435914304680996E+00, 0.1252490967185870E+00, 0.1188442793922463E+01, 0.1312087633296770E+01, 0.1815610143690141E-01, 0.2542350442532051E-01, 0.8537168361686087E+00, 0.8487220155581966E+00, 0.5279290053521807E-01, 0.2862783537884150E-01, 0.1600922365848029E+01, 0.1574348441670908E+01, 0.5696064828871891E-01, 0.8484373804386666E-01, 0.1196485614439627E+01, 0.1074794415251550E+01, 0.1090859328498209E+00, 0.1841287608083240E+00, 0.1569564714711448E+01, 0.7088504197799252E+00, 0.2684440171930437E+00, 0.9752771411128711E+00, 0.3739226419241781E+00, 0.4320210527858890E+00, 0.4123490187575627E+00, 0.2158225196628609E+00, 0.3349394365677468E+00, 0.7144140228254550E+00, 0.1593084209965356E+00, 0.3835134583138245E-01, 0.1622117829797102E+01, 0.1391311611427577E+01, 0.2659792301064862E+00, 0.3955915526975877E+00, 0.1299881336702450E+01, 0.1090644248011184E+01, 0.1698038973078534E+00, 0.4806932321292802E+00, 0.4555835838996722E+00, 0.4591804133211825E+00, 0.6447216871142054E+00, 0.1716839812178710E+00, 0.1519370786864500E+01, 0.1710844627144272E+01, 0.1628548339973444E+00, 0.4587247601659613E+00, 0.6501219520571339E+00, 0.5394016167466366E+00, 0.3261720586281595E+00, 0.2790520826393697E+00, 0.5795909120773233E+00, 0.7598997773325197E+00, 0.2650819619788820E-01, 0.1513401067044909E+00, 0.9168290082662192E-01, 0.1015190880092246E+00, 0.4790492926791300E-02, 0.2214892234681356E-01, 0.1119088044887405E-01, 0.1982969108114764E-01, 0.2781732372482683E-02, 0.1628595626045726E-02, 0.1085600370518675E-01, 0.1911464940919745E-01, 0.1605699355811189E-03, 0.3616937915375072E-03},
			i0:       1,
			n0:       20,
			pp:       0,
			n0in:     20,
			dmin:     9.5622848228283271E-003,
			dmin1:    9.5622848228283271E-003,
			dmin2:    6.9533978479808370E-002,
			dn:       1.0856003705186750E-002,
			dn1:      9.5622848228283271E-003,
			dn2:      6.9533978479808370E-002,
			tau:      5.4769133315280185E-003,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2468318984233055E+01, 0.2471445466333236E+01, 0.1123474100024551E-02, 0.2350431231346416E-02, 0.1173584145846428E+01, 0.1179822839486443E+01, 0.1605699355811189E-03, 0.3616937915375072E-03, 0.5204201986859162E+00, 0.5209996276036221E+00, 0.1005707814522541E-01, 0.5058054349403302E-02, 0.1144481884123012E+01, 0.1034766778881179E+01, 0.1435914304680996E+00, 0.1252490967185870E+00, 0.1188442793922463E+01, 0.1312087633296770E+01, 0.1815610143690141E-01, 0.2542350442532051E-01, 0.8537168361686087E+00, 0.8487220155581966E+00, 0.5279290053521807E-01, 0.2862783537884150E-01, 0.1600922365848029E+01, 0.1574348441670908E+01, 0.5696064828871891E-01, 0.8484373804386666E-01, 0.1196485614439627E+01, 0.1074794415251550E+01, 0.1090859328498209E+00, 0.1841287608083240E+00, 0.1569564714711448E+01, 0.7088504197799252E+00, 0.2684440171930437E+00, 0.9752771411128711E+00, 0.3739226419241781E+00, 0.4320210527858890E+00, 0.4123490187575627E+00, 0.2158225196628609E+00, 0.3349394365677468E+00, 0.7144140228254550E+00, 0.1593084209965356E+00, 0.3835134583138245E-01, 0.1622117829797102E+01, 0.1391311611427577E+01, 0.2659792301064862E+00, 0.3955915526975877E+00, 0.1299881336702450E+01, 0.1090644248011184E+01, 0.1698038973078534E+00, 0.4806932321292802E+00, 0.4555835838996722E+00, 0.4591804133211825E+00, 0.6447216871142054E+00, 0.1716839812178710E+00, 0.1519370786864500E+01, 0.1710844627144272E+01, 0.1628548339973444E+00, 0.4587247601659613E+00, 0.6501219520571339E+00, 0.5394016167466366E+00, 0.3261720586281595E+00, 0.2790520826393697E+00, 0.5795909120773233E+00, 0.7598997773325197E+00, 0.2650819619788820E-01, 0.1513401067044909E+00, 0.9168290082662192E-01, 0.1015190880092246E+00, 0.4790492926791300E-02, 0.2214892234681356E-01, 0.1119088044887405E-01, 0.1982969108114764E-01, 0.2781732372482683E-02, 0.1628595626045726E-02, 0.1085600370518675E-01, 0.1911464940919745E-01, 0.1605699355811189E-03, 0.3616937915375072E-03},
			tauOut:   5.1216063611655054E-003,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2468318984233055E+01, 0.2464320851971913E+01, 0.1123474100024551E-02, 0.5350323562789559E-03, 0.1173584145846428E+01, 0.1168088077064565E+01, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.5204201986859162E+00, 0.5252841314829605E+00, 0.1005707814522541E-01, 0.2191222436498315E-01, 0.1144481884123012E+01, 0.1261039483864963E+01, 0.1435914304680996E+00, 0.1353250258951489E+00, 0.1188442793922463E+01, 0.1066152263103050E+01, 0.1815610143690141E-01, 0.1453842008528346E-01, 0.8537168361686087E+00, 0.8868497102573779E+00, 0.5279290053521807E-01, 0.9530062900995111E-01, 0.1600922365848029E+01, 0.1557460778765631E+01, 0.5696064828871891E-01, 0.4375878814786067E-01, 0.1196485614439627E+01, 0.1256691152780422E+01, 0.1090859328498209E+00, 0.1362446379077657E+00, 0.1569564714711448E+01, 0.1696642487635560E+01, 0.2684440171930437E+00, 0.5916231430550117E-01, 0.3739226419241781E+00, 0.7219877400150740E+00, 0.4123490187575627E+00, 0.1912940350054112E+00, 0.3349394365677468E+00, 0.2978322161977056E+00, 0.1593084209965356E+00, 0.8676597630518320E+00, 0.1622117829797102E+01, 0.1015315690490590E+01, 0.2659792301064862E+00, 0.3405260456467969E+00, 0.1299881336702450E+01, 0.1124037582002341E+01, 0.1698038973078534E+00, 0.6882320425428856E-01, 0.4555835838996722E+00, 0.1026360460398424E+01, 0.6447216871142054E+00, 0.9544125430154021E+00, 0.1519370786864500E+01, 0.7226914714852769E+00, 0.1628548339973444E+00, 0.1465016632377001E+00, 0.6501219520571339E+00, 0.8246707410864278E+00, 0.3261720586281595E+00, 0.2292385937027206E+00, 0.5795909120773233E+00, 0.3717389082113253E+00, 0.2650819619788820E-01, 0.6537783023029759E-02, 0.9168290082662192E-01, 0.8481400436921797E-01, 0.4790492926791300E-02, 0.6320870478125323E-03, 0.1119088044887405E-01, 0.8218919412378699E-02, 0.2781732372482683E-02, 0.3674266095981827E-02, 0.1085600370518675E-01, 0.2060131248039419E-02, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			n0in:     20,
			dmin:     2.0601312480394186E-003,
			dmin1:    5.4371870398960158E-003,
			dmin2:    8.0023511442426670E-002,
			dn:       2.0601312480394186E-003,
			dn1:      5.4371870398960158E-003,
			dn2:      8.0023511442426670E-002,
			tau:      5.1216063611655054E-003,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2468318984233055E+01, 0.2464320851971913E+01, 0.1123474100024551E-02, 0.5350323562789559E-03, 0.1173584145846428E+01, 0.1168088077064565E+01, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.5204201986859162E+00, 0.5252841314829605E+00, 0.1005707814522541E-01, 0.2191222436498315E-01, 0.1144481884123012E+01, 0.1261039483864963E+01, 0.1435914304680996E+00, 0.1353250258951489E+00, 0.1188442793922463E+01, 0.1066152263103050E+01, 0.1815610143690141E-01, 0.1453842008528346E-01, 0.8537168361686087E+00, 0.8868497102573779E+00, 0.5279290053521807E-01, 0.9530062900995111E-01, 0.1600922365848029E+01, 0.1557460778765631E+01, 0.5696064828871891E-01, 0.4375878814786067E-01, 0.1196485614439627E+01, 0.1256691152780422E+01, 0.1090859328498209E+00, 0.1362446379077657E+00, 0.1569564714711448E+01, 0.1696642487635560E+01, 0.2684440171930437E+00, 0.5916231430550117E-01, 0.3739226419241781E+00, 0.7219877400150740E+00, 0.4123490187575627E+00, 0.1912940350054112E+00, 0.3349394365677468E+00, 0.2978322161977056E+00, 0.1593084209965356E+00, 0.8676597630518320E+00, 0.1622117829797102E+01, 0.1015315690490590E+01, 0.2659792301064862E+00, 0.3405260456467969E+00, 0.1299881336702450E+01, 0.1124037582002341E+01, 0.1698038973078534E+00, 0.6882320425428856E-01, 0.4555835838996722E+00, 0.1026360460398424E+01, 0.6447216871142054E+00, 0.9544125430154021E+00, 0.1519370786864500E+01, 0.7226914714852769E+00, 0.1628548339973444E+00, 0.1465016632377001E+00, 0.6501219520571339E+00, 0.8246707410864278E+00, 0.3261720586281595E+00, 0.2292385937027206E+00, 0.5795909120773233E+00, 0.3717389082113253E+00, 0.2650819619788820E-01, 0.6537783023029759E-02, 0.9168290082662192E-01, 0.8481400436921797E-01, 0.4790492926791300E-02, 0.6320870478125323E-03, 0.1119088044887405E-01, 0.8218919412378699E-02, 0.2781732372482683E-02, 0.3674266095981827E-02, 0.1085600370518675E-01, 0.2060131248039419E-02, 0.1605699355811189E-03, 0.7153898701552432E-04, 0.2117542506861687E-04},
			tauOut:   1.2817878169164906E-003,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2463574096511276E+01, 0.2464320851971913E+01, 0.2536822079344948E-03, 0.5350323562789559E-03, 0.1166624146026729E+01, 0.1168088077064565E+01, 0.3221114082852138E-04, 0.7153898701552432E-04, 0.5458823568901986E+00, 0.5252841314829605E+00, 0.5061929508212644E-01, 0.2191222436498315E-01, 0.1344463426861069E+01, 0.1261039483864963E+01, 0.1073120173669855E+00, 0.1353250258951489E+00, 0.9720968780044319E+00, 0.1066152263103050E+01, 0.1326348631702415E-01, 0.1453842008528346E-01, 0.9676050651333883E+00, 0.8868497102573779E+00, 0.1533962535161303E+00, 0.9530062900995111E-01, 0.1446541525580445E+01, 0.1557460778765631E+01, 0.3801569533217738E-01, 0.4375878814786067E-01, 0.1353638307539094E+01, 0.1256691152780422E+01, 0.1707682473962209E+00, 0.1362446379077657E+00, 0.1583754766727924E+01, 0.1696642487635560E+01, 0.2697037855661164E-01, 0.5916231430550117E-01, 0.8850296086469572E+00, 0.7219877400150740E+00, 0.6437471225190403E-01, 0.1912940350054112E+00, 0.1099835479180717E+01, 0.2978322161977056E+00, 0.8009821360646626E+00, 0.8676597630518320E+00, 0.5535778122558079E+00, 0.1015315690490590E+01, 0.6914368034330997E+00, 0.3405260456467969E+00, 0.5001421950066134E+00, 0.1124037582002341E+01, 0.1412346654806686E+00, 0.6882320425428856E-01, 0.1838256550116241E+01, 0.1026360460398424E+01, 0.3752173792456719E+00, 0.9544125430154021E+00, 0.4926939676603885E+00, 0.7226914714852769E+00, 0.2452143584512202E+00, 0.1465016632377001E+00, 0.8074131885210117E+00, 0.8246707410864278E+00, 0.1055431169003394E+00, 0.2292385937027206E+00, 0.2714517865170992E+00, 0.3717389082113253E+00, 0.2042703660177667E-02, 0.6537783023029759E-02, 0.8212159993993635E-01, 0.8481400436921797E-01, 0.6326073178529442E-04, 0.6320870478125323E-03, 0.1054813695965874E-01, 0.8218919412378699E-02, 0.7176120699696391E-03, 0.3674266095981827E-02, 0.6073136115328898E-04, 0.2060131248039419E-02, 0.3221114082852138E-04, 0.7153898701552432E-04},
			i0:       1,
			n0:       20,
			pp:       0,
			n0in:     20,
			dmin:     6.0731361153288982E-005,
			dmin1:    6.8738708636769136E-003,
			dmin2:    8.1489512892123819E-002,
			dn:       6.0731361153288982E-005,
			dn1:      6.8738708636769136E-003,
			dn2:      8.1489512892123819E-002,
			tau:      1.2817878169164906E-003,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2463574096511276E+01, 0.2464320851971913E+01, 0.2536822079344948E-03, 0.5350323562789559E-03, 0.1166624146026729E+01, 0.1168088077064565E+01, 0.3221114082852138E-04, 0.7153898701552432E-04, 0.5458823568901986E+00, 0.5252841314829605E+00, 0.5061929508212644E-01, 0.2191222436498315E-01, 0.1344463426861069E+01, 0.1261039483864963E+01, 0.1073120173669855E+00, 0.1353250258951489E+00, 0.9720968780044319E+00, 0.1066152263103050E+01, 0.1326348631702415E-01, 0.1453842008528346E-01, 0.9676050651333883E+00, 0.8868497102573779E+00, 0.1533962535161303E+00, 0.9530062900995111E-01, 0.1446541525580445E+01, 0.1557460778765631E+01, 0.3801569533217738E-01, 0.4375878814786067E-01, 0.1353638307539094E+01, 0.1256691152780422E+01, 0.1707682473962209E+00, 0.1362446379077657E+00, 0.1583754766727924E+01, 0.1696642487635560E+01, 0.2697037855661164E-01, 0.5916231430550117E-01, 0.8850296086469572E+00, 0.7219877400150740E+00, 0.6437471225190403E-01, 0.1912940350054112E+00, 0.1099835479180717E+01, 0.2978322161977056E+00, 0.8009821360646626E+00, 0.8676597630518320E+00, 0.5535778122558079E+00, 0.1015315690490590E+01, 0.6914368034330997E+00, 0.3405260456467969E+00, 0.5001421950066134E+00, 0.1124037582002341E+01, 0.1412346654806686E+00, 0.6882320425428856E-01, 0.1838256550116241E+01, 0.1026360460398424E+01, 0.3752173792456719E+00, 0.9544125430154021E+00, 0.4926939676603885E+00, 0.7226914714852769E+00, 0.2452143584512202E+00, 0.1465016632377001E+00, 0.8074131885210117E+00, 0.8246707410864278E+00, 0.1055431169003394E+00, 0.2292385937027206E+00, 0.2714517865170992E+00, 0.3717389082113253E+00, 0.2042703660177667E-02, 0.6537783023029759E-02, 0.8212159993993635E-01, 0.8481400436921797E-01, 0.6326073178529442E-04, 0.6320870478125323E-03, 0.1054813695965874E-01, 0.8218919412378699E-02, 0.7176120699696391E-03, 0.3674266095981827E-02, 0.6073136115328898E-04, 0.2060131248039419E-02, 0.3221114082852138E-04, 0.7153898701552432E-04},
			tauOut:   5.6837241251038845E-005,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2463574096511276E+01, 0.2463770941477959E+01, 0.2536822079344948E-03, 0.1201214707955848E-03, 0.1166624146026729E+01, 0.1166479398455512E+01, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.5458823568901986E+00, 0.5964297407456295E+00, 0.5061929508212644E-01, 0.1141052940222717E+00, 0.1344463426861069E+01, 0.1337613312964532E+01, 0.1073120173669855E+00, 0.7798791776646297E-01, 0.9720968780044319E+00, 0.9073156093137420E+00, 0.1326348631702415E-01, 0.1414482062243694E-01, 0.9676050651333883E+00, 0.1106799660785830E+01, 0.1533962535161303E+00, 0.2004825791345134E+00, 0.1446541525580445E+01, 0.1284017804536858E+01, 0.3801569533217738E-01, 0.4007693764646178E-01, 0.1353638307539094E+01, 0.1484272780047602E+01, 0.1707682473962209E+00, 0.1822138285193538E+00, 0.1583754766727924E+01, 0.1428454479523931E+01, 0.2697037855661164E-01, 0.1671007646458111E-01, 0.8850296086469572E+00, 0.9326374071930291E+00, 0.6437471225190403E-01, 0.7591545433480534E-01, 0.1099835479180717E+01, 0.1824845323669324E+01, 0.8009821360646626E+00, 0.2429827519008994E+00, 0.5535778122558079E+00, 0.1001975026546757E+01, 0.6914368034330997E+00, 0.3451350696526060E+00, 0.5001421950066134E+00, 0.2961849535934249E+00, 0.1412346654806686E+00, 0.8765656248686587E+00, 0.1838256550116241E+01, 0.1336851467252003E+01, 0.3752173792456719E+00, 0.1382856239786244E+00, 0.4926939676603885E+00, 0.5995658648917332E+00, 0.2452143584512202E+00, 0.3302211126778973E+00, 0.8074131885210117E+00, 0.5826783555022028E+00, 0.1055431169003394E+00, 0.4916926700063749E-01, 0.2714517865170992E+00, 0.2242683859353883E+00, 0.2042703660177667E-02, 0.7479881396448043E-03, 0.8212159993993635E-01, 0.8138003529082581E-01, 0.6326073178529442E-04, 0.8199589256196194E-05, 0.1054813695965874E-01, 0.1120071219912114E-01, 0.7176120699696391E-03, 0.3890963093641941E-05, 0.6073136115328898E-04, 0.3156808608191942E-08, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			n0in:     20,
			dmin:     3.1568086081919418E-009,
			dmin1:    1.0483100129151506E-002,
			dmin2:    8.1316774559040517E-002,
			dn:       3.1568086081919418E-009,
			dn1:      1.0483100129151506E-002,
			dn2:      8.1316774559040517E-002,
			tau:      5.6837241251038845E-005,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2463574096511276E+01, 0.2463770941477959E+01, 0.2536822079344948E-03, 0.1201214707955848E-03, 0.1166624146026729E+01, 0.1166479398455512E+01, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.5458823568901986E+00, 0.5964297407456295E+00, 0.5061929508212644E-01, 0.1141052940222717E+00, 0.1344463426861069E+01, 0.1337613312964532E+01, 0.1073120173669855E+00, 0.7798791776646297E-01, 0.9720968780044319E+00, 0.9073156093137420E+00, 0.1326348631702415E-01, 0.1414482062243694E-01, 0.9676050651333883E+00, 0.1106799660785830E+01, 0.1533962535161303E+00, 0.2004825791345134E+00, 0.1446541525580445E+01, 0.1284017804536858E+01, 0.3801569533217738E-01, 0.4007693764646178E-01, 0.1353638307539094E+01, 0.1484272780047602E+01, 0.1707682473962209E+00, 0.1822138285193538E+00, 0.1583754766727924E+01, 0.1428454479523931E+01, 0.2697037855661164E-01, 0.1671007646458111E-01, 0.8850296086469572E+00, 0.9326374071930291E+00, 0.6437471225190403E-01, 0.7591545433480534E-01, 0.1099835479180717E+01, 0.1824845323669324E+01, 0.8009821360646626E+00, 0.2429827519008994E+00, 0.5535778122558079E+00, 0.1001975026546757E+01, 0.6914368034330997E+00, 0.3451350696526060E+00, 0.5001421950066134E+00, 0.2961849535934249E+00, 0.1412346654806686E+00, 0.8765656248686587E+00, 0.1838256550116241E+01, 0.1336851467252003E+01, 0.3752173792456719E+00, 0.1382856239786244E+00, 0.4926939676603885E+00, 0.5995658648917332E+00, 0.2452143584512202E+00, 0.3302211126778973E+00, 0.8074131885210117E+00, 0.5826783555022028E+00, 0.1055431169003394E+00, 0.4916926700063749E-01, 0.2714517865170992E+00, 0.2242683859353883E+00, 0.2042703660177667E-02, 0.7479881396448043E-03, 0.8212159993993635E-01, 0.8138003529082581E-01, 0.6326073178529442E-04, 0.8199589256196194E-05, 0.1054813695965874E-01, 0.1120071219912114E-01, 0.7176120699696391E-03, 0.3890963093641941E-05, 0.6073136115328898E-04, 0.3156808608191942E-08, 0.3221114082852138E-04, 0.1507398544447245E-04, 0.2117542506861687E-04},
			tauOut:   3.1557121791797713E-009,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2463891059793043E+01, 0.2463770941477959E+01, 0.5686908130061341E-04, 0.1201214707955848E-03, 0.1166437600203943E+01, 0.1166479398455512E+01, 0.7707718980490818E-05, 0.1507398544447245E-04, 0.7105273238932086E+00, 0.5964297407456295E+00, 0.2148105431436762E+00, 0.1141052940222717E+00, 0.1200790684431606E+01, 0.1337613312964532E+01, 0.5892755169139442E-01, 0.7798791776646297E-01, 0.8625328750890724E+00, 0.9073156093137420E+00, 0.1815059242254727E-01, 0.1414482062243694E-01, 0.1289131644342084E+01, 0.1106799660785830E+01, 0.1996872873596725E+00, 0.2004825791345134E+00, 0.1124407451667935E+01, 0.1284017804536858E+01, 0.5290351604133232E-01, 0.4007693764646178E-01, 0.1613583089369911E+01, 0.1484272780047602E+01, 0.1613081850537457E+00, 0.1822138285193538E+00, 0.1283856367779054E+01, 0.1428454479523931E+01, 0.1213877407087503E-01, 0.1671007646458111E-01, 0.9964140843012472E+00, 0.9326374071930291E+00, 0.1390325207358455E+00, 0.7591545433480534E-01, 0.1928795551678665E+01, 0.1824845323669324E+01, 0.1262252233392066E+00, 0.2429827519008994E+00, 0.1220884869704444E+01, 0.1001975026546757E+01, 0.8372928285471114E-01, 0.3451350696526060E+00, 0.1089021292451660E+01, 0.2961849535934249E+00, 0.1076046951396362E+01, 0.8765656248686587E+00, 0.3990901366785531E+00, 0.1336851467252003E+01, 0.2077509116934600E+00, 0.1382856239786244E+00, 0.7220360627204584E+00, 0.5995658648917332E+00, 0.2664862668525171E+00, 0.3302211126778973E+00, 0.3653613524946110E+00, 0.5826783555022028E+00, 0.3018138637972599E-01, 0.4916926700063749E-01, 0.1948349845395949E+00, 0.2242683859353883E+00, 0.3124249032854923E-03, 0.7479881396448043E-03, 0.8107580682108434E-01, 0.8138003529082581E-01, 0.1132782305976083E-05, 0.8199589256196194E-05, 0.1120346722419663E-01, 0.1120071219912114E-01, 0.1096359327194516E-11, 0.3890963093641941E-05, 0.6968497581336674E-16, 0.3156808608191942E-08, 0.7707718980490818E-05, 0.1507398544447245E-04},
			i0:       1,
			n0:       20,
			pp:       0,
			n0in:     20,
			dmin:     6.9684975813366743E-017,
			dmin1:    1.1199576261102989E-002,
			dmin2:    8.1067607231828140E-002,
			dn:       6.9684975813366743E-017,
			dn1:      1.1199576261102989E-002,
			dn2:      8.1067607231828140E-002,
			tau:      3.1557121791797713E-009,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2463891059793043E+01, 0.2463770941477959E+01, 0.5686908130061341E-04, 0.1201214707955848E-03, 0.1166437600203943E+01, 0.1166479398455512E+01, 0.7707718980490818E-05, 0.1507398544447245E-04, 0.7105273238932086E+00, 0.5964297407456295E+00, 0.2148105431436762E+00, 0.1141052940222717E+00, 0.1200790684431606E+01, 0.1337613312964532E+01, 0.5892755169139442E-01, 0.7798791776646297E-01, 0.8625328750890724E+00, 0.9073156093137420E+00, 0.1815059242254727E-01, 0.1414482062243694E-01, 0.1289131644342084E+01, 0.1106799660785830E+01, 0.1996872873596725E+00, 0.2004825791345134E+00, 0.1124407451667935E+01, 0.1284017804536858E+01, 0.5290351604133232E-01, 0.4007693764646178E-01, 0.1613583089369911E+01, 0.1484272780047602E+01, 0.1613081850537457E+00, 0.1822138285193538E+00, 0.1283856367779054E+01, 0.1428454479523931E+01, 0.1213877407087503E-01, 0.1671007646458111E-01, 0.9964140843012472E+00, 0.9326374071930291E+00, 0.1390325207358455E+00, 0.7591545433480534E-01, 0.1928795551678665E+01, 0.1824845323669324E+01, 0.1262252233392066E+00, 0.2429827519008994E+00, 0.1220884869704444E+01, 0.1001975026546757E+01, 0.8372928285471114E-01, 0.3451350696526060E+00, 0.1089021292451660E+01, 0.2961849535934249E+00, 0.1076046951396362E+01, 0.8765656248686587E+00, 0.3990901366785531E+00, 0.1336851467252003E+01, 0.2077509116934600E+00, 0.1382856239786244E+00, 0.7220360627204584E+00, 0.5995658648917332E+00, 0.2664862668525171E+00, 0.3302211126778973E+00, 0.3653613524946110E+00, 0.5826783555022028E+00, 0.3018138637972599E-01, 0.4916926700063749E-01, 0.1948349845395949E+00, 0.2242683859353883E+00, 0.3124249032854923E-03, 0.7479881396448043E-03, 0.8107580682108434E-01, 0.8138003529082581E-01, 0.1132782305976083E-05, 0.8199589256196194E-05, 0.1120346722419663E-01, 0.1120071219912114E-01, 0.1096359327194516E-11, 0.3890963093641941E-05, 0.6968497581336674E-16, 0.3156808608191942E-08, 0.7707718980490818E-05, 0.1507398544447245E-04},
			tauOut:   6.9684975806547287E-017,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2463891059793043E+01, 0.2463947928874343E+01, 0.5686908130061341E-04, 0.2692193042748079E-04, 0.1166437600203943E+01, 0.1166418385992496E+01, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.7105273238932086E+00, 0.9253331718563164E+00, 0.2148105431436762E+00, 0.2787563517334627E+00, 0.1200790684431606E+01, 0.9809618843895378E+00, 0.5892755169139442E-01, 0.5181337969514327E-01, 0.8625328750890724E+00, 0.8288700878164763E+00, 0.1815059242254727E-01, 0.2822939734392020E-01, 0.1289131644342084E+01, 0.1460589534357837E+01, 0.1996872873596725E+00, 0.1537255119449346E+00, 0.1124407451667935E+01, 0.1023585455764333E+01, 0.5290351604133232E-01, 0.8339725654733963E-01, 0.1613583089369911E+01, 0.1691494017876317E+01, 0.1613081850537457E+00, 0.1224340957564512E+00, 0.1283856367779054E+01, 0.1173561046093478E+01, 0.1213877407087503E-01, 0.1030644761994533E-01, 0.9964140843012472E+00, 0.1125140157417147E+01, 0.1390325207358455E+00, 0.2383394688796517E+00, 0.1928795551678665E+01, 0.1816681306138221E+01, 0.1262252233392066E+00, 0.8482856339700598E-01, 0.1220884869704444E+01, 0.1219785589162149E+01, 0.8372928285471114E-01, 0.7475327847832687E-01, 0.1089021292451660E+01, 0.2090314965369696E+01, 0.1076046951396362E+01, 0.2054425921547012E+00, 0.3990901366785531E+00, 0.4013984562173118E+00, 0.2077509116934600E+00, 0.3737026089221466E+00, 0.7220360627204584E+00, 0.6148197206508288E+00, 0.2664862668525171E+00, 0.1583615157552351E+00, 0.3653613524946110E+00, 0.2371812231191019E+00, 0.3018138637972599E-01, 0.2479281399828426E-01, 0.1948349845395949E+00, 0.1703545954445960E+00, 0.3124249032854923E-03, 0.1486904479375115E-03, 0.8107580682108434E-01, 0.8092824915545274E-01, 0.1132782305976083E-05, 0.1568190288260776E-06, 0.1120346722419663E-01, 0.1120331040626409E-01, 0.1096359327194516E-11, 0.6819392699821255E-26, 0.6968497581336674E-16, 0.6162975822039155E-31, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.2117542506861687E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			n0in:     20,
			dmin:     6.1629758220391547E-032,
			dmin1:    1.1203310405167735E-002,
			dmin2:    8.0927116373146771E-002,
			dn:       6.1629758220391547E-032,
			dn1:      1.1203310405167735E-002,
			dn2:      8.0927116373146771E-002,
			tau:      6.9684975806547287E-017,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2463891059793043E+01, 0.2463947928874343E+01, 0.5686908130061341E-04, 0.2692193042748079E-04, 0.1166437600203943E+01, 0.1166418385992496E+01, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.7105273238932086E+00, 0.9253331718563164E+00, 0.2148105431436762E+00, 0.2787563517334627E+00, 0.1200790684431606E+01, 0.9809618843895378E+00, 0.5892755169139442E-01, 0.5181337969514327E-01, 0.8625328750890724E+00, 0.8288700878164763E+00, 0.1815059242254727E-01, 0.2822939734392020E-01, 0.1289131644342084E+01, 0.1460589534357837E+01, 0.1996872873596725E+00, 0.1537255119449346E+00, 0.1124407451667935E+01, 0.1023585455764333E+01, 0.5290351604133232E-01, 0.8339725654733963E-01, 0.1613583089369911E+01, 0.1691494017876317E+01, 0.1613081850537457E+00, 0.1224340957564512E+00, 0.1283856367779054E+01, 0.1173561046093478E+01, 0.1213877407087503E-01, 0.1030644761994533E-01, 0.9964140843012472E+00, 0.1125140157417147E+01, 0.1390325207358455E+00, 0.2383394688796517E+00, 0.1928795551678665E+01, 0.1816681306138221E+01, 0.1262252233392066E+00, 0.8482856339700598E-01, 0.1220884869704444E+01, 0.1219785589162149E+01, 0.8372928285471114E-01, 0.7475327847832687E-01, 0.1089021292451660E+01, 0.2090314965369696E+01, 0.1076046951396362E+01, 0.2054425921547012E+00, 0.3990901366785531E+00, 0.4013984562173118E+00, 0.2077509116934600E+00, 0.3737026089221466E+00, 0.7220360627204584E+00, 0.6148197206508288E+00, 0.2664862668525171E+00, 0.1583615157552351E+00, 0.3653613524946110E+00, 0.2371812231191019E+00, 0.3018138637972599E-01, 0.2479281399828426E-01, 0.1948349845395949E+00, 0.1703545954445960E+00, 0.3124249032854923E-03, 0.1486904479375115E-03, 0.8107580682108434E-01, 0.8092824915545274E-01, 0.1132782305976083E-05, 0.1568190288260776E-06, 0.1120346722419663E-01, 0.1120331040626409E-01, 0.1096359327194516E-11, 0.6819392699821255E-26, 0.6968497581336674E-16, 0.6162975822039155E-31, 0.7707718980490818E-05, 0.4695180568393632E-05, 0.2117542506861687E-04},
			tauOut:   6.1629758220391547E-032,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2463974850804771E+01, 0.2463947928874343E+01, 0.1274454348702788E-04, 0.2692193042748079E-04, 0.1166410336629578E+01, 0.1166418385992496E+01, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.1204085798823124E+01, 0.9253331718563164E+00, 0.2271012218143261E+00, 0.2787563517334627E+00, 0.8056740422703550E+00, 0.9809618843895378E+00, 0.5330513126246473E-01, 0.5181337969514327E-01, 0.8037943538979316E+00, 0.8288700878164763E+00, 0.5129615818002433E-01, 0.2822939734392020E-01, 0.1563018888122747E+01, 0.1460589534357837E+01, 0.1006713350698832E+00, 0.1537255119449346E+00, 0.1006311377241790E+01, 0.1023585455764333E+01, 0.1401812239704283E+00, 0.8339725654733963E-01, 0.1673746889662340E+01, 0.1691494017876317E+01, 0.8584564749956700E-01, 0.1224340957564512E+00, 0.1098021846213856E+01, 0.1173561046093478E+01, 0.1056099032774466E-01, 0.1030644761994533E-01, 0.1352918635969054E+01, 0.1125140157417147E+01, 0.3200390963041470E+00, 0.2383394688796517E+00, 0.1581470773231080E+01, 0.1816681306138221E+01, 0.6542811978092533E-01, 0.8482856339700598E-01, 0.1229110747859551E+01, 0.1219785589162149E+01, 0.1271308521106110E+00, 0.7475327847832687E-01, 0.2168626705413786E+01, 0.2090314965369696E+01, 0.3802606466401751E-01, 0.2054425921547012E+00, 0.7370750004754409E+00, 0.4013984562173118E+00, 0.3117182559112661E+00, 0.3737026089221466E+00, 0.4614629804947978E+00, 0.6148197206508288E+00, 0.8139413038408401E-01, 0.1583615157552351E+00, 0.1805799067333021E+00, 0.2371812231191019E+00, 0.2338892446571373E-01, 0.2479281399828426E-01, 0.1471143614268198E+00, 0.1703545954445960E+00, 0.8179526119010886E-04, 0.1486904479375115E-03, 0.8084661071329148E-01, 0.8092824915545274E-01, 0.2173118009582292E-07, 0.1568190288260776E-06, 0.1120328867508400E-01, 0.1120331040626409E-01, 0.3751376363572422E-55, 0.6819392699821255E-26},
			i0:       1,
			n0:       19,
			pp:       0,
			n0in:     20,
			dmin:     6.1629758220391547E-032,
			dmin1:    1.1203288675083998E-002,
			dmin2:    8.0846453894262649E-002,
			dn:       6.1629758220391547E-032,
			dn1:      1.1203288675083998E-002,
			dn2:      8.0846453894262649E-002,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2463974850804771E+01, 0.2463947928874343E+01, 0.1274454348702788E-04, 0.2692193042748079E-04, 0.1166410336629578E+01, 0.1166418385992496E+01, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.1204085798823124E+01, 0.9253331718563164E+00, 0.2271012218143261E+00, 0.2787563517334627E+00, 0.8056740422703550E+00, 0.9809618843895378E+00, 0.5330513126246473E-01, 0.5181337969514327E-01, 0.8037943538979316E+00, 0.8288700878164763E+00, 0.5129615818002433E-01, 0.2822939734392020E-01, 0.1563018888122747E+01, 0.1460589534357837E+01, 0.1006713350698832E+00, 0.1537255119449346E+00, 0.1006311377241790E+01, 0.1023585455764333E+01, 0.1401812239704283E+00, 0.8339725654733963E-01, 0.1673746889662340E+01, 0.1691494017876317E+01, 0.8584564749956700E-01, 0.1224340957564512E+00, 0.1098021846213856E+01, 0.1173561046093478E+01, 0.1056099032774466E-01, 0.1030644761994533E-01, 0.1352918635969054E+01, 0.1125140157417147E+01, 0.3200390963041470E+00, 0.2383394688796517E+00, 0.1581470773231080E+01, 0.1816681306138221E+01, 0.6542811978092533E-01, 0.8482856339700598E-01, 0.1229110747859551E+01, 0.1219785589162149E+01, 0.1271308521106110E+00, 0.7475327847832687E-01, 0.2168626705413786E+01, 0.2090314965369696E+01, 0.3802606466401751E-01, 0.2054425921547012E+00, 0.7370750004754409E+00, 0.4013984562173118E+00, 0.3117182559112661E+00, 0.3737026089221466E+00, 0.4614629804947978E+00, 0.6148197206508288E+00, 0.8139413038408401E-01, 0.1583615157552351E+00, 0.1805799067333021E+00, 0.2371812231191019E+00, 0.2338892446571373E-01, 0.2479281399828426E-01, 0.1471143614268198E+00, 0.1703545954445960E+00, 0.8179526119010886E-04, 0.1486904479375115E-03, 0.8084661071329148E-01, 0.8092824915545274E-01, 0.2173118009582292E-07, 0.1568190288260776E-06, 0.1120328867508400E-01, 0.1120331040626409E-01, 0.3751376363572422E-55, 0.6819392699821255E-26},
			tauOut:   1.1203284285913290E-002,
			ttypeOut: -7,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2463974850804771E+01, 0.2452784311062345E+01, 0.1274454348702788E-04, 0.6060609239813679E-05, 0.1166410336629578E+01, 0.1155204716501079E+01, 0.3724766654883956E-05, 0.3882375624867434E-05, 0.1204085798823124E+01, 0.1419979853975912E+01, 0.2271012218143261E+00, 0.1288536304732592E+00, 0.8056740422703550E+00, 0.7189222587736472E+00, 0.5330513126246473E-01, 0.5959804835594534E-01, 0.8037943538979316E+00, 0.7842891794360973E+00, 0.5129615818002433E-01, 0.1022287011292917E+00, 0.1563018888122747E+01, 0.1550258237777425E+01, 0.1006713350698832E+00, 0.6534828028921505E-01, 0.1006311377241790E+01, 0.1069941036637090E+01, 0.1401812239704283E+00, 0.2192904838448094E+00, 0.1673746889662340E+01, 0.1529098769031184E+01, 0.8584564749956700E-01, 0.6164441320989392E-01, 0.1098021846213856E+01, 0.1035735139045794E+01, 0.1056099032774466E-01, 0.1379518767882891E-01, 0.1352918635969054E+01, 0.1647959260308459E+01, 0.3200390963041470E+00, 0.3071268139247320E+00, 0.1581470773231080E+01, 0.1328568794801360E+01, 0.6542811978092533E-01, 0.6053010243026304E-01, 0.1229110747859551E+01, 0.1284508213253985E+01, 0.1271308521106110E+00, 0.2146341752620367E+00, 0.2168626705413786E+01, 0.1980815310529854E+01, 0.3802606466401751E-01, 0.1414976019284330E-01, 0.7370750004754409E+00, 0.1023440211907950E+01, 0.3117182559112661E+00, 0.1405518698344743E+00, 0.4614629804947978E+00, 0.3911019567584944E+00, 0.8139413038408401E-01, 0.3758136265851575E-01, 0.1805799067333021E+00, 0.1551841842545868E+00, 0.2338892446571373E-01, 0.2217266343062855E-01, 0.1471143614268198E+00, 0.1138202089714681E+00, 0.8179526119010886E-04, 0.5809925758690545E-04, 0.8084661071329148E-01, 0.6958524890097138E-01, 0.2173118009582292E-07, 0.3498739858072780E-08, 0.1120328867508400E-01, 0.8904308505808256E-09, 0.3751376363572422E-55, 0.3882375624867434E-05, 0.1697846193036144E-01},
			i0:       1,
			n0:       19,
			pp:       1,
			n0in:     19,
			dmin:     8.9043085058082561E-010,
			dmin1:    6.9585227169791292E-002,
			dmin2:    0.11373841371027797,
			dn:       8.9043085058082561E-010,
			dn1:      6.9585227169791292E-002,
			dn2:      0.11373841371027797,
			tau:      1.1203284285913290E-002,
			ttype:    -7,
			g:        0.0000000000000000,
			zOut:     []float64{0.2463974850804771E+01, 0.2452784311062345E+01, 0.1274454348702788E-04, 0.6060609239813679E-05, 0.1166410336629578E+01, 0.1155204716501079E+01, 0.3724766654883956E-05, 0.3882375624867434E-05, 0.1204085798823124E+01, 0.1419979853975912E+01, 0.2271012218143261E+00, 0.1288536304732592E+00, 0.8056740422703550E+00, 0.7189222587736472E+00, 0.5330513126246473E-01, 0.5959804835594534E-01, 0.8037943538979316E+00, 0.7842891794360973E+00, 0.5129615818002433E-01, 0.1022287011292917E+00, 0.1563018888122747E+01, 0.1550258237777425E+01, 0.1006713350698832E+00, 0.6534828028921505E-01, 0.1006311377241790E+01, 0.1069941036637090E+01, 0.1401812239704283E+00, 0.2192904838448094E+00, 0.1673746889662340E+01, 0.1529098769031184E+01, 0.8584564749956700E-01, 0.6164441320989392E-01, 0.1098021846213856E+01, 0.1035735139045794E+01, 0.1056099032774466E-01, 0.1379518767882891E-01, 0.1352918635969054E+01, 0.1647959260308459E+01, 0.3200390963041470E+00, 0.3071268139247320E+00, 0.1581470773231080E+01, 0.1328568794801360E+01, 0.6542811978092533E-01, 0.6053010243026304E-01, 0.1229110747859551E+01, 0.1284508213253985E+01, 0.1271308521106110E+00, 0.2146341752620367E+00, 0.2168626705413786E+01, 0.1980815310529854E+01, 0.3802606466401751E-01, 0.1414976019284330E-01, 0.7370750004754409E+00, 0.1023440211907950E+01, 0.3117182559112661E+00, 0.1405518698344743E+00, 0.4614629804947978E+00, 0.3911019567584944E+00, 0.8139413038408401E-01, 0.3758136265851575E-01, 0.1805799067333021E+00, 0.1551841842545868E+00, 0.2338892446571373E-01, 0.2217266343062855E-01, 0.1471143614268198E+00, 0.1138202089714681E+00, 0.8179526119010886E-04, 0.5809925758690545E-04, 0.8084661071329148E-01, 0.6958524890097138E-01, 0.2173118009582292E-07, 0.3498739858072780E-08, 0.1120328867508400E-01, 0.8904308505808256E-09, 0.3751376363572422E-55, 0.3882375624867434E-05, 0.1697846193036144E-01},
			tauOut:   8.9043080564395014E-010,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2452790370781154E+01, 0.2452784311062345E+01, 0.2854399814229969E-05, 0.6060609239813679E-05, 0.1155205743586459E+01, 0.1155204716501079E+01, 0.4772219324121025E-05, 0.3882375624867434E-05, 0.1548828711339416E+01, 0.1419979853975912E+01, 0.5981019230390531E-01, 0.1288536304732592E+00, 0.7187101139352565E+00, 0.7189222587736472E+00, 0.6503610222645050E-01, 0.5959804835594534E-01, 0.8214817774485077E+00, 0.7842891794360973E+00, 0.1929207566298143E+00, 0.1022287011292917E+00, 0.1422685760546395E+01, 0.1550258237777425E+01, 0.4914564318703864E-01, 0.6534828028921505E-01, 0.1240085876404429E+01, 0.1069941036637090E+01, 0.2703980549150243E+00, 0.2192904838448094E+00, 0.1320345126435623E+01, 0.1529098769031184E+01, 0.4835651195207321E-01, 0.6164441320989392E-01, 0.1001173813882118E+01, 0.1035735139045794E+01, 0.2270725319399537E-01, 0.1379518767882891E-01, 0.1932378820148765E+01, 0.1647959260308459E+01, 0.2111589595024383E+00, 0.3071268139247320E+00, 0.1177939936838754E+01, 0.1328568794801360E+01, 0.6600626338337756E-01, 0.6053010243026304E-01, 0.1433136124242214E+01, 0.1284508213253985E+01, 0.2966575563411978E+00, 0.2146341752620367E+00, 0.1698307513491068E+01, 0.1980815310529854E+01, 0.8526979628348903E-02, 0.1414976019284330E-01, 0.1155465101223645E+01, 0.1023440211907950E+01, 0.4757401262929915E-01, 0.1405518698344743E+00, 0.3811093058972801E+00, 0.3911019567584944E+00, 0.1530278326215801E-01, 0.3758136265851575E-01, 0.1620540635326265E+00, 0.1551841842545868E+00, 0.1557318051836492E-01, 0.2217266343062855E-01, 0.9830512682025924E-01, 0.1138202089714681E+00, 0.4112553872737890E-04, 0.5809925758690545E-04, 0.6954412597055305E-01, 0.6958524890097138E-01, 0.4479725446695418E-16, 0.3498739858072780E-08, 0.1396210563637444E-18, 0.8904308505808256E-09, 0.2854399814229969E-05, 0.3882375624867434E-05},
			i0:       1,
			n0:       19,
			pp:       0,
			n0in:     19,
			dmin:     1.3962105636374437E-019,
			dmin1:    6.9544122471813200E-002,
			dmin2:    9.8247027562672340E-002,
			dn:       1.3962105636374437E-019,
			dn1:      6.9544122471813200E-002,
			dn2:      9.8247027562672340E-002,
			tau:      8.9043080564395014E-010,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2452790370781154E+01, 0.2452784311062345E+01, 0.2854399814229969E-05, 0.6060609239813679E-05, 0.1155205743586459E+01, 0.1155204716501079E+01, 0.4772219324121025E-05, 0.3882375624867434E-05, 0.1548828711339416E+01, 0.1419979853975912E+01, 0.5981019230390531E-01, 0.1288536304732592E+00, 0.7187101139352565E+00, 0.7189222587736472E+00, 0.6503610222645050E-01, 0.5959804835594534E-01, 0.8214817774485077E+00, 0.7842891794360973E+00, 0.1929207566298143E+00, 0.1022287011292917E+00, 0.1422685760546395E+01, 0.1550258237777425E+01, 0.4914564318703864E-01, 0.6534828028921505E-01, 0.1240085876404429E+01, 0.1069941036637090E+01, 0.2703980549150243E+00, 0.2192904838448094E+00, 0.1320345126435623E+01, 0.1529098769031184E+01, 0.4835651195207321E-01, 0.6164441320989392E-01, 0.1001173813882118E+01, 0.1035735139045794E+01, 0.2270725319399537E-01, 0.1379518767882891E-01, 0.1932378820148765E+01, 0.1647959260308459E+01, 0.2111589595024383E+00, 0.3071268139247320E+00, 0.1177939936838754E+01, 0.1328568794801360E+01, 0.6600626338337756E-01, 0.6053010243026304E-01, 0.1433136124242214E+01, 0.1284508213253985E+01, 0.2966575563411978E+00, 0.2146341752620367E+00, 0.1698307513491068E+01, 0.1980815310529854E+01, 0.8526979628348903E-02, 0.1414976019284330E-01, 0.1155465101223645E+01, 0.1023440211907950E+01, 0.4757401262929915E-01, 0.1405518698344743E+00, 0.3811093058972801E+00, 0.3911019567584944E+00, 0.1530278326215801E-01, 0.3758136265851575E-01, 0.1620540635326265E+00, 0.1551841842545868E+00, 0.1557318051836492E-01, 0.2217266343062855E-01, 0.9830512682025924E-01, 0.1138202089714681E+00, 0.4112553872737890E-04, 0.5809925758690545E-04, 0.6954412597055305E-01, 0.6958524890097138E-01, 0.4479725446695418E-16, 0.3498739858072780E-08, 0.1396210563637444E-18, 0.8904308505808256E-09, 0.2854399814229969E-05, 0.3882375624867434E-05},
			tauOut:   1.3962105636374427E-019,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2452790370781154E+01, 0.2452793225180968E+01, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1155205743586459E+01, 0.1155209171453138E+01, 0.4772219324121025E-05, 0.6398278760815103E-05, 0.1548828711339416E+01, 0.1608632505364561E+01, 0.5981019230390531E-01, 0.2672219415054498E-01, 0.7187101139352565E+00, 0.7570240220111620E+00, 0.6503610222645050E-01, 0.7057368234283014E-01, 0.8214817774485077E+00, 0.9438288517354918E+00, 0.1929207566298143E+00, 0.2908001941945216E+00, 0.1422685760546395E+01, 0.1181031209538913E+01, 0.4914564318703864E-01, 0.5160305461093759E-01, 0.1240085876404429E+01, 0.1458880876708516E+01, 0.2703980549150243E+00, 0.2447209773632920E+00, 0.1320345126435623E+01, 0.1123980661024405E+01, 0.4835651195207321E-01, 0.4307304847484577E-01, 0.1001173813882118E+01, 0.9808080186012680E+00, 0.2270725319399537E-01, 0.4473761868138884E-01, 0.1932378820148765E+01, 0.2098800160969815E+01, 0.2111589595024383E+00, 0.1185117935689049E+00, 0.1177939936838754E+01, 0.1125434406653226E+01, 0.6600626338337756E-01, 0.8405284210411720E-01, 0.1433136124242214E+01, 0.1645740838479294E+01, 0.2966575563411978E+00, 0.3061331074057168E+00, 0.1698307513491068E+01, 0.1400701385713701E+01, 0.8526979628348903E-02, 0.7034066989504621E-02, 0.1155465101223645E+01, 0.1196005046863439E+01, 0.4757401262929915E-01, 0.1515955052150448E-01, 0.3811093058972801E+00, 0.3812525386379337E+00, 0.1530278326215801E-01, 0.6504555274179689E-02, 0.1620540635326265E+00, 0.1711226887768117E+00, 0.1557318051836492E-01, 0.8946350111698947E-02, 0.9830512682025924E-01, 0.8939990224728768E-01, 0.4112553872737890E-04, 0.3199152990069927E-04, 0.6954412597055305E-01, 0.6951213444065239E-01, 0.4479725446695418E-16, 0.8997910999570377E-34, 0.2818174710670554E-01},
			i0:       1,
			n0:       18,
			pp:       1,
			n0in:     19,
			dmin:     1.3962105636374430E-019,
			dmin1:    6.9512134440652351E-002,
			dmin2:    8.9358776708560295E-002,
			dn:       1.3962105636374430E-019,
			dn1:      6.9512134440652351E-002,
			dn2:      8.9358776708560295E-002,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2452790370781154E+01, 0.2452793225180968E+01, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1155205743586459E+01, 0.1155209171453138E+01, 0.4772219324121025E-05, 0.6398278760815103E-05, 0.1548828711339416E+01, 0.1608632505364561E+01, 0.5981019230390531E-01, 0.2672219415054498E-01, 0.7187101139352565E+00, 0.7570240220111620E+00, 0.6503610222645050E-01, 0.7057368234283014E-01, 0.8214817774485077E+00, 0.9438288517354918E+00, 0.1929207566298143E+00, 0.2908001941945216E+00, 0.1422685760546395E+01, 0.1181031209538913E+01, 0.4914564318703864E-01, 0.5160305461093759E-01, 0.1240085876404429E+01, 0.1458880876708516E+01, 0.2703980549150243E+00, 0.2447209773632920E+00, 0.1320345126435623E+01, 0.1123980661024405E+01, 0.4835651195207321E-01, 0.4307304847484577E-01, 0.1001173813882118E+01, 0.9808080186012680E+00, 0.2270725319399537E-01, 0.4473761868138884E-01, 0.1932378820148765E+01, 0.2098800160969815E+01, 0.2111589595024383E+00, 0.1185117935689049E+00, 0.1177939936838754E+01, 0.1125434406653226E+01, 0.6600626338337756E-01, 0.8405284210411720E-01, 0.1433136124242214E+01, 0.1645740838479294E+01, 0.2966575563411978E+00, 0.3061331074057168E+00, 0.1698307513491068E+01, 0.1400701385713701E+01, 0.8526979628348903E-02, 0.7034066989504621E-02, 0.1155465101223645E+01, 0.1196005046863439E+01, 0.4757401262929915E-01, 0.1515955052150448E-01, 0.3811093058972801E+00, 0.3812525386379337E+00, 0.1530278326215801E-01, 0.6504555274179689E-02, 0.1620540635326265E+00, 0.1711226887768117E+00, 0.1557318051836492E-01, 0.8946350111698947E-02, 0.9830512682025924E-01, 0.8939990224728768E-01, 0.4112553872737890E-04, 0.3199152990069927E-04, 0.6954412597055305E-01, 0.6951213444065239E-01, 0.4479725446695418E-16, 0.8997910999570377E-34, 0.2818174710670554E-01},
			tauOut:   6.8088572105595116E-002,
			ttypeOut: -8,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2384705997428018E+01, 0.2452793225180968E+01, 0.6512368848977837E-06, 0.1344352644992036E-05, 0.1087126346389419E+01, 0.1155209171453138E+01, 0.9467601652019932E-05, 0.6398278760815103E-05, 0.1567256659807859E+01, 0.1608632505364561E+01, 0.1290748567965171E-01, 0.2672219415054498E-01, 0.7466016465687454E+00, 0.7570240220111620E+00, 0.8921689079377818E-01, 0.7057368234283014E-01, 0.1077323583030640E+01, 0.9438288517354918E+00, 0.3187938243378626E+00, 0.2908001941945216E+00, 0.8457518677063927E+00, 0.1181031209538913E+01, 0.8901276181133644E-01, 0.5160305461093759E-01, 0.1546500520154876E+01, 0.1458880876708516E+01, 0.1778606876095877E+00, 0.2447209773632920E+00, 0.9211044497840675E+00, 0.1123980661024405E+01, 0.4586493023634136E-01, 0.4307304847484577E-01, 0.9115921349407204E+00, 0.9808080186012680E+00, 0.1030014605117353E+00, 0.4473761868138884E-01, 0.2046221921921390E+01, 0.2098800160969815E+01, 0.6518220171905387E-01, 0.1185117935689049E+00, 0.1076216474932694E+01, 0.1125434406653226E+01, 0.1285328723941422E+00, 0.8405284210411720E-01, 0.1755252501385274E+01, 0.1645740838479294E+01, 0.2442959445536252E+00, 0.3061331074057168E+00, 0.1095350936043985E+01, 0.1400701385713701E+01, 0.7680442260639305E-02, 0.7034066989504621E-02, 0.1135395583018709E+01, 0.1196005046863439E+01, 0.5090399511302622E-02, 0.1515955052150448E-01, 0.3145781222952156E+00, 0.3812525386379337E+00, 0.3538316586334169E-02, 0.6504555274179689E-02, 0.1084421501965814E+00, 0.1711226887768117E+00, 0.7375387005938499E-02, 0.8946350111698947E-02, 0.1396793466565477E-01, 0.8939990224728768E-01, 0.1592074691534444E-03, 0.3199152990069927E-04, 0.1264354865903830E-02, 0.6951213444065239E-01, 0.6512368848977837E-06, 0.1344352644992036E-05},
			i0:       1,
			n0:       18,
			pp:       0,
			n0in:     18,
			dmin:     1.2643548659038301E-003,
			dmin1:    1.3935943135754067E-002,
			dmin2:    9.9495800084882416E-002,
			dn:       1.2643548659038301E-003,
			dn1:      1.3935943135754067E-002,
			dn2:      9.9495800084882416E-002,
			tau:      6.8088572105595116E-002,
			ttype:    -8,
			g:        0.0000000000000000,
			zOut:     []float64{0.2384705997428018E+01, 0.2452793225180968E+01, 0.6512368848977837E-06, 0.1344352644992036E-05, 0.1087126346389419E+01, 0.1155209171453138E+01, 0.9467601652019932E-05, 0.6398278760815103E-05, 0.1567256659807859E+01, 0.1608632505364561E+01, 0.1290748567965171E-01, 0.2672219415054498E-01, 0.7466016465687454E+00, 0.7570240220111620E+00, 0.8921689079377818E-01, 0.7057368234283014E-01, 0.1077323583030640E+01, 0.9438288517354918E+00, 0.3187938243378626E+00, 0.2908001941945216E+00, 0.8457518677063927E+00, 0.1181031209538913E+01, 0.8901276181133644E-01, 0.5160305461093759E-01, 0.1546500520154876E+01, 0.1458880876708516E+01, 0.1778606876095877E+00, 0.2447209773632920E+00, 0.9211044497840675E+00, 0.1123980661024405E+01, 0.4586493023634136E-01, 0.4307304847484577E-01, 0.9115921349407204E+00, 0.9808080186012680E+00, 0.1030014605117353E+00, 0.4473761868138884E-01, 0.2046221921921390E+01, 0.2098800160969815E+01, 0.6518220171905387E-01, 0.1185117935689049E+00, 0.1076216474932694E+01, 0.1125434406653226E+01, 0.1285328723941422E+00, 0.8405284210411720E-01, 0.1755252501385274E+01, 0.1645740838479294E+01, 0.2442959445536252E+00, 0.3061331074057168E+00, 0.1095350936043985E+01, 0.1400701385713701E+01, 0.7680442260639305E-02, 0.7034066989504621E-02, 0.1135395583018709E+01, 0.1196005046863439E+01, 0.5090399511302622E-02, 0.1515955052150448E-01, 0.3145781222952156E+00, 0.3812525386379337E+00, 0.3538316586334169E-02, 0.6504555274179689E-02, 0.1084421501965814E+00, 0.1711226887768117E+00, 0.7375387005938499E-02, 0.8946350111698947E-02, 0.1396793466565477E-01, 0.8939990224728768E-01, 0.1592074691534444E-03, 0.3199152990069927E-04, 0.1264354865903830E-02, 0.6951213444065239E-01, 0.6512368848977837E-06, 0.1344352644992036E-05},
			tauOut:   1.2463174543591322E-003,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2384705997428018E+01, 0.2383460331210544E+01, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.1087126346389419E+01, 0.1085889199499347E+01, 0.9467601652019932E-05, 0.1366452649899944E-04, 0.1567256659807859E+01, 0.1578904163506652E+01, 0.1290748567965171E-01, 0.6103442048115082E-02, 0.7466016465687454E+00, 0.8284687778600495E+00, 0.8921689079377818E-01, 0.1160157908485996E+00, 0.1077323583030640E+01, 0.1278855299065544E+01, 0.3187938243378626E+00, 0.2108295383723412E+00, 0.8457518677063927E+00, 0.7226887736910288E+00, 0.8901276181133644E-01, 0.1904807262171572E+00, 0.1546500520154876E+01, 0.1532634164092948E+01, 0.1778606876095877E+00, 0.1068932656188067E+00, 0.9211044497840675E+00, 0.8588297969472432E+00, 0.4586493023634136E-01, 0.4868264913684867E-01, 0.9115921349407204E+00, 0.9646646288612479E+00, 0.1030014605117353E+00, 0.2184840619043245E+00, 0.2046221921921390E+01, 0.1891673744281760E+01, 0.6518220171905387E-01, 0.3708364593761750E-01, 0.1076216474932694E+01, 0.1166419383934860E+01, 0.1285328723941422E+00, 0.1934189785315259E+00, 0.1755252501385274E+01, 0.1804883149953014E+01, 0.2442959445536252E+00, 0.1482587897978486E+00, 0.1095350936043985E+01, 0.9535262710524167E+00, 0.7680442260639305E-02, 0.9145359161143373E-02, 0.1135395583018709E+01, 0.1130094305914509E+01, 0.5090399511302622E-02, 0.1416986451145964E-02, 0.3145781222952156E+00, 0.3154531349760448E+00, 0.3538316586334169E-02, 0.1216353924418735E-02, 0.1084421501965814E+00, 0.1133548658237420E+00, 0.7375387005938499E-02, 0.9088178357782586E-03, 0.1396793466565477E-01, 0.1197200684467082E-01, 0.1592074691534444E-03, 0.1681378409852773E-04, 0.1264354865903830E-02, 0.1223627446170173E-05, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.2818174710670554E-01},
			i0:       1,
			n0:       18,
			pp:       1,
			n0in:     18,
			dmin:     1.2236274461701734E-006,
			dmin1:    1.1812799375517376E-002,
			dmin2:    0.10597947881780349,
			dn:       1.2236274461701734E-006,
			dn1:      1.1812799375517376E-002,
			dn2:      0.10597947881780349,
			tau:      1.2463174543591322E-003,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2384705997428018E+01, 0.2383460331210544E+01, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.1087126346389419E+01, 0.1085889199499347E+01, 0.9467601652019932E-05, 0.1366452649899944E-04, 0.1567256659807859E+01, 0.1578904163506652E+01, 0.1290748567965171E-01, 0.6103442048115082E-02, 0.7466016465687454E+00, 0.8284687778600495E+00, 0.8921689079377818E-01, 0.1160157908485996E+00, 0.1077323583030640E+01, 0.1278855299065544E+01, 0.3187938243378626E+00, 0.2108295383723412E+00, 0.8457518677063927E+00, 0.7226887736910288E+00, 0.8901276181133644E-01, 0.1904807262171572E+00, 0.1546500520154876E+01, 0.1532634164092948E+01, 0.1778606876095877E+00, 0.1068932656188067E+00, 0.9211044497840675E+00, 0.8588297969472432E+00, 0.4586493023634136E-01, 0.4868264913684867E-01, 0.9115921349407204E+00, 0.9646646288612479E+00, 0.1030014605117353E+00, 0.2184840619043245E+00, 0.2046221921921390E+01, 0.1891673744281760E+01, 0.6518220171905387E-01, 0.3708364593761750E-01, 0.1076216474932694E+01, 0.1166419383934860E+01, 0.1285328723941422E+00, 0.1934189785315259E+00, 0.1755252501385274E+01, 0.1804883149953014E+01, 0.2442959445536252E+00, 0.1482587897978486E+00, 0.1095350936043985E+01, 0.9535262710524167E+00, 0.7680442260639305E-02, 0.9145359161143373E-02, 0.1135395583018709E+01, 0.1130094305914509E+01, 0.5090399511302622E-02, 0.1416986451145964E-02, 0.3145781222952156E+00, 0.3154531349760448E+00, 0.3538316586334169E-02, 0.1216353924418735E-02, 0.1084421501965814E+00, 0.1133548658237420E+00, 0.7375387005938499E-02, 0.9088178357782586E-03, 0.1396793466565477E-01, 0.1197200684467082E-01, 0.1592074691534444E-03, 0.1681378409852773E-04, 0.1264354865903830E-02, 0.1223627446170173E-05, 0.6512368848977837E-06, 0.2970373645586866E-06, 0.2818174710670554E-01},
			tauOut:   1.2218877946904154E-006,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2383459406360114E+01, 0.2383460331210544E+01, 0.1353283656358167E-06, 0.2970373645586866E-06, 0.1085901506809686E+01, 0.1085889199499347E+01, 0.1986826396898849E-04, 0.1366452649899944E-04, 0.1584986515403004E+01, 0.1578904163506652E+01, 0.3190255011763210E-02, 0.6103442048115082E-02, 0.9412930918090910E+00, 0.8284687778600495E+00, 0.1576208411525267E+00, 0.1160157908485996E+00, 0.1332062774397564E+01, 0.1278855299065544E+01, 0.1143821023097510E+00, 0.2108295383723412E+00, 0.7987861757106404E+00, 0.7226887736910288E+00, 0.3654761154847582E+00, 0.1904807262171572E+00, 0.1274050092339202E+01, 0.1532634164092948E+01, 0.7205613198290631E-01, 0.1068932656188067E+00, 0.8354550922133908E+00, 0.8588297969472432E+00, 0.5621179414582513E-01, 0.4868264913684867E-01, 0.1126935674731953E+01, 0.9646646288612479E+00, 0.3667472533840468E+00, 0.2184840619043245E+00, 0.1562008914947536E+01, 0.1891673744281760E+01, 0.2769195683500124E-01, 0.3708364593761750E-01, 0.1332145183743590E+01, 0.1166419383934860E+01, 0.2620575140703801E+00, 0.1934189785315259E+00, 0.1691083203792688E+01, 0.1804883149953014E+01, 0.8359650824372877E-01, 0.1482587897978486E+00, 0.8790739000820365E+00, 0.9535262710524167E+00, 0.1175682535061811E-01, 0.9145359161143373E-02, 0.1119753245127243E+01, 0.1130094305914509E+01, 0.3991886785572840E-03, 0.1416986451145964E-02, 0.3162690783341116E+00, 0.3154531349760448E+00, 0.4359567385560517E-03, 0.1216353924418735E-02, 0.1138265050331695E+00, 0.1133548658237420E+00, 0.9558734450579545E-04, 0.9088178357782586E-03, 0.1189201139646886E-01, 0.1197200684467082E-01, 0.1730052806966466E-08, 0.1681378409852773E-04, 0.9598672791582074E-11, 0.1223627446170173E-05, 0.1353283656358167E-06, 0.2970373645586866E-06},
			i0:       1,
			n0:       18,
			pp:       0,
			n0in:     18,
			dmin:     9.5986727915820745E-012,
			dmin1:    1.1875197612370336E-002,
			dmin2:    0.11291768719739126,
			dn:       9.5986727915820745E-012,
			dn1:      1.1875197612370336E-002,
			dn2:      0.11291768719739126,
			tau:      1.2218877946904154E-006,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2383459406360114E+01, 0.2383460331210544E+01, 0.1353283656358167E-06, 0.2970373645586866E-06, 0.1085901506809686E+01, 0.1085889199499347E+01, 0.1986826396898849E-04, 0.1366452649899944E-04, 0.1584986515403004E+01, 0.1578904163506652E+01, 0.3190255011763210E-02, 0.6103442048115082E-02, 0.9412930918090910E+00, 0.8284687778600495E+00, 0.1576208411525267E+00, 0.1160157908485996E+00, 0.1332062774397564E+01, 0.1278855299065544E+01, 0.1143821023097510E+00, 0.2108295383723412E+00, 0.7987861757106404E+00, 0.7226887736910288E+00, 0.3654761154847582E+00, 0.1904807262171572E+00, 0.1274050092339202E+01, 0.1532634164092948E+01, 0.7205613198290631E-01, 0.1068932656188067E+00, 0.8354550922133908E+00, 0.8588297969472432E+00, 0.5621179414582513E-01, 0.4868264913684867E-01, 0.1126935674731953E+01, 0.9646646288612479E+00, 0.3667472533840468E+00, 0.2184840619043245E+00, 0.1562008914947536E+01, 0.1891673744281760E+01, 0.2769195683500124E-01, 0.3708364593761750E-01, 0.1332145183743590E+01, 0.1166419383934860E+01, 0.2620575140703801E+00, 0.1934189785315259E+00, 0.1691083203792688E+01, 0.1804883149953014E+01, 0.8359650824372877E-01, 0.1482587897978486E+00, 0.8790739000820365E+00, 0.9535262710524167E+00, 0.1175682535061811E-01, 0.9145359161143373E-02, 0.1119753245127243E+01, 0.1130094305914509E+01, 0.3991886785572840E-03, 0.1416986451145964E-02, 0.3162690783341116E+00, 0.3154531349760448E+00, 0.4359567385560517E-03, 0.1216353924418735E-02, 0.1138265050331695E+00, 0.1133548658237420E+00, 0.9558734450579545E-04, 0.9088178357782586E-03, 0.1189201139646886E-01, 0.1197200684467082E-01, 0.1730052806966466E-08, 0.1681378409852773E-04, 0.9598672791582074E-11, 0.1223627446170173E-05, 0.1353283656358167E-06, 0.2970373645586866E-06},
			tauOut:   9.5986713933289272E-012,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2383459406360114E+01, 0.2383459541678881E+01, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.1085901506809686E+01, 0.1085921313408604E+01, 0.1986826396898849E-04, 0.2899927470478232E-04, 0.1584986515403004E+01, 0.1588147771130463E+01, 0.3190255011763210E-02, 0.1890859942802736E-02, 0.9412930918090910E+00, 0.1097023073009216E+01, 0.1576208411525267E+00, 0.1913914667196325E+00, 0.1332062774397564E+01, 0.1255053409978084E+01, 0.1143821023097510E+00, 0.7279916643176543E-01, 0.7987861757106404E+00, 0.1091463124754034E+01, 0.3654761154847582E+00, 0.4266153094141972E+00, 0.1274050092339202E+01, 0.9194909148983121E+00, 0.7205613198290631E-01, 0.6547064404326042E-01, 0.8354550922133908E+00, 0.8261962423063568E+00, 0.5621179414582513E-01, 0.7667315937770838E-01, 0.1126935674731953E+01, 0.1417009768728692E+01, 0.3667472533840468E+00, 0.4042756034295816E+00, 0.1562008914947536E+01, 0.1185425268343357E+01, 0.2769195683500124E-01, 0.3111938635974581E-01, 0.1332145183743590E+01, 0.1563083311444625E+01, 0.2620575140703801E+00, 0.2835172362390010E+00, 0.1691083203792688E+01, 0.1491162475787817E+01, 0.8359650824372877E-01, 0.4928202642453804E-01, 0.8790739000820365E+00, 0.8415486989985177E+00, 0.1175682535061811E-01, 0.1564347179719430E-01, 0.1119753245127243E+01, 0.1104508961999007E+01, 0.3991886785572840E-03, 0.1143051254380294E-03, 0.3162690783341116E+00, 0.3165907299376309E+00, 0.4359567385560517E-03, 0.1567431614478116E-03, 0.1138265050331695E+00, 0.1137653492066288E+00, 0.9558734450579545E-04, 0.9991845479738400E-05, 0.1189201139646886E-01, 0.1188202127144326E-01, 0.1730052806966466E-08, 0.1397591405272086E-17, 0.9598672791582074E-11, 0.6617412588681544E-21, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.2818174710670554E-01},
			i0:       1,
			n0:       18,
			pp:       1,
			n0in:     18,
			dmin:     6.6174125886815435E-022,
			dmin1:    1.1882019541390456E-002,
			dmin2:    0.11366976186212303,
			dn:       6.6174125886815435E-022,
			dn1:      1.1882019541390456E-002,
			dn2:      0.11366976186212303,
			tau:      9.5986713933289272E-012,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2383459406360114E+01, 0.2383459541678881E+01, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.1085901506809686E+01, 0.1085921313408604E+01, 0.1986826396898849E-04, 0.2899927470478232E-04, 0.1584986515403004E+01, 0.1588147771130463E+01, 0.3190255011763210E-02, 0.1890859942802736E-02, 0.9412930918090910E+00, 0.1097023073009216E+01, 0.1576208411525267E+00, 0.1913914667196325E+00, 0.1332062774397564E+01, 0.1255053409978084E+01, 0.1143821023097510E+00, 0.7279916643176543E-01, 0.7987861757106404E+00, 0.1091463124754034E+01, 0.3654761154847582E+00, 0.4266153094141972E+00, 0.1274050092339202E+01, 0.9194909148983121E+00, 0.7205613198290631E-01, 0.6547064404326042E-01, 0.8354550922133908E+00, 0.8261962423063568E+00, 0.5621179414582513E-01, 0.7667315937770838E-01, 0.1126935674731953E+01, 0.1417009768728692E+01, 0.3667472533840468E+00, 0.4042756034295816E+00, 0.1562008914947536E+01, 0.1185425268343357E+01, 0.2769195683500124E-01, 0.3111938635974581E-01, 0.1332145183743590E+01, 0.1563083311444625E+01, 0.2620575140703801E+00, 0.2835172362390010E+00, 0.1691083203792688E+01, 0.1491162475787817E+01, 0.8359650824372877E-01, 0.4928202642453804E-01, 0.8790739000820365E+00, 0.8415486989985177E+00, 0.1175682535061811E-01, 0.1564347179719430E-01, 0.1119753245127243E+01, 0.1104508961999007E+01, 0.3991886785572840E-03, 0.1143051254380294E-03, 0.3162690783341116E+00, 0.3165907299376309E+00, 0.4359567385560517E-03, 0.1567431614478116E-03, 0.1138265050331695E+00, 0.1137653492066288E+00, 0.9558734450579545E-04, 0.9991845479738400E-05, 0.1189201139646886E-01, 0.1188202127144326E-01, 0.1730052806966466E-08, 0.1397591405272086E-17, 0.9598672791582074E-11, 0.6617412588681544E-21, 0.1353283656358167E-06, 0.6165545233233256E-07, 0.2818174710670554E-01},
			tauOut:   6.6174125886815426E-022,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2383459603334333E+01, 0.2383459541678881E+01, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.1085950284592642E+01, 0.1085921313408604E+01, 0.4240998334843304E-04, 0.2899927470478232E-04, 0.1589996221089918E+01, 0.1588147771130463E+01, 0.1304604978030436E-02, 0.1890859942802736E-02, 0.1287109934750819E+01, 0.1097023073009216E+01, 0.1866247058326724E+00, 0.1913914667196325E+00, 0.1141227870577177E+01, 0.1255053409978084E+01, 0.6962466280543776E-01, 0.7279916643176543E-01, 0.1448453771362794E+01, 0.1091463124754034E+01, 0.2708190685256155E+00, 0.4266153094141972E+00, 0.7141424904159570E+00, 0.9194909148983121E+00, 0.7574342769943966E-01, 0.6547064404326042E-01, 0.8271259739846255E+00, 0.8261962423063568E+00, 0.1313543755784947E+00, 0.7667315937770838E-01, 0.1689930996579779E+01, 0.1417009768728692E+01, 0.2835846650840220E+00, 0.4042756034295816E+00, 0.9329599896190804E+00, 0.1185425268343357E+01, 0.5213749144931325E-01, 0.3111938635974581E-01, 0.1794463056234313E+01, 0.1563083311444625E+01, 0.2355970842920850E+00, 0.2835172362390010E+00, 0.1304847417920270E+01, 0.1491162475787817E+01, 0.3178396542921673E-01, 0.4928202642453804E-01, 0.8254082053664953E+00, 0.8415486989985177E+00, 0.2093310277804656E-01, 0.1564347179719430E-01, 0.1083690164346398E+01, 0.1104508961999007E+01, 0.3339325601415243E-04, 0.1143051254380294E-03, 0.3167140798430647E+00, 0.3165907299376309E+00, 0.5630296103885627E-04, 0.1567431614478116E-03, 0.1137190380910697E+00, 0.1137653492066288E+00, 0.1044005669799533E-05, 0.9991845479738400E-05, 0.1188097726577346E-01, 0.1188202127144326E-01, 0.7784240935906335E-37, 0.1397591405272086E-17},
			i0:       1,
			n0:       17,
			pp:       0,
			n0in:     18,
			dmin:     6.6174125886815426E-022,
			dmin1:    1.1880977265773463E-002,
			dmin2:    0.11370904624558997,
			dn:       6.6174125886815426E-022,
			dn1:      1.1880977265773463E-002,
			dn2:      0.11370904624558997,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2383459603334333E+01, 0.2383459541678881E+01, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.1085950284592642E+01, 0.1085921313408604E+01, 0.4240998334843304E-04, 0.2899927470478232E-04, 0.1589996221089918E+01, 0.1588147771130463E+01, 0.1304604978030436E-02, 0.1890859942802736E-02, 0.1287109934750819E+01, 0.1097023073009216E+01, 0.1866247058326724E+00, 0.1913914667196325E+00, 0.1141227870577177E+01, 0.1255053409978084E+01, 0.6962466280543776E-01, 0.7279916643176543E-01, 0.1448453771362794E+01, 0.1091463124754034E+01, 0.2708190685256155E+00, 0.4266153094141972E+00, 0.7141424904159570E+00, 0.9194909148983121E+00, 0.7574342769943966E-01, 0.6547064404326042E-01, 0.8271259739846255E+00, 0.8261962423063568E+00, 0.1313543755784947E+00, 0.7667315937770838E-01, 0.1689930996579779E+01, 0.1417009768728692E+01, 0.2835846650840220E+00, 0.4042756034295816E+00, 0.9329599896190804E+00, 0.1185425268343357E+01, 0.5213749144931325E-01, 0.3111938635974581E-01, 0.1794463056234313E+01, 0.1563083311444625E+01, 0.2355970842920850E+00, 0.2835172362390010E+00, 0.1304847417920270E+01, 0.1491162475787817E+01, 0.3178396542921673E-01, 0.4928202642453804E-01, 0.8254082053664953E+00, 0.8415486989985177E+00, 0.2093310277804656E-01, 0.1564347179719430E-01, 0.1083690164346398E+01, 0.1104508961999007E+01, 0.3339325601415243E-04, 0.1143051254380294E-03, 0.3167140798430647E+00, 0.3165907299376309E+00, 0.5630296103885627E-04, 0.1567431614478116E-03, 0.1137190380910697E+00, 0.1137653492066288E+00, 0.1044005669799533E-05, 0.9991845479738400E-05, 0.1188097726577346E-01, 0.1188202127144326E-01, 0.7784240935906335E-37, 0.1397591405272086E-17},
			tauOut:   1.1880832155707781E-002,
			ttypeOut: -7,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2383459603334333E+01, 0.2371578799269292E+01, 0.2809066689524106E-07, 0.1286276792433928E-07, 0.1085950284592642E+01, 0.1074111849557515E+01, 0.4240998334843304E-04, 0.6277904232066118E-04, 0.1589996221089918E+01, 0.1579357214869920E+01, 0.1304604978030436E-02, 0.1063198377376995E-02, 0.1287109934750819E+01, 0.1460790610050406E+01, 0.1866247058326724E+00, 0.1457986614708343E+00, 0.1141227870577177E+01, 0.1053173039756072E+01, 0.6962466280543776E-01, 0.9575644420574685E-01, 0.1448453771362794E+01, 0.1611635563526955E+01, 0.2708190685256155E+00, 0.1200044280642223E+00, 0.7141424904159570E+00, 0.6580006578954666E+00, 0.7574342769943966E-01, 0.9521169265880262E-01, 0.8271259739846255E+00, 0.8513878247486099E+00, 0.1313543755784947E+00, 0.2607270439790754E+00, 0.1689930996579779E+01, 0.1700907785529018E+01, 0.2835846650840220E+00, 0.1555482010511415E+00, 0.9329599896190804E+00, 0.8176684478615445E+00, 0.5213749144931325E-01, 0.1144214412274440E+00, 0.1794463056234313E+01, 0.1903757867143247E+01, 0.2355970842920850E+00, 0.1614796988702029E+00, 0.1304847417920270E+01, 0.1163270852323576E+01, 0.3178396542921673E-01, 0.2255256874351997E-01, 0.8254082053664953E+00, 0.8119079072453143E+00, 0.2093310277804656E-01, 0.2794035799797570E-01, 0.1083690164346398E+01, 0.1043902367448729E+01, 0.3339325601415243E-04, 0.1013132519024162E-04, 0.3167140798430647E+00, 0.3048794193232055E+00, 0.5630296103885627E-04, 0.2100082250625822E-04, 0.1137190380910697E+00, 0.1018182491185255E+00, 0.1044005669799533E-05, 0.1218230301111122E-06, 0.1188097726577346E-01, 0.2328703557243073E-07, 0.7784240935906335E-37, 0.1286276792433928E-07, 0.9751785856405315E-01},
			i0:       1,
			n0:       17,
			pp:       1,
			n0in:     17,
			dmin:     2.3287035572430725E-008,
			dmin1:    0.10181720511285566,
			dmin2:    0.30482311636216664,
			dn:       2.3287035572430725E-008,
			dn1:      0.10181720511285566,
			dn2:      0.30482311636216664,
			tau:      1.1880832155707781E-002,
			ttype:    -7,
			g:        0.0000000000000000,
			zOut:     []float64{0.2383459603334333E+01, 0.2371578799269292E+01, 0.2809066689524106E-07, 0.1286276792433928E-07, 0.1085950284592642E+01, 0.1074111849557515E+01, 0.4240998334843304E-04, 0.6277904232066118E-04, 0.1589996221089918E+01, 0.1579357214869920E+01, 0.1304604978030436E-02, 0.1063198377376995E-02, 0.1287109934750819E+01, 0.1460790610050406E+01, 0.1866247058326724E+00, 0.1457986614708343E+00, 0.1141227870577177E+01, 0.1053173039756072E+01, 0.6962466280543776E-01, 0.9575644420574685E-01, 0.1448453771362794E+01, 0.1611635563526955E+01, 0.2708190685256155E+00, 0.1200044280642223E+00, 0.7141424904159570E+00, 0.6580006578954666E+00, 0.7574342769943966E-01, 0.9521169265880262E-01, 0.8271259739846255E+00, 0.8513878247486099E+00, 0.1313543755784947E+00, 0.2607270439790754E+00, 0.1689930996579779E+01, 0.1700907785529018E+01, 0.2835846650840220E+00, 0.1555482010511415E+00, 0.9329599896190804E+00, 0.8176684478615445E+00, 0.5213749144931325E-01, 0.1144214412274440E+00, 0.1794463056234313E+01, 0.1903757867143247E+01, 0.2355970842920850E+00, 0.1614796988702029E+00, 0.1304847417920270E+01, 0.1163270852323576E+01, 0.3178396542921673E-01, 0.2255256874351997E-01, 0.8254082053664953E+00, 0.8119079072453143E+00, 0.2093310277804656E-01, 0.2794035799797570E-01, 0.1083690164346398E+01, 0.1043902367448729E+01, 0.3339325601415243E-04, 0.1013132519024162E-04, 0.3167140798430647E+00, 0.3048794193232055E+00, 0.5630296103885627E-04, 0.2100082250625822E-04, 0.1137190380910697E+00, 0.1018182491185255E+00, 0.1044005669799533E-05, 0.1218230301111122E-06, 0.1188097726577346E-01, 0.2328703557243073E-07, 0.7784240935906335E-37, 0.1286276792433928E-07, 0.9751785856405315E-01},
			tauOut:   2.3287007705477136E-008,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2371578788845052E+01, 0.2371578799269292E+01, 0.5825676764620370E-08, 0.1286276792433928E-07, 0.1074174599487151E+01, 0.1074111849557515E+01, 0.9230392664199863E-04, 0.6277904232066118E-04, 0.1580328086033647E+01, 0.1579357214869920E+01, 0.9827770701659698E-03, 0.1063198377376995E-02, 0.1605606471164067E+01, 0.1460790610050406E+01, 0.9563440497488795E-01, 0.1457986614708343E+00, 0.1053295055699924E+01, 0.1053173039756072E+01, 0.1465159169633776E+00, 0.9575644420574685E-01, 0.1585124051340792E+01, 0.1611635563526955E+01, 0.4981502397231050E-01, 0.1200044280642223E+00, 0.7033973032949510E+00, 0.6580006578954666E+00, 0.1152436546510613E+00, 0.9521169265880262E-01, 0.9968711907896164E+00, 0.8513878247486099E+00, 0.4448645553200346E+00, 0.2607270439790754E+00, 0.1411591407973117E+01, 0.1700907785529018E+01, 0.9010174998427344E-01, 0.1555482010511415E+00, 0.8419881158177074E+00, 0.8176684478615445E+00, 0.2587099684834221E+00, 0.1144214412274440E+00, 0.1806527574243020E+01, 0.1903757867143247E+01, 0.1039810460775319E+00, 0.1614796988702029E+00, 0.1081842351702556E+01, 0.1163270852323576E+01, 0.1692539477932339E-01, 0.2255256874351997E-01, 0.8229228471769590E+00, 0.8119079072453143E+00, 0.3544318396494814E-01, 0.2794035799797570E-01, 0.1008469291521964E+01, 0.1043902367448729E+01, 0.3062892015595061E-05, 0.1013132519024162E-04, 0.3048973339666884E+00, 0.3048794193232055E+00, 0.7013072071892081E-05, 0.2100082250625822E-04, 0.1018113345824760E+00, 0.1018182491185255E+00, 0.2786425742647189E-13, 0.1218230301111122E-06, 0.2696165428114542E-17, 0.2328703557243073E-07, 0.5825676764620370E-08, 0.1286276792433928E-07},
			i0:       1,
			n0:       17,
			pp:       0,
			n0in:     17,
			dmin:     2.6961654281145418E-018,
			dmin1:    0.10181121275944585,
			dmin2:    0.30487633314418217,
			dn:       2.6961654281145418E-018,
			dn1:      0.10181121275944585,
			dn2:      0.30487633314418217,
			tau:      2.3287007705477136E-008,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2371578788845052E+01, 0.2371578799269292E+01, 0.5825676764620370E-08, 0.1286276792433928E-07, 0.1074174599487151E+01, 0.1074111849557515E+01, 0.9230392664199863E-04, 0.6277904232066118E-04, 0.1580328086033647E+01, 0.1579357214869920E+01, 0.9827770701659698E-03, 0.1063198377376995E-02, 0.1605606471164067E+01, 0.1460790610050406E+01, 0.9563440497488795E-01, 0.1457986614708343E+00, 0.1053295055699924E+01, 0.1053173039756072E+01, 0.1465159169633776E+00, 0.9575644420574685E-01, 0.1585124051340792E+01, 0.1611635563526955E+01, 0.4981502397231050E-01, 0.1200044280642223E+00, 0.7033973032949510E+00, 0.6580006578954666E+00, 0.1152436546510613E+00, 0.9521169265880262E-01, 0.9968711907896164E+00, 0.8513878247486099E+00, 0.4448645553200346E+00, 0.2607270439790754E+00, 0.1411591407973117E+01, 0.1700907785529018E+01, 0.9010174998427344E-01, 0.1555482010511415E+00, 0.8419881158177074E+00, 0.8176684478615445E+00, 0.2587099684834221E+00, 0.1144214412274440E+00, 0.1806527574243020E+01, 0.1903757867143247E+01, 0.1039810460775319E+00, 0.1614796988702029E+00, 0.1081842351702556E+01, 0.1163270852323576E+01, 0.1692539477932339E-01, 0.2255256874351997E-01, 0.8229228471769590E+00, 0.8119079072453143E+00, 0.3544318396494814E-01, 0.2794035799797570E-01, 0.1008469291521964E+01, 0.1043902367448729E+01, 0.3062892015595061E-05, 0.1013132519024162E-04, 0.3048973339666884E+00, 0.3048794193232055E+00, 0.7013072071892081E-05, 0.2100082250625822E-04, 0.1018113345824760E+00, 0.1018182491185255E+00, 0.2786425742647189E-13, 0.1218230301111122E-06, 0.2696165428114542E-17, 0.2328703557243073E-07, 0.5825676764620370E-08, 0.1286276792433928E-07},
			tauOut:   2.6961654281138038E-018,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2371578788845052E+01, 0.2371578794670729E+01, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.1074174599487151E+01, 0.1074266900775131E+01, 0.9230392664199863E-04, 0.1357860766428602E-03, 0.1580328086033647E+01, 0.1581175077027170E+01, 0.9827770701659698E-03, 0.9979623676695663E-03, 0.1605606471164067E+01, 0.1700242913771285E+01, 0.9563440497488795E-01, 0.5924520849284005E-01, 0.1053295055699924E+01, 0.1140565764170461E+01, 0.1465159169633776E+00, 0.2036234219705987E+00, 0.1585124051340792E+01, 0.1431315653342504E+01, 0.4981502397231050E-01, 0.2448080089382756E-01, 0.7033973032949510E+00, 0.7941601570521848E+00, 0.1152436546510613E+00, 0.1446598374682775E+00, 0.9968711907896164E+00, 0.1297075908641373E+01, 0.4448645553200346E+00, 0.4841405038964208E+00, 0.1411591407973117E+01, 0.1017552654060970E+01, 0.9010174998427344E-01, 0.7455594793877945E-01, 0.8419881158177074E+00, 0.1026142136362350E+01, 0.2587099684834221E+00, 0.4554599945126984E+00, 0.1806527574243020E+01, 0.1455048625807853E+01, 0.1039810460775319E+00, 0.7731088667813635E-01, 0.1081842351702556E+01, 0.1021456859803743E+01, 0.1692539477932339E-01, 0.1363571444815687E-01, 0.8229228471769590E+00, 0.8447303166937503E+00, 0.3544318396494814E-01, 0.4231334180394086E-01, 0.1008469291521964E+01, 0.9661590126100381E+00, 0.3062892015595061E-05, 0.9665775484099522E-06, 0.3048973339666884E+00, 0.3049033804612119E+00, 0.7013072071892081E-05, 0.2341758973227438E-05, 0.1018113345824760E+00, 0.1018089928235306E+00, 0.2786425742647189E-13, 0.7379175991216932E-30, 0.1093987140067686E+00},
			i0:       1,
			n0:       16,
			pp:       1,
			n0in:     17,
			dmin:     2.6961654281138038E-018,
			dmin1:    0.10180899282350273,
			dmin2:    0.30489636738914000,
			dn:       2.6961654281138038E-018,
			dn1:      0.10180899282350273,
			dn2:      0.30489636738914000,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2371578788845052E+01, 0.2371578794670729E+01, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.1074174599487151E+01, 0.1074266900775131E+01, 0.9230392664199863E-04, 0.1357860766428602E-03, 0.1580328086033647E+01, 0.1581175077027170E+01, 0.9827770701659698E-03, 0.9979623676695663E-03, 0.1605606471164067E+01, 0.1700242913771285E+01, 0.9563440497488795E-01, 0.5924520849284005E-01, 0.1053295055699924E+01, 0.1140565764170461E+01, 0.1465159169633776E+00, 0.2036234219705987E+00, 0.1585124051340792E+01, 0.1431315653342504E+01, 0.4981502397231050E-01, 0.2448080089382756E-01, 0.7033973032949510E+00, 0.7941601570521848E+00, 0.1152436546510613E+00, 0.1446598374682775E+00, 0.9968711907896164E+00, 0.1297075908641373E+01, 0.4448645553200346E+00, 0.4841405038964208E+00, 0.1411591407973117E+01, 0.1017552654060970E+01, 0.9010174998427344E-01, 0.7455594793877945E-01, 0.8419881158177074E+00, 0.1026142136362350E+01, 0.2587099684834221E+00, 0.4554599945126984E+00, 0.1806527574243020E+01, 0.1455048625807853E+01, 0.1039810460775319E+00, 0.7731088667813635E-01, 0.1081842351702556E+01, 0.1021456859803743E+01, 0.1692539477932339E-01, 0.1363571444815687E-01, 0.8229228471769590E+00, 0.8447303166937503E+00, 0.3544318396494814E-01, 0.4231334180394086E-01, 0.1008469291521964E+01, 0.9661590126100381E+00, 0.3062892015595061E-05, 0.9665775484099522E-06, 0.3048973339666884E+00, 0.3049033804612119E+00, 0.7013072071892081E-05, 0.2341758973227438E-05, 0.1018113345824760E+00, 0.1018089928235306E+00, 0.2786425742647189E-13, 0.7379175991216932E-30, 0.1093987140067686E+00},
			tauOut:   0.10180650470263587,
			ttypeOut: -7,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2269772292606755E+01, 0.2371578794670729E+01, 0.1248859578385863E-08, 0.2638661645752538E-08, 0.9725961809002787E+00, 0.1074266900775131E+01, 0.2207509801202943E-03, 0.1357860766428602E-03, 0.1480145783712083E+01, 0.1581175077027170E+01, 0.1146359002276934E-02, 0.9979623676695663E-03, 0.1656535258559212E+01, 0.1700242913771285E+01, 0.4079180092843103E-01, 0.5924520849284005E-01, 0.1201590880509993E+01, 0.1140565764170461E+01, 0.2425529321011353E+00, 0.2036234219705987E+00, 0.1111437017432560E+01, 0.1431315653342504E+01, 0.1749237822536809E-01, 0.2448080089382756E-01, 0.8195211115924584E+00, 0.7941601570521848E+00, 0.2289566278084960E+00, 0.1446598374682775E+00, 0.1450453280026662E+01, 0.1297075908641373E+01, 0.3396444831847066E+00, 0.4841405038964208E+00, 0.6506576141124067E+00, 0.1017552654060970E+01, 0.1175810411452473E+00, 0.7455594793877945E-01, 0.1262214585027166E+01, 0.1026142136362350E+01, 0.5250426092262996E+00, 0.4554599945126984E+00, 0.9055103985570538E+00, 0.1455048625807853E+01, 0.8721019179982008E-01, 0.7731088667813635E-01, 0.8460758777494440E+00, 0.1021456859803743E+01, 0.1361402882064932E-01, 0.1363571444815687E-01, 0.7716231249744060E+00, 0.8447303166937503E+00, 0.5298106707064092E-01, 0.4231334180394086E-01, 0.8113724074143097E+00, 0.9661590126100381E+00, 0.3632274887524200E-06, 0.9665775484099522E-06, 0.2030988542900605E+00, 0.3049033804612119E+00, 0.1173872316183806E-05, 0.2341758973227438E-05, 0.1314248578535016E-05, 0.1018089928235306E+00, 0.1248859578385863E-08, 0.2638661645752538E-08},
			i0:       1,
			n0:       16,
			pp:       0,
			n0in:     16,
			dmin:     1.3142485785350155E-006,
			dmin1:    0.20309651253108729,
			dmin2:    0.57610166617362735,
			dn:       1.3142485785350155E-006,
			dn1:      0.20309651253108729,
			dn2:      0.81137144083676127,
			tau:      0.10180650470263587,
			ttype:    -7,
			g:        0.0000000000000000,
			zOut:     []float64{0.2269772292606755E+01, 0.2371578794670729E+01, 0.1248859578385863E-08, 0.2638661645752538E-08, 0.9725961809002787E+00, 0.1074266900775131E+01, 0.2207509801202943E-03, 0.1357860766428602E-03, 0.1480145783712083E+01, 0.1581175077027170E+01, 0.1146359002276934E-02, 0.9979623676695663E-03, 0.1656535258559212E+01, 0.1700242913771285E+01, 0.4079180092843103E-01, 0.5924520849284005E-01, 0.1201590880509993E+01, 0.1140565764170461E+01, 0.2425529321011353E+00, 0.2036234219705987E+00, 0.1111437017432560E+01, 0.1431315653342504E+01, 0.1749237822536809E-01, 0.2448080089382756E-01, 0.8195211115924584E+00, 0.7941601570521848E+00, 0.2289566278084960E+00, 0.1446598374682775E+00, 0.1450453280026662E+01, 0.1297075908641373E+01, 0.3396444831847066E+00, 0.4841405038964208E+00, 0.6506576141124067E+00, 0.1017552654060970E+01, 0.1175810411452473E+00, 0.7455594793877945E-01, 0.1262214585027166E+01, 0.1026142136362350E+01, 0.5250426092262996E+00, 0.4554599945126984E+00, 0.9055103985570538E+00, 0.1455048625807853E+01, 0.8721019179982008E-01, 0.7731088667813635E-01, 0.8460758777494440E+00, 0.1021456859803743E+01, 0.1361402882064932E-01, 0.1363571444815687E-01, 0.7716231249744060E+00, 0.8447303166937503E+00, 0.5298106707064092E-01, 0.4231334180394086E-01, 0.8113724074143097E+00, 0.9661590126100381E+00, 0.3632274887524200E-06, 0.9665775484099522E-06, 0.2030988542900605E+00, 0.3049033804612119E+00, 0.1173872316183806E-05, 0.2341758973227438E-05, 0.1314248578535016E-05, 0.1018089928235306E+00, 0.1248859578385863E-08, 0.2638661645752538E-08},
			tauOut:   1.3142409824136984E-006,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2269772292606755E+01, 0.2269770979614632E+01, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.9725961809002787E+00, 0.9728156171042807E+00, 0.2207509801202943E-03, 0.3358741643642203E-03, 0.1480145783712083E+01, 0.1480954954309013E+01, 0.1146359002276934E-02, 0.1282269997958535E-02, 0.1656535258559212E+01, 0.1696043475248702E+01, 0.4079180092843103E-01, 0.2889964597634762E-01, 0.1201590880509993E+01, 0.1415242852393798E+01, 0.2425529321011353E+00, 0.1904848393814713E+00, 0.1111437017432560E+01, 0.9384432420354740E+00, 0.1749237822536809E-01, 0.1527569554079381E-01, 0.8195211115924584E+00, 0.1033200729619178E+01, 0.2289566278084960E+00, 0.3214195289148511E+00, 0.1450453280026662E+01, 0.1468676920055535E+01, 0.3396444831847066E+00, 0.1504703083827628E+00, 0.6506576141124067E+00, 0.6177670326339089E+00, 0.1175810411452473E+00, 0.2402402478867145E+00, 0.1262214585027166E+01, 0.1547015632125768E+01, 0.5250426092262996E+00, 0.3073217441808570E+00, 0.9055103985570538E+00, 0.6853975319350345E+00, 0.8721019179982008E-01, 0.1076549537133787E+00, 0.8460758777494440E+00, 0.7520336386157322E+00, 0.1361402882064932E-01, 0.1396865635082151E-01, 0.7716231249744060E+00, 0.8106342214532430E+00, 0.5298106707064092E-01, 0.5302931309687425E-01, 0.8113724074143097E+00, 0.7583421433039418E+00, 0.3632274887524200E-06, 0.9727942389020712E-07, 0.2030988542900605E+00, 0.2030986166419704E+00, 0.1173872316183806E-05, 0.7596112905317353E-11, 0.1314248578535016E-05, 0.8411884199182457E-17, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.1093987140067686E+00},
			i0:       1,
			n0:       16,
			pp:       1,
			n0in:     16,
			dmin:     8.4118841991824567E-018,
			dmin1:    0.20309744276965425,
			dmin2:    0.50018599148866161,
			dn:       8.4118841991824567E-018,
			dn1:      0.20309744276965425,
			dn2:      0.75834178007645303,
			tau:      1.3142409824136984E-006,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2269772292606755E+01, 0.2269770979614632E+01, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.9725961809002787E+00, 0.9728156171042807E+00, 0.2207509801202943E-03, 0.3358741643642203E-03, 0.1480145783712083E+01, 0.1480954954309013E+01, 0.1146359002276934E-02, 0.1282269997958535E-02, 0.1656535258559212E+01, 0.1696043475248702E+01, 0.4079180092843103E-01, 0.2889964597634762E-01, 0.1201590880509993E+01, 0.1415242852393798E+01, 0.2425529321011353E+00, 0.1904848393814713E+00, 0.1111437017432560E+01, 0.9384432420354740E+00, 0.1749237822536809E-01, 0.1527569554079381E-01, 0.8195211115924584E+00, 0.1033200729619178E+01, 0.2289566278084960E+00, 0.3214195289148511E+00, 0.1450453280026662E+01, 0.1468676920055535E+01, 0.3396444831847066E+00, 0.1504703083827628E+00, 0.6506576141124067E+00, 0.6177670326339089E+00, 0.1175810411452473E+00, 0.2402402478867145E+00, 0.1262214585027166E+01, 0.1547015632125768E+01, 0.5250426092262996E+00, 0.3073217441808570E+00, 0.9055103985570538E+00, 0.6853975319350345E+00, 0.8721019179982008E-01, 0.1076549537133787E+00, 0.8460758777494440E+00, 0.7520336386157322E+00, 0.1361402882064932E-01, 0.1396865635082151E-01, 0.7716231249744060E+00, 0.8106342214532430E+00, 0.5298106707064092E-01, 0.5302931309687425E-01, 0.8113724074143097E+00, 0.7583421433039418E+00, 0.3632274887524200E-06, 0.9727942389020712E-07, 0.2030988542900605E+00, 0.2030986166419704E+00, 0.1173872316183806E-05, 0.7596112905317353E-11, 0.1314248578535016E-05, 0.8411884199182457E-17, 0.1248859578385863E-08, 0.5351359530665278E-09, 0.1093987140067686E+00},
			tauOut:   8.4118841988678429E-018,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2269770980149768E+01, 0.2269770979614632E+01, 0.2293573303077261E-09, 0.5351359530665278E-09, 0.9731514910392876E+00, 0.9728156171042807E+00, 0.5111377954200868E-03, 0.3358741643642203E-03, 0.1481726086511552E+01, 0.1480954954309013E+01, 0.1467737987028945E-02, 0.1282269997958535E-02, 0.1723475383238021E+01, 0.1696043475248702E+01, 0.2373112943910766E-01, 0.2889964597634762E-01, 0.1581996562336162E+01, 0.1415242852393798E+01, 0.1129959536471923E+00, 0.1904848393814713E+00, 0.8407229839290754E+00, 0.9384432420354740E+00, 0.1877296098701645E-01, 0.1527569554079381E-01, 0.1335847297547013E+01, 0.1033200729619178E+01, 0.3533797947109679E+00, 0.3214195289148511E+00, 0.1265767433727330E+01, 0.1468676920055535E+01, 0.7343813202351114E-01, 0.1504703083827628E+00, 0.7845691484971122E+00, 0.6177670326339089E+00, 0.4737063898809231E+00, 0.2402402478867145E+00, 0.1380630986425702E+01, 0.1547015632125768E+01, 0.1525661578238559E+00, 0.3073217441808570E+00, 0.6404863278245572E+00, 0.6853975319350345E+00, 0.1264041760751794E+00, 0.1076549537133787E+00, 0.6395981188913744E+00, 0.7520336386157322E+00, 0.1770404028911661E-01, 0.1396865635082151E-01, 0.8459594942610007E+00, 0.8106342214532430E+00, 0.4753698401003136E-01, 0.5302931309687425E-01, 0.7108052565733343E+00, 0.7583421433039418E+00, 0.2779568135873871E-07, 0.9727942389020712E-07, 0.2030985888538852E+00, 0.2030986166419704E+00, 0.3146138162949754E-27, 0.7596112905317353E-11, 0.8411884198867843E-17, 0.8411884199182457E-17, 0.2293573303077261E-09, 0.5351359530665278E-09},
			i0:       1,
			n0:       16,
			pp:       0,
			n0in:     16,
			dmin:     8.4118841988678429E-018,
			dmin1:    0.20309858884628909,
			dmin2:    0.53283137411117854,
			dn:       8.4118841988678429E-018,
			dn1:      0.20309858884628909,
			dn2:      0.71080515929391042,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2269770980149768E+01, 0.2269770979614632E+01, 0.2293573303077261E-09, 0.5351359530665278E-09, 0.9731514910392876E+00, 0.9728156171042807E+00, 0.5111377954200868E-03, 0.3358741643642203E-03, 0.1481726086511552E+01, 0.1480954954309013E+01, 0.1467737987028945E-02, 0.1282269997958535E-02, 0.1723475383238021E+01, 0.1696043475248702E+01, 0.2373112943910766E-01, 0.2889964597634762E-01, 0.1581996562336162E+01, 0.1415242852393798E+01, 0.1129959536471923E+00, 0.1904848393814713E+00, 0.8407229839290754E+00, 0.9384432420354740E+00, 0.1877296098701645E-01, 0.1527569554079381E-01, 0.1335847297547013E+01, 0.1033200729619178E+01, 0.3533797947109679E+00, 0.3214195289148511E+00, 0.1265767433727330E+01, 0.1468676920055535E+01, 0.7343813202351114E-01, 0.1504703083827628E+00, 0.7845691484971122E+00, 0.6177670326339089E+00, 0.4737063898809231E+00, 0.2402402478867145E+00, 0.1380630986425702E+01, 0.1547015632125768E+01, 0.1525661578238559E+00, 0.3073217441808570E+00, 0.6404863278245572E+00, 0.6853975319350345E+00, 0.1264041760751794E+00, 0.1076549537133787E+00, 0.6395981188913744E+00, 0.7520336386157322E+00, 0.1770404028911661E-01, 0.1396865635082151E-01, 0.8459594942610007E+00, 0.8106342214532430E+00, 0.4753698401003136E-01, 0.5302931309687425E-01, 0.7108052565733343E+00, 0.7583421433039418E+00, 0.2779568135873871E-07, 0.9727942389020712E-07, 0.2030985888538852E+00, 0.2030986166419704E+00, 0.3146138162949754E-27, 0.7596112905317353E-11, 0.8411884198867843E-17, 0.8411884199182457E-17, 0.2293573303077261E-09, 0.5351359530665278E-09},
			tauOut:   8.4118841988678429E-018,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2269770980149768E+01, 0.2269770980379126E+01, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.9731514910392876E+00, 0.9736626287363720E+00, 0.5111377954200868E-03, 0.7778528033461282E-03, 0.1481726086511552E+01, 0.1482415971695234E+01, 0.1467737987028945E-02, 0.1706410574351102E-02, 0.1723475383238021E+01, 0.1745500102102777E+01, 0.2373112943910766E-01, 0.2150819994097728E-01, 0.1581996562336162E+01, 0.1673484316042377E+01, 0.1129959536471923E+00, 0.5676676764251993E-01, 0.8407229839290754E+00, 0.8027291772735718E+00, 0.1877296098701645E-01, 0.3124068479314146E-01, 0.1335847297547013E+01, 0.1657986407464839E+01, 0.3533797947109679E+00, 0.2697830536296953E+00, 0.1265767433727330E+01, 0.1069422512121146E+01, 0.7343813202351114E-01, 0.5387701498318342E-01, 0.7845691484971122E+00, 0.1204398523394852E+01, 0.4737063898809231E+00, 0.5430210247136315E+00, 0.1380630986425702E+01, 0.9901761195359265E+00, 0.1525661578238559E+00, 0.9868601781741700E-01, 0.6404863278245572E+00, 0.6682044860823195E+00, 0.1264041760751794E+00, 0.1209927124430272E+00, 0.6395981188913744E+00, 0.5363094467374639E+00, 0.1770404028911661E-01, 0.2792585709699239E-01, 0.8459594942610007E+00, 0.8655706211740396E+00, 0.4753698401003136E-01, 0.3903729781186600E-01, 0.7108052565733343E+00, 0.6717679865571495E+00, 0.2779568135873871E-07, 0.8403591378512072E-08, 0.2030985888538852E+00, 0.2030985804502939E+00, 0.3146138162949754E-27, 0.1303059324279677E-43, 0.2112065329503869E+00},
			i0:       1,
			n0:       15,
			pp:       1,
			n0in:     16,
			dmin:     8.4118841988678429E-018,
			dmin1:    0.20309858045029386,
			dmin2:    0.51860540644834729,
			dn:       8.4118841988678429E-018,
			dn1:      0.20309858045029386,
			dn2:      0.67176795876146822,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2269770980149768E+01, 0.2269770980379126E+01, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.9731514910392876E+00, 0.9736626287363720E+00, 0.5111377954200868E-03, 0.7778528033461282E-03, 0.1481726086511552E+01, 0.1482415971695234E+01, 0.1467737987028945E-02, 0.1706410574351102E-02, 0.1723475383238021E+01, 0.1745500102102777E+01, 0.2373112943910766E-01, 0.2150819994097728E-01, 0.1581996562336162E+01, 0.1673484316042377E+01, 0.1129959536471923E+00, 0.5676676764251993E-01, 0.8407229839290754E+00, 0.8027291772735718E+00, 0.1877296098701645E-01, 0.3124068479314146E-01, 0.1335847297547013E+01, 0.1657986407464839E+01, 0.3533797947109679E+00, 0.2697830536296953E+00, 0.1265767433727330E+01, 0.1069422512121146E+01, 0.7343813202351114E-01, 0.5387701498318342E-01, 0.7845691484971122E+00, 0.1204398523394852E+01, 0.4737063898809231E+00, 0.5430210247136315E+00, 0.1380630986425702E+01, 0.9901761195359265E+00, 0.1525661578238559E+00, 0.9868601781741700E-01, 0.6404863278245572E+00, 0.6682044860823195E+00, 0.1264041760751794E+00, 0.1209927124430272E+00, 0.6395981188913744E+00, 0.5363094467374639E+00, 0.1770404028911661E-01, 0.2792585709699239E-01, 0.8459594942610007E+00, 0.8655706211740396E+00, 0.4753698401003136E-01, 0.3903729781186600E-01, 0.7108052565733343E+00, 0.6717679865571495E+00, 0.2779568135873871E-07, 0.8403591378512072E-08, 0.2030985888538852E+00, 0.2030985804502939E+00, 0.3146138162949754E-27, 0.1303059324279677E-43, 0.2112065329503869E+00},
			tauOut:   0.10154929022514693,
			ttypeOut: -9,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2168221690252314E+01, 0.2269770980379126E+01, 0.4415865678637858E-10, 0.9833566024906726E-10, 0.8728911912704126E+00, 0.9736626287363720E+00, 0.1321013925721922E-02, 0.7778528033461282E-03, 0.1381252078118717E+01, 0.1482415971695234E+01, 0.2156405683614187E-02, 0.1706410574351102E-02, 0.1663302606134993E+01, 0.1745500102102777E+01, 0.2163985983955575E-01, 0.2150819994097728E-01, 0.1607061933620194E+01, 0.1673484316042377E+01, 0.2835506195054301E-01, 0.5676676764251993E-01, 0.7040655098910235E+00, 0.8027291772735718E+00, 0.7356791380810462E-01, 0.3124068479314146E-01, 0.1752652257061283E+01, 0.1657986407464839E+01, 0.1646145547572216E+00, 0.2697830536296953E+00, 0.8571356821219610E+00, 0.1069422512121146E+01, 0.7570493055431493E-01, 0.5387701498318342E-01, 0.1570165327329021E+01, 0.1204398523394852E+01, 0.3424393862982660E+00, 0.5430210247136315E+00, 0.6448734608299306E+00, 0.9901761195359265E+00, 0.1022564019526126E+00, 0.9868601781741700E-01, 0.5853915063475871E+00, 0.6682044860823195E+00, 0.1108480973262629E+00, 0.1209927124430272E+00, 0.3518379162830466E+00, 0.5363094467374639E+00, 0.6870152520689508E-01, 0.2792585709699239E-01, 0.7343571035538636E+00, 0.8655706211740396E+00, 0.3571015630515456E-01, 0.3903729781186600E-01, 0.5345085484304394E+00, 0.6717679865571495E+00, 0.3193134112956561E-08, 0.8403591378512072E-08, 0.1015492870320128E+00, 0.2030985804502939E+00, 0.4415865678637858E-10, 0.9833566024906726E-10},
			i0:       1,
			n0:       15,
			pp:       0,
			n0in:     15,
			dmin:     0.10154928703201281,
			dmin1:    0.32391205918605420,
			dmin2:    0.32391205918605420,
			dn:       0.10154928703201281,
			dn1:      0.53450854002684800,
			dn2:      0.69531980574199759,
			tau:      0.10154929022514693,
			ttype:    -9,
			g:        0.0000000000000000,
			zOut:     []float64{0.2168221690252314E+01, 0.2269770980379126E+01, 0.4415865678637858E-10, 0.9833566024906726E-10, 0.8728911912704126E+00, 0.9736626287363720E+00, 0.1321013925721922E-02, 0.7778528033461282E-03, 0.1381252078118717E+01, 0.1482415971695234E+01, 0.2156405683614187E-02, 0.1706410574351102E-02, 0.1663302606134993E+01, 0.1745500102102777E+01, 0.2163985983955575E-01, 0.2150819994097728E-01, 0.1607061933620194E+01, 0.1673484316042377E+01, 0.2835506195054301E-01, 0.5676676764251993E-01, 0.7040655098910235E+00, 0.8027291772735718E+00, 0.7356791380810462E-01, 0.3124068479314146E-01, 0.1752652257061283E+01, 0.1657986407464839E+01, 0.1646145547572216E+00, 0.2697830536296953E+00, 0.8571356821219610E+00, 0.1069422512121146E+01, 0.7570493055431493E-01, 0.5387701498318342E-01, 0.1570165327329021E+01, 0.1204398523394852E+01, 0.3424393862982660E+00, 0.5430210247136315E+00, 0.6448734608299306E+00, 0.9901761195359265E+00, 0.1022564019526126E+00, 0.9868601781741700E-01, 0.5853915063475871E+00, 0.6682044860823195E+00, 0.1108480973262629E+00, 0.1209927124430272E+00, 0.3518379162830466E+00, 0.5363094467374639E+00, 0.6870152520689508E-01, 0.2792585709699239E-01, 0.7343571035538636E+00, 0.8655706211740396E+00, 0.3571015630515456E-01, 0.3903729781186600E-01, 0.5345085484304394E+00, 0.6717679865571495E+00, 0.3193134112956561E-08, 0.8403591378512072E-08, 0.1015492870320128E+00, 0.2030985804502939E+00, 0.4415865678637858E-10, 0.9833566024906726E-10},
			tauOut:   0.10154100618138810,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2168221690252314E+01, 0.2066680684115085E+01, 0.4415865678637858E-10, 0.1865101988102620E-10, 0.8728911912704126E+00, 0.7726711989960955E+00, 0.1321013925721922E-02, 0.2361487308570419E-02, 0.1381252078118717E+01, 0.1279505990312372E+01, 0.2156405683614187E-02, 0.2803234389363145E-02, 0.1663302606134993E+01, 0.1580598225403798E+01, 0.2163985983955575E-01, 0.2200217261925751E-01, 0.1607061933620194E+01, 0.1511873816770092E+01, 0.2835506195054301E-01, 0.1320468740761088E-01, 0.7040655098910235E+00, 0.6628877301101292E+00, 0.7356791380810462E-01, 0.1945110224949301E+00, 0.1752652257061283E+01, 0.1621214783142186E+01, 0.1646145547572216E+00, 0.8703165684534678E-01, 0.8571356821219610E+00, 0.7442679496495411E+00, 0.7570493055431493E-01, 0.1597129865933493E+00, 0.1570165327329021E+01, 0.1651350720852550E+01, 0.3424393862982660E+00, 0.1337269360034139E+00, 0.6448734608299306E+00, 0.5118619205977412E+00, 0.1022564019526126E+00, 0.1169456581236225E+00, 0.5853915063475871E+00, 0.4777529393688394E+00, 0.1108480973262629E+00, 0.8163333047984263E-01, 0.3518379162830466E+00, 0.2373651048287110E+00, 0.6870152520689508E-01, 0.2125478936639627E+00, 0.7343571035538636E+00, 0.4559783600136673E+00, 0.3571015630515456E-01, 0.4186028435717903E-01, 0.5345085484304394E+00, 0.3911072610850064E+00, 0.3193134112956561E-08, 0.8290832843879624E-09, 0.1015492870320128E+00, 0.8280021541434701E-05, 0.4415865678637858E-10, 0.1865101988102620E-10, 0.2112065329503869E+00},
			i0:       1,
			n0:       15,
			pp:       1,
			n0in:     15,
			dmin:     8.2800215414347011E-006,
			dmin1:    0.16866357962181588,
			dmin2:    0.16866357962181588,
			dn:       8.2800215414347011E-006,
			dn1:      0.39110725789187228,
			dn2:      0.42026820370851276,
			tau:      0.10154100618138810,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2168221690252314E+01, 0.2066680684115085E+01, 0.4415865678637858E-10, 0.1865101988102620E-10, 0.8728911912704126E+00, 0.7726711989960955E+00, 0.1321013925721922E-02, 0.2361487308570419E-02, 0.1381252078118717E+01, 0.1279505990312372E+01, 0.2156405683614187E-02, 0.2803234389363145E-02, 0.1663302606134993E+01, 0.1580598225403798E+01, 0.2163985983955575E-01, 0.2200217261925751E-01, 0.1607061933620194E+01, 0.1511873816770092E+01, 0.2835506195054301E-01, 0.1320468740761088E-01, 0.7040655098910235E+00, 0.6628877301101292E+00, 0.7356791380810462E-01, 0.1945110224949301E+00, 0.1752652257061283E+01, 0.1621214783142186E+01, 0.1646145547572216E+00, 0.8703165684534678E-01, 0.8571356821219610E+00, 0.7442679496495411E+00, 0.7570493055431493E-01, 0.1597129865933493E+00, 0.1570165327329021E+01, 0.1651350720852550E+01, 0.3424393862982660E+00, 0.1337269360034139E+00, 0.6448734608299306E+00, 0.5118619205977412E+00, 0.1022564019526126E+00, 0.1169456581236225E+00, 0.5853915063475871E+00, 0.4777529393688394E+00, 0.1108480973262629E+00, 0.8163333047984263E-01, 0.3518379162830466E+00, 0.2373651048287110E+00, 0.6870152520689508E-01, 0.2125478936639627E+00, 0.7343571035538636E+00, 0.4559783600136673E+00, 0.3571015630515456E-01, 0.4186028435717903E-01, 0.5345085484304394E+00, 0.3911072610850064E+00, 0.3193134112956561E-08, 0.8290832843879624E-09, 0.1015492870320128E+00, 0.8280021541434701E-05, 0.4415865678637858E-10, 0.1865101988102620E-10, 0.2112065329503869E+00},
			tauOut:   8.2795951083136037E-006,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2066672404538628E+01, 0.2066680684115085E+01, 0.6973096395115281E-11, 0.1865101988102620E-10, 0.7750244067025847E+00, 0.7726711989960955E+00, 0.3898634844569491E-02, 0.2361487308570419E-02, 0.1278402310262057E+01, 0.1279505990312372E+01, 0.3465878671879142E-02, 0.2803234389363145E-02, 0.1599126239756068E+01, 0.1580598225403798E+01, 0.2080167773382635E-01, 0.2200217261925751E-01, 0.1504268546848768E+01, 0.1511873816770092E+01, 0.5818924606767697E-02, 0.1320468740761088E-01, 0.8515715484031834E+00, 0.6628877301101292E+00, 0.3703084558710277E+00, 0.1945110224949301E+00, 0.1337929704521397E+01, 0.1621214783142186E+01, 0.4841425717359337E-01, 0.8703165684534678E-01, 0.8555583994741887E+00, 0.7442679496495411E+00, 0.3082690272254148E+00, 0.1597129865933493E+00, 0.1476800350035441E+01, 0.1651350720852550E+01, 0.4635002036444261E-01, 0.1337269360034139E+00, 0.5824492787618127E+00, 0.5118619205977412E+00, 0.9592445892242578E-01, 0.1169456581236225E+00, 0.4634535313311479E+00, 0.4777529393688394E+00, 0.4180980990954932E-01, 0.8163333047984263E-01, 0.4080949089880160E+00, 0.2373651048287110E+00, 0.2374870106014949E+00, 0.2125478936639627E+00, 0.2603433541742431E+00, 0.4559783600136673E+00, 0.6288565043307552E-01, 0.4186028435717903E-01, 0.3282133318859058E+00, 0.3911072610850064E+00, 0.2091574834858362E-13, 0.8290832843879624E-09, 0.4264122053484397E-09, 0.8280021541434701E-05, 0.6973096395115281E-11, 0.1865101988102620E-10},
			i0:       1,
			n0:       15,
			pp:       0,
			n0in:     15,
			dmin:     4.2641220534843968E-010,
			dmin1:    0.19554701532405336,
			dmin2:    0.19554701532405336,
			dn:       4.2641220534843968E-010,
			dn1:      0.32821333105682254,
			dn2:      0.21848306981706411,
			tau:      8.2795951083136037E-006,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2066672404538628E+01, 0.2066680684115085E+01, 0.6973096395115281E-11, 0.1865101988102620E-10, 0.7750244067025847E+00, 0.7726711989960955E+00, 0.3898634844569491E-02, 0.2361487308570419E-02, 0.1278402310262057E+01, 0.1279505990312372E+01, 0.3465878671879142E-02, 0.2803234389363145E-02, 0.1599126239756068E+01, 0.1580598225403798E+01, 0.2080167773382635E-01, 0.2200217261925751E-01, 0.1504268546848768E+01, 0.1511873816770092E+01, 0.5818924606767697E-02, 0.1320468740761088E-01, 0.8515715484031834E+00, 0.6628877301101292E+00, 0.3703084558710277E+00, 0.1945110224949301E+00, 0.1337929704521397E+01, 0.1621214783142186E+01, 0.4841425717359337E-01, 0.8703165684534678E-01, 0.8555583994741887E+00, 0.7442679496495411E+00, 0.3082690272254148E+00, 0.1597129865933493E+00, 0.1476800350035441E+01, 0.1651350720852550E+01, 0.4635002036444261E-01, 0.1337269360034139E+00, 0.5824492787618127E+00, 0.5118619205977412E+00, 0.9592445892242578E-01, 0.1169456581236225E+00, 0.4634535313311479E+00, 0.4777529393688394E+00, 0.4180980990954932E-01, 0.8163333047984263E-01, 0.4080949089880160E+00, 0.2373651048287110E+00, 0.2374870106014949E+00, 0.2125478936639627E+00, 0.2603433541742431E+00, 0.4559783600136673E+00, 0.6288565043307552E-01, 0.4186028435717903E-01, 0.3282133318859058E+00, 0.3911072610850064E+00, 0.2091574834858362E-13, 0.8290832843879624E-09, 0.4264122053484397E-09, 0.8280021541434701E-05, 0.6973096395115281E-11, 0.1865101988102620E-10},
			tauOut:   4.2641207498271701E-010,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2066672404538628E+01, 0.2066672404119188E+01, 0.6973096395115281E-11, 0.2614986238618434E-11, 0.7750244067025847E+00, 0.7789230411181270E+00, 0.3898634844569491E-02, 0.6398608757305904E-02, 0.1278402310262057E+01, 0.1275469579750218E+01, 0.3465878671879142E-02, 0.4345362379476143E-02, 0.1599126239756068E+01, 0.1615582554684006E+01, 0.2080167773382635E-01, 0.1936843737632438E-01, 0.1504268546848768E+01, 0.1490719033652799E+01, 0.5818924606767697E-02, 0.3324054047451484E-02, 0.8515715484031834E+00, 0.1218555949800348E+01, 0.3703084558710277E+00, 0.4065850919906259E+00, 0.1337929704521397E+01, 0.9797588692779520E+00, 0.4841425717359337E-01, 0.4227695780870786E-01, 0.8555583994741887E+00, 0.1121550468464483E+01, 0.3082690272254148E+00, 0.4059129037098644E+00, 0.1476800350035441E+01, 0.1117237466263607E+01, 0.4635002036444261E-01, 0.2416365074306882E-01, 0.5824492787618127E+00, 0.6542100865147574E+00, 0.9592445892242578E-01, 0.6795451513972497E-01, 0.4634535313311479E+00, 0.4373088256745602E+00, 0.4180980990954932E-01, 0.3901675330591518E-01, 0.4080949089880160E+00, 0.6065651658571838E+00, 0.2374870106014949E+00, 0.1019316116273065E+00, 0.2603433541742431E+00, 0.2212973925536000E+00, 0.6288565043307552E-01, 0.9326774535516916E-01, 0.3282133318859058E+00, 0.2349455861043456E+00, 0.2091574834858362E-13, 0.3796083394336032E-22, 0.4264122053484397E-09, 0.1303656847202082E-15, 0.6973096395115281E-11, 0.2614986238618434E-11, 0.2112065329503869E+00},
			i0:       1,
			n0:       15,
			pp:       1,
			n0in:     15,
			dmin:     1.3036568472020817E-016,
			dmin1:    0.15841174212052453,
			dmin2:    0.15841174212052453,
			dn:       1.3036568472020817E-016,
			dn1:      0.23494558610432464,
			dn2:      0.15841174212052453,
			tau:      4.2641207498271701E-010,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2066672404538628E+01, 0.2066672404119188E+01, 0.6973096395115281E-11, 0.2614986238618434E-11, 0.7750244067025847E+00, 0.7789230411181270E+00, 0.3898634844569491E-02, 0.6398608757305904E-02, 0.1278402310262057E+01, 0.1275469579750218E+01, 0.3465878671879142E-02, 0.4345362379476143E-02, 0.1599126239756068E+01, 0.1615582554684006E+01, 0.2080167773382635E-01, 0.1936843737632438E-01, 0.1504268546848768E+01, 0.1490719033652799E+01, 0.5818924606767697E-02, 0.3324054047451484E-02, 0.8515715484031834E+00, 0.1218555949800348E+01, 0.3703084558710277E+00, 0.4065850919906259E+00, 0.1337929704521397E+01, 0.9797588692779520E+00, 0.4841425717359337E-01, 0.4227695780870786E-01, 0.8555583994741887E+00, 0.1121550468464483E+01, 0.3082690272254148E+00, 0.4059129037098644E+00, 0.1476800350035441E+01, 0.1117237466263607E+01, 0.4635002036444261E-01, 0.2416365074306882E-01, 0.5824492787618127E+00, 0.6542100865147574E+00, 0.9592445892242578E-01, 0.6795451513972497E-01, 0.4634535313311479E+00, 0.4373088256745602E+00, 0.4180980990954932E-01, 0.3901675330591518E-01, 0.4080949089880160E+00, 0.6065651658571838E+00, 0.2374870106014949E+00, 0.1019316116273065E+00, 0.2603433541742431E+00, 0.2212973925536000E+00, 0.6288565043307552E-01, 0.9326774535516916E-01, 0.3282133318859058E+00, 0.2349455861043456E+00, 0.2091574834858362E-13, 0.3796083394336032E-22, 0.4264122053484397E-09, 0.1303656847202082E-15, 0.6973096395115281E-11, 0.2614986238618434E-11, 0.2112065329503869E+00},
			tauOut:   1.3036568471812905E-016,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2066672404121803E+01, 0.2066672404119188E+01, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.7853216498744473E+00, 0.7789230411181270E+00, 0.1039221422709001E-01, 0.6398608757305904E-02, 0.1269422727902604E+01, 0.1275469579750218E+01, 0.5530302475095172E-02, 0.4345362379476143E-02, 0.1629420689585235E+01, 0.1615582554684006E+01, 0.1771973219288664E-01, 0.1936843737632438E-01, 0.1476323355507364E+01, 0.1490719033652799E+01, 0.2743671176012720E-02, 0.3324054047451484E-02, 0.1622397370614960E+01, 0.1218555949800348E+01, 0.2455350071499520E+00, 0.4065850919906259E+00, 0.7765008199367077E+00, 0.9797588692779520E+00, 0.6106335063429082E-01, 0.4227695780870786E-01, 0.1466400021540057E+01, 0.1121550468464483E+01, 0.3092615230516922E+00, 0.4059129037098644E+00, 0.8321395939549830E+00, 0.1117237466263607E+01, 0.1899693772291602E-01, 0.2416365074306882E-01, 0.7031676639315663E+00, 0.6542100865147574E+00, 0.4226176876348708E-01, 0.6795451513972497E-01, 0.4340638102169883E+00, 0.4373088256745602E+00, 0.5452240634477346E-01, 0.3901675330591518E-01, 0.6539743711397168E+00, 0.6065651658571838E+00, 0.3449248298919955E-01, 0.1019316116273065E+00, 0.2800726549195695E+00, 0.2212973925536000E+00, 0.7823985923721809E-01, 0.9326774535516916E-01, 0.1567057268671274E+00, 0.2349455861043456E+00, 0.3158014839988858E-37, 0.3796083394336032E-22},
			i0:       1,
			n0:       14,
			pp:       0,
			n0in:     15,
			dmin:     2.0791168714198411E-027,
			dmin1:    0.15670572686712736,
			dmin2:    0.18680490956440032,
			dn:       2.0791168714198411E-027,
			dn1:      0.15670572686712736,
			dn2:      0.18680490956440032,
			tau:      1.3036568471812905E-016,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2066672404121803E+01, 0.2066672404119188E+01, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.7853216498744473E+00, 0.7789230411181270E+00, 0.1039221422709001E-01, 0.6398608757305904E-02, 0.1269422727902604E+01, 0.1275469579750218E+01, 0.5530302475095172E-02, 0.4345362379476143E-02, 0.1629420689585235E+01, 0.1615582554684006E+01, 0.1771973219288664E-01, 0.1936843737632438E-01, 0.1476323355507364E+01, 0.1490719033652799E+01, 0.2743671176012720E-02, 0.3324054047451484E-02, 0.1622397370614960E+01, 0.1218555949800348E+01, 0.2455350071499520E+00, 0.4065850919906259E+00, 0.7765008199367077E+00, 0.9797588692779520E+00, 0.6106335063429082E-01, 0.4227695780870786E-01, 0.1466400021540057E+01, 0.1121550468464483E+01, 0.3092615230516922E+00, 0.4059129037098644E+00, 0.8321395939549830E+00, 0.1117237466263607E+01, 0.1899693772291602E-01, 0.2416365074306882E-01, 0.7031676639315663E+00, 0.6542100865147574E+00, 0.4226176876348708E-01, 0.6795451513972497E-01, 0.4340638102169883E+00, 0.4373088256745602E+00, 0.5452240634477346E-01, 0.3901675330591518E-01, 0.6539743711397168E+00, 0.6065651658571838E+00, 0.3449248298919955E-01, 0.1019316116273065E+00, 0.2800726549195695E+00, 0.2212973925536000E+00, 0.7823985923721809E-01, 0.9326774535516916E-01, 0.1567057268671274E+00, 0.2349455861043456E+00, 0.3158014839988858E-37, 0.3796083394336032E-22},
			tauOut:   5.2226904068357879E-002,
			ttypeOut: -8,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2066672404121803E+01, 0.2014445500054431E+01, 0.9855809897129084E-12, 0.3842238913414725E-12, 0.7853216498744473E+00, 0.7434869600327952E+00, 0.1039221422709001E-01, 0.1774357001838869E-01, 0.1269422727902604E+01, 0.1204982556290953E+01, 0.5530302475095172E-02, 0.7478273627729330E-02, 0.1629420689585235E+01, 0.1587435244082034E+01, 0.1771973219288664E-01, 0.1647944669694028E-01, 0.1476323355507364E+01, 0.1410360675918078E+01, 0.2743671176012720E-02, 0.3156160674217245E-02, 0.1622397370614960E+01, 0.1812549313022337E+01, 0.2455350071499520E+00, 0.1051878329628395E+00, 0.7765008199367077E+00, 0.6801494335398012E+00, 0.6106335063429082E-01, 0.1316523902981273E+00, 0.1466400021540057E+01, 0.1591782250225264E+01, 0.3092615230516922E+00, 0.1616733433116970E+00, 0.8321395939549830E+00, 0.6372362842978442E+00, 0.1899693772291602E-01, 0.2096244776016672E-01, 0.7031676639315663E+00, 0.6722400808665288E+00, 0.4226176876348708E-01, 0.2728832287468248E-01, 0.4340638102169883E+00, 0.4090709896187213E+00, 0.5452240634477346E-01, 0.8716398206477828E-01, 0.6539743711397168E+00, 0.5490759679957803E+00, 0.3449248298919955E-01, 0.1759392479116377E-01, 0.2800726549195695E+00, 0.2884916852972659E+00, 0.7823985923721809E-01, 0.4249908970207047E-01, 0.1567057268671274E+00, 0.6197973309669901E-01, 0.3158014839988858E-37, 0.3842238913414725E-12, 0.4143051093784424E+00},
			i0:       1,
			n0:       14,
			pp:       1,
			n0in:     14,
			dmin:     6.1979733096699005E-002,
			dmin1:    0.21025182606004778,
			dmin2:    0.35454858327394784,
			dn:       6.1979733096699005E-002,
			dn1:      0.21025182606004778,
			dn2:      0.51458348500658069,
			tau:      5.2226904068357879E-002,
			ttype:    -8,
			g:        0.0000000000000000,
			zOut:     []float64{0.2066672404121803E+01, 0.2014445500054431E+01, 0.9855809897129084E-12, 0.3842238913414725E-12, 0.7853216498744473E+00, 0.7434869600327952E+00, 0.1039221422709001E-01, 0.1774357001838869E-01, 0.1269422727902604E+01, 0.1204982556290953E+01, 0.5530302475095172E-02, 0.7478273627729330E-02, 0.1629420689585235E+01, 0.1587435244082034E+01, 0.1771973219288664E-01, 0.1647944669694028E-01, 0.1476323355507364E+01, 0.1410360675918078E+01, 0.2743671176012720E-02, 0.3156160674217245E-02, 0.1622397370614960E+01, 0.1812549313022337E+01, 0.2455350071499520E+00, 0.1051878329628395E+00, 0.7765008199367077E+00, 0.6801494335398012E+00, 0.6106335063429082E-01, 0.1316523902981273E+00, 0.1466400021540057E+01, 0.1591782250225264E+01, 0.3092615230516922E+00, 0.1616733433116970E+00, 0.8321395939549830E+00, 0.6372362842978442E+00, 0.1899693772291602E-01, 0.2096244776016672E-01, 0.7031676639315663E+00, 0.6722400808665288E+00, 0.4226176876348708E-01, 0.2728832287468248E-01, 0.4340638102169883E+00, 0.4090709896187213E+00, 0.5452240634477346E-01, 0.8716398206477828E-01, 0.6539743711397168E+00, 0.5490759679957803E+00, 0.3449248298919955E-01, 0.1759392479116377E-01, 0.2800726549195695E+00, 0.2884916852972659E+00, 0.7823985923721809E-01, 0.4249908970207047E-01, 0.1567057268671274E+00, 0.6197973309669901E-01, 0.3158014839988858E-37, 0.3842238913414725E-12, 0.4143051093784424E+00},
			tauOut:   4.3992746597899146E-002,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1970452753456916E+01, 0.2014445500054431E+01, 0.1449745254963753E-12, 0.3842238913414725E-12, 0.7172377834531399E+00, 0.7434869600327952E+00, 0.2980976860358389E-01, 0.1774357001838869E-01, 0.1138658314717199E+01, 0.1204982556290953E+01, 0.1042566937606312E-01, 0.7478273627729330E-02, 0.1549496274805013E+01, 0.1587435244082034E+01, 0.1499968987352189E-01, 0.1647944669694028E-01, 0.1354524400120875E+01, 0.1410360675918078E+01, 0.4223398900256122E-02, 0.3156160674217245E-02, 0.1869521000487022E+01, 0.1812549313022337E+01, 0.3826832915293116E-01, 0.1051878329628395E+00, 0.7295407480870981E+00, 0.6801494335398012E+00, 0.2872518617030954E+00, 0.1316523902981273E+00, 0.1422210985235967E+01, 0.1591782250225264E+01, 0.7243940711431240E-01, 0.1616733433116970E+00, 0.5417665783457993E+00, 0.6372362842978442E+00, 0.2601082853889214E-01, 0.2096244776016672E-01, 0.6295248286044199E+00, 0.6722400808665288E+00, 0.1773220171177086E-01, 0.2728832287468248E-01, 0.4345100233738295E+00, 0.4090709896187213E+00, 0.1101462457758059E+00, 0.8716398206477828E-01, 0.4125309004132391E+00, 0.5490759679957803E+00, 0.1230380805149813E-01, 0.1759392479116377E-01, 0.2746942203499391E+00, 0.2884916852972659E+00, 0.9589143278047064E-02, 0.4249908970207047E-01, 0.8397843220752800E-02, 0.6197973309669901E-01, 0.1449745254963753E-12, 0.3842238913414725E-12},
			i0:       1,
			n0:       14,
			pp:       0,
			n0in:     14,
			dmin:     8.3978432207528000E-003,
			dmin1:    0.23219513064786862,
			dmin2:    0.34734604130905122,
			dn:       8.3978432207528000E-003,
			dn1:      0.23219513064786862,
			dn2:      0.39493697562207530,
			tau:      4.3992746597899146E-002,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1970452753456916E+01, 0.2014445500054431E+01, 0.1449745254963753E-12, 0.3842238913414725E-12, 0.7172377834531399E+00, 0.7434869600327952E+00, 0.2980976860358389E-01, 0.1774357001838869E-01, 0.1138658314717199E+01, 0.1204982556290953E+01, 0.1042566937606312E-01, 0.7478273627729330E-02, 0.1549496274805013E+01, 0.1587435244082034E+01, 0.1499968987352189E-01, 0.1647944669694028E-01, 0.1354524400120875E+01, 0.1410360675918078E+01, 0.4223398900256122E-02, 0.3156160674217245E-02, 0.1869521000487022E+01, 0.1812549313022337E+01, 0.3826832915293116E-01, 0.1051878329628395E+00, 0.7295407480870981E+00, 0.6801494335398012E+00, 0.2872518617030954E+00, 0.1316523902981273E+00, 0.1422210985235967E+01, 0.1591782250225264E+01, 0.7243940711431240E-01, 0.1616733433116970E+00, 0.5417665783457993E+00, 0.6372362842978442E+00, 0.2601082853889214E-01, 0.2096244776016672E-01, 0.6295248286044199E+00, 0.6722400808665288E+00, 0.1773220171177086E-01, 0.2728832287468248E-01, 0.4345100233738295E+00, 0.4090709896187213E+00, 0.1101462457758059E+00, 0.8716398206477828E-01, 0.4125309004132391E+00, 0.5490759679957803E+00, 0.1230380805149813E-01, 0.1759392479116377E-01, 0.2746942203499391E+00, 0.2884916852972659E+00, 0.9589143278047064E-02, 0.4249908970207047E-01, 0.8397843220752800E-02, 0.6197973309669901E-01, 0.1449745254963753E-12, 0.3842238913414725E-12},
			tauOut:   8.0121275381568099E-003,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1970452753456916E+01, 0.1962440625918904E+01, 0.1449745254963753E-12, 0.5298565773193886E-13, 0.7172377834531399E+00, 0.7390354245185139E+00, 0.2980976860358389E-01, 0.4592897681783073E-01, 0.1138658314717199E+01, 0.1095142879737275E+01, 0.1042566937606312E-01, 0.1475107600976594E-01, 0.1549496274805013E+01, 0.1541732761130612E+01, 0.1499968987352189E-01, 0.1317831886314191E-01, 0.1354524400120875E+01, 0.1337557352619832E+01, 0.4223398900256122E-02, 0.5903098601340335E-02, 0.1869521000487022E+01, 0.1893874103500456E+01, 0.3826832915293116E-01, 0.1474137347708128E-01, 0.7295407480870981E+00, 0.9940391087749554E+00, 0.2872518617030954E+00, 0.4109825756725979E+00, 0.1422210985235967E+01, 0.1075655689139524E+01, 0.7243940711431240E-01, 0.3648495529374628E-01, 0.5417665783457993E+00, 0.5232803240527883E+00, 0.2601082853889214E-01, 0.3129195122603765E-01, 0.6295248286044199E+00, 0.6079529515519964E+00, 0.1773220171177086E-01, 0.1267338099203561E-01, 0.4345100233738295E+00, 0.5239707606194429E+00, 0.1101462457758059E+00, 0.8671997249104711E-01, 0.4125309004132391E+00, 0.3301026084355332E+00, 0.1230380805149813E-01, 0.1023858907404432E-01, 0.2746942203499391E+00, 0.2660326470157850E+00, 0.9589143278047064E-02, 0.3027001489241909E-03, 0.8397843220752800E-02, 0.8301553367179998E-04, 0.1449745254963753E-12, 0.5298565773193886E-13, 0.4143051093784424E+00},
			i0:       1,
			n0:       14,
			pp:       1,
			n0in:     14,
			dmin:     8.3015533671799979E-005,
			dmin1:    0.25644350373773794,
			dmin2:    0.31779880038403513,
			dn:       8.3015533671799979E-005,
			dn1:      0.25644350373773794,
			dn2:      0.31779880038403513,
			tau:      8.0121275381568099E-003,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1970452753456916E+01, 0.1962440625918904E+01, 0.1449745254963753E-12, 0.5298565773193886E-13, 0.7172377834531399E+00, 0.7390354245185139E+00, 0.2980976860358389E-01, 0.4592897681783073E-01, 0.1138658314717199E+01, 0.1095142879737275E+01, 0.1042566937606312E-01, 0.1475107600976594E-01, 0.1549496274805013E+01, 0.1541732761130612E+01, 0.1499968987352189E-01, 0.1317831886314191E-01, 0.1354524400120875E+01, 0.1337557352619832E+01, 0.4223398900256122E-02, 0.5903098601340335E-02, 0.1869521000487022E+01, 0.1893874103500456E+01, 0.3826832915293116E-01, 0.1474137347708128E-01, 0.7295407480870981E+00, 0.9940391087749554E+00, 0.2872518617030954E+00, 0.4109825756725979E+00, 0.1422210985235967E+01, 0.1075655689139524E+01, 0.7243940711431240E-01, 0.3648495529374628E-01, 0.5417665783457993E+00, 0.5232803240527883E+00, 0.2601082853889214E-01, 0.3129195122603765E-01, 0.6295248286044199E+00, 0.6079529515519964E+00, 0.1773220171177086E-01, 0.1267338099203561E-01, 0.4345100233738295E+00, 0.5239707606194429E+00, 0.1101462457758059E+00, 0.8671997249104711E-01, 0.4125309004132391E+00, 0.3301026084355332E+00, 0.1230380805149813E-01, 0.1023858907404432E-01, 0.2746942203499391E+00, 0.2660326470157850E+00, 0.9589143278047064E-02, 0.3027001489241909E-03, 0.8397843220752800E-02, 0.8301553367179998E-04, 0.1449745254963753E-12, 0.5298565773193886E-13, 0.4143051093784424E+00},
			tauOut:   8.2898056486573887E-005,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1962357727862470E+01, 0.1962440625918904E+01, 0.1995470932711637E-13, 0.5298565773193886E-13, 0.7848815032798381E+00, 0.7390354245185139E+00, 0.6408456783027588E-01, 0.4592897681783073E-01, 0.1045726489860278E+01, 0.1095142879737275E+01, 0.2174776805091989E-01, 0.1475107600976594E-01, 0.1533080413886347E+01, 0.1541732761130612E+01, 0.1149760777771624E-01, 0.1317831886314191E-01, 0.1331879945386970E+01, 0.1337557352619832E+01, 0.8393943921304423E-02, 0.5903098601340335E-02, 0.1900138634999746E+01, 0.1893874103500456E+01, 0.7711806645770664E-02, 0.1474137347708128E-01, 0.1397226979745296E+01, 0.9940391087749554E+00, 0.3163950825942626E+00, 0.4109825756725979E+00, 0.7956626637825215E+00, 0.1075655689139524E+01, 0.2399491656225486E-01, 0.3648495529374628E-01, 0.5304944606600845E+00, 0.5232803240527883E+00, 0.3586094769777509E-01, 0.3129195122603765E-01, 0.5846824867897703E+00, 0.6079529515519964E+00, 0.1135741402906866E-01, 0.1267338099203561E-01, 0.5992504210249349E+00, 0.5239707606194429E+00, 0.4777049480214081E-01, 0.8671997249104711E-01, 0.2924878046509502E+00, 0.3301026084355332E+00, 0.9312521444528056E-02, 0.1023858907404432E-01, 0.2569399276636945E+00, 0.2660326470157850E+00, 0.9780034825247492E-07, 0.3027001489241909E-03, 0.1967683697362142E-07, 0.8301553367179998E-04, 0.1995470932711637E-13, 0.5298565773193886E-13},
			i0:       1,
			n0:       14,
			pp:       0,
			n0in:     14,
			dmin:     1.9676836973621424E-008,
			dmin1:    0.25663722751477031,
			dmin2:    0.28224921557690585,
			dn:       1.9676836973621424E-008,
			dn1:      0.25663722751477031,
			dn2:      0.28224921557690585,
			tau:      8.2898056486573887E-005,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1962357727862470E+01, 0.1962440625918904E+01, 0.1995470932711637E-13, 0.5298565773193886E-13, 0.7848815032798381E+00, 0.7390354245185139E+00, 0.6408456783027588E-01, 0.4592897681783073E-01, 0.1045726489860278E+01, 0.1095142879737275E+01, 0.2174776805091989E-01, 0.1475107600976594E-01, 0.1533080413886347E+01, 0.1541732761130612E+01, 0.1149760777771624E-01, 0.1317831886314191E-01, 0.1331879945386970E+01, 0.1337557352619832E+01, 0.8393943921304423E-02, 0.5903098601340335E-02, 0.1900138634999746E+01, 0.1893874103500456E+01, 0.7711806645770664E-02, 0.1474137347708128E-01, 0.1397226979745296E+01, 0.9940391087749554E+00, 0.3163950825942626E+00, 0.4109825756725979E+00, 0.7956626637825215E+00, 0.1075655689139524E+01, 0.2399491656225486E-01, 0.3648495529374628E-01, 0.5304944606600845E+00, 0.5232803240527883E+00, 0.3586094769777509E-01, 0.3129195122603765E-01, 0.5846824867897703E+00, 0.6079529515519964E+00, 0.1135741402906866E-01, 0.1267338099203561E-01, 0.5992504210249349E+00, 0.5239707606194429E+00, 0.4777049480214081E-01, 0.8671997249104711E-01, 0.2924878046509502E+00, 0.3301026084355332E+00, 0.9312521444528056E-02, 0.1023858907404432E-01, 0.2569399276636945E+00, 0.2660326470157850E+00, 0.9780034825247492E-07, 0.3027001489241909E-03, 0.1967683697362142E-07, 0.8301553367179998E-04, 0.1995470932711637E-13, 0.5298565773193886E-13},
			tauOut:   1.9676827722764296E-008,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1962357727862470E+01, 0.1962357708185662E+01, 0.1995470932711637E-13, 0.7981257539768321E-14, 0.7848815032798381E+00, 0.8489660514332782E+00, 0.6408456783027588E-01, 0.7893711422056097E-01, 0.1045726489860278E+01, 0.9885371240138099E+00, 0.2174776805091989E-01, 0.3372769361380378E-01, 0.1533080413886347E+01, 0.1510850308373431E+01, 0.1149760777771624E-01, 0.1013563894066502E-01, 0.1331879945386970E+01, 0.1330138230690781E+01, 0.8393943921304423E-02, 0.1199097716077874E-01, 0.1900138634999746E+01, 0.1895859444807910E+01, 0.7711806645770664E-02, 0.5683514322519618E-02, 0.1397226979745296E+01, 0.1707938528340212E+01, 0.3163950825942626E+00, 0.1473962616612956E+00, 0.7956626637825215E+00, 0.6722612990066530E+00, 0.2399491656225486E-01, 0.1893485515094503E-01, 0.5304944606600845E+00, 0.5474205335300870E+00, 0.3586094769777509E-01, 0.3830193935796279E-01, 0.5846824867897703E+00, 0.5577379417840483E+00, 0.1135741402906866E-01, 0.1220274725600271E-01, 0.5992504210249349E+00, 0.6348181488942451E+00, 0.4777049480214081E-01, 0.2200990500366971E-01, 0.2924878046509502E+00, 0.2797904014149808E+00, 0.9312521444528056E-02, 0.8551968095484232E-02, 0.2569399276636945E+00, 0.2483880376917308E+00, 0.9780034825247492E-07, 0.7747561140265878E-14, 0.1967683697362142E-07, 0.1503295986001297E-14, 0.1995470932711637E-13, 0.7981257539768321E-14, 0.4143051093784424E+00},
			i0:       1,
			n0:       14,
			pp:       1,
			n0in:     14,
			dmin:     1.5032959860012969E-015,
			dmin1:    0.24838793989138258,
			dmin2:    0.27047787997045275,
			dn:       1.5032959860012969E-015,
			dn1:      0.24838793989138258,
			dn2:      0.27047787997045275,
			tau:      1.9676827722764296E-008,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1962357727862470E+01, 0.1962357708185662E+01, 0.1995470932711637E-13, 0.7981257539768321E-14, 0.7848815032798381E+00, 0.8489660514332782E+00, 0.6408456783027588E-01, 0.7893711422056097E-01, 0.1045726489860278E+01, 0.9885371240138099E+00, 0.2174776805091989E-01, 0.3372769361380378E-01, 0.1533080413886347E+01, 0.1510850308373431E+01, 0.1149760777771624E-01, 0.1013563894066502E-01, 0.1331879945386970E+01, 0.1330138230690781E+01, 0.8393943921304423E-02, 0.1199097716077874E-01, 0.1900138634999746E+01, 0.1895859444807910E+01, 0.7711806645770664E-02, 0.5683514322519618E-02, 0.1397226979745296E+01, 0.1707938528340212E+01, 0.3163950825942626E+00, 0.1473962616612956E+00, 0.7956626637825215E+00, 0.6722612990066530E+00, 0.2399491656225486E-01, 0.1893485515094503E-01, 0.5304944606600845E+00, 0.5474205335300870E+00, 0.3586094769777509E-01, 0.3830193935796279E-01, 0.5846824867897703E+00, 0.5577379417840483E+00, 0.1135741402906866E-01, 0.1220274725600271E-01, 0.5992504210249349E+00, 0.6348181488942451E+00, 0.4777049480214081E-01, 0.2200990500366971E-01, 0.2924878046509502E+00, 0.2797904014149808E+00, 0.9312521444528056E-02, 0.8551968095484232E-02, 0.2569399276636945E+00, 0.2483880376917308E+00, 0.9780034825247492E-07, 0.7747561140265878E-14, 0.1967683697362142E-07, 0.1503295986001297E-14, 0.1995470932711637E-13, 0.7981257539768321E-14, 0.4143051093784424E+00},
			tauOut:   1.5032959860012393E-015,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1962357708185669E+01, 0.1962357708185662E+01, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.9279031656538340E+00, 0.8489660514332782E+00, 0.8409527066820451E-01, 0.7893711422056097E-01, 0.9381695469594078E+00, 0.9885371240138099E+00, 0.5431587122209677E-01, 0.3372769361380378E-01, 0.1466670076091998E+01, 0.1510850308373431E+01, 0.9192115573380727E-02, 0.1013563894066502E-01, 0.1332937092278178E+01, 0.1330138230690781E+01, 0.1705497388769041E-01, 0.1199097716077874E-01, 0.1884487985242738E+01, 0.1895859444807910E+01, 0.5151050664063701E-02, 0.5683514322519618E-02, 0.1850183739337442E+01, 0.1707938528340212E+01, 0.5355619565040129E-01, 0.1473962616612956E+00, 0.6376399585071952E+00, 0.6722612990066530E+00, 0.1625577000116482E-01, 0.1893485515094503E-01, 0.5694667028868834E+00, 0.5474205335300870E+00, 0.3751307094084999E-01, 0.3830193935796279E-01, 0.5324276180991995E+00, 0.5577379417840483E+00, 0.1454944326918194E-01, 0.1220274725600271E-01, 0.6422786106287314E+00, 0.6348181488942451E+00, 0.9587988848101409E-02, 0.2200990500366971E-01, 0.2787543806623621E+00, 0.2797904014149808E+00, 0.7620352256320365E-02, 0.8551968095484232E-02, 0.2407676854354167E+00, 0.2483880376917308E+00, 0.4837392336267432E-28, 0.7747561140265878E-14},
			i0:       1,
			n0:       13,
			pp:       0,
			n0in:     14,
			dmin:     9.2691156363468887E-030,
			dmin1:    0.24076768543540897,
			dmin2:    0.27020241256687788,
			dn:       9.2691156363468887E-030,
			dn1:      0.24076768543540897,
			dn2:      0.27020241256687788,
			tau:      1.5032959860012393E-015,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1962357708185669E+01, 0.1962357708185662E+01, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.9279031656538340E+00, 0.8489660514332782E+00, 0.8409527066820451E-01, 0.7893711422056097E-01, 0.9381695469594078E+00, 0.9885371240138099E+00, 0.5431587122209677E-01, 0.3372769361380378E-01, 0.1466670076091998E+01, 0.1510850308373431E+01, 0.9192115573380727E-02, 0.1013563894066502E-01, 0.1332937092278178E+01, 0.1330138230690781E+01, 0.1705497388769041E-01, 0.1199097716077874E-01, 0.1884487985242738E+01, 0.1895859444807910E+01, 0.5151050664063701E-02, 0.5683514322519618E-02, 0.1850183739337442E+01, 0.1707938528340212E+01, 0.5355619565040129E-01, 0.1473962616612956E+00, 0.6376399585071952E+00, 0.6722612990066530E+00, 0.1625577000116482E-01, 0.1893485515094503E-01, 0.5694667028868834E+00, 0.5474205335300870E+00, 0.3751307094084999E-01, 0.3830193935796279E-01, 0.5324276180991995E+00, 0.5577379417840483E+00, 0.1454944326918194E-01, 0.1220274725600271E-01, 0.6422786106287314E+00, 0.6348181488942451E+00, 0.9587988848101409E-02, 0.2200990500366971E-01, 0.2787543806623621E+00, 0.2797904014149808E+00, 0.7620352256320365E-02, 0.8551968095484232E-02, 0.2407676854354167E+00, 0.2483880376917308E+00, 0.4837392336267432E-28, 0.7747561140265878E-14},
			tauOut:   0.19361025134591583,
			ttypeOut: -8,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1962357708185669E+01, 0.1768747456839757E+01, 0.3452895805257589E-14, 0.1811424766142649E-14, 0.9279031656538340E+00, 0.8183881849761209E+00, 0.8409527066820451E-01, 0.9640366690596863E-01, 0.9381695469594078E+00, 0.7024714999296202E+00, 0.5431587122209677E-01, 0.1134045480653624E+00, 0.1466670076091998E+01, 0.1168847392254101E+01, 0.9192115573380727E-02, 0.1048255904531585E-01, 0.1332937092278178E+01, 0.1145899255774637E+01, 0.1705497388769041E-01, 0.2804774784346500E-01, 0.1884487985242738E+01, 0.1667981036717421E+01, 0.5151050664063701E-02, 0.5713728135608638E-02, 0.1850183739337442E+01, 0.1704415955506319E+01, 0.5355619565040129E-01, 0.2003593680404179E-01, 0.6376399585071952E+00, 0.4402495403584025E+00, 0.1625577000116482E-01, 0.2102698332839761E-01, 0.5694667028868834E+00, 0.3923425391534199E+00, 0.3751307094084999E-01, 0.5090703407211448E-01, 0.5324276180991995E+00, 0.3024597759503511E+00, 0.1454944326918194E-01, 0.3089599659653809E-01, 0.6422786106287314E+00, 0.4273603515343789E+00, 0.9587988848101409E-02, 0.6253958476854957E-02, 0.2787543806623621E+00, 0.8651052309591171E-01, 0.7620352256320365E-02, 0.2120822426333837E-01, 0.2407676854354167E+00, 0.2594920982616250E-01, 0.4837392336267432E-28, 0.1811424766142649E-14, 0.5186198053161721E+00},
			i0:       1,
			n0:       13,
			pp:       1,
			n0in:     13,
			dmin:     2.5949209826162500E-002,
			dmin1:    7.8890170839591350E-002,
			dmin2:    0.28791033268116917,
			dn:       2.5949209826162500E-002,
			dn1:      7.8890170839591350E-002,
			dn2:      0.41777236268627749,
			tau:      0.19361025134591583,
			ttype:    -8,
			g:        0.0000000000000000,
			zOut:     []float64{0.1962357708185669E+01, 0.1768747456839757E+01, 0.3452895805257589E-14, 0.1811424766142649E-14, 0.9279031656538340E+00, 0.8183881849761209E+00, 0.8409527066820451E-01, 0.9640366690596863E-01, 0.9381695469594078E+00, 0.7024714999296202E+00, 0.5431587122209677E-01, 0.1134045480653624E+00, 0.1466670076091998E+01, 0.1168847392254101E+01, 0.9192115573380727E-02, 0.1048255904531585E-01, 0.1332937092278178E+01, 0.1145899255774637E+01, 0.1705497388769041E-01, 0.2804774784346500E-01, 0.1884487985242738E+01, 0.1667981036717421E+01, 0.5151050664063701E-02, 0.5713728135608638E-02, 0.1850183739337442E+01, 0.1704415955506319E+01, 0.5355619565040129E-01, 0.2003593680404179E-01, 0.6376399585071952E+00, 0.4402495403584025E+00, 0.1625577000116482E-01, 0.2102698332839761E-01, 0.5694667028868834E+00, 0.3923425391534199E+00, 0.3751307094084999E-01, 0.5090703407211448E-01, 0.5324276180991995E+00, 0.3024597759503511E+00, 0.1454944326918194E-01, 0.3089599659653809E-01, 0.6422786106287314E+00, 0.4273603515343789E+00, 0.9587988848101409E-02, 0.6253958476854957E-02, 0.2787543806623621E+00, 0.8651052309591171E-01, 0.7620352256320365E-02, 0.2120822426333837E-01, 0.2407676854354167E+00, 0.2594920982616250E-01, 0.4837392336267432E-28, 0.1811424766142649E-14, 0.5186198053161721E+00},
			tauOut:   1.8780556192507153E-002,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1749966900647251E+01, 0.1768747456839757E+01, 0.8471295234418270E-15, 0.1811424766142649E-14, 0.8960112956895816E+00, 0.8183881849761209E+00, 0.7558032897122403E-01, 0.9640366690596863E-01, 0.7215151628312513E+00, 0.7024714999296202E+00, 0.1837142406762631E+00, 0.1134045480653624E+00, 0.9768351544306461E+00, 0.1168847392254101E+01, 0.1229681032071614E-01, 0.1048255904531585E-01, 0.1142869637104878E+01, 0.1145899255774637E+01, 0.4093477506677201E-01, 0.2804774784346500E-01, 0.1613979433593750E+01, 0.1667981036717421E+01, 0.6033886923870183E-02, 0.5713728135608638E-02, 0.1699637449193983E+01, 0.1704415955506319E+01, 0.5189819730562233E-02, 0.2003593680404179E-01, 0.4373061477637308E+00, 0.4402495403584025E+00, 0.1886499897608887E-01, 0.2102698332839761E-01, 0.4056040180569384E+00, 0.3923425391534199E+00, 0.3796148320598534E-01, 0.5090703407211448E-01, 0.2766137331483968E+00, 0.3024597759503511E+00, 0.4773343613933298E-01, 0.3089599659653809E-01, 0.3671003176793937E+00, 0.4273603515343789E+00, 0.1473802100398464E-02, 0.6253958476854957E-02, 0.8746438906634448E-01, 0.8651052309591171E-01, 0.6292122626412339E-02, 0.2120822426333837E-01, 0.8765310072430081E-03, 0.2594920982616250E-01, 0.8471295234418270E-15, 0.1811424766142649E-14},
			i0:       1,
			n0:       13,
			pp:       0,
			n0in:     13,
			dmin:     8.7653100724300811E-004,
			dmin1:    6.6256164803006098E-002,
			dmin2:    0.24571773655185866,
			dn:       8.7653100724300811E-004,
			dn1:      6.6256164803006098E-002,
			dn2:      0.36084635920253871,
			tau:      1.8780556192507153E-002,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1749966900647251E+01, 0.1768747456839757E+01, 0.8471295234418270E-15, 0.1811424766142649E-14, 0.8960112956895816E+00, 0.8183881849761209E+00, 0.7558032897122403E-01, 0.9640366690596863E-01, 0.7215151628312513E+00, 0.7024714999296202E+00, 0.1837142406762631E+00, 0.1134045480653624E+00, 0.9768351544306461E+00, 0.1168847392254101E+01, 0.1229681032071614E-01, 0.1048255904531585E-01, 0.1142869637104878E+01, 0.1145899255774637E+01, 0.4093477506677201E-01, 0.2804774784346500E-01, 0.1613979433593750E+01, 0.1667981036717421E+01, 0.6033886923870183E-02, 0.5713728135608638E-02, 0.1699637449193983E+01, 0.1704415955506319E+01, 0.5189819730562233E-02, 0.2003593680404179E-01, 0.4373061477637308E+00, 0.4402495403584025E+00, 0.1886499897608887E-01, 0.2102698332839761E-01, 0.4056040180569384E+00, 0.3923425391534199E+00, 0.3796148320598534E-01, 0.5090703407211448E-01, 0.2766137331483968E+00, 0.3024597759503511E+00, 0.4773343613933298E-01, 0.3089599659653809E-01, 0.3671003176793937E+00, 0.4273603515343789E+00, 0.1473802100398464E-02, 0.6253958476854957E-02, 0.8746438906634448E-01, 0.8651052309591171E-01, 0.6292122626412339E-02, 0.2120822426333837E-01, 0.8765310072430081E-03, 0.2594920982616250E-01, 0.8471295234418270E-15, 0.1811424766142649E-14},
			tauOut:   8.1622622999092049E-004,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1749966900647251E+01, 0.1749150674417261E+01, 0.8471295234418270E-15, 0.4339463906783712E-15, 0.8960112956895816E+00, 0.9707753984308143E+00, 0.7558032897122403E-01, 0.5617401661873561E-01, 0.7215151628312513E+00, 0.8482391606587878E+00, 0.1837142406762631E+00, 0.2115659556707206E+00, 0.9768351544306461E+00, 0.7767497828506508E+00, 0.1229681032071614E-01, 0.1809289356632693E-01, 0.1142869637104878E+01, 0.1164895292375332E+01, 0.4093477506677201E-01, 0.5671572845129930E-01, 0.1613979433593750E+01, 0.1562481365836330E+01, 0.6033886923870183E-02, 0.6563547191183529E-02, 0.1699637449193983E+01, 0.1697447495503371E+01, 0.5189819730562233E-02, 0.1337031089310571E-02, 0.4373061477637308E+00, 0.4540178894205181E+00, 0.1886499897608887E-01, 0.1685334336738995E-01, 0.4056040180569384E+00, 0.4258959316655428E+00, 0.3796148320598534E-01, 0.2465547755855056E-01, 0.2766137331483968E+00, 0.2988754654991882E+00, 0.4773343613933298E-01, 0.5862963539483231E-01, 0.3671003176793937E+00, 0.3091282581549689E+00, 0.1473802100398464E-02, 0.4169958485368410E-03, 0.8746438906634448E-01, 0.9252328961422907E-01, 0.6292122626412339E-02, 0.5960921413863723E-04, 0.8765310072430081E-03, 0.6955631134505013E-06, 0.8471295234418270E-15, 0.4339463906783712E-15, 0.5186198053161721E+00},
			i0:       1,
			n0:       13,
			pp:       1,
			n0in:     13,
			dmin:     6.9556311345050133E-007,
			dmin1:    8.6231166987816729E-002,
			dmin2:    0.25114202935985525,
			dn:       6.9556311345050133E-007,
			dn1:      8.6231166987816729E-002,
			dn2:      0.30765445605457048,
			tau:      8.1622622999092049E-004,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1749966900647251E+01, 0.1749150674417261E+01, 0.8471295234418270E-15, 0.4339463906783712E-15, 0.8960112956895816E+00, 0.9707753984308143E+00, 0.7558032897122403E-01, 0.5617401661873561E-01, 0.7215151628312513E+00, 0.8482391606587878E+00, 0.1837142406762631E+00, 0.2115659556707206E+00, 0.9768351544306461E+00, 0.7767497828506508E+00, 0.1229681032071614E-01, 0.1809289356632693E-01, 0.1142869637104878E+01, 0.1164895292375332E+01, 0.4093477506677201E-01, 0.5671572845129930E-01, 0.1613979433593750E+01, 0.1562481365836330E+01, 0.6033886923870183E-02, 0.6563547191183529E-02, 0.1699637449193983E+01, 0.1697447495503371E+01, 0.5189819730562233E-02, 0.1337031089310571E-02, 0.4373061477637308E+00, 0.4540178894205181E+00, 0.1886499897608887E-01, 0.1685334336738995E-01, 0.4056040180569384E+00, 0.4258959316655428E+00, 0.3796148320598534E-01, 0.2465547755855056E-01, 0.2766137331483968E+00, 0.2988754654991882E+00, 0.4773343613933298E-01, 0.5862963539483231E-01, 0.3671003176793937E+00, 0.3091282581549689E+00, 0.1473802100398464E-02, 0.4169958485368410E-03, 0.8746438906634448E-01, 0.9252328961422907E-01, 0.6292122626412339E-02, 0.5960921413863723E-04, 0.8765310072430081E-03, 0.6955631134505013E-06, 0.8471295234418270E-15, 0.4339463906783712E-15, 0.5186198053161721E+00},
			tauOut:   6.9511331676175615E-007,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1749149979303945E+01, 0.1749150674417261E+01, 0.2408395422307052E-15, 0.4339463906783712E-15, 0.1026948719936233E+01, 0.9707753984308143E+00, 0.4639861736277134E-01, 0.5617401661873561E-01, 0.1013405803853420E+01, 0.8482391606587878E+00, 0.1621599259654447E+00, 0.2115659556707206E+00, 0.6326820553382161E+00, 0.7767497828506508E+00, 0.3331266686486833E-01, 0.1809289356632693E-01, 0.1188297658848447E+01, 0.1164895292375332E+01, 0.7457497555020480E-01, 0.5671572845129930E-01, 0.1494469242363992E+01, 0.1562481365836330E+01, 0.7455005714048079E-02, 0.6563547191183529E-02, 0.1691328825765317E+01, 0.1697447495503371E+01, 0.3589107121045615E-03, 0.1337031089310571E-02, 0.4705116269624868E+00, 0.4540178894205181E+00, 0.1525524548983380E-01, 0.1685334336738995E-01, 0.4352954686209429E+00, 0.4258959316655428E+00, 0.1692854133253905E-01, 0.2465547755855056E-01, 0.3405758644481647E+00, 0.2988754654991882E+00, 0.5321597610926388E-01, 0.5862963539483231E-01, 0.2563285827809251E+00, 0.3091282581549689E+00, 0.1505170716567330E-03, 0.4169958485368410E-03, 0.9243168664339420E-01, 0.9252328961422907E-01, 0.4485687980202113E-09, 0.5960921413863723E-04, 0.1227890724922389E-11, 0.6955631134505013E-06, 0.2408395422307052E-15, 0.4339463906783712E-15},
			i0:       1,
			n0:       13,
			pp:       0,
			n0in:     13,
			dmin:     1.2278907249223888E-012,
			dmin1:    9.2372077429255559E-002,
			dmin2:    0.25591158693238830,
			dn:       1.2278907249223888E-012,
			dn1:      9.2372077429255559E-002,
			dn2:      0.25591158693238830,
			tau:      6.9511331676175615E-007,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1749149979303945E+01, 0.1749150674417261E+01, 0.2408395422307052E-15, 0.4339463906783712E-15, 0.1026948719936233E+01, 0.9707753984308143E+00, 0.4639861736277134E-01, 0.5617401661873561E-01, 0.1013405803853420E+01, 0.8482391606587878E+00, 0.1621599259654447E+00, 0.2115659556707206E+00, 0.6326820553382161E+00, 0.7767497828506508E+00, 0.3331266686486833E-01, 0.1809289356632693E-01, 0.1188297658848447E+01, 0.1164895292375332E+01, 0.7457497555020480E-01, 0.5671572845129930E-01, 0.1494469242363992E+01, 0.1562481365836330E+01, 0.7455005714048079E-02, 0.6563547191183529E-02, 0.1691328825765317E+01, 0.1697447495503371E+01, 0.3589107121045615E-03, 0.1337031089310571E-02, 0.4705116269624868E+00, 0.4540178894205181E+00, 0.1525524548983380E-01, 0.1685334336738995E-01, 0.4352954686209429E+00, 0.4258959316655428E+00, 0.1692854133253905E-01, 0.2465547755855056E-01, 0.3405758644481647E+00, 0.2988754654991882E+00, 0.5321597610926388E-01, 0.5862963539483231E-01, 0.2563285827809251E+00, 0.3091282581549689E+00, 0.1505170716567330E-03, 0.4169958485368410E-03, 0.9243168664339420E-01, 0.9252328961422907E-01, 0.4485687980202113E-09, 0.5960921413863723E-04, 0.1227890724922389E-11, 0.6955631134505013E-06, 0.2408395422307052E-15, 0.4339463906783712E-15},
			tauOut:   1.2278907189544363E-012,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1749149979303945E+01, 0.1749149979302717E+01, 0.2408395422307052E-15, 0.1414000300319855E-15, 0.1026948719936233E+01, 0.1073347337297776E+01, 0.4639861736277134E-01, 0.4380746706334982E-01, 0.1013405803853420E+01, 0.1131758262754287E+01, 0.1621599259654447E+00, 0.9065158049178278E-01, 0.6326820553382161E+00, 0.5753431417100738E+00, 0.3331266686486833E-01, 0.6880305191066147E-01, 0.1188297658848447E+01, 0.1194069582486762E+01, 0.7457497555020480E-01, 0.9333627524262254E-01, 0.1494469242363992E+01, 0.1408587972834190E+01, 0.7455005714048079E-02, 0.8951422490882578E-02, 0.1691328825765317E+01, 0.1682736313985311E+01, 0.3589107121045615E-03, 0.1003553923945662E-03, 0.4705116269624868E+00, 0.4856665170586981E+00, 0.1525524548983380E-01, 0.1367304312976171E-01, 0.4352954686209429E+00, 0.4385509668224923E+00, 0.1692854133253905E-01, 0.1314659648329904E-01, 0.3405758644481647E+00, 0.3806452440729016E+00, 0.5321597610926388E-01, 0.3583592846566260E-01, 0.2563285827809251E+00, 0.2206431713856914E+00, 0.1505170716567330E-03, 0.6305450884558253E-04, 0.9243168664339420E-01, 0.9236863258188953E-01, 0.4485687980202113E-09, 0.5962992535266723E-20, 0.1227890724922389E-11, 0.4959852501050381E-23, 0.2408395422307052E-15, 0.1414000300319855E-15, 0.5186198053161721E+00},
			i0:       1,
			n0:       13,
			pp:       1,
			n0in:     13,
			dmin:     4.9598525010503808E-024,
			dmin1:    9.2368632133320736E-002,
			dmin2:    0.22049265431403467,
			dn:       4.9598525010503808E-024,
			dn1:      9.2368632133320736E-002,
			dn2:      0.22049265431403467,
			tau:      1.2278907189544363E-012,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1749149979303945E+01, 0.1749149979302717E+01, 0.2408395422307052E-15, 0.1414000300319855E-15, 0.1026948719936233E+01, 0.1073347337297776E+01, 0.4639861736277134E-01, 0.4380746706334982E-01, 0.1013405803853420E+01, 0.1131758262754287E+01, 0.1621599259654447E+00, 0.9065158049178278E-01, 0.6326820553382161E+00, 0.5753431417100738E+00, 0.3331266686486833E-01, 0.6880305191066147E-01, 0.1188297658848447E+01, 0.1194069582486762E+01, 0.7457497555020480E-01, 0.9333627524262254E-01, 0.1494469242363992E+01, 0.1408587972834190E+01, 0.7455005714048079E-02, 0.8951422490882578E-02, 0.1691328825765317E+01, 0.1682736313985311E+01, 0.3589107121045615E-03, 0.1003553923945662E-03, 0.4705116269624868E+00, 0.4856665170586981E+00, 0.1525524548983380E-01, 0.1367304312976171E-01, 0.4352954686209429E+00, 0.4385509668224923E+00, 0.1692854133253905E-01, 0.1314659648329904E-01, 0.3405758644481647E+00, 0.3806452440729016E+00, 0.5321597610926388E-01, 0.3583592846566260E-01, 0.2563285827809251E+00, 0.2206431713856914E+00, 0.1505170716567330E-03, 0.6305450884558253E-04, 0.9243168664339420E-01, 0.9236863258188953E-01, 0.4485687980202113E-09, 0.5962992535266723E-20, 0.1227890724922389E-11, 0.4959852501050381E-23, 0.2408395422307052E-15, 0.1414000300319855E-15, 0.5186198053161721E+00},
			tauOut:   4.9598525010503808E-024,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1749149979302717E+01, 0.1749149979302717E+01, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.1117154804361126E+01, 0.1073347337297776E+01, 0.4438011869593646E-01, 0.4380746706334982E-01, 0.1178029724550133E+01, 0.1131758262754287E+01, 0.4427372589519608E-01, 0.9065158049178278E-01, 0.5998724677255393E+00, 0.5753431417100738E+00, 0.1369551627869799E+00, 0.6880305191066147E-01, 0.1150450694942405E+01, 0.1194069582486762E+01, 0.1142789997988411E+00, 0.9333627524262254E-01, 0.1303260395526232E+01, 0.1408587972834190E+01, 0.1155784656615063E-01, 0.8951422490882578E-02, 0.1671278822811555E+01, 0.1682736313985311E+01, 0.2916285016424428E-04, 0.1003553923945662E-03, 0.4993103973382956E+00, 0.4856665170586981E+00, 0.1200921574220688E-01, 0.1367304312976171E-01, 0.4396883475635844E+00, 0.4385509668224923E+00, 0.1138121911768345E-01, 0.1314659648329904E-01, 0.4050999534208807E+00, 0.3806452440729016E+00, 0.1951852336551518E-01, 0.3583592846566260E-01, 0.2011877025290218E+00, 0.2206431713856914E+00, 0.2894937755625969E-04, 0.6305450884558253E-04, 0.9233968320433328E-01, 0.9236863258188953E-01, 0.3202909346606844E-42, 0.5962992535266723E-20},
			i0:       1,
			n0:       12,
			pp:       0,
			n0in:     13,
			dmin:     4.9598525010503808E-024,
			dmin1:    9.2339683204333278E-002,
			dmin2:    0.20112464802017624,
			dn:       4.9598525010503808E-024,
			dn1:      9.2339683204333278E-002,
			dn2:      0.20112464802017624,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1749149979302717E+01, 0.1749149979302717E+01, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.1117154804361126E+01, 0.1073347337297776E+01, 0.4438011869593646E-01, 0.4380746706334982E-01, 0.1178029724550133E+01, 0.1131758262754287E+01, 0.4427372589519608E-01, 0.9065158049178278E-01, 0.5998724677255393E+00, 0.5753431417100738E+00, 0.1369551627869799E+00, 0.6880305191066147E-01, 0.1150450694942405E+01, 0.1194069582486762E+01, 0.1142789997988411E+00, 0.9333627524262254E-01, 0.1303260395526232E+01, 0.1408587972834190E+01, 0.1155784656615063E-01, 0.8951422490882578E-02, 0.1671278822811555E+01, 0.1682736313985311E+01, 0.2916285016424428E-04, 0.1003553923945662E-03, 0.4993103973382956E+00, 0.4856665170586981E+00, 0.1200921574220688E-01, 0.1367304312976171E-01, 0.4396883475635844E+00, 0.4385509668224923E+00, 0.1138121911768345E-01, 0.1314659648329904E-01, 0.4050999534208807E+00, 0.3806452440729016E+00, 0.1951852336551518E-01, 0.3583592846566260E-01, 0.2011877025290218E+00, 0.2206431713856914E+00, 0.2894937755625969E-04, 0.6305450884558253E-04, 0.9233968320433328E-01, 0.9236863258188953E-01, 0.3202909346606844E-42, 0.5962992535266723E-20},
			tauOut:   9.2159326345418235E-002,
			ttypeOut: -7,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1749149979302717E+01, 0.1656990652957299E+01, 0.8676862906242004E-16, 0.5850002270797901E-16, 0.1117154804361126E+01, 0.1069375596711644E+01, 0.4438011869593646E-01, 0.4888936980013561E-01, 0.1178029724550133E+01, 0.1081254754299776E+01, 0.4427372589519608E-01, 0.2456274906772994E-01, 0.5998724677255393E+00, 0.6201055550993710E+00, 0.1369551627869799E+00, 0.2540860356894923E+00, 0.1150450694942405E+01, 0.9184843327063353E+00, 0.1142789997988411E+00, 0.1621533314992303E+00, 0.1303260395526232E+01, 0.1060505584247734E+01, 0.1155784656615063E-01, 0.1821431635083262E-01, 0.1671278822811555E+01, 0.1560934342965469E+01, 0.2916285016424428E-04, 0.9328588590960435E-05, 0.4993103973382956E+00, 0.4191509581464933E+00, 0.1200921574220688E-01, 0.1259763844648080E-01, 0.4396883475635844E+00, 0.3463126018893689E+00, 0.1138121911768345E-01, 0.1331320693873929E-01, 0.4050999534208807E+00, 0.3191459435022383E+00, 0.1951852336551518E-01, 0.1230436091267282E-01, 0.2011877025290218E+00, 0.9675296464848704E-01, 0.2894937755625969E-04, 0.2762888312745317E-04, 0.9233968320433328E-01, 0.1527279757875810E-03, 0.3202909346606844E-42, 0.5850002270797901E-16, 0.7318275341991307E+00},
			i0:       1,
			n0:       12,
			pp:       1,
			n0in:     12,
			dmin:     1.5272797578758102E-004,
			dmin1:    9.6724015270930774E-002,
			dmin2:    0.29962742013672317,
			dn:       1.5272797578758102E-004,
			dn1:      9.6724015270930774E-002,
			dn2:      0.29962742013672317,
			tau:      9.2159326345418235E-002,
			ttype:    -7,
			g:        0.0000000000000000,
			zOut:     []float64{0.1749149979302717E+01, 0.1656990652957299E+01, 0.8676862906242004E-16, 0.5850002270797901E-16, 0.1117154804361126E+01, 0.1069375596711644E+01, 0.4438011869593646E-01, 0.4888936980013561E-01, 0.1178029724550133E+01, 0.1081254754299776E+01, 0.4427372589519608E-01, 0.2456274906772994E-01, 0.5998724677255393E+00, 0.6201055550993710E+00, 0.1369551627869799E+00, 0.2540860356894923E+00, 0.1150450694942405E+01, 0.9184843327063353E+00, 0.1142789997988411E+00, 0.1621533314992303E+00, 0.1303260395526232E+01, 0.1060505584247734E+01, 0.1155784656615063E-01, 0.1821431635083262E-01, 0.1671278822811555E+01, 0.1560934342965469E+01, 0.2916285016424428E-04, 0.9328588590960435E-05, 0.4993103973382956E+00, 0.4191509581464933E+00, 0.1200921574220688E-01, 0.1259763844648080E-01, 0.4396883475635844E+00, 0.3463126018893689E+00, 0.1138121911768345E-01, 0.1331320693873929E-01, 0.4050999534208807E+00, 0.3191459435022383E+00, 0.1951852336551518E-01, 0.1230436091267282E-01, 0.2011877025290218E+00, 0.9675296464848704E-01, 0.2894937755625969E-04, 0.2762888312745317E-04, 0.9233968320433328E-01, 0.1527279757875810E-03, 0.3202909346606844E-42, 0.5850002270797901E-16, 0.7318275341991307E+00},
			tauOut:   1.5267965277267402E-004,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1656837973304527E+01, 0.1656990652957299E+01, 0.3775776370348287E-16, 0.5850002270797901E-16, 0.1118112286859007E+01, 0.1069375596711644E+01, 0.4727777715386321E-01, 0.4888936980013561E-01, 0.1058387046560870E+01, 0.1081254754299776E+01, 0.1439123541326830E-01, 0.2456274906772994E-01, 0.8596476757228223E+00, 0.6201055550993710E+00, 0.2714763844897646E+00, 0.2540860356894923E+00, 0.8090086000630283E+00, 0.9184843327063353E+00, 0.2125620340079330E+00, 0.1621533314992303E+00, 0.8660051869378607E+00, 0.1060505584247734E+01, 0.3283046378299830E-01, 0.1821431635083262E-01, 0.1527960528118289E+01, 0.1560934342965469E+01, 0.2559023465659061E-05, 0.9328588590960435E-05, 0.4315933579167358E+00, 0.4191509581464933E+00, 0.1010840613748274E-01, 0.1259763844648080E-01, 0.3493647230378528E+00, 0.3463126018893689E+00, 0.1216166289646864E-01, 0.1331320693873929E-01, 0.3191359618656699E+00, 0.3191459435022383E+00, 0.3730332957296599E-02, 0.1230436091267282E-01, 0.9289758092154521E-01, 0.9675296464848704E-01, 0.4542317842367971E-07, 0.2762888312745317E-04, 0.2899836483324881E-08, 0.1527279757875810E-03, 0.3775776370348287E-16, 0.5850002270797901E-16},
			i0:       1,
			n0:       12,
			pp:       0,
			n0in:     12,
			dmin:     2.8998364833248812E-009,
			dmin1:    9.2869952038417761E-002,
			dmin2:    0.30683160095299705,
			dn:       2.8998364833248812E-009,
			dn1:      9.2869952038417761E-002,
			dn2:      0.30683160095299705,
			tau:      1.5267965277267402E-004,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1656837973304527E+01, 0.1656990652957299E+01, 0.3775776370348287E-16, 0.5850002270797901E-16, 0.1118112286859007E+01, 0.1069375596711644E+01, 0.4727777715386321E-01, 0.4888936980013561E-01, 0.1058387046560870E+01, 0.1081254754299776E+01, 0.1439123541326830E-01, 0.2456274906772994E-01, 0.8596476757228223E+00, 0.6201055550993710E+00, 0.2714763844897646E+00, 0.2540860356894923E+00, 0.8090086000630283E+00, 0.9184843327063353E+00, 0.2125620340079330E+00, 0.1621533314992303E+00, 0.8660051869378607E+00, 0.1060505584247734E+01, 0.3283046378299830E-01, 0.1821431635083262E-01, 0.1527960528118289E+01, 0.1560934342965469E+01, 0.2559023465659061E-05, 0.9328588590960435E-05, 0.4315933579167358E+00, 0.4191509581464933E+00, 0.1010840613748274E-01, 0.1259763844648080E-01, 0.3493647230378528E+00, 0.3463126018893689E+00, 0.1216166289646864E-01, 0.1331320693873929E-01, 0.3191359618656699E+00, 0.3191459435022383E+00, 0.3730332957296599E-02, 0.1230436091267282E-01, 0.9289758092154521E-01, 0.9675296464848704E-01, 0.4542317842367971E-07, 0.2762888312745317E-04, 0.2899836483324881E-08, 0.1527279757875810E-03, 0.3775776370348287E-16, 0.5850002270797901E-16},
			tauOut:   2.8998350258011044E-009,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1656837973304527E+01, 0.1656837970404692E+01, 0.3775776370348287E-16, 0.2548071704976161E-16, 0.1118112286859007E+01, 0.1165390061113035E+01, 0.4727777715386321E-01, 0.4293685745187325E-01, 0.1058387046560870E+01, 0.1029841421622430E+01, 0.1439123541326830E-01, 0.1201290976848257E-01, 0.8596476757228223E+00, 0.1119111147544269E+01, 0.2714763844897646E+00, 0.1962510428460807E+00, 0.8090086000630283E+00, 0.8253195883250457E+00, 0.2125620340079330E+00, 0.2230406579474441E+00, 0.8660051869378607E+00, 0.6757949898735797E+00, 0.3283046378299830E-01, 0.7422909836846008E-01, 0.1527960528118289E+01, 0.1453733985873459E+01, 0.2559023465659061E-05, 0.7597383986781573E-06, 0.4315933579167358E+00, 0.4417010014159849E+00, 0.1010840613748274E-01, 0.7995273950601426E-02, 0.3493647230378528E+00, 0.3535311090838850E+00, 0.1216166289646864E-01, 0.1097845108004235E-01, 0.3191359618656699E+00, 0.3118878408430891E+00, 0.3730332957296599E-02, 0.1111101051031713E-02, 0.9289758092154521E-01, 0.9178652239385691E-01, 0.4542317842367971E-07, 0.1435066789177947E-14, 0.2899836483324881E-08, 0.2245698748385924E-16, 0.3775776370348287E-16, 0.2548071704976161E-16, 0.7318275341991307E+00},
			i0:       1,
			n0:       12,
			pp:       1,
			n0in:     12,
			dmin:     2.2456987483859239E-017,
			dmin1:    9.1786476970678488E-002,
			dmin2:    0.30815750788579249,
			dn:       2.2456987483859239E-017,
			dn1:      9.1786476970678488E-002,
			dn2:      0.30815750788579249,
			tau:      2.8998350258011044E-009,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1656837973304527E+01, 0.1656837970404692E+01, 0.3775776370348287E-16, 0.2548071704976161E-16, 0.1118112286859007E+01, 0.1165390061113035E+01, 0.4727777715386321E-01, 0.4293685745187325E-01, 0.1058387046560870E+01, 0.1029841421622430E+01, 0.1439123541326830E-01, 0.1201290976848257E-01, 0.8596476757228223E+00, 0.1119111147544269E+01, 0.2714763844897646E+00, 0.1962510428460807E+00, 0.8090086000630283E+00, 0.8253195883250457E+00, 0.2125620340079330E+00, 0.2230406579474441E+00, 0.8660051869378607E+00, 0.6757949898735797E+00, 0.3283046378299830E-01, 0.7422909836846008E-01, 0.1527960528118289E+01, 0.1453733985873459E+01, 0.2559023465659061E-05, 0.7597383986781573E-06, 0.4315933579167358E+00, 0.4417010014159849E+00, 0.1010840613748274E-01, 0.7995273950601426E-02, 0.3493647230378528E+00, 0.3535311090838850E+00, 0.1216166289646864E-01, 0.1097845108004235E-01, 0.3191359618656699E+00, 0.3118878408430891E+00, 0.3730332957296599E-02, 0.1111101051031713E-02, 0.9289758092154521E-01, 0.9178652239385691E-01, 0.4542317842367971E-07, 0.1435066789177947E-14, 0.2899836483324881E-08, 0.2245698748385924E-16, 0.3775776370348287E-16, 0.2548071704976161E-16, 0.7318275341991307E+00},
			tauOut:   2.2456987483858885E-017,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1656837970404692E+01, 0.1656837970404692E+01, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.1208326918564908E+01, 0.1165390061113035E+01, 0.3659452888027460E-01, 0.4293685745187325E-01, 0.1005259802510638E+01, 0.1029841421622430E+01, 0.1337343958524596E-01, 0.1201290976848257E-01, 0.1301988750805104E+01, 0.1119111147544269E+01, 0.1244018658302015E+00, 0.1962510428460807E+00, 0.9239583804422884E+00, 0.8253195883250457E+00, 0.1631347930486186E+00, 0.2230406579474441E+00, 0.5868892951934211E+00, 0.6757949898735797E+00, 0.1838666404767374E+00, 0.7422909836846008E-01, 0.1269868105135121E+01, 0.1453733985873459E+01, 0.2642614694812039E-06, 0.7597383986781573E-06, 0.4496960111051168E+00, 0.4417010014159849E+00, 0.6285530663790794E-02, 0.7995273950601426E-02, 0.3582240295001366E+00, 0.3535311090838850E+00, 0.9558391177537082E-02, 0.1097845108004235E-01, 0.3034405507165837E+00, 0.3118878408430891E+00, 0.3360925270585024E-03, 0.1111101051031713E-02, 0.9145042986679984E-01, 0.9178652239385691E-01, 0.3524015903480299E-30, 0.1435066789177947E-14},
			i0:       1,
			n0:       11,
			pp:       0,
			n0in:     12,
			dmin:     2.2456987483858888E-017,
			dmin1:    9.1450429866798411E-002,
			dmin2:    0.30232944966555197,
			dn:       2.2456987483858888E-017,
			dn1:      9.1450429866798411E-002,
			dn2:      0.30232944966555197,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1656837970404692E+01, 0.1656837970404692E+01, 0.1792267857826344E-16, 0.2548071704976161E-16, 0.1208326918564908E+01, 0.1165390061113035E+01, 0.3659452888027460E-01, 0.4293685745187325E-01, 0.1005259802510638E+01, 0.1029841421622430E+01, 0.1337343958524596E-01, 0.1201290976848257E-01, 0.1301988750805104E+01, 0.1119111147544269E+01, 0.1244018658302015E+00, 0.1962510428460807E+00, 0.9239583804422884E+00, 0.8253195883250457E+00, 0.1631347930486186E+00, 0.2230406579474441E+00, 0.5868892951934211E+00, 0.6757949898735797E+00, 0.1838666404767374E+00, 0.7422909836846008E-01, 0.1269868105135121E+01, 0.1453733985873459E+01, 0.2642614694812039E-06, 0.7597383986781573E-06, 0.4496960111051168E+00, 0.4417010014159849E+00, 0.6285530663790794E-02, 0.7995273950601426E-02, 0.3582240295001366E+00, 0.3535311090838850E+00, 0.9558391177537082E-02, 0.1097845108004235E-01, 0.3034405507165837E+00, 0.3118878408430891E+00, 0.3360925270585024E-03, 0.1111101051031713E-02, 0.9145042986679984E-01, 0.9178652239385691E-01, 0.3524015903480299E-30, 0.1435066789177947E-14},
			tauOut:   9.1173077708044642E-002,
			ttypeOut: -7,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1656837970404692E+01, 0.1565664892696647E+01, 0.1792267857826344E-16, 0.1383211380667930E-16, 0.1208326918564908E+01, 0.1153748369737138E+01, 0.3659452888027460E-01, 0.3188477647299814E-01, 0.1005259802510638E+01, 0.8955753879148407E+00, 0.1337343958524596E-01, 0.1944232516271161E-01, 0.1301988750805104E+01, 0.1315775213764550E+01, 0.1244018658302015E+00, 0.8735697805677015E-01, 0.9239583804422884E+00, 0.9085631177260922E+00, 0.1631347930486186E+00, 0.1053774491236745E+00, 0.5868892951934211E+00, 0.5742054088384394E+00, 0.1838666404767374E+00, 0.4066251880351924E+00, 0.1269868105135121E+01, 0.7720701036533529E+00, 0.2642614694812039E-06, 0.1539203864417861E-06, 0.4496960111051168E+00, 0.3648083101404764E+00, 0.6285530663790794E-02, 0.6172085611379779E-02, 0.3582240295001366E+00, 0.2704372573582493E+00, 0.9558391177537082E-02, 0.1072486650400470E-01, 0.3034405507165837E+00, 0.2018786990315928E+00, 0.3360925270585024E-03, 0.1522488812438262E-03, 0.9145042986679984E-01, 0.1251032775113764E-03, 0.3524015903480299E-30, 0.1383211380667930E-16, 0.8241395430971566E+00},
			i0:       1,
			n0:       11,
			pp:       1,
			n0in:     11,
			dmin:     1.2510327751137640E-004,
			dmin1:    0.20154260650453434,
			dmin2:    0.26087886618071221,
			dn:       1.2510327751137640E-004,
			dn1:      0.20154260650453434,
			dn2:      0.26087886618071221,
			tau:      9.1173077708044642E-002,
			ttype:    -7,
			g:        0.0000000000000000,
			zOut:     []float64{0.1656837970404692E+01, 0.1565664892696647E+01, 0.1792267857826344E-16, 0.1383211380667930E-16, 0.1208326918564908E+01, 0.1153748369737138E+01, 0.3659452888027460E-01, 0.3188477647299814E-01, 0.1005259802510638E+01, 0.8955753879148407E+00, 0.1337343958524596E-01, 0.1944232516271161E-01, 0.1301988750805104E+01, 0.1315775213764550E+01, 0.1244018658302015E+00, 0.8735697805677015E-01, 0.9239583804422884E+00, 0.9085631177260922E+00, 0.1631347930486186E+00, 0.1053774491236745E+00, 0.5868892951934211E+00, 0.5742054088384394E+00, 0.1838666404767374E+00, 0.4066251880351924E+00, 0.1269868105135121E+01, 0.7720701036533529E+00, 0.2642614694812039E-06, 0.1539203864417861E-06, 0.4496960111051168E+00, 0.3648083101404764E+00, 0.6285530663790794E-02, 0.6172085611379779E-02, 0.3582240295001366E+00, 0.2704372573582493E+00, 0.9558391177537082E-02, 0.1072486650400470E-01, 0.3034405507165837E+00, 0.2018786990315928E+00, 0.3360925270585024E-03, 0.1522488812438262E-03, 0.9145042986679984E-01, 0.1251032775113764E-03, 0.3524015903480299E-30, 0.1383211380667930E-16, 0.8241395430971566E+00},
			tauOut:   1.2498058228587147E-004,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1565539912114361E+01, 0.1565664892696647E+01, 0.1019378594629470E-16, 0.1383211380667930E-16, 0.1185508165627851E+01, 0.1153748369737138E+01, 0.2408690373149840E-01, 0.3188477647299814E-01, 0.8908058287637680E+00, 0.8955753879148407E+00, 0.2871751477260568E-01, 0.1944232516271161E-01, 0.1374289696466428E+01, 0.1315775213764550E+01, 0.5775298217141787E-01, 0.8735697805677015E-01, 0.9560626040960629E+00, 0.9085631177260922E+00, 0.6328905763825028E-01, 0.1053774491236745E+00, 0.9174165586530958E+00, 0.5742054088384394E+00, 0.3422034931823232E+00, 0.4066251880351924E+00, 0.4297417838091302E+00, 0.7720701036533529E+00, 0.1306631986684747E-06, 0.1539203864417861E-06, 0.3708552845063717E+00, 0.3648083101404764E+00, 0.4500844331080801E-02, 0.6172085611379779E-02, 0.2765362989488873E+00, 0.2704372573582493E+00, 0.7829431815445537E-02, 0.1072486650400470E-01, 0.1940765355151052E+00, 0.2018786990315928E+00, 0.9814083907923291E-07, 0.1522488812438262E-03, 0.2455438642568072E-07, 0.1251032775113764E-03, 0.1019378594629470E-16, 0.1383211380667930E-16},
			i0:       1,
			n0:       11,
			pp:       0,
			n0in:     11,
			dmin:     2.4554386425680716E-008,
			dmin1:    0.19392428663386141,
			dmin2:    0.26581143244488259,
			dn:       2.4554386425680716E-008,
			dn1:      0.19392428663386141,
			dn2:      0.26581143244488259,
			tau:      1.2498058228587147E-004,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1565539912114361E+01, 0.1565664892696647E+01, 0.1019378594629470E-16, 0.1383211380667930E-16, 0.1185508165627851E+01, 0.1153748369737138E+01, 0.2408690373149840E-01, 0.3188477647299814E-01, 0.8908058287637680E+00, 0.8955753879148407E+00, 0.2871751477260568E-01, 0.1944232516271161E-01, 0.1374289696466428E+01, 0.1315775213764550E+01, 0.5775298217141787E-01, 0.8735697805677015E-01, 0.9560626040960629E+00, 0.9085631177260922E+00, 0.6328905763825028E-01, 0.1053774491236745E+00, 0.9174165586530958E+00, 0.5742054088384394E+00, 0.3422034931823232E+00, 0.4066251880351924E+00, 0.4297417838091302E+00, 0.7720701036533529E+00, 0.1306631986684747E-06, 0.1539203864417861E-06, 0.3708552845063717E+00, 0.3648083101404764E+00, 0.4500844331080801E-02, 0.6172085611379779E-02, 0.2765362989488873E+00, 0.2704372573582493E+00, 0.7829431815445537E-02, 0.1072486650400470E-01, 0.1940765355151052E+00, 0.2018786990315928E+00, 0.9814083907923291E-07, 0.1522488812438262E-03, 0.2455438642568072E-07, 0.1251032775113764E-03, 0.1019378594629470E-16, 0.1383211380667930E-16},
			tauOut:   2.4554370888251911E-008,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1565539912114361E+01, 0.1565539887559990E+01, 0.1019378594629470E-16, 0.7719264500395369E-17, 0.1185508165627851E+01, 0.1209595044804978E+01, 0.2408690373149840E-01, 0.1773879145177052E-01, 0.8908058287637680E+00, 0.9017845275302323E+00, 0.2871751477260568E-01, 0.4376453959373496E-01, 0.1374289696466428E+01, 0.1388278114489740E+01, 0.5775298217141787E-01, 0.3977262621431848E-01, 0.9560626040960629E+00, 0.9795790109656241E+00, 0.6328905763825028E-01, 0.5927283946360358E-01, 0.9174165586530958E+00, 0.1200347187817445E+01, 0.3422034931823232E+00, 0.1225138368952072E+00, 0.4297417838091302E+00, 0.3072280530227507E+00, 0.1306631986684747E-06, 0.1577236754259593E-06, 0.3708552845063717E+00, 0.3753559465594062E+00, 0.4500844331080801E-02, 0.3315910790466669E-02, 0.2765362989488873E+00, 0.2810497954194954E+00, 0.7829431815445537E-02, 0.5406547261581845E-02, 0.1940765355151052E+00, 0.1886700618399915E+00, 0.9814083907923291E-07, 0.1277249852674422E-13, 0.2455438642568072E-07, 0.2764930279233778E-14, 0.1019378594629470E-16, 0.7719264500395369E-17, 0.8241395430971566E+00},
			i0:       1,
			n0:       11,
			pp:       1,
			n0in:     11,
			dmin:     2.7649302792337775E-015,
			dmin1:    0.18866996369915248,
			dmin2:    0.27322036360404983,
			dn:       2.7649302792337775E-015,
			dn1:      0.18866996369915248,
			dn2:      0.27322036360404983,
			tau:      2.4554370888251911E-008,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1565539912114361E+01, 0.1565539887559990E+01, 0.1019378594629470E-16, 0.7719264500395369E-17, 0.1185508165627851E+01, 0.1209595044804978E+01, 0.2408690373149840E-01, 0.1773879145177052E-01, 0.8908058287637680E+00, 0.9017845275302323E+00, 0.2871751477260568E-01, 0.4376453959373496E-01, 0.1374289696466428E+01, 0.1388278114489740E+01, 0.5775298217141787E-01, 0.3977262621431848E-01, 0.9560626040960629E+00, 0.9795790109656241E+00, 0.6328905763825028E-01, 0.5927283946360358E-01, 0.9174165586530958E+00, 0.1200347187817445E+01, 0.3422034931823232E+00, 0.1225138368952072E+00, 0.4297417838091302E+00, 0.3072280530227507E+00, 0.1306631986684747E-06, 0.1577236754259593E-06, 0.3708552845063717E+00, 0.3753559465594062E+00, 0.4500844331080801E-02, 0.3315910790466669E-02, 0.2765362989488873E+00, 0.2810497954194954E+00, 0.7829431815445537E-02, 0.5406547261581845E-02, 0.1940765355151052E+00, 0.1886700618399915E+00, 0.9814083907923291E-07, 0.1277249852674422E-13, 0.2455438642568072E-07, 0.2764930279233778E-14, 0.1019378594629470E-16, 0.7719264500395369E-17, 0.8241395430971566E+00},
			tauOut:   2.7649302792335523E-015,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1565539887559988E+01, 0.1565539887559990E+01, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.1227333836256746E+01, 0.1209595044804978E+01, 0.1303359134714337E-01, 0.1773879145177052E-01, 0.9325154757768211E+00, 0.9017845275302323E+00, 0.6515425651042277E-01, 0.4376453959373496E-01, 0.1362896484193633E+01, 0.1388278114489740E+01, 0.2858649229958118E-01, 0.3977262621431848E-01, 0.1010265358129644E+01, 0.9795790109656241E+00, 0.7042504782685149E-01, 0.5927283946360358E-01, 0.1252435976885798E+01, 0.1200347187817445E+01, 0.3005318297487194E-01, 0.1225138368952072E+00, 0.2771750277715515E+00, 0.3072280530227507E+00, 0.2135925446109612E-06, 0.1577236754259593E-06, 0.3786716437573254E+00, 0.3753559465594062E+00, 0.2461066374135985E-02, 0.3315910790466669E-02, 0.2839952763069384E+00, 0.2810497954194954E+00, 0.3591797791316158E-02, 0.5406547261581845E-02, 0.1850782640486854E+00, 0.1886700618399915E+00, 0.1908115364037247E-27, 0.1277249852674422E-13},
			i0:       1,
			n0:       10,
			pp:       0,
			n0in:     11,
			dmin:     3.4709879829724519E-029,
			dmin1:    0.18507826404867261,
			dmin2:    0.27717487004787600,
			dn:       3.4709879829724519E-029,
			dn1:      0.18507826404867261,
			dn2:      0.27858872904535659,
			tau:      2.7649302792335523E-015,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1565539887559988E+01, 0.1565539887559990E+01, 0.5964194309842799E-17, 0.7719264500395369E-17, 0.1227333836256746E+01, 0.1209595044804978E+01, 0.1303359134714337E-01, 0.1773879145177052E-01, 0.9325154757768211E+00, 0.9017845275302323E+00, 0.6515425651042277E-01, 0.4376453959373496E-01, 0.1362896484193633E+01, 0.1388278114489740E+01, 0.2858649229958118E-01, 0.3977262621431848E-01, 0.1010265358129644E+01, 0.9795790109656241E+00, 0.7042504782685149E-01, 0.5927283946360358E-01, 0.1252435976885798E+01, 0.1200347187817445E+01, 0.3005318297487194E-01, 0.1225138368952072E+00, 0.2771750277715515E+00, 0.3072280530227507E+00, 0.2135925446109612E-06, 0.1577236754259593E-06, 0.3786716437573254E+00, 0.3753559465594062E+00, 0.2461066374135985E-02, 0.3315910790466669E-02, 0.2839952763069384E+00, 0.2810497954194954E+00, 0.3591797791316158E-02, 0.5406547261581845E-02, 0.1850782640486854E+00, 0.1886700618399915E+00, 0.1908115364037247E-27, 0.1277249852674422E-13},
			tauOut:   9.2539132024336307E-002,
			ttypeOut: -9,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1565539887559988E+01, 0.1473000755535651E+01, 0.5964194309842799E-17, 0.4969486576955697E-17, 0.1227333836256746E+01, 0.1147828295579553E+01, 0.1303359134714337E-01, 0.1058871408116432E-01, 0.9325154757768211E+00, 0.8945418861817434E+00, 0.6515425651042277E-01, 0.9926701980086396E-01, 0.1362896484193633E+01, 0.1199676824668014E+01, 0.2858649229958118E-01, 0.2407310226126826E-01, 0.1010265358129644E+01, 0.9640781716708908E+00, 0.7042504782685149E-01, 0.9148932748822959E-01, 0.1252435976885798E+01, 0.1098460700348104E+01, 0.3005318297487194E-01, 0.7583331677723075E-02, 0.2771750277715515E+00, 0.1770527776620367E+00, 0.2135925446109612E-06, 0.4568210735249311E-06, 0.3786716437573254E+00, 0.2885931212860515E+00, 0.2461066374135985E-02, 0.2421856840585221E-02, 0.2839952763069384E+00, 0.1926260852333330E+00, 0.3591797791316158E-02, 0.3451057520197492E-02, 0.1850782640486854E+00, 0.8908807450415157E-01, 0.1908115364037247E-27, 0.4969486576955697E-17, 0.9154376259418607E+00},
			i0:       1,
			n0:       10,
			pp:       1,
			n0in:     10,
			dmin:     8.9088074504151571E-002,
			dmin1:    0.17705256406949207,
			dmin2:    0.17705256406949207,
			dn:       8.9088074504151571E-002,
			dn1:      0.18903428744201686,
			dn2:      0.28613205491191551,
			tau:      9.2539132024336307E-002,
			ttype:    -9,
			g:        0.0000000000000000,
			zOut:     []float64{0.1565539887559988E+01, 0.1473000755535651E+01, 0.5964194309842799E-17, 0.4969486576955697E-17, 0.1227333836256746E+01, 0.1147828295579553E+01, 0.1303359134714337E-01, 0.1058871408116432E-01, 0.9325154757768211E+00, 0.8945418861817434E+00, 0.6515425651042277E-01, 0.9926701980086396E-01, 0.1362896484193633E+01, 0.1199676824668014E+01, 0.2858649229958118E-01, 0.2407310226126826E-01, 0.1010265358129644E+01, 0.9640781716708908E+00, 0.7042504782685149E-01, 0.9148932748822959E-01, 0.1252435976885798E+01, 0.1098460700348104E+01, 0.3005318297487194E-01, 0.7583331677723075E-02, 0.2771750277715515E+00, 0.1770527776620367E+00, 0.2135925446109612E-06, 0.4568210735249311E-06, 0.3786716437573254E+00, 0.2885931212860515E+00, 0.2461066374135985E-02, 0.2421856840585221E-02, 0.2839952763069384E+00, 0.1926260852333330E+00, 0.3591797791316158E-02, 0.3451057520197492E-02, 0.1850782640486854E+00, 0.8908807450415157E-01, 0.1908115364037247E-27, 0.4969486576955697E-17, 0.9154376259418607E+00},
			tauOut:   7.5387921832037685E-002,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1397612833703614E+01, 0.1473000755535651E+01, 0.4081328655531061E-17, 0.4969486576955697E-17, 0.1083029087828680E+01, 0.1147828295579553E+01, 0.8745885380967966E-02, 0.1058871408116432E-01, 0.9096750987696016E+00, 0.8945418861817434E+00, 0.1309130515609722E+00, 0.9926701980086396E-01, 0.1017448953536272E+01, 0.1199676824668014E+01, 0.2281033592282574E-01, 0.2407310226126826E-01, 0.9573692414042568E+00, 0.9640781716708908E+00, 0.1049724875218357E+00, 0.9148932748822959E-01, 0.9256836226719531E+00, 0.1098460700348104E+01, 0.1450441494900679E-02, 0.7583331677723075E-02, 0.1002148711561719E+00, 0.1770527776620367E+00, 0.1315527505616969E-05, 0.4568210735249311E-06, 0.2156257407670935E+00, 0.2885931212860515E+00, 0.2163530200698068E-02, 0.2421856840585221E-02, 0.1185256907207948E+00, 0.1926260852333330E+00, 0.2593936112987583E-02, 0.3451057520197492E-02, 0.1110621655912630E-01, 0.8908807450415157E-01, 0.4081328655531061E-17, 0.4969486576955697E-17},
			i0:       1,
			n0:       10,
			pp:       0,
			n0in:     10,
			dmin:     1.1106216559126303E-002,
			dmin1:    0.10021441433509834,
			dmin2:    0.10021441433509834,
			dn:       1.1106216559126303E-002,
			dn1:      0.11507463320059727,
			dn2:      0.21320388392650824,
			tau:      7.5387921832037685E-002,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1397612833703614E+01, 0.1473000755535651E+01, 0.4081328655531061E-17, 0.4969486576955697E-17, 0.1083029087828680E+01, 0.1147828295579553E+01, 0.8745885380967966E-02, 0.1058871408116432E-01, 0.9096750987696016E+00, 0.8945418861817434E+00, 0.1309130515609722E+00, 0.9926701980086396E-01, 0.1017448953536272E+01, 0.1199676824668014E+01, 0.2281033592282574E-01, 0.2407310226126826E-01, 0.9573692414042568E+00, 0.9640781716708908E+00, 0.1049724875218357E+00, 0.9148932748822959E-01, 0.9256836226719531E+00, 0.1098460700348104E+01, 0.1450441494900679E-02, 0.7583331677723075E-02, 0.1002148711561719E+00, 0.1770527776620367E+00, 0.1315527505616969E-05, 0.4568210735249311E-06, 0.2156257407670935E+00, 0.2885931212860515E+00, 0.2163530200698068E-02, 0.2421856840585221E-02, 0.1185256907207948E+00, 0.1926260852333330E+00, 0.2593936112987583E-02, 0.3451057520197492E-02, 0.1110621655912630E-01, 0.8908807450415157E-01, 0.4081328655531061E-17, 0.4969486576955697E-17},
			tauOut:   9.2006637361187298E-003,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1397612833703614E+01, 0.1388412169967495E+01, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.1083029087828680E+01, 0.1082574309473529E+01, 0.8745885380967966E-02, 0.7349069784991225E-02, 0.9096750987696016E+00, 0.1024038416809464E+01, 0.1309130515609722E+00, 0.1300706547025319E+00, 0.1017448953536272E+01, 0.9009879710204475E+00, 0.2281033592282574E-01, 0.2423774201322422E-01, 0.9573692414042568E+00, 0.1028903323176749E+01, 0.1049724875218357E+00, 0.9444163542020824E-01, 0.9256836226719531E+00, 0.8234917650105269E+00, 0.1450441494900679E-02, 0.1765115496075164E-03, 0.1002148711561719E+00, 0.9083901139795122E-01, 0.1315527505616969E-05, 0.3122684720284652E-05, 0.2156257407670935E+00, 0.2085854845469525E+00, 0.2163530200698068E-02, 0.1229394806594584E-02, 0.1185256907207948E+00, 0.1106895682910690E+00, 0.2593936112987583E-02, 0.2602667681892373E-03, 0.1110621655912630E-01, 0.1645286054818337E-02, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.9154376259418607E+00},
			i0:       1,
			n0:       10,
			pp:       1,
			n0in:     10,
			dmin:     1.6452860548183366E-003,
			dmin1:    9.0837695870445614E-002,
			dmin2:    9.0837695870445614E-002,
			dn:       1.6452860548183366E-003,
			dn1:      0.10809563217808144,
			dn2:      0.20642195434625446,
			tau:      9.2006637361187298E-003,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1397612833703614E+01, 0.1388412169967495E+01, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.1083029087828680E+01, 0.1082574309473529E+01, 0.8745885380967966E-02, 0.7349069784991225E-02, 0.9096750987696016E+00, 0.1024038416809464E+01, 0.1309130515609722E+00, 0.1300706547025319E+00, 0.1017448953536272E+01, 0.9009879710204475E+00, 0.2281033592282574E-01, 0.2423774201322422E-01, 0.9573692414042568E+00, 0.1028903323176749E+01, 0.1049724875218357E+00, 0.9444163542020824E-01, 0.9256836226719531E+00, 0.8234917650105269E+00, 0.1450441494900679E-02, 0.1765115496075164E-03, 0.1002148711561719E+00, 0.9083901139795122E-01, 0.1315527505616969E-05, 0.3122684720284652E-05, 0.2156257407670935E+00, 0.2085854845469525E+00, 0.2163530200698068E-02, 0.1229394806594584E-02, 0.1185256907207948E+00, 0.1106895682910690E+00, 0.2593936112987583E-02, 0.2602667681892373E-03, 0.1110621655912630E-01, 0.1645286054818337E-02, 0.4081328655531061E-17, 0.3183635051997810E-17, 0.9154376259418607E+00},
			tauOut:   1.5594219393746818E-003,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1386852748028120E+01, 0.1388412169967495E+01, 0.2485138759635906E-17, 0.3183635051997810E-17, 0.1088363957319145E+01, 0.1082574309473529E+01, 0.6914717946174950E-02, 0.7349069784991225E-02, 0.1145634931626446E+01, 0.1024038416809464E+01, 0.1022944500333619E+00, 0.1300706547025319E+00, 0.8213718410609353E+00, 0.9009879710204475E+00, 0.3036175828902931E-01, 0.2423774201322422E-01, 0.1091423778368554E+01, 0.1028903323176749E+01, 0.7125729765473905E-01, 0.9444163542020824E-01, 0.7508515569660207E+00, 0.8234917650105269E+00, 0.2135460001102832E-04, 0.1765115496075164E-03, 0.8926135754328580E-01, 0.9083901139795122E-01, 0.7297073710223142E-05, 0.3122684720284652E-05, 0.2082481603404622E+00, 0.2085854845469525E+00, 0.6534568189162364E-03, 0.1229394806594584E-02, 0.1087369563009673E+00, 0.1106895682910690E+00, 0.3938065757966957E-05, 0.2602667681892373E-03, 0.8192604968568760E-04, 0.1645286054818337E-02, 0.2485138759635906E-17, 0.3183635051997810E-17},
			i0:       1,
			n0:       10,
			pp:       0,
			n0in:     10,
			dmin:     8.1926049685687600E-005,
			dmin1:    8.9258234858565516E-002,
			dmin2:    8.9258234858565516E-002,
			dn:       8.1926049685687600E-005,
			dn1:      0.10847668953277810,
			dn2:      0.20701876553386761,
			tau:      1.5594219393746818E-003,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1386852748028120E+01, 0.1388412169967495E+01, 0.2485138759635906E-17, 0.3183635051997810E-17, 0.1088363957319145E+01, 0.1082574309473529E+01, 0.6914717946174950E-02, 0.7349069784991225E-02, 0.1145634931626446E+01, 0.1024038416809464E+01, 0.1022944500333619E+00, 0.1300706547025319E+00, 0.8213718410609353E+00, 0.9009879710204475E+00, 0.3036175828902931E-01, 0.2423774201322422E-01, 0.1091423778368554E+01, 0.1028903323176749E+01, 0.7125729765473905E-01, 0.9444163542020824E-01, 0.7508515569660207E+00, 0.8234917650105269E+00, 0.2135460001102832E-04, 0.1765115496075164E-03, 0.8926135754328580E-01, 0.9083901139795122E-01, 0.7297073710223142E-05, 0.3122684720284652E-05, 0.2082481603404622E+00, 0.2085854845469525E+00, 0.6534568189162364E-03, 0.1229394806594584E-02, 0.1087369563009673E+00, 0.1106895682910690E+00, 0.3938065757966957E-05, 0.2602667681892373E-03, 0.8192604968568760E-04, 0.1645286054818337E-02, 0.2485138759635906E-17, 0.3183635051997810E-17},
			tauOut:   8.1416944590412474E-005,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1386852748028120E+01, 0.1386771331083530E+01, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.1088363957319145E+01, 0.1095197258320730E+01, 0.6914717946174950E-02, 0.7233164949324962E-02, 0.1145634931626446E+01, 0.1240614799765893E+01, 0.1022944500333619E+00, 0.6772592167212049E-01, 0.8213718410609353E+00, 0.7839262607332537E+00, 0.3036175828902931E-01, 0.4227125255215914E-01, 0.1091423778368554E+01, 0.1120328406526543E+01, 0.7125729765473905E-01, 0.4775711530437247E-01, 0.7508515569660207E+00, 0.7030343793170689E+00, 0.2135460001102832E-04, 0.2711304941630149E-05, 0.8926135754328580E-01, 0.8918452636746399E-01, 0.7297073710223142E-05, 0.1703885458517271E-04, 0.2082481603404622E+00, 0.2088031613602029E+00, 0.6534568189162364E-03, 0.3402961195615630E-03, 0.1087369563009673E+00, 0.1083191813025733E+00, 0.3938065757966957E-05, 0.2978513750500819E-08, 0.8192604968568760E-04, 0.5061265815246250E-06, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.9154376259418607E+00},
			i0:       1,
			n0:       10,
			pp:       1,
			n0in:     10,
			dmin:     5.0612658152462498E-007,
			dmin1:    8.9177229293753768E-002,
			dmin2:    8.9177229293753768E-002,
			dn:       5.0612658152462498E-007,
			dn1:      0.10831524323681536,
			dn2:      0.20814970454128662,
			tau:      8.1416944590412474E-005,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1386852748028120E+01, 0.1386771331083530E+01, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.1088363957319145E+01, 0.1095197258320730E+01, 0.6914717946174950E-02, 0.7233164949324962E-02, 0.1145634931626446E+01, 0.1240614799765893E+01, 0.1022944500333619E+00, 0.6772592167212049E-01, 0.8213718410609353E+00, 0.7839262607332537E+00, 0.3036175828902931E-01, 0.4227125255215914E-01, 0.1091423778368554E+01, 0.1120328406526543E+01, 0.7125729765473905E-01, 0.4775711530437247E-01, 0.7508515569660207E+00, 0.7030343793170689E+00, 0.2135460001102832E-04, 0.2711304941630149E-05, 0.8926135754328580E-01, 0.8918452636746399E-01, 0.7297073710223142E-05, 0.1703885458517271E-04, 0.2082481603404622E+00, 0.2088031613602029E+00, 0.6534568189162364E-03, 0.3402961195615630E-03, 0.1087369563009673E+00, 0.1083191813025733E+00, 0.3938065757966957E-05, 0.2978513750500819E-08, 0.8192604968568760E-04, 0.5061265815246250E-06, 0.2485138759635906E-17, 0.1950383162890474E-17, 0.9154376259418607E+00},
			tauOut:   5.0604049633765406E-007,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1386770825043033E+01, 0.1386771331083530E+01, 0.1540308069724697E-17, 0.1950383162890474E-17, 0.1102429917229558E+01, 0.1095197258320730E+01, 0.8139811288713328E-02, 0.7233164949324962E-02, 0.1300200404108803E+01, 0.1240614799765893E+01, 0.4083380405309871E-01, 0.6772592167212049E-01, 0.7853632031918177E+00, 0.7839262607332537E+00, 0.6030036143936175E-01, 0.4227125255215914E-01, 0.1107784654351057E+01, 0.1120328406526543E+01, 0.3030814137396716E-01, 0.4775711530437247E-01, 0.6727284432075471E+00, 0.7030343793170689E+00, 0.3594413905024206E-06, 0.2711304941630149E-05, 0.8920069974016231E-01, 0.8918452636746399E-01, 0.3988496406087022E-04, 0.1703885458517271E-04, 0.2091030664752072E+00, 0.2088031613602029E+00, 0.1762795624794033E-03, 0.3402961195615630E-03, 0.1081423986781113E+00, 0.1083191813025733E+00, 0.1393999949133917E-13, 0.2978513750500819E-08, 0.8607124697139263E-10, 0.5061265815246250E-06, 0.1540308069724697E-17, 0.1950383162890474E-17},
			i0:       1,
			n0:       10,
			pp:       0,
			n0in:     10,
			dmin:     8.6071246971392626E-011,
			dmin1:    8.9183660885577137E-002,
			dmin2:    8.9183660885577137E-002,
			dn:       8.6071246971392626E-011,
			dn1:      0.10814239569959758,
			dn2:      0.20876277035564564,
			tau:      5.0604049633765406E-007,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1386770825043033E+01, 0.1386771331083530E+01, 0.1540308069724697E-17, 0.1950383162890474E-17, 0.1102429917229558E+01, 0.1095197258320730E+01, 0.8139811288713328E-02, 0.7233164949324962E-02, 0.1300200404108803E+01, 0.1240614799765893E+01, 0.4083380405309871E-01, 0.6772592167212049E-01, 0.7853632031918177E+00, 0.7839262607332537E+00, 0.6030036143936175E-01, 0.4227125255215914E-01, 0.1107784654351057E+01, 0.1120328406526543E+01, 0.3030814137396716E-01, 0.4775711530437247E-01, 0.6727284432075471E+00, 0.7030343793170689E+00, 0.3594413905024206E-06, 0.2711304941630149E-05, 0.8920069974016231E-01, 0.8918452636746399E-01, 0.3988496406087022E-04, 0.1703885458517271E-04, 0.2091030664752072E+00, 0.2088031613602029E+00, 0.1762795624794033E-03, 0.3402961195615630E-03, 0.1081423986781113E+00, 0.1083191813025733E+00, 0.1393999949133917E-13, 0.2978513750500819E-08, 0.8607124697139263E-10, 0.5061265815246250E-06, 0.1540308069724697E-17, 0.1950383162890474E-17},
			tauOut:   8.6071215292546838E-011,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1386770825043033E+01, 0.1386770824956962E+01, 0.1540308069724697E-17, 0.1224486171222500E-17, 0.1102429917229558E+01, 0.1110569728432201E+01, 0.8139811288713328E-02, 0.9529690622753703E-02, 0.1300200404108803E+01, 0.1331504517453077E+01, 0.4083380405309871E-01, 0.2408506071837550E-01, 0.7853632031918177E+00, 0.8215785038267327E+00, 0.6030036143936175E-01, 0.8130667336500198E-01, 0.1107784654351057E+01, 0.1056786122273951E+01, 0.3030814137396716E-01, 0.1929354325655847E-01, 0.6727284432075471E+00, 0.6534352593063080E+00, 0.3594413905024206E-06, 0.4906748310831983E-07, 0.8920069974016231E-01, 0.8924053555066887E-01, 0.3988496406087022E-04, 0.9345605379795243E-04, 0.2091030664752072E+00, 0.2091858898978174E+00, 0.1762795624794033E-03, 0.9113088236382798E-04, 0.1081423986781113E+00, 0.1080512677096902E+00, 0.1393999949133917E-13, 0.1110429488179469E-22, 0.8607124697139263E-10, 0.3167883469916549E-16, 0.1540308069724697E-17, 0.1224486171222500E-17, 0.9154376259418607E+00},
			i0:       1,
			n0:       10,
			pp:       1,
			n0in:     10,
			dmin:     3.1678834699165494E-017,
			dmin1:    8.9200650586607991E-002,
			dmin2:    8.9200650586607991E-002,
			dn:       3.1678834699165494E-017,
			dn1:      0.10805126770967630,
			dn2:      0.20900961033533805,
			tau:      8.6071215292546838E-011,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1386770825043033E+01, 0.1386770824956962E+01, 0.1540308069724697E-17, 0.1224486171222500E-17, 0.1102429917229558E+01, 0.1110569728432201E+01, 0.8139811288713328E-02, 0.9529690622753703E-02, 0.1300200404108803E+01, 0.1331504517453077E+01, 0.4083380405309871E-01, 0.2408506071837550E-01, 0.7853632031918177E+00, 0.8215785038267327E+00, 0.6030036143936175E-01, 0.8130667336500198E-01, 0.1107784654351057E+01, 0.1056786122273951E+01, 0.3030814137396716E-01, 0.1929354325655847E-01, 0.6727284432075471E+00, 0.6534352593063080E+00, 0.3594413905024206E-06, 0.4906748310831983E-07, 0.8920069974016231E-01, 0.8924053555066887E-01, 0.3988496406087022E-04, 0.9345605379795243E-04, 0.2091030664752072E+00, 0.2091858898978174E+00, 0.1762795624794033E-03, 0.9113088236382798E-04, 0.1081423986781113E+00, 0.1080512677096902E+00, 0.1393999949133917E-13, 0.1110429488179469E-22, 0.8607124697139263E-10, 0.3167883469916549E-16, 0.1540308069724697E-17, 0.1224486171222500E-17, 0.9154376259418607E+00},
			tauOut:   3.1678834698836348E-017,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1386770824956962E+01, 0.1386770824956962E+01, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.1120099419054954E+01, 0.1110569728432201E+01, 0.1132830345080667E-01, 0.9529690622753703E-02, 0.1344261274720646E+01, 0.1331504517453077E+01, 0.1472018016266302E-01, 0.2408506071837550E-01, 0.8881649970290716E+00, 0.8215785038267327E+00, 0.9674301998819117E-01, 0.8130667336500198E-01, 0.9793366455423188E+00, 0.1056786122273951E+01, 0.1287308250760436E-01, 0.1929354325655847E-01, 0.6405622258661866E+00, 0.6534352593063080E+00, 0.6835883063177366E-08, 0.4906748310831983E-07, 0.8933398476858376E-01, 0.8924053555066887E-01, 0.2188381927740679E-03, 0.9345605379795243E-04, 0.2090581825874072E+00, 0.2091858898978174E+00, 0.4710079866305792E-04, 0.9113088236382798E-04, 0.1080041669110272E+00, 0.1080512677096902E+00, 0.3257014354834561E-38, 0.1110429488179469E-22},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     10,
			dmin:     3.1678834699165494E-017,
			dmin1:    8.9240528714785800E-002,
			dmin2:    8.9240528714785800E-002,
			dn:       3.1678834699165494E-017,
			dn1:      0.10800416691102718,
			dn2:      0.20896705170504334,
			tau:      0.0000000000000000,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1386770824956962E+01, 0.1386770824956962E+01, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.1120099419054954E+01, 0.1110569728432201E+01, 0.1132830345080667E-01, 0.9529690622753703E-02, 0.1344261274720646E+01, 0.1331504517453077E+01, 0.1472018016266302E-01, 0.2408506071837550E-01, 0.8881649970290716E+00, 0.8215785038267327E+00, 0.9674301998819117E-01, 0.8130667336500198E-01, 0.9793366455423188E+00, 0.1056786122273951E+01, 0.1287308250760436E-01, 0.1929354325655847E-01, 0.6405622258661866E+00, 0.6534352593063080E+00, 0.6835883063177366E-08, 0.4906748310831983E-07, 0.8933398476858376E-01, 0.8924053555066887E-01, 0.2188381927740679E-03, 0.9345605379795243E-04, 0.2090581825874072E+00, 0.2091858898978174E+00, 0.4710079866305792E-04, 0.9113088236382798E-04, 0.1080041669110272E+00, 0.1080512677096902E+00, 0.3257014354834561E-38, 0.1110429488179469E-22},
			tauOut:   2.2310132178696450E-002,
			ttypeOut: -9,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1386770824956962E+01, 0.1364460692778266E+01, 0.9806070694382835E-18, 0.8049901434408684E-18, 0.1120099419054954E+01, 0.1109117590327065E+01, 0.1132830345080667E-01, 0.1373001363427395E-01, 0.1344261274720646E+01, 0.1322941309070339E+01, 0.1472018016266302E-01, 0.9882485852396858E-02, 0.8881649970290716E+00, 0.9527153989861693E+00, 0.9674301998819117E-01, 0.9944626147083412E-01, 0.9793366455423188E+00, 0.8704533344003925E+00, 0.1287308250760436E-01, 0.9473236598617136E-02, 0.6405622258661866E+00, 0.6087788639247561E+00, 0.6835883063177366E-08, 0.1003117403762534E-08, 0.8933398476858376E-01, 0.6724268977954398E-01, 0.2188381927740679E-03, 0.6803700894781421E-03, 0.2090581825874072E+00, 0.1861147811178956E+00, 0.4710079866305792E-04, 0.2733303872960605E-04, 0.1080041669110272E+00, 0.8566670169360113E-01, 0.3257014354834561E-38, 0.8049901434408684E-18, 0.1094206688544886E+01},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     9,
			dmin:     6.7023851586769906E-002,
			dmin1:    6.7023851586769906E-002,
			dmin2:    6.7023851586769906E-002,
			dn:       8.5666701693601133E-002,
			dn1:      0.18606768031923254,
			dn2:      6.7023851586769906E-002,
			tau:      2.2310132178696450E-002,
			ttype:    -9,
			g:        0.0000000000000000,
			zOut:     []float64{0.1386770824956962E+01, 0.1364460692778266E+01, 0.9806070694382835E-18, 0.8049901434408684E-18, 0.1120099419054954E+01, 0.1109117590327065E+01, 0.1132830345080667E-01, 0.1373001363427395E-01, 0.1344261274720646E+01, 0.1322941309070339E+01, 0.1472018016266302E-01, 0.9882485852396858E-02, 0.8881649970290716E+00, 0.9527153989861693E+00, 0.9674301998819117E-01, 0.9944626147083412E-01, 0.9793366455423188E+00, 0.8704533344003925E+00, 0.1287308250760436E-01, 0.9473236598617136E-02, 0.6405622258661866E+00, 0.6087788639247561E+00, 0.6835883063177366E-08, 0.1003117403762534E-08, 0.8933398476858376E-01, 0.6724268977954398E-01, 0.2188381927740679E-03, 0.6803700894781421E-03, 0.2090581825874072E+00, 0.1861147811178956E+00, 0.4710079866305792E-04, 0.2733303872960605E-04, 0.1080041669110272E+00, 0.8566670169360113E-01, 0.3257014354834561E-38, 0.8049901434408684E-18, 0.1094206688544886E+01},
			tauOut:   6.4730147312741043E-002,
			ttypeOut: -5,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1299730545465525E+01, 0.1364460692778266E+01, 0.6869337119490330E-18, 0.8049901434408684E-18, 0.1058117456648598E+01, 0.1109117590327065E+01, 0.1716633828952343E-01, 0.1373001363427395E-01, 0.1250927309320471E+01, 0.1322941309070339E+01, 0.7526573591998700E-02, 0.9882485852396858E-02, 0.9799049395522637E+00, 0.9527153989861693E+00, 0.8833849733474472E-01, 0.9944626147083412E-01, 0.7268579263515238E+00, 0.8704533344003925E+00, 0.7934296380510910E-02, 0.9473236598617136E-02, 0.5361144212346215E+00, 0.6087788639247561E+00, 0.1258170079408221E-09, 0.1003117403762534E-08, 0.3192912430464083E-02, 0.6724268977954398E-01, 0.3965875451961033E-01, 0.6803700894781421E-03, 0.8175321232427381E-01, 0.1861147811178956E+00, 0.2864145895504550E-04, 0.2733303872960605E-04, 0.2090791292190505E-01, 0.8566670169360113E-01, 0.6869337119490330E-18, 0.8049901434408684E-18},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     2.5125423409859404E-003,
			dmin1:    2.5125423409859404E-003,
			dmin2:    2.5125423409859404E-003,
			dn:       2.0907912921905053E-002,
			dn1:      8.1725879285544201E-002,
			dn2:      2.5125423409859404E-003,
			tau:      6.4730147312741043E-002,
			ttype:    -5,
			g:        0.0000000000000000,
			zOut:     []float64{0.1299730545465525E+01, 0.1364460692778266E+01, 0.6869337119490330E-18, 0.8049901434408684E-18, 0.1058117456648598E+01, 0.1109117590327065E+01, 0.1716633828952343E-01, 0.1373001363427395E-01, 0.1250927309320471E+01, 0.1322941309070339E+01, 0.7526573591998700E-02, 0.9882485852396858E-02, 0.9799049395522637E+00, 0.9527153989861693E+00, 0.8833849733474472E-01, 0.9944626147083412E-01, 0.7268579263515238E+00, 0.8704533344003925E+00, 0.7934296380510910E-02, 0.9473236598617136E-02, 0.5361144212346215E+00, 0.6087788639247561E+00, 0.1258170079408221E-09, 0.1003117403762534E-08, 0.3192912430464083E-02, 0.6724268977954398E-01, 0.3965875451961033E-01, 0.6803700894781421E-03, 0.8175321232427381E-01, 0.1861147811178956E+00, 0.2864145895504550E-04, 0.2733303872960605E-04, 0.2090791292190505E-01, 0.8566670169360113E-01, 0.6869337119490330E-18, 0.8049901434408684E-18},
			tauOut:   2.3478378904869292E-003,
			ttypeOut: -5,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1299730545465525E+01, 0.1299143585992903E+01, 0.6869337119490330E-18, 0.5594890049187079E-18, 0.1058117456648598E+01, 0.1074696835465499E+01, 0.1716633828952343E-01, 0.1998130138542489E-01, 0.1250927309320471E+01, 0.1237885622054423E+01, 0.7526573591998700E-02, 0.5958003315736791E-02, 0.9799049395522637E+00, 0.1061698474098650E+01, 0.8833849733474472E-01, 0.6047812873071532E-01, 0.7268579263515238E+00, 0.6737271345286977E+00, 0.7934296380510910E-02, 0.6313669873066943E-02, 0.5361144212346215E+00, 0.5292137920147498E+00, 0.1258170079408221E-09, 0.7590933847144573E-12, 0.3192912430464083E-02, 0.4226470747669359E-01, 0.3965875451961033E-01, 0.7671248122433710E-01, 0.8175321232427381E-01, 0.4482413086270017E-02, 0.2864145895504550E-04, 0.1335961497218277E-03, 0.2090791292190505E-01, 0.2018735729956149E-01, 0.6869337119490330E-18, 0.5594890049187079E-18, 0.1094206688544886E+01},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     9,
			dmin:     2.6059529570832572E-003,
			dmin1:    2.6059529570832572E-003,
			dmin2:    2.6059529570832572E-003,
			dn:       2.0187357299561493E-002,
			dn1:      4.4537716273149721E-003,
			dn2:      2.6059529570832572E-003,
			tau:      5.8695947262173229E-004,
			ttype:    -17,
			g:        0.0000000000000000,
			zOut:     []float64{0.1299730545465525E+01, 0.1299143585992903E+01, 0.6869337119490330E-18, 0.5594890049187079E-18, 0.1058117456648598E+01, 0.1074696835465499E+01, 0.1716633828952343E-01, 0.1998130138542489E-01, 0.1250927309320471E+01, 0.1237885622054423E+01, 0.7526573591998700E-02, 0.5958003315736791E-02, 0.9799049395522637E+00, 0.1061698474098650E+01, 0.8833849733474472E-01, 0.6047812873071532E-01, 0.7268579263515238E+00, 0.6737271345286977E+00, 0.7934296380510910E-02, 0.6313669873066943E-02, 0.5361144212346215E+00, 0.5292137920147498E+00, 0.1258170079408221E-09, 0.7590933847144573E-12, 0.3192912430464083E-02, 0.4226470747669359E-01, 0.3965875451961033E-01, 0.7671248122433710E-01, 0.8175321232427381E-01, 0.4482413086270017E-02, 0.2864145895504550E-04, 0.1335961497218277E-03, 0.2090791292190505E-01, 0.2018735729956149E-01, 0.6869337119490330E-18, 0.5594890049187079E-18, 0.1094206688544886E+01},
			tauOut:   4.9324499328963489E-004,
			ttypeOut: -5,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1298650340999613E+01, 0.1299143585992903E+01, 0.4630045856693428E-18, 0.5594890049187079E-18, 0.1094184891857634E+01, 0.1074696835465499E+01, 0.2260547177996662E-01, 0.1998130138542489E-01, 0.1220744908596903E+01, 0.1237885622054423E+01, 0.5181756634367578E-02, 0.5958003315736791E-02, 0.1116501601201708E+01, 0.1061698474098650E+01, 0.3649413160495897E-01, 0.6047812873071532E-01, 0.6430534278035160E+00, 0.6737271345286977E+00, 0.5195962000339362E-02, 0.6313669873066943E-02, 0.5235245850218799E+00, 0.5292137920147498E+00, 0.6128243213469890E-13, 0.7590933847144573E-12, 0.1184839437076798E+00, 0.4226470747669359E-01, 0.2902140315050332E-02, 0.7671248122433710E-01, 0.1220623927651878E-02, 0.4482413086270017E-02, 0.2209487416380896E-02, 0.1335961497218277E-03, 0.1748462488989096E-01, 0.2018735729956149E-01, 0.4630045856693428E-18, 0.5594890049187079E-18},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     1.0870277779300500E-003,
			dmin1:    1.0870277779300500E-003,
			dmin2:    4.1771462483342674E-002,
			dn:       1.7484624889890960E-002,
			dn1:      1.0870277779300500E-003,
			dn2:      4.1771462483342674E-002,
			tau:      4.9324499328963489E-004,
			ttype:    -5,
			g:        0.0000000000000000,
			zOut:     []float64{0.1298650340999613E+01, 0.1299143585992903E+01, 0.4630045856693428E-18, 0.5594890049187079E-18, 0.1094184891857634E+01, 0.1074696835465499E+01, 0.2260547177996662E-01, 0.1998130138542489E-01, 0.1220744908596903E+01, 0.1237885622054423E+01, 0.5181756634367578E-02, 0.5958003315736791E-02, 0.1116501601201708E+01, 0.1061698474098650E+01, 0.3649413160495897E-01, 0.6047812873071532E-01, 0.6430534278035160E+00, 0.6737271345286977E+00, 0.5195962000339362E-02, 0.6313669873066943E-02, 0.5235245850218799E+00, 0.5292137920147498E+00, 0.6128243213469890E-13, 0.7590933847144573E-12, 0.1184839437076798E+00, 0.4226470747669359E-01, 0.2902140315050332E-02, 0.7671248122433710E-01, 0.1220623927651878E-02, 0.4482413086270017E-02, 0.2209487416380896E-02, 0.1335961497218277E-03, 0.1748462488989096E-01, 0.2018735729956149E-01, 0.4630045856693428E-18, 0.5594890049187079E-18},
			tauOut:   8.6238530354903250E-004,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1298650340999613E+01, 0.1297787955696064E+01, 0.4630045856693428E-18, 0.3903662538064461E-18, 0.1094184891857634E+01, 0.1115927978334052E+01, 0.2260547177996662E-01, 0.2472875948770642E-01, 0.1220744908596903E+01, 0.1200335520440015E+01, 0.5181756634367578E-02, 0.4819852017032840E-02, 0.1116501601201708E+01, 0.1147313495486085E+01, 0.3649413160495897E-01, 0.2045445862496274E-01, 0.6430534278035160E+00, 0.6269325458753436E+00, 0.5195962000339362E-02, 0.4338925882718484E-02, 0.5235245850218799E+00, 0.5183232738356738E+00, 0.6128243213469890E-13, 0.1400860159256393E-13, 0.1184839437076798E+00, 0.1205236987191671E+00, 0.2902140315050332E-02, 0.2939191169537379E-04, 0.1220623927651878E-02, 0.2538334128788367E-02, 0.2209487416380896E-02, 0.1521945367089819E-01, 0.1748462488989096E-01, 0.1402785915443734E-02, 0.4630045856693428E-18, 0.3903662538064461E-18, 0.1094206688544886E+01},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     9,
			dmin:     3.2884671240747138E-004,
			dmin1:    3.2884671240747138E-004,
			dmin2:    0.11762155840411674,
			dn:       1.4027859154437344E-003,
			dn1:      3.2884671240747138E-004,
			dn2:      0.11762155840411674,
			tau:      8.6238530354903250E-004,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1298650340999613E+01, 0.1297787955696064E+01, 0.4630045856693428E-18, 0.3903662538064461E-18, 0.1094184891857634E+01, 0.1115927978334052E+01, 0.2260547177996662E-01, 0.2472875948770642E-01, 0.1220744908596903E+01, 0.1200335520440015E+01, 0.5181756634367578E-02, 0.4819852017032840E-02, 0.1116501601201708E+01, 0.1147313495486085E+01, 0.3649413160495897E-01, 0.2045445862496274E-01, 0.6430534278035160E+00, 0.6269325458753436E+00, 0.5195962000339362E-02, 0.4338925882718484E-02, 0.5235245850218799E+00, 0.5183232738356738E+00, 0.6128243213469890E-13, 0.1400860159256393E-13, 0.1184839437076798E+00, 0.1205236987191671E+00, 0.2902140315050332E-02, 0.2939191169537379E-04, 0.1220623927651878E-02, 0.2538334128788367E-02, 0.2209487416380896E-02, 0.1521945367089819E-01, 0.1748462488989096E-01, 0.1402785915443734E-02, 0.4630045856693428E-18, 0.3903662538064461E-18, 0.1094206688544886E+01},
			tauOut:   1.8442717265434725E-004,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1297603528523410E+01, 0.1297787955696064E+01, 0.3357116521683426E-18, 0.3903662538064461E-18, 0.1140472310649104E+01, 0.1115927978334052E+01, 0.2602676813137004E-01, 0.2472875948770642E-01, 0.1178944177153024E+01, 0.1200335520440015E+01, 0.4690536984322235E-02, 0.4819852017032840E-02, 0.1162892989954071E+01, 0.1147313495486085E+01, 0.1102729651913737E-01, 0.2045445862496274E-01, 0.6200597480662705E+00, 0.6269325458753436E+00, 0.3627015421456806E-02, 0.4338925882718484E-02, 0.5145118312415766E+00, 0.5183232738356738E+00, 0.3281495925457712E-14, 0.1400860159256393E-13, 0.1203686634582048E+00, 0.1205236987191671E+00, 0.6198165737098726E-06, 0.2939191169537379E-04, 0.1757274081045850E-01, 0.2538334128788367E-02, 0.1214929161054836E-02, 0.1521945367089819E-01, 0.3429581734551261E-05, 0.1402785915443734E-02, 0.3357116521683426E-18, 0.3903662538064461E-18},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     3.4295817345512611E-006,
			dmin1:    2.3532871395603098E-003,
			dmin2:    0.12033927154650945,
			dn:       3.4295817345512611E-006,
			dn1:      2.3532871395603098E-003,
			dn2:      0.12033927154650945,
			tau:      1.8442717265434725E-004,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1297603528523410E+01, 0.1297787955696064E+01, 0.3357116521683426E-18, 0.3903662538064461E-18, 0.1140472310649104E+01, 0.1115927978334052E+01, 0.2602676813137004E-01, 0.2472875948770642E-01, 0.1178944177153024E+01, 0.1200335520440015E+01, 0.4690536984322235E-02, 0.4819852017032840E-02, 0.1162892989954071E+01, 0.1147313495486085E+01, 0.1102729651913737E-01, 0.2045445862496274E-01, 0.6200597480662705E+00, 0.6269325458753436E+00, 0.3627015421456806E-02, 0.4338925882718484E-02, 0.5145118312415766E+00, 0.5183232738356738E+00, 0.3281495925457712E-14, 0.1400860159256393E-13, 0.1203686634582048E+00, 0.1205236987191671E+00, 0.6198165737098726E-06, 0.2939191169537379E-04, 0.1757274081045850E-01, 0.2538334128788367E-02, 0.1214929161054836E-02, 0.1521945367089819E-01, 0.3429581734551261E-05, 0.1402785915443734E-02, 0.3357116521683426E-18, 0.3903662538064461E-18},
			tauOut:   3.2077610710809750E-006,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1297603528523410E+01, 0.1297600320762339E+01, 0.3357116521683426E-18, 0.2950599175525191E-18, 0.1140472310649104E+01, 0.1166495871019403E+01, 0.2602676813137004E-01, 0.2630451380147252E-01, 0.1178944177153024E+01, 0.1157326992574803E+01, 0.4690536984322235E-02, 0.4713095445958055E-02, 0.1162892989954071E+01, 0.1169203983266179E+01, 0.1102729651913737E-01, 0.5848066547299594E-02, 0.6200597480662705E+00, 0.6178354891793567E+00, 0.3627015421456806E-02, 0.3020451850239120E-02, 0.5145118312415766E+00, 0.5114881716302697E+00, 0.3281495925457712E-14, 0.7722354114894537E-15, 0.1203686634582048E+00, 0.1203660755137067E+00, 0.6198165737098726E-06, 0.9048958315991394E-07, 0.1757274081045850E-01, 0.1878437172085910E-01, 0.1214929161054836E-02, 0.2218173128942313E-06, 0.3429581734551261E-05, 0.3350576054907334E-11, 0.3357116521683426E-18, 0.2950599175525191E-18, 0.1094206688544886E+01},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     9,
			dmin:     3.3505760549073344E-012,
			dmin1:    1.7569442559804263E-002,
			dmin2:    0.12036545569713296,
			dn:       3.3505760549073344E-012,
			dn1:      1.7569442559804263E-002,
			dn2:      0.12036545569713296,
			tau:      3.2077610710809750E-006,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1297603528523410E+01, 0.1297600320762339E+01, 0.3357116521683426E-18, 0.2950599175525191E-18, 0.1140472310649104E+01, 0.1166495871019403E+01, 0.2602676813137004E-01, 0.2630451380147252E-01, 0.1178944177153024E+01, 0.1157326992574803E+01, 0.4690536984322235E-02, 0.4713095445958055E-02, 0.1162892989954071E+01, 0.1169203983266179E+01, 0.1102729651913737E-01, 0.5848066547299594E-02, 0.6200597480662705E+00, 0.6178354891793567E+00, 0.3627015421456806E-02, 0.3020451850239120E-02, 0.5145118312415766E+00, 0.5114881716302697E+00, 0.3281495925457712E-14, 0.7722354114894537E-15, 0.1203686634582048E+00, 0.1203660755137067E+00, 0.6198165737098726E-06, 0.9048958315991394E-07, 0.1757274081045850E-01, 0.1878437172085910E-01, 0.1214929161054836E-02, 0.2218173128942313E-06, 0.3429581734551261E-05, 0.3350576054907334E-11, 0.3357116521683426E-18, 0.2950599175525191E-18, 0.1094206688544886E+01},
			tauOut:   3.3505364896797715E-012,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1297600320758988E+01, 0.1297600320762339E+01, 0.2652482201353177E-18, 0.2950599175525191E-18, 0.1192800384817525E+01, 0.1166495871019403E+01, 0.2552222839336001E-01, 0.2630451380147252E-01, 0.1136517859624050E+01, 0.1157326992574803E+01, 0.4848643531876122E-02, 0.4713095445958055E-02, 0.1170203406278252E+01, 0.1169203983266179E+01, 0.3087619670750762E-02, 0.5848066547299594E-02, 0.6177683213554945E+00, 0.6178354891793567E+00, 0.2500816796475141E-02, 0.3020451850239120E-02, 0.5089873548304448E+00, 0.5114881716302697E+00, 0.1826193617023393E-15, 0.7722354114894537E-15, 0.1203661659999391E+00, 0.1203660755137067E+00, 0.1412182528886294E-07, 0.9048958315991394E-07, 0.1878457941299617E-01, 0.1878437172085910E-01, 0.3956520722700361E-16, 0.2218173128942313E-06, 0.2033579915109003E-22, 0.3350576054907334E-11, 0.2652482201353177E-18, 0.2950599175525191E-18},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     2.0335799151090034E-023,
			dmin1:    1.8784357595683275E-002,
			dmin2:    0.12036607551035594,
			dn:       2.0335799151090034E-023,
			dn1:      1.8784357595683275E-002,
			dn2:      0.12036607551035594,
			tau:      3.3505364896797715E-012,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1297600320758988E+01, 0.1297600320762339E+01, 0.2652482201353177E-18, 0.2950599175525191E-18, 0.1192800384817525E+01, 0.1166495871019403E+01, 0.2552222839336001E-01, 0.2630451380147252E-01, 0.1136517859624050E+01, 0.1157326992574803E+01, 0.4848643531876122E-02, 0.4713095445958055E-02, 0.1170203406278252E+01, 0.1169203983266179E+01, 0.3087619670750762E-02, 0.5848066547299594E-02, 0.6177683213554945E+00, 0.6178354891793567E+00, 0.2500816796475141E-02, 0.3020451850239120E-02, 0.5089873548304448E+00, 0.5114881716302697E+00, 0.1826193617023393E-15, 0.7722354114894537E-15, 0.1203661659999391E+00, 0.1203660755137067E+00, 0.1412182528886294E-07, 0.9048958315991394E-07, 0.1878457941299617E-01, 0.1878437172085910E-01, 0.3956520722700361E-16, 0.2218173128942313E-06, 0.2033579915109003E-22, 0.3350576054907334E-11, 0.2652482201353177E-18, 0.2950599175525191E-18},
			tauOut:   2.0335799151089990E-023,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1297600320758988E+01, 0.1297600320758988E+01, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1192800384817525E+01, 0.1218322613210885E+01, 0.2552222839336001E-01, 0.2380852827644004E-01, 0.1136517859624050E+01, 0.1117557974879486E+01, 0.4848643531876122E-02, 0.5077051306839188E-02, 0.1170203406278252E+01, 0.1168213974642164E+01, 0.3087619670750762E-02, 0.1632777609571200E-02, 0.6177683213554945E+00, 0.6186363605423986E+00, 0.2500816796475141E-02, 0.2057564358223963E-02, 0.5089873548304448E+00, 0.5069297904722210E+00, 0.1826193617023393E-15, 0.4336141378669131E-16, 0.1203661659999391E+00, 0.1203661801217644E+00, 0.1412182528886294E-07, 0.2203879431304952E-08, 0.1878457941299617E-01, 0.1878457720911678E-01, 0.3956520722700361E-16, 0.4283248425464200E-37, 0.1183377192742860E+01},
			i0:       1,
			n0:       8,
			pp:       1,
			n0in:     9,
			dmin:     2.0335799151089993E-023,
			dmin1:    1.8784577209116738E-002,
			dmin2:    0.12036616599993906,
			dn:       2.0335799151089993E-023,
			dn1:      1.8784577209116738E-002,
			dn2:      0.12036616599993906,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1297600320758988E+01, 0.1297600320758988E+01, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1192800384817525E+01, 0.1218322613210885E+01, 0.2552222839336001E-01, 0.2380852827644004E-01, 0.1136517859624050E+01, 0.1117557974879486E+01, 0.4848643531876122E-02, 0.5077051306839188E-02, 0.1170203406278252E+01, 0.1168213974642164E+01, 0.3087619670750762E-02, 0.1632777609571200E-02, 0.6177683213554945E+00, 0.6186363605423986E+00, 0.2500816796475141E-02, 0.2057564358223963E-02, 0.5089873548304448E+00, 0.5069297904722210E+00, 0.1826193617023393E-15, 0.4336141378669131E-16, 0.1203661659999391E+00, 0.1203661801217644E+00, 0.1412182528886294E-07, 0.2203879431304952E-08, 0.1878457941299617E-01, 0.1878457720911678E-01, 0.3956520722700361E-16, 0.4283248425464200E-37, 0.1183377192742860E+01},
			tauOut:   1.8784576682472597E-002,
			ttypeOut: -7,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1278815744076516E+01, 0.1297600320758988E+01, 0.2322916695987584E-18, 0.2438256017573345E-18, 0.1223346564804853E+01, 0.1218322613210885E+01, 0.2174969171530204E-01, 0.2380852827644004E-01, 0.1082100757788551E+01, 0.1117557974879486E+01, 0.5481081353963683E-02, 0.5077051306839188E-02, 0.1145581094215299E+01, 0.1168213974642164E+01, 0.8817320773368219E-03, 0.1632777609571200E-02, 0.6010276161408131E+00, 0.6186363605423986E+00, 0.1735428857154564E-02, 0.2057564358223963E-02, 0.4864097849325938E+00, 0.5069297904722210E+00, 0.1073014545319338E-16, 0.4336141378669131E-16, 0.1015816056431712E+00, 0.1203661801217644E+00, 0.4075436992240071E-09, 0.2203879431304952E-08, 0.1191004794787176E-09, 0.1878457720911678E-01, 0.2322916695987584E-18, 0.2438256017573345E-18},
			i0:       1,
			n0:       8,
			pp:       0,
			n0in:     8,
			dmin:     1.1910047947871760E-010,
			dmin1:    0.10158160343929173,
			dmin2:    0.48640978493259379,
			dn:       1.1910047947871760E-010,
			dn1:      0.10158160343929173,
			dn2:      0.48640978493259379,
			tau:      1.8784576682472597E-002,
			ttype:    -7,
			g:        0.0000000000000000,
			zOut:     []float64{0.1278815744076516E+01, 0.1297600320758988E+01, 0.2322916695987584E-18, 0.2438256017573345E-18, 0.1223346564804853E+01, 0.1218322613210885E+01, 0.2174969171530204E-01, 0.2380852827644004E-01, 0.1082100757788551E+01, 0.1117557974879486E+01, 0.5481081353963683E-02, 0.5077051306839188E-02, 0.1145581094215299E+01, 0.1168213974642164E+01, 0.8817320773368219E-03, 0.1632777609571200E-02, 0.6010276161408131E+00, 0.6186363605423986E+00, 0.1735428857154564E-02, 0.2057564358223963E-02, 0.4864097849325938E+00, 0.5069297904722210E+00, 0.1073014545319338E-16, 0.4336141378669131E-16, 0.1015816056431712E+00, 0.1203661801217644E+00, 0.4075436992240071E-09, 0.2203879431304952E-08, 0.1191004794787176E-09, 0.1878457720911678E-01, 0.2322916695987584E-18, 0.2438256017573345E-18},
			tauOut:   1.1910047900088848E-010,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1278815744076516E+01, 0.1278815743957415E+01, 0.2322916695987584E-18, 0.2222159192042978E-18, 0.1223346564804853E+01, 0.1245096256401054E+01, 0.2174969171530204E-01, 0.1890244048666934E-01, 0.1082100757788551E+01, 0.1068679398536745E+01, 0.5481081353963683E-02, 0.5875497537946494E-02, 0.1145581094215299E+01, 0.1140587328635589E+01, 0.8817320773368219E-03, 0.4646249482278370E-03, 0.6010276161408131E+00, 0.6022984199306394E+00, 0.1735428857154564E-02, 0.1401513849681988E-02, 0.4864097849325938E+00, 0.4850082709638114E+00, 0.1073014545319338E-16, 0.2247354260070927E-17, 0.1015816056431712E+00, 0.1015816059316144E+00, 0.4075436992240071E-09, 0.4778291260603437E-18, 0.1191004794787176E-09, 0.0000000000000000E+00, 0.2322916695987584E-18, 0.2222159192042978E-18, 0.1183377192742860E+01},
			i0:       1,
			n0:       8,
			pp:       1,
			n0in:     8,
			dmin:     0.0000000000000000,
			dmin1:    0.10158160552407068,
			dmin2:    0.48500827096381138,
			dn:       0.0000000000000000,
			dn1:      0.10158160552407068,
			dn2:      0.48500827096381138,
			tau:      1.1910047900088848E-010,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1278815744076516E+01, 0.1278815743957415E+01, 0.2322916695987584E-18, 0.2222159192042978E-18, 0.1223346564804853E+01, 0.1245096256401054E+01, 0.2174969171530204E-01, 0.1890244048666934E-01, 0.1082100757788551E+01, 0.1068679398536745E+01, 0.5481081353963683E-02, 0.5875497537946494E-02, 0.1145581094215299E+01, 0.1140587328635589E+01, 0.8817320773368219E-03, 0.4646249482278370E-03, 0.6010276161408131E+00, 0.6022984199306394E+00, 0.1735428857154564E-02, 0.1401513849681988E-02, 0.4864097849325938E+00, 0.4850082709638114E+00, 0.1073014545319338E-16, 0.2247354260070927E-17, 0.1015816056431712E+00, 0.1015816059316144E+00, 0.4075436992240071E-09, 0.4778291260603437E-18, 0.1191004794787176E-09, 0.0000000000000000E+00, 0.2322916695987584E-18, 0.2222159192042978E-18, 0.1183377192742860E+01},
			tauOut:   -0.0000000000000000,
			ttypeOut: -1,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1278815743957415E+01, 0.1278815743957415E+01, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1263998696887723E+01, 0.1245096256401054E+01, 0.1598154237018549E-01, 0.1890244048666934E-01, 0.1058573353704506E+01, 0.1068679398536745E+01, 0.6330707284251229E-02, 0.5875497537946494E-02, 0.1134721246299565E+01, 0.1140587328635589E+01, 0.2466181655543824E-03, 0.4646249482278370E-03, 0.6034533156147670E+00, 0.6022984199306394E+00, 0.1126426504548419E-02, 0.1401513849681988E-02, 0.4838818444592630E+00, 0.4850082709638114E+00, 0.4717884282068346E-18, 0.2247354260070927E-17, 0.1015816059316144E+00, 0.1015816059316144E+00, 0.0000000000000000E+00, 0.4778291260603437E-18},
			i0:       1,
			n0:       7,
			pp:       0,
			n0in:     8,
			dmin:     0.0000000000000000,
			dmin1:    0.10158160593161437,
			dmin2:    0.48388184445926297,
			dn:       0.0000000000000000,
			dn1:      0.10158160593161437,
			dn2:      0.48388184445926297,
			tau:      0.0000000000000000,
			ttype:    -1,
			g:        0.0000000000000000,
			zOut:     []float64{0.1278815743957415E+01, 0.1278815743957415E+01, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1263998696887723E+01, 0.1245096256401054E+01, 0.1598154237018549E-01, 0.1890244048666934E-01, 0.1058573353704506E+01, 0.1068679398536745E+01, 0.6330707284251229E-02, 0.5875497537946494E-02, 0.1134721246299565E+01, 0.1140587328635589E+01, 0.2466181655543824E-03, 0.4646249482278370E-03, 0.6034533156147670E+00, 0.6022984199306394E+00, 0.1126426504548419E-02, 0.1401513849681988E-02, 0.4838818444592630E+00, 0.4850082709638114E+00, 0.4717884282068346E-18, 0.2247354260070927E-17, 0.1015816059316144E+00, 0.1015816059316144E+00, 0.0000000000000000E+00, 0.4778291260603437E-18},
			tauOut:   -0.0000000000000000,
			ttypeOut: -1,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1278815743957415E+01, 0.1278815743957415E+01, 0.2163565864913247E-18, 0.2138497627045302E-18, 0.1263998696887723E+01, 0.1279980239257909E+01, 0.1598154237018549E-01, 0.1321710631563048E-01, 0.1058573353704506E+01, 0.1051686954673127E+01, 0.6330707284251229E-02, 0.6830538334266978E-02, 0.1134721246299565E+01, 0.1128137326130852E+01, 0.2466181655543824E-03, 0.1319188242844840E-03, 0.6034533156147670E+00, 0.6044478232950309E+00, 0.1126426504548419E-02, 0.9017442261557238E-03, 0.4838818444592630E+00, 0.4829801002331072E+00, 0.4717884282068346E-18, 0.9922774494036444E-19, 0.1015816059316144E+00, 0.1015816059316144E+00, 0.0000000000000000E+00, 0.2138497627045302E-18, 0.1202161769544433E+01},
			i0:       1,
			n0:       7,
			pp:       1,
			n0in:     7,
			dmin:     0.10158160593161437,
			dmin1:    0.48298010023310722,
			dmin2:    0.60332139679048247,
			dn:       0.10158160593161437,
			dn1:      0.48298010023310722,
			dn2:      0.60332139679048247,
			tau:      0.0000000000000000,
			ttype:    -1,
			g:        0.0000000000000000,
			zOut:     []float64{0.1278815743957415E+01, 0.1278815743957415E+01, 0.2163565864913247E-18, 0.2138497627045302E-18, 0.1263998696887723E+01, 0.1279980239257909E+01, 0.1598154237018549E-01, 0.1321710631563048E-01, 0.1058573353704506E+01, 0.1051686954673127E+01, 0.6330707284251229E-02, 0.6830538334266978E-02, 0.1134721246299565E+01, 0.1128137326130852E+01, 0.2466181655543824E-03, 0.1319188242844840E-03, 0.6034533156147670E+00, 0.6044478232950309E+00, 0.1126426504548419E-02, 0.9017442261557238E-03, 0.4838818444592630E+00, 0.4829801002331072E+00, 0.4717884282068346E-18, 0.9922774494036444E-19, 0.1015816059316144E+00, 0.1015816059316144E+00, 0.0000000000000000E+00, 0.2138497627045302E-18, 0.1202161769544433E+01},
			tauOut:   0.10158160593161437,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1177234138025801E+01, 0.1278815743957415E+01, 0.2325140442247288E-18, 0.2138497627045302E-18, 0.1191615739641925E+01, 0.1279980239257909E+01, 0.1166505092896250E-01, 0.1321710631563048E-01, 0.9452708361468171E+00, 0.1051686954673127E+01, 0.8151933771558135E-02, 0.6830538334266978E-02, 0.1018535705251965E+01, 0.1128137326130852E+01, 0.7828694249915420E-04, 0.1319188242844840E-03, 0.5036896746470730E+00, 0.6044478232950309E+00, 0.8646683437346259E-03, 0.9017442261557238E-03, 0.3805338259577582E+00, 0.4829801002331072E+00, 0.2648835135390502E-19, 0.9922774494036444E-19, 0.0000000000000000E+00, 0.1015816059316144E+00, 0.2325140442247288E-18, 0.2138497627045302E-18},
			i0:       1,
			n0:       7,
			pp:       0,
			n0in:     7,
			dmin:     0.0000000000000000,
			dmin1:    0.38053382595775820,
			dmin2:    0.50278793042091730,
			dn:       0.0000000000000000,
			dn1:      0.38053382595775820,
			dn2:      0.50278793042091730,
			tau:      0.10158160593161437,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1177234138025801E+01, 0.1278815743957415E+01, 0.2325140442247288E-18, 0.2138497627045302E-18, 0.1191615739641925E+01, 0.1279980239257909E+01, 0.1166505092896250E-01, 0.1321710631563048E-01, 0.9452708361468171E+00, 0.1051686954673127E+01, 0.8151933771558135E-02, 0.6830538334266978E-02, 0.1018535705251965E+01, 0.1128137326130852E+01, 0.7828694249915420E-04, 0.1319188242844840E-03, 0.5036896746470730E+00, 0.6044478232950309E+00, 0.8646683437346259E-03, 0.9017442261557238E-03, 0.3805338259577582E+00, 0.4829801002331072E+00, 0.2648835135390502E-19, 0.9922774494036444E-19, 0.0000000000000000E+00, 0.1015816059316144E+00, 0.2325140442247288E-18, 0.2138497627045302E-18},
			tauOut:   -0.0000000000000000,
			ttypeOut: -1,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1177234138025801E+01, 0.1177234138025801E+01, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1191615739641925E+01, 0.1203280790570888E+01, 0.1166505092896250E-01, 0.9163806595868684E-02, 0.9452708361468171E+00, 0.9442589633225065E+00, 0.8151933771558135E-02, 0.8793176380307672E-02, 0.1018535705251965E+01, 0.1009820815814156E+01, 0.7828694249915420E-04, 0.3904883319791864E-04, 0.5036896746470730E+00, 0.5045152941576099E+00, 0.8646683437346259E-03, 0.6521815232088979E-03, 0.3805338259577582E+00, 0.3798816444345493E+00, 0.2648835135390502E-19, 0.0000000000000000E+00, 0.1303743375476047E+01},
			i0:       1,
			n0:       6,
			pp:       1,
			n0in:     7,
			dmin:     0.0000000000000000,
			dmin1:    0.37988164443454930,
			dmin2:    0.50365062581387521,
			dn:       0.0000000000000000,
			dn1:      0.37988164443454930,
			dn2:      0.50365062581387521,
			tau:      0.0000000000000000,
			ttype:    -1,
			g:        0.0000000000000000,
			zOut:     []float64{0.1177234138025801E+01, 0.1177234138025801E+01, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1191615739641925E+01, 0.1203280790570888E+01, 0.1166505092896250E-01, 0.9163806595868684E-02, 0.9452708361468171E+00, 0.9442589633225065E+00, 0.8151933771558135E-02, 0.8793176380307672E-02, 0.1018535705251965E+01, 0.1009820815814156E+01, 0.7828694249915420E-04, 0.3904883319791864E-04, 0.5036896746470730E+00, 0.5045152941576099E+00, 0.8646683437346259E-03, 0.6521815232088979E-03, 0.3805338259577582E+00, 0.3798816444345493E+00, 0.2648835135390502E-19, 0.0000000000000000E+00, 0.1303743375476047E+01},
			tauOut:   -0.0000000000000000,
			ttypeOut: -1,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1177234138025801E+01, 0.1177234138025801E+01, 0.2405618249359890E-18, 0.2353545363971710E-18, 0.1212444597166756E+01, 0.1203280790570888E+01, 0.7136826323052847E-02, 0.9163806595868684E-02, 0.9459153133797613E+00, 0.9442589633225065E+00, 0.9387238392656353E-02, 0.8793176380307672E-02, 0.1000472626254697E+01, 0.1009820815814156E+01, 0.1969142688202247E-04, 0.3904883319791864E-04, 0.5051477842539368E+00, 0.5045152941576099E+00, 0.4904540754787929E-03, 0.6521815232088979E-03, 0.3793911903590705E+00, 0.3798816444345493E+00, 0.2405618249359890E-18, 0.2353545363971710E-18},
			i0:       1,
			n0:       6,
			pp:       0,
			n0in:     6,
			dmin:     0.37939119035907048,
			dmin1:    0.50449560273072791,
			dmin2:    0.93712213699945368,
			dn:       0.37939119035907048,
			dn1:      0.50449560273072791,
			dn2:      1.0004335774214996,
			tau:      0.0000000000000000,
			ttype:    -1,
			g:        0.0000000000000000,
			zOut:     []float64{0.1177234138025801E+01, 0.1177234138025801E+01, 0.2405618249359890E-18, 0.2353545363971710E-18, 0.1212444597166756E+01, 0.1203280790570888E+01, 0.7136826323052847E-02, 0.9163806595868684E-02, 0.9459153133797613E+00, 0.9442589633225065E+00, 0.9387238392656353E-02, 0.8793176380307672E-02, 0.1000472626254697E+01, 0.1009820815814156E+01, 0.1969142688202247E-04, 0.3904883319791864E-04, 0.5051477842539368E+00, 0.5045152941576099E+00, 0.4904540754787929E-03, 0.6521815232088979E-03, 0.3793911903590705E+00, 0.3798816444345493E+00, 0.2405618249359890E-18, 0.2353545363971710E-18},
			tauOut:   0.37791671367087804,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1177234138025801E+01, 0.7993174243549228E+00, 0.2405618249359890E-18, 0.3648961927279404E-18, 0.1212444597166756E+01, 0.8416647098189309E+00, 0.7136826323052847E-02, 0.8020810697124018E-02, 0.9459153133797613E+00, 0.5693650274044155E+00, 0.9387238392656353E-02, 0.1649499810480807E-01, 0.1000472626254697E+01, 0.6060806059058935E+00, 0.1969142688202247E-04, 0.1641214148963635E-04, 0.5051477842539368E+00, 0.1277051125170478E+00, 0.4904540754787929E-03, 0.1457059563590431E-02, 0.3793911903590705E+00, 0.1741712460201494E-04, 0.2405618249359890E-18, 0.3648961927279404E-18, 0.1303743375476047E+01},
			i0:       1,
			n0:       6,
			pp:       1,
			n0in:     6,
			dmin:     1.7417124602014944E-005,
			dmin1:    0.12721465844156904,
			dmin2:    0.55997778901175921,
			dn:       1.7417124602014944E-005,
			dn1:      0.12721465844156904,
			dn2:      0.60606091447901145,
			tau:      0.37791671367087804,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1177234138025801E+01, 0.7993174243549228E+00, 0.2405618249359890E-18, 0.3648961927279404E-18, 0.1212444597166756E+01, 0.8416647098189309E+00, 0.7136826323052847E-02, 0.8020810697124018E-02, 0.9459153133797613E+00, 0.5693650274044155E+00, 0.9387238392656353E-02, 0.1649499810480807E-01, 0.1000472626254697E+01, 0.6060806059058935E+00, 0.1969142688202247E-04, 0.1641214148963635E-04, 0.5051477842539368E+00, 0.1277051125170478E+00, 0.4904540754787929E-03, 0.1457059563590431E-02, 0.3793911903590705E+00, 0.1741712460201494E-04, 0.2405618249359890E-18, 0.3648961927279404E-18, 0.1303743375476047E+01},
			tauOut:   1.7220607103034587E-005,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.7993002037478197E+00, 0.7993174243549228E+00, 0.3842364192156412E-18, 0.3648961927279404E-18, 0.8496682999089519E+00, 0.8416647098189309E+00, 0.5374766956544110E-02, 0.8020810697124018E-02, 0.5804680379455763E+00, 0.5693650274044155E+00, 0.1722282329473577E-01, 0.1649499810480807E-01, 0.5888569741455443E+00, 0.6060806059058935E+00, 0.3559292778388127E-05, 0.1641214148963635E-04, 0.1291413921807568E+00, 0.1277051125170478E+00, 0.1965116493098606E-06, 0.1457059563590431E-02, 0.5849670496393478E-11, 0.1741712460201494E-04, 0.3842364192156412E-18, 0.3648961927279404E-18},
			i0:       1,
			n0:       6,
			pp:       0,
			n0in:     6,
			dmin:     5.8496704963934779E-012,
			dmin1:    0.12768433261716639,
			dmin2:    0.56397303984076830,
			dn:       5.8496704963934779E-012,
			dn1:      0.12768433261716639,
			dn2:      0.58884056200405466,
			tau:      1.7220607103034587E-005,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.7993002037478197E+00, 0.7993174243549228E+00, 0.3842364192156412E-18, 0.3648961927279404E-18, 0.8496682999089519E+00, 0.8416647098189309E+00, 0.5374766956544110E-02, 0.8020810697124018E-02, 0.5804680379455763E+00, 0.5693650274044155E+00, 0.1722282329473577E-01, 0.1649499810480807E-01, 0.5888569741455443E+00, 0.6060806059058935E+00, 0.3559292778388127E-05, 0.1641214148963635E-04, 0.1291413921807568E+00, 0.1277051125170478E+00, 0.1965116493098606E-06, 0.1457059563590431E-02, 0.5849670496393478E-11, 0.1741712460201494E-04, 0.3842364192156412E-18, 0.3648961927279404E-18},
			tauOut:   5.8496615949827599E-012,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.7993002037478197E+00, 0.7993002037419701E+00, 0.3842364192156412E-18, 0.4084491703488284E-18, 0.8496682999089519E+00, 0.8550430668596465E+00, 0.5374766956544110E-02, 0.3648799166500929E-02, 0.5804680379455763E+00, 0.5940420620679615E+00, 0.1722282329473577E-01, 0.1707249411982080E-01, 0.5888569741455443E+00, 0.5717880393126522E+00, 0.3559292778388127E-05, 0.8038853438286423E-06, 0.1291413921807568E+00, 0.1291407848012127E+00, 0.1965116493098606E-06, 0.8901358303923820E-17, 0.5849670496393478E-11, 0.5241368559131172E-22, 0.3842364192156412E-18, 0.4084491703488284E-18, 0.1303743375476047E+01},
			i0:       1,
			n0:       6,
			pp:       1,
			n0in:     6,
			dmin:     5.2413685591311719E-023,
			dmin1:    0.12914058828956335,
			dmin2:    0.57178448001987381,
			dn:       5.2413685591311719E-023,
			dn1:      0.12914058828956335,
			dn2:      0.57178448001987381,
			tau:      5.8496615949827599E-012,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.7993002037478197E+00, 0.7993002037419701E+00, 0.3842364192156412E-18, 0.4084491703488284E-18, 0.8496682999089519E+00, 0.8550430668596465E+00, 0.5374766956544110E-02, 0.3648799166500929E-02, 0.5804680379455763E+00, 0.5940420620679615E+00, 0.1722282329473577E-01, 0.1707249411982080E-01, 0.5888569741455443E+00, 0.5717880393126522E+00, 0.3559292778388127E-05, 0.8038853438286423E-06, 0.1291413921807568E+00, 0.1291407848012127E+00, 0.1965116493098606E-06, 0.8901358303923820E-17, 0.5849670496393478E-11, 0.5241368559131172E-22, 0.3842364192156412E-18, 0.4084491703488284E-18, 0.1303743375476047E+01},
			tauOut:   5.2413685591311714E-023,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.7993002037419701E+00, 0.7993002037419701E+00, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.8586918660261474E+00, 0.8550430668596465E+00, 0.2524235138002424E-02, 0.3648799166500929E-02, 0.6085903210497798E+00, 0.5940420620679615E+00, 0.1604009725641142E-01, 0.1707249411982080E-01, 0.5557487459415846E+00, 0.5717880393126522E+00, 0.1868009328861996E-06, 0.8038853438286423E-06, 0.1291405980002798E+00, 0.1291407848012127E+00, 0.3612752323451872E-38, 0.8901358303923820E-17},
			i0:       1,
			n0:       5,
			pp:       0,
			n0in:     6,
			dmin:     5.2413685591311719E-023,
			dmin1:    0.12914059800027977,
			dmin2:    0.55574794205624078,
			dn:       5.2413685591311719E-023,
			dn1:      0.12914059800027977,
			dn2:      0.55574794205624078,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.7993002037419701E+00, 0.7993002037419701E+00, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.8586918660261474E+00, 0.8550430668596465E+00, 0.2524235138002424E-02, 0.3648799166500929E-02, 0.6085903210497798E+00, 0.5940420620679615E+00, 0.1604009725641142E-01, 0.1707249411982080E-01, 0.5557487459415846E+00, 0.5717880393126522E+00, 0.1868009328861996E-06, 0.8038853438286423E-06, 0.1291405980002798E+00, 0.1291407848012127E+00, 0.3612752323451872E-38, 0.8901358303923820E-17},
			tauOut:   0.12914051019182277,
			ttypeOut: -7,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.7993002037419701E+00, 0.6701596935501473E+00, 0.4369342452764876E-18, 0.5598544436768911E-18, 0.8586918660261474E+00, 0.7320755909723271E+00, 0.2524235138002424E-02, 0.2098451433139096E-02, 0.6085903210497798E+00, 0.4933914566812295E+00, 0.1604009725641142E-01, 0.1806732527351204E-01, 0.5557487459415846E+00, 0.4085410972771827E+00, 0.1868009328861996E-06, 0.5904812108429529E-07, 0.1291405980002798E+00, 0.2876033591836524E-07, 0.3612752323451872E-38, 0.5598544436768911E-18, 0.1681677309759878E+01},
			i0:       1,
			n0:       5,
			pp:       1,
			n0in:     5,
			dmin:     2.8760335918365243E-008,
			dmin1:    0.40854091047624980,
			dmin2:    0.47735135942481804,
			dn:       2.8760335918365243E-008,
			dn1:      0.40854091047624980,
			dn2:      0.47735135942481804,
			tau:      0.12914051019182277,
			ttype:    -7,
			g:        0.0000000000000000,
			zOut:     []float64{0.7993002037419701E+00, 0.6701596935501473E+00, 0.4369342452764876E-18, 0.5598544436768911E-18, 0.8586918660261474E+00, 0.7320755909723271E+00, 0.2524235138002424E-02, 0.2098451433139096E-02, 0.6085903210497798E+00, 0.4933914566812295E+00, 0.1604009725641142E-01, 0.1806732527351204E-01, 0.5557487459415846E+00, 0.4085410972771827E+00, 0.1868009328861996E-06, 0.5904812108429529E-07, 0.1291405980002798E+00, 0.2876033591836524E-07, 0.3612752323451872E-38, 0.5598544436768911E-18, 0.1681677309759878E+01},
			tauOut:   2.8760330654564486E-008,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.6701596647898166E+00, 0.6701596935501473E+00, 0.6115792910959321E-18, 0.5598544436768911E-18, 0.7341740136451356E+00, 0.7320755909723271E+00, 0.1410235162411720E-02, 0.2098451433139096E-02, 0.5100485180319990E+00, 0.4933914566812295E+00, 0.1447165246275905E-01, 0.1806732527351204E-01, 0.3940694751022140E+00, 0.4085410972771827E+00, 0.4309503539425748E-14, 0.5904812108429529E-07, 0.9542972193014773E-15, 0.2876033591836524E-07, 0.6115792910959321E-18, 0.5598544436768911E-18},
			i0:       1,
			n0:       5,
			pp:       0,
			n0in:     5,
			dmin:     9.5429721930147733E-016,
			dmin1:    0.39406941605409296,
			dmin2:    0.49198119275848701,
			dn:       9.5429721930147733E-016,
			dn1:      0.39406941605409296,
			dn2:      0.49198119275848701,
			tau:      2.8760330654564486E-008,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.6701596647898166E+00, 0.6701596935501473E+00, 0.6115792910959321E-18, 0.5598544436768911E-18, 0.7341740136451356E+00, 0.7320755909723271E+00, 0.1410235162411720E-02, 0.2098451433139096E-02, 0.5100485180319990E+00, 0.4933914566812295E+00, 0.1447165246275905E-01, 0.1806732527351204E-01, 0.3940694751022140E+00, 0.4085410972771827E+00, 0.4309503539425748E-14, 0.5904812108429529E-07, 0.9542972193014773E-15, 0.2876033591836524E-07, 0.6115792910959321E-18, 0.5598544436768911E-18},
			tauOut:   9.5429721930146451E-016,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.6701596647898166E+00, 0.6701596647898156E+00, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.7341740136451356E+00, 0.7355842488075462E+00, 0.1410235162411720E-02, 0.9778463253267723E-03, 0.5100485180319990E+00, 0.5235423241694304E+00, 0.1447165246275905E-01, 0.1089278980244500E-01, 0.3940694751022140E+00, 0.3831766852997724E+00, 0.4309503539425748E-14, 0.1073277003016631E-28, 0.1810817848712033E+01},
			i0:       1,
			n0:       4,
			pp:       1,
			n0in:     5,
			dmin:     1.9721522630525295E-030,
			dmin1:    0.38317668529976806,
			dmin2:    0.50907067170667131,
			dn:       1.9721522630525295E-030,
			dn1:      0.38317668529976806,
			dn2:      0.50907067170667131,
			tau:      9.5429721930146451E-016,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.6701596647898166E+00, 0.6701596647898156E+00, 0.6115792910959321E-18, 0.6699979816704878E-18, 0.7341740136451356E+00, 0.7355842488075462E+00, 0.1410235162411720E-02, 0.9778463253267723E-03, 0.5100485180319990E+00, 0.5235423241694304E+00, 0.1447165246275905E-01, 0.1089278980244500E-01, 0.3940694751022140E+00, 0.3831766852997724E+00, 0.4309503539425748E-14, 0.1073277003016631E-28, 0.1810817848712033E+01},
			tauOut:   0.31895966905528556,
			ttypeOut: -8,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.3511999957345300E+00, 0.6701596647898156E+00, 0.1403302870260262E-17, 0.6699979816704878E-18, 0.4176024260775874E+00, 0.7355842488075462E+00, 0.1225912269357841E-02, 0.9778463253267723E-03, 0.2142495326472319E+00, 0.5235423241694304E+00, 0.1948131712866055E-01, 0.1089278980244500E-01, 0.4473569911582631E-01, 0.3831766852997724E+00, 0.1403302870260262E-17, 0.6699979816704878E-18},
			i0:       1,
			n0:       4,
			pp:       0,
			n0in:     4,
			dmin:     4.4735699115826311E-002,
			dmin1:    0.20335674284478694,
			dmin2:    0.35119999573453003,
			dn:       4.4735699115826311E-002,
			dn1:      0.20335674284478694,
			dn2:      0.41662457975226058,
			tau:      0.31895966905528556,
			ttype:    -8,
			g:        0.0000000000000000,
			zOut:     []float64{0.3511999957345300E+00, 0.6701596647898156E+00, 0.1403302870260262E-17, 0.6699979816704878E-18, 0.4176024260775874E+00, 0.7355842488075462E+00, 0.1225912269357841E-02, 0.9778463253267723E-03, 0.2142495326472319E+00, 0.5235423241694304E+00, 0.1948131712866055E-01, 0.1089278980244500E-01, 0.4473569911582631E-01, 0.3831766852997724E+00, 0.1403302870260262E-17, 0.6699979816704878E-18},
			tauOut:   3.9897804510299034E-002,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.3511999957345300E+00, 0.3113021912242310E+00, 0.1403302870260262E-17, 0.1882488140663987E-17, 0.4176024260775874E+00, 0.3789305338366462E+00, 0.1225912269357841E-02, 0.6931379430343076E-03, 0.2142495326472319E+00, 0.1931399073225591E+00, 0.1948131712866055E-01, 0.4512326600593524E-02, 0.4473569911582631E-01, 0.3255680049337531E-03, 0.1403302870260262E-17, 0.1882488140663987E-17, 0.1810817848712033E+01},
			i0:       1,
			n0:       4,
			pp:       1,
			n0in:     4,
			dmin:     3.2556800493375310E-004,
			dmin1:    0.17365859019389857,
			dmin2:    0.31130219122423097,
			dn:       3.2556800493375310E-004,
			dn1:      0.17365859019389857,
			dn2:      0.37770462156728835,
			tau:      3.9897804510299034E-002,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.3511999957345300E+00, 0.3113021912242310E+00, 0.1403302870260262E-17, 0.1882488140663987E-17, 0.4176024260775874E+00, 0.3789305338366462E+00, 0.1225912269357841E-02, 0.6931379430343076E-03, 0.2142495326472319E+00, 0.1931399073225591E+00, 0.1948131712866055E-01, 0.4512326600593524E-02, 0.4473569911582631E-01, 0.3255680049337531E-03, 0.1403302870260262E-17, 0.1882488140663987E-17, 0.1810817848712033E+01},
			tauOut:   3.1797943619548145E-004,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.3109842117880355E+00, 0.3113021912242310E+00, 0.2293789231233262E-17, 0.1882488140663987E-17, 0.3793056923434850E+00, 0.3789305338366462E+00, 0.3529411785314453E-03, 0.6931379430343076E-03, 0.1969813133084257E+00, 0.1931399073225591E+00, 0.7457911333267062E-05, 0.4512326600593524E-02, 0.1306574050045526E-06, 0.3255680049337531E-03, 0.2293789231233262E-17, 0.1882488140663987E-17},
			i0:       1,
			n0:       4,
			pp:       0,
			n0in:     4,
			dmin:     1.3065740500455263E-007,
			dmin1:    0.19246898670783216,
			dmin2:    0.31098421178803548,
			dn:       1.3065740500455263E-007,
			dn1:      0.19246898670783216,
			dn2:      0.37861255440045072,
			tau:      3.1797943619548145E-004,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.3109842117880355E+00, 0.3113021912242310E+00, 0.2293789231233262E-17, 0.1882488140663987E-17, 0.3793056923434850E+00, 0.3789305338366462E+00, 0.3529411785314453E-03, 0.6931379430343076E-03, 0.1969813133084257E+00, 0.1931399073225591E+00, 0.7457911333267062E-05, 0.4512326600593524E-02, 0.1306574050045526E-06, 0.3255680049337531E-03, 0.2293789231233262E-17, 0.1882488140663987E-17},
			tauOut:   1.3065240973281283E-007,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.3109842117880355E+00, 0.3109840811356258E+00, 0.2293789231233262E-17, 0.2797722987188916E-17, 0.3793056923434850E+00, 0.3796585028696068E+00, 0.3529411785314453E-03, 0.1831193463132450E-03, 0.1969813133084257E+00, 0.1968055212210360E+00, 0.7457911333267062E-05, 0.4951239861123181E-11, 0.1306574050045526E-06, 0.4403187866809611E-13, 0.2293789231233262E-17, 0.2797722987188916E-17, 0.1810817848712033E+01},
			i0:       1,
			n0:       4,
			pp:       1,
			n0in:     4,
			dmin:     4.4031878668096113E-014,
			dmin1:    0.19679806330970273,
			dmin2:    0.31098408113562576,
			dn:       4.4031878668096113E-014,
			dn1:      0.19679806330970273,
			dn2:      0.37930556169107532,
			tau:      1.3065240973281283E-007,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.3109842117880355E+00, 0.3109840811356258E+00, 0.2293789231233262E-17, 0.2797722987188916E-17, 0.3793056923434850E+00, 0.3796585028696068E+00, 0.3529411785314453E-03, 0.1831193463132450E-03, 0.1969813133084257E+00, 0.1968055212210360E+00, 0.7457911333267062E-05, 0.4951239861123181E-11, 0.1306574050045526E-06, 0.4403187866809611E-13, 0.2293789231233262E-17, 0.2797722987188916E-17, 0.1810817848712033E+01},
			tauOut:   4.4031878666982759E-014,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.3109840811355817E+00, 0.3109840811356258E+00, 0.3415542419024794E-17, 0.2797722987188916E-17, 0.3798416222158761E+00, 0.3796585028696068E+00, 0.9487875021856221E-04, 0.1831193463132450E-03, 0.1967106424757246E+00, 0.1968055212210360E+00, 0.1108289770587888E-23, 0.4951239861123181E-11, 0.5067642455139780E-26, 0.4403187866809611E-13, 0.3415542419024794E-17, 0.2797722987188916E-17},
			i0:       1,
			n0:       4,
			pp:       0,
			n0in:     4,
			dmin:     5.0676424551397798E-027,
			dmin1:    0.19671064247077341,
			dmin2:    0.31098408113558174,
			dn:       5.0676424551397798E-027,
			dn1:      0.19671064247077341,
			dn2:      0.37965850286956282,
			tau:      4.4031878666982759E-014,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.3109840811355817E+00, 0.3109840811356258E+00, 0.3415542419024794E-17, 0.2797722987188916E-17, 0.3798416222158761E+00, 0.3796585028696068E+00, 0.9487875021856221E-04, 0.1831193463132450E-03, 0.1967106424757246E+00, 0.1968055212210360E+00, 0.1108289770587888E-23, 0.4951239861123181E-11, 0.5067642455139780E-26, 0.4403187866809611E-13, 0.3415542419024794E-17, 0.2797722987188916E-17},
			tauOut:   5.0676424551397798E-027,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.3109840811355817E+00, 0.3109840811355817E+00, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.3798416222158761E+00, 0.3799365009660947E+00, 0.9487875021856221E-04, 0.4912310311151891E-04, 0.1967106424757246E+00, 0.1966615193726131E+00, 0.1108289770587888E-23, 0.2855879641297252E-49, 0.2169993432366266E+01},
			i0:       1,
			n0:       3,
			pp:       1,
			n0in:     4,
			dmin:     5.0676424551397798E-027,
			dmin1:    0.19666151937261311,
			dmin2:    0.31098408113558174,
			dn:       5.0676424551397798E-027,
			dn1:      0.19666151937261311,
			dn2:      0.37984162221587608,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.3109840811355817E+00, 0.3109840811355817E+00, 0.3415542419024794E-17, 0.4171805735046273E-17, 0.3798416222158761E+00, 0.3799365009660947E+00, 0.9487875021856221E-04, 0.4912310311151891E-04, 0.1967106424757246E+00, 0.1966615193726131E+00, 0.1108289770587888E-23, 0.2855879641297252E-49, 0.2169993432366266E+01},
			tauOut:   9.8330759686306557E-002,
			ttypeOut: -9,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2126533214492752E+00, 0.3109840811355817E+00, 0.7453545812882342E-17, 0.4171805735046273E-17, 0.2816548643828996E+00, 0.3799365009660947E+00, 0.3429951091160839E-04, 0.4912310311151891E-04, 0.9829646017539498E-01, 0.1966615193726131E+00, 0.3799365009660947E+00, 0.2855879641297252E-49},
			i0:       1,
			n0:       3,
			pp:       0,
			n0in:     3,
			dmin:     9.8296460175394978E-002,
			dmin1:    0.21265332144927518,
			dmin2:    0.21265332144927518,
			dn:       9.8296460175394978E-002,
			dn1:      0.28160574127978810,
			dn2:      0.21265332144927518,
			tau:      9.8330759686306557E-002,
			ttype:    -9,
			g:        0.0000000000000000,
			zOut:     []float64{0.2126533214492752E+00, 0.3109840811355817E+00, 0.7453545812882342E-17, 0.4171805735046273E-17, 0.2816548643828996E+00, 0.3799365009660947E+00, 0.3429951091160839E-04, 0.4912310311151891E-04, 0.9829646017539498E-01, 0.1966615193726131E+00, 0.3799365009660947E+00, 0.2855879641297252E-49},
			tauOut:   9.7172513485343004E-002,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2126533214492752E+00, 0.1154808079639322E+00, 0.7453545812882342E-17, 0.1817901582187390E-16, 0.2816548643828996E+00, 0.1845166504084682E+00, 0.3429951091160839E-04, 0.1827217490071947E-04, 0.9829646017539498E-01, 0.1105674515151256E-02, 0.3799365009660947E+00, 0.2816548643828996E+00, 0.2169993432366266E+01},
			i0:       1,
			n0:       3,
			pp:       1,
			n0in:     3,
			dmin:     1.1056745151512559E-003,
			dmin1:    0.11548080796393217,
			dmin2:    0.11548080796393217,
			dn:       1.1056745151512559E-003,
			dn1:      0.18448235089755655,
			dn2:      0.11548080796393217,
			tau:      9.7172513485343004E-002,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2126533214492752E+00, 0.1154808079639322E+00, 0.7453545812882342E-17, 0.1817901582187390E-16, 0.2816548643828996E+00, 0.1845166504084682E+00, 0.3429951091160839E-04, 0.1827217490071947E-04, 0.9829646017539498E-01, 0.1105674515151256E-02, 0.3799365009660947E+00, 0.2816548643828996E+00, 0.2169993432366266E+01},
			tauOut:   1.0942861809085330E-003,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1143865217830237E+00, 0.1154808079639322E+00, 0.2932453102768040E-16, 0.1817901582187390E-16, 0.1834406364024603E+00, 0.1845166504084682E+00, 0.1101341475930520E-06, 0.1827217490071947E-04, 0.1127820009512990E-04, 0.1105674515151256E-02, 0.1845166504084682E+00, 0.2816548643828996E+00},
			i0:       1,
			n0:       3,
			pp:       0,
			n0in:     3,
			dmin:     1.1278200095129901E-005,
			dmin1:    0.11438652178302365,
			dmin2:    0.11438652178302365,
			dn:       1.1278200095129901E-005,
			dn1:      0.18342236422755959,
			dn2:      0.11438652178302365,
			tau:      1.0942861809085330E-003,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1143865217830237E+00, 0.1154808079639322E+00, 0.2932453102768040E-16, 0.1817901582187390E-16, 0.1834406364024603E+00, 0.1845166504084682E+00, 0.1101341475930520E-06, 0.1827217490071947E-04, 0.1127820009512990E-04, 0.1105674515151256E-02, 0.1845166504084682E+00, 0.2816548643828996E+00},
			tauOut:   1.1269238360546607E-005,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1143865217830237E+00, 0.1143752525446631E+00, 0.2932453102768040E-16, 0.4703212027287794E-16, 0.1834406364024603E+00, 0.1834294772982473E+00, 0.1101341475930520E-06, 0.6771621290952006E-11, 0.1127820009512990E-04, 0.8954962962003413E-08, 0.1845166504084682E+00, 0.1834406364024603E+00, 0.2169993432366266E+01},
			i0:       1,
			n0:       3,
			pp:       1,
			n0in:     3,
			dmin:     8.9549629620034135E-009,
			dmin1:    0.11437525254466312,
			dmin2:    0.11437525254466312,
			dn:       8.9549629620034135E-009,
			dn1:      0.18342936716409974,
			dn2:      0.11437525254466312,
			tau:      1.1269238360546607E-005,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1143865217830237E+00, 0.1143752525446631E+00, 0.2932453102768040E-16, 0.4703212027287794E-16, 0.1834406364024603E+00, 0.1834294772982473E+00, 0.1101341475930520E-06, 0.6771621290952006E-11, 0.1127820009512990E-04, 0.8954962962003413E-08, 0.1845166504084682E+00, 0.1834406364024603E+00, 0.2169993432366266E+01},
			tauOut:   8.9549072084035346E-009,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1143752435897560E+00, 0.1143752525446631E+00, 0.7542783706608855E-16, 0.4703212027287794E-16, 0.1834294683501117E+00, 0.1834294772982473E+00, 0.3305882004599510E-18, 0.6771621290952006E-11, 0.5575326929115112E-13, 0.8954962962003413E-08, 0.1834294772982473E+00, 0.1834406364024603E+00},
			i0:       1,
			n0:       3,
			pp:       0,
			n0in:     3,
			dmin:     5.5753269291151117E-014,
			dmin1:    0.11437524358975594,
			dmin2:    0.11437524358975594,
			dn:       5.5753269291151117E-014,
			dn1:      0.18342946834334006,
			dn2:      0.11437524358975594,
			tau:      8.9549072084035346E-009,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1143752435897560E+00, 0.1143752525446631E+00, 0.7542783706608855E-16, 0.4703212027287794E-16, 0.1834294683501117E+00, 0.1834294772982473E+00, 0.3305882004599510E-18, 0.6771621290952006E-11, 0.5575326929115112E-13, 0.8954962962003413E-08, 0.1834294772982473E+00, 0.1834406364024603E+00},
			tauOut:   5.5753269214454873E-014,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1648283185136998E+01, 0.1396221235720571E+01, 0.1712714336271993E+00, 0.2520619494164272E+00, 0.1510753432847732E+01, 0.1119977945086946E+01, 0.3792800633372563E+00, 0.5620469213879850E+00, 0.1328441621586708E+01, 0.1019485448443405E+01, 0.1388998759717073E+00, 0.6882362364805590E+00, 0.4466938077796418E+00, 0.2681061628164644E+00, 0.7093206047768255E+00, 0.3174875209348847E+00, 0.6843112870203156E+00, 0.9979892152967577E+00, 0.6490582981441884E+00, 0.3956426765003833E+00, 0.5085572738629487E+00, 0.1122623886995757E+01, 0.9750235054014829E-02, 0.3499168501137979E-01, 0.1364886053450573E+00, 0.1417066070690837E+00, 0.2301225778544498E-01, 0.4532233329988395E-02, 0.1646009972289452E+01, 0.6930161671496210E+00, 0.2362515608142310E+00, 0.9760060629252760E+00, 0.5818602562677768E+00, 0.3984323866837953E+00, 0.1797665269485310E-01, 0.4196794303982125E+00, 0.5600419521166516E+00, 0.2492354636952108E-01, 0.2195137569256029E+00, 0.5530950584419837E+00, 0.4184071984843414E+00, 0.2222708575473020E+00, 0.2727864547293006E+00, 0.4156500978626423E+00, 0.6774373914466536E-01, 0.2745959086613283E+00, 0.1050967099374242E+00, 0.6593428521263771E-01, 0.2040338718098096E+00, 0.1079809097801335E+00, 0.1271971985482246E+00, 0.2011496719671002E+00, 0.4444741998443960E-01, 0.1290210252363728E+00, 0.5776327498150620E+00, 0.4262359329629137E-01, 0.3402556968467140E+00, 0.6023491555328507E+00, 0.1086565805630269E+00, 0.3155392911289253E+00, 0.9382999256694983E+00, 0.1171677238805356E+00, 0.7901625299559836E+00, 0.9297887823519896E+00, 0.6095853796269167E+00, 0.7973955560628040E+00, 0.8018125008387630E+00, 0.6023523535200964E+00, 0.2443177602187348E-01, 0.8114406374558937E+00, 0.2277830378453201E+00, 0.1480363940474286E-01, 0.6969081780841352E+00, 0.3759308106650992E+00, 0.8012406122589412E-01, 0.5487604052643561E+00, 0.9781022865072954E-01, 0.1017549972519246E+00, 0.2961248981181939E-03, 0.7617929262469909E-01, 0.8408398800007548E-04, 0.3802088861182694E-03, 0.0000000000000000E+00, 0.0000000000000000E+00},
			i0:       1,
			n0:       21,
			pp:       0,
			n0in:     21,
			dmin:     -0.0000000000000000,
			dmin1:    0.0000000000000000,
			dmin2:    0.0000000000000000,
			dn:       0.0000000000000000,
			dn1:      0.0000000000000000,
			dn2:      0.0000000000000000,
			tau:      0.0000000000000000,
			ttype:    0,
			g:        0.0000000000000000,
			zOut:     []float64{0.1648283185136998E+01, 0.1396221235720571E+01, 0.1712714336271993E+00, 0.2520619494164272E+00, 0.1510753432847732E+01, 0.1119977945086946E+01, 0.3792800633372563E+00, 0.5620469213879850E+00, 0.1328441621586708E+01, 0.1019485448443405E+01, 0.1388998759717073E+00, 0.6882362364805590E+00, 0.4466938077796418E+00, 0.2681061628164644E+00, 0.7093206047768255E+00, 0.3174875209348847E+00, 0.6843112870203156E+00, 0.9979892152967577E+00, 0.6490582981441884E+00, 0.3956426765003833E+00, 0.5085572738629487E+00, 0.1122623886995757E+01, 0.9750235054014829E-02, 0.3499168501137979E-01, 0.1364886053450573E+00, 0.1417066070690837E+00, 0.2301225778544498E-01, 0.4532233329988395E-02, 0.1646009972289452E+01, 0.6930161671496210E+00, 0.2362515608142310E+00, 0.9760060629252760E+00, 0.5818602562677768E+00, 0.3984323866837953E+00, 0.1797665269485310E-01, 0.4196794303982125E+00, 0.5600419521166516E+00, 0.2492354636952108E-01, 0.2195137569256029E+00, 0.5530950584419837E+00, 0.4184071984843414E+00, 0.2222708575473020E+00, 0.2727864547293006E+00, 0.4156500978626423E+00, 0.6774373914466536E-01, 0.2745959086613283E+00, 0.1050967099374242E+00, 0.6593428521263771E-01, 0.2040338718098096E+00, 0.1079809097801335E+00, 0.1271971985482246E+00, 0.2011496719671002E+00, 0.4444741998443960E-01, 0.1290210252363728E+00, 0.5776327498150620E+00, 0.4262359329629137E-01, 0.3402556968467140E+00, 0.6023491555328507E+00, 0.1086565805630269E+00, 0.3155392911289253E+00, 0.9382999256694983E+00, 0.1171677238805356E+00, 0.7901625299559836E+00, 0.9297887823519896E+00, 0.6095853796269167E+00, 0.7973955560628040E+00, 0.8018125008387630E+00, 0.6023523535200964E+00, 0.2443177602187348E-01, 0.8114406374558937E+00, 0.2277830378453201E+00, 0.1480363940474286E-01, 0.6969081780841352E+00, 0.3759308106650992E+00, 0.8012406122589412E-01, 0.5487604052643561E+00, 0.9781022865072954E-01, 0.1017549972519246E+00, 0.2961248981181939E-03, 0.7617929262469909E-01, 0.8408398800007548E-04, 0.3802088861182694E-03, 0.0000000000000000E+00, 0.0000000000000000E+00},
			tauOut:   0.0000000000000000,
			ttypeOut: -1,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1648283185136998E+01, 0.1819554618764197E+01, 0.1712714336271993E+00, 0.1422045283129674E+00, 0.1510753432847732E+01, 0.1747828967872021E+01, 0.3792800633372563E+00, 0.2882727267008810E+00, 0.1328441621586708E+01, 0.1179068770857534E+01, 0.1388998759717073E+00, 0.5262264257308433E-01, 0.4466938077796418E+00, 0.1103391769983383E+01, 0.7093206047768255E+00, 0.4399127392187891E+00, 0.6843112870203156E+00, 0.8934568459457148E+00, 0.6490582981441884E+00, 0.3694451726238032E+00, 0.5085572738629487E+00, 0.1488623362931603E+00, 0.9750235054014829E-02, 0.8939776288934402E-02, 0.1364886053450573E+00, 0.1505610868415679E+00, 0.2301225778544498E-01, 0.2515816443301624E+00, 0.1646009972289452E+01, 0.1630679888773521E+01, 0.2362515608142310E+00, 0.8429943526342391E-01, 0.5818602562677768E+00, 0.5155374736992060E+00, 0.1797665269485310E-01, 0.1952851185677853E-01, 0.5600419521166516E+00, 0.7600271971854760E+00, 0.2195137569256029E+00, 0.1208458544696003E+00, 0.4184071984843414E+00, 0.5703477987440417E+00, 0.2727864547293006E+00, 0.3240053608004366E-01, 0.6774373914466536E-01, 0.1404399130020459E+00, 0.1050967099374242E+00, 0.1526865702536626E+00, 0.2040338718098096E+00, 0.1785445001043715E+00, 0.1271971985482246E+00, 0.3166486394939177E-01, 0.4444741998443960E-01, 0.5904153058501098E+00, 0.5776327498150620E+00, 0.3328891237445398E+00, 0.3402556968467140E+00, 0.1160231536652011E+00, 0.1086565805630269E+00, 0.8787251358464724E+00, 0.9382999256694983E+00, 0.8497373197790092E+00, 0.7901625299559836E+00, 0.5668475593321608E+00, 0.6095853796269167E+00, 0.8445503211335190E+00, 0.8018125008387630E+00, 0.2319542476253924E-01, 0.2443177602187348E-01, 0.2290193891046544E+00, 0.2277830378453201E+00, 0.6931459494493321E+00, 0.6969081780841352E+00, 0.8388628986069724E-01, 0.8012406122589412E-01, 0.9342352322344821E-01, 0.9781022865072954E-01, 0.4682830325399513E-02, 0.2961248981181939E-03, 0.5317160915449039E-05, 0.8408398800007548E-04, 0.7876682708462645E-04, 0.0000000000000000E+00, 0.8939776288934402E-02, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			n0in:     21,
			dmin:     7.8766827084626452E-005,
			dmin1:    1.2363512593342330E-003,
			dmin2:    1.2363512593342330E-003,
			dn:       7.8766827084626452E-005,
			dn1:      4.3867054272813191E-003,
			dn2:      3.7622286348031123E-003,
			tau:      0.0000000000000000,
			ttype:    -1,
			g:        0.0000000000000000,
			zOut:     []float64{0.1648283185136998E+01, 0.1819554618764197E+01, 0.1712714336271993E+00, 0.1422045283129674E+00, 0.1510753432847732E+01, 0.1747828967872021E+01, 0.3792800633372563E+00, 0.2882727267008810E+00, 0.1328441621586708E+01, 0.1179068770857534E+01, 0.1388998759717073E+00, 0.5262264257308433E-01, 0.4466938077796418E+00, 0.1103391769983383E+01, 0.7093206047768255E+00, 0.4399127392187891E+00, 0.6843112870203156E+00, 0.8934568459457148E+00, 0.6490582981441884E+00, 0.3694451726238032E+00, 0.5085572738629487E+00, 0.1488623362931603E+00, 0.9750235054014829E-02, 0.8939776288934402E-02, 0.1364886053450573E+00, 0.1505610868415679E+00, 0.2301225778544498E-01, 0.2515816443301624E+00, 0.1646009972289452E+01, 0.1630679888773521E+01, 0.2362515608142310E+00, 0.8429943526342391E-01, 0.5818602562677768E+00, 0.5155374736992060E+00, 0.1797665269485310E-01, 0.1952851185677853E-01, 0.5600419521166516E+00, 0.7600271971854760E+00, 0.2195137569256029E+00, 0.1208458544696003E+00, 0.4184071984843414E+00, 0.5703477987440417E+00, 0.2727864547293006E+00, 0.3240053608004366E-01, 0.6774373914466536E-01, 0.1404399130020459E+00, 0.1050967099374242E+00, 0.1526865702536626E+00, 0.2040338718098096E+00, 0.1785445001043715E+00, 0.1271971985482246E+00, 0.3166486394939177E-01, 0.4444741998443960E-01, 0.5904153058501098E+00, 0.5776327498150620E+00, 0.3328891237445398E+00, 0.3402556968467140E+00, 0.1160231536652011E+00, 0.1086565805630269E+00, 0.8787251358464724E+00, 0.9382999256694983E+00, 0.8497373197790092E+00, 0.7901625299559836E+00, 0.5668475593321608E+00, 0.6095853796269167E+00, 0.8445503211335190E+00, 0.8018125008387630E+00, 0.2319542476253924E-01, 0.2443177602187348E-01, 0.2290193891046544E+00, 0.2277830378453201E+00, 0.6931459494493321E+00, 0.6969081780841352E+00, 0.8388628986069724E-01, 0.8012406122589412E-01, 0.9342352322344821E-01, 0.9781022865072954E-01, 0.4682830325399513E-02, 0.2961248981181939E-03, 0.5317160915449039E-05, 0.8408398800007548E-04, 0.7876682708462645E-04, 0.0000000000000000E+00, 0.8939776288934402E-02, 0.1037537856266618 - 321},
			tauOut:   0.0000000000000000,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1961759147077164E+01, 0.1819554618764197E+01, 0.1266970995487882E+00, 0.1422045283129674E+00, 0.1909404595024114E+01, 0.1747828967872021E+01, 0.1780101349021133E+00, 0.2882727267008810E+00, 0.1053681278528505E+01, 0.1179068770857534E+01, 0.5510526941411123E-01, 0.5262264257308433E-01, 0.1488199239788061E+01, 0.1103391769983383E+01, 0.2641064704009213E+00, 0.4399127392187891E+00, 0.9987955481685968E+00, 0.8934568459457148E+00, 0.5506279200968747E-01, 0.3694451726238032E+00, 0.1027393205724072E+00, 0.1488623362931603E+00, 0.1310094739466220E-01, 0.8939776288934402E-02, 0.3890417837770681E+00, 0.1505610868415679E+00, 0.1054511738587064E+01, 0.2515816443301624E+00, 0.6604675854498806E+00, 0.1630679888773521E+01, 0.6580113672099847E-01, 0.8429943526342391E-01, 0.4692648488349861E+00, 0.5155374736992060E+00, 0.3162862116895929E-01, 0.1952851185677853E-01, 0.8492444304861170E+00, 0.7600271971854760E+00, 0.8115939841327705E-01, 0.1208458544696003E+00, 0.5215889364108083E+00, 0.5703477987440417E+00, 0.8723974284448969E-02, 0.3240053608004366E-01, 0.2844025089712595E+00, 0.1404399130020459E+00, 0.9585480612390133E-01, 0.1526865702536626E+00, 0.1143545579298620E+00, 0.1785445001043715E+00, 0.1634864466429828E+00, 0.3166486394939177E-01, 0.7598179829516669E+00, 0.5904153058501098E+00, 0.5083170815153470E-01, 0.3328891237445398E+00, 0.9439165813601388E+00, 0.1160231536652011E+00, 0.7910503496831139E+00, 0.8787251358464724E+00, 0.6255345294280562E+00, 0.8497373197790092E+00, 0.7653155273545736E+00, 0.5668475593321608E+00, 0.1024302185414846E+00, 0.8445503211335190E+00, 0.5186166821452450E-01, 0.2319542476253924E-01, 0.8703036703394620E+00, 0.2290193891046544E+00, 0.6681052144545734E-01, 0.6931459494493321E+00, 0.1104992916386881E+00, 0.8388628986069724E-01, 0.3959179295799719E-02, 0.9342352322344821E-01, 0.7289681905152429E-03, 0.4682830325399513E-02, 0.5745324691222600E-06, 0.5317160915449039E-05, 0.7819229461550419E-04, 0.7876682708462645E-04, 0.8723974284448969E-02, 0.8939776288934402E-02},
			i0:       1,
			n0:       21,
			pp:       0,
			n0in:     21,
			dmin:     7.8192294615504193E-005,
			dmin1:    7.2365102959979382E-004,
			dmin2:    1.7075768415239889E-002,
			dn:       7.8192294615504193E-005,
			dn1:      7.2365102959979382E-004,
			dn2:      1.7075768415239889E-002,
			tau:      0.0000000000000000,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1961759147077164E+01, 0.1819554618764197E+01, 0.1266970995487882E+00, 0.1422045283129674E+00, 0.1909404595024114E+01, 0.1747828967872021E+01, 0.1780101349021133E+00, 0.2882727267008810E+00, 0.1053681278528505E+01, 0.1179068770857534E+01, 0.5510526941411123E-01, 0.5262264257308433E-01, 0.1488199239788061E+01, 0.1103391769983383E+01, 0.2641064704009213E+00, 0.4399127392187891E+00, 0.9987955481685968E+00, 0.8934568459457148E+00, 0.5506279200968747E-01, 0.3694451726238032E+00, 0.1027393205724072E+00, 0.1488623362931603E+00, 0.1310094739466220E-01, 0.8939776288934402E-02, 0.3890417837770681E+00, 0.1505610868415679E+00, 0.1054511738587064E+01, 0.2515816443301624E+00, 0.6604675854498806E+00, 0.1630679888773521E+01, 0.6580113672099847E-01, 0.8429943526342391E-01, 0.4692648488349861E+00, 0.5155374736992060E+00, 0.3162862116895929E-01, 0.1952851185677853E-01, 0.8492444304861170E+00, 0.7600271971854760E+00, 0.8115939841327705E-01, 0.1208458544696003E+00, 0.5215889364108083E+00, 0.5703477987440417E+00, 0.8723974284448969E-02, 0.3240053608004366E-01, 0.2844025089712595E+00, 0.1404399130020459E+00, 0.9585480612390133E-01, 0.1526865702536626E+00, 0.1143545579298620E+00, 0.1785445001043715E+00, 0.1634864466429828E+00, 0.3166486394939177E-01, 0.7598179829516669E+00, 0.5904153058501098E+00, 0.5083170815153470E-01, 0.3328891237445398E+00, 0.9439165813601388E+00, 0.1160231536652011E+00, 0.7910503496831139E+00, 0.8787251358464724E+00, 0.6255345294280562E+00, 0.8497373197790092E+00, 0.7653155273545736E+00, 0.5668475593321608E+00, 0.1024302185414846E+00, 0.8445503211335190E+00, 0.5186166821452450E-01, 0.2319542476253924E-01, 0.8703036703394620E+00, 0.2290193891046544E+00, 0.6681052144545734E-01, 0.6931459494493321E+00, 0.1104992916386881E+00, 0.8388628986069724E-01, 0.3959179295799719E-02, 0.9342352322344821E-01, 0.7289681905152429E-03, 0.4682830325399513E-02, 0.5745324691222600E-06, 0.5317160915449039E-05, 0.7819229461550419E-04, 0.7876682708462645E-04, 0.8723974284448969E-02, 0.8939776288934402E-02},
			tauOut:   7.8083356181667918E-005,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1961759147077164E+01, 0.2088378163269771E+01, 0.1266970995487882E+00, 0.1158391848322702E+00, 0.1909404595024114E+01, 0.1971497461737776E+01, 0.1780101349021133E+00, 0.9513882222772962E-01, 0.1053681278528505E+01, 0.1013569642358705E+01, 0.5510526941411123E-01, 0.8090970429970110E-01, 0.1488199239788061E+01, 0.1671317922533099E+01, 0.2641064704009213E+00, 0.1578325483874163E+00, 0.9987955481685968E+00, 0.8959477084346864E+00, 0.5506279200968747E-01, 0.6314111623521673E-02, 0.1027393205724072E+00, 0.1094480729873660E+00, 0.1310094739466220E-01, 0.4656834793406785E-01, 0.3890417837770681E+00, 0.1396907091073883E+01, 0.1054511738587064E+01, 0.4985806330739840E+00, 0.6604675854498806E+00, 0.2276100057407134E+00, 0.6580113672099847E-01, 0.1356625794022653E+00, 0.4692648488349861E+00, 0.3651528072454984E+00, 0.3162862116895929E-01, 0.7355942454424362E-01, 0.8492444304861170E+00, 0.8567663209989688E+00, 0.8115939841327705E-01, 0.4940885660487250E-01, 0.5215889364108083E+00, 0.4808259707342031E+00, 0.8723974284448969E-02, 0.5160120970399038E-02, 0.2844025089712595E+00, 0.3750191107685802E+00, 0.9585480612390133E-01, 0.2922900104287094E-01, 0.1143545579298620E+00, 0.2485339201737921E+00, 0.1634864466429828E+00, 0.4998108187459615E+00, 0.7598179829516669E+00, 0.3107607890010585E+00, 0.5083170815153470E-01, 0.1543981540828483E+00, 0.9439165813601388E+00, 0.1580490693604223E+01, 0.7910503496831139E+00, 0.3130858727896049E+00, 0.6255345294280562E+00, 0.1077686100636843E+01, 0.7653155273545736E+00, 0.7274051012980143E-01, 0.1024302185414846E+00, 0.8147329327002600E-01, 0.5186166821452450E-01, 0.5539901283655780E+00, 0.8703036703394620E+00, 0.3830459800631597E+00, 0.6681052144545734E-01, 0.1927318305890360E-01, 0.1104992916386881E+00, 0.9510720451940254E-01, 0.3959179295799719E-02, 0.3034592154998885E-04, 0.7289681905152429E-03, 0.6211134452527084E-03, 0.5745324691222600E-06, 0.7232819130731082E-07, 0.7819229461550419E-04, 0.3661024252896476E-07, 0.8723974284448969E-02, 0.5160120970399038E-02, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			n0in:     21,
			dmin:     3.6610242528964756E-008,
			dmin1:    6.2053891278358614E-004,
			dmin2:    2.9611625055501498E-002,
			dn:       3.6610242528964756E-008,
			dn1:      6.2053891278358614E-004,
			dn2:      9.1148025223602810E-002,
			tau:      7.8083356181667918E-005,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.1961759147077164E+01, 0.2088378163269771E+01, 0.1266970995487882E+00, 0.1158391848322702E+00, 0.1909404595024114E+01, 0.1971497461737776E+01, 0.1780101349021133E+00, 0.9513882222772962E-01, 0.1053681278528505E+01, 0.1013569642358705E+01, 0.5510526941411123E-01, 0.8090970429970110E-01, 0.1488199239788061E+01, 0.1671317922533099E+01, 0.2641064704009213E+00, 0.1578325483874163E+00, 0.9987955481685968E+00, 0.8959477084346864E+00, 0.5506279200968747E-01, 0.6314111623521673E-02, 0.1027393205724072E+00, 0.1094480729873660E+00, 0.1310094739466220E-01, 0.4656834793406785E-01, 0.3890417837770681E+00, 0.1396907091073883E+01, 0.1054511738587064E+01, 0.4985806330739840E+00, 0.6604675854498806E+00, 0.2276100057407134E+00, 0.6580113672099847E-01, 0.1356625794022653E+00, 0.4692648488349861E+00, 0.3651528072454984E+00, 0.3162862116895929E-01, 0.7355942454424362E-01, 0.8492444304861170E+00, 0.8567663209989688E+00, 0.8115939841327705E-01, 0.4940885660487250E-01, 0.5215889364108083E+00, 0.4808259707342031E+00, 0.8723974284448969E-02, 0.5160120970399038E-02, 0.2844025089712595E+00, 0.3750191107685802E+00, 0.9585480612390133E-01, 0.2922900104287094E-01, 0.1143545579298620E+00, 0.2485339201737921E+00, 0.1634864466429828E+00, 0.4998108187459615E+00, 0.7598179829516669E+00, 0.3107607890010585E+00, 0.5083170815153470E-01, 0.1543981540828483E+00, 0.9439165813601388E+00, 0.1580490693604223E+01, 0.7910503496831139E+00, 0.3130858727896049E+00, 0.6255345294280562E+00, 0.1077686100636843E+01, 0.7653155273545736E+00, 0.7274051012980143E-01, 0.1024302185414846E+00, 0.8147329327002600E-01, 0.5186166821452450E-01, 0.5539901283655780E+00, 0.8703036703394620E+00, 0.3830459800631597E+00, 0.6681052144545734E-01, 0.1927318305890360E-01, 0.1104992916386881E+00, 0.9510720451940254E-01, 0.3959179295799719E-02, 0.3034592154998885E-04, 0.7289681905152429E-03, 0.6211134452527084E-03, 0.5745324691222600E-06, 0.7232819130731082E-07, 0.7819229461550419E-04, 0.3661024252896476E-07, 0.8723974284448969E-02, 0.5160120970399038E-02, 0.1037537856266618 - 321},
			tauOut:   3.6605973538398975E-008,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2204217311496068E+01, 0.2088378163269771E+01, 0.1036089580076783E+00, 0.1158391848322702E+00, 0.1963027289351853E+01, 0.1971497461737776E+01, 0.4912301654839819E-01, 0.9513882222772962E-01, 0.1045356293504034E+01, 0.1013569642358705E+01, 0.1293586117415210E+00, 0.8090970429970110E-01, 0.1699791822573021E+01, 0.1671317922533099E+01, 0.8319236989271818E-01, 0.1578325483874163E+00, 0.8190694135595162E+00, 0.8959477084346864E+00, 0.8437225690290790E-03, 0.6314111623521673E-02, 0.1551726617464312E+00, 0.1094480729873660E+00, 0.4192211096758560E+00, 0.4656834793406785E-01, 0.1476266577866037E+01, 0.1396907091073883E+01, 0.7687090018675194E-01, 0.4985806330739840E+00, 0.2864016483502532E+00, 0.2276100057407134E+00, 0.1729653861709652E+00, 0.1356625794022653E+00, 0.2657468090128033E+00, 0.3651528072454984E+00, 0.2371551996266359E+00, 0.7355942454424362E-01, 0.6690199413712319E+00, 0.8567663209989688E+00, 0.3551024412099299E-01, 0.4940885660487250E-01, 0.4504758109776356E+00, 0.4808259707342031E+00, 0.4295777776785939E-02, 0.5160120970399038E-02, 0.3999522974286917E+00, 0.3750191107685802E+00, 0.1816316160365039E-01, 0.2922900104287094E-01, 0.7301815407101296E+00, 0.2485339201737921E+00, 0.2127164214993764E+00, 0.4998108187459615E+00, 0.2524424849785568E+00, 0.3107607890010585E+00, 0.9666552191416621E+00, 0.1543981540828483E+00, 0.9269213106461920E+00, 0.1580490693604223E+01, 0.3640096408786760E+00, 0.3130858727896049E+00, 0.7864169332819952E+00, 0.1077686100636843E+01, 0.7535963003344312E-02, 0.7274051012980143E-01, 0.6279274220262862E+00, 0.8147329327002600E-01, 0.3379430237022286E+00, 0.5539901283655780E+00, 0.6437610281386123E-01, 0.3830459800631597E+00, 0.2847358698029713E-01, 0.1927318305890360E-01, 0.6666392685468185E-01, 0.9510720451940254E-01, 0.2827355178816361E-06, 0.3034592154998885E-04, 0.6208664319525956E-03, 0.6211134452527084E-03, 0.4264931214133004E-11, 0.7232819130731082E-07, 0.4059351646211308E-14, 0.3661024252896476E-07, 0.8437225690290790E-03, 0.5160120970399038E-02},
			i0:       1,
			n0:       21,
			pp:       0,
			n0in:     21,
			dmin:     4.0593516462113082E-015,
			dmin1:    6.2079410376128833E-004,
			dmin2:    4.5102919754957636E-002,
			dn:       4.0593516462113082E-015,
			dn1:      6.2079410376128833E-004,
			dn2:      6.6633580933131861E-002,
			tau:      3.6605973538398975E-008,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2204217311496068E+01, 0.2088378163269771E+01, 0.1036089580076783E+00, 0.1158391848322702E+00, 0.1963027289351853E+01, 0.1971497461737776E+01, 0.4912301654839819E-01, 0.9513882222772962E-01, 0.1045356293504034E+01, 0.1013569642358705E+01, 0.1293586117415210E+00, 0.8090970429970110E-01, 0.1699791822573021E+01, 0.1671317922533099E+01, 0.8319236989271818E-01, 0.1578325483874163E+00, 0.8190694135595162E+00, 0.8959477084346864E+00, 0.8437225690290790E-03, 0.6314111623521673E-02, 0.1551726617464312E+00, 0.1094480729873660E+00, 0.4192211096758560E+00, 0.4656834793406785E-01, 0.1476266577866037E+01, 0.1396907091073883E+01, 0.7687090018675194E-01, 0.4985806330739840E+00, 0.2864016483502532E+00, 0.2276100057407134E+00, 0.1729653861709652E+00, 0.1356625794022653E+00, 0.2657468090128033E+00, 0.3651528072454984E+00, 0.2371551996266359E+00, 0.7355942454424362E-01, 0.6690199413712319E+00, 0.8567663209989688E+00, 0.3551024412099299E-01, 0.4940885660487250E-01, 0.4504758109776356E+00, 0.4808259707342031E+00, 0.4295777776785939E-02, 0.5160120970399038E-02, 0.3999522974286917E+00, 0.3750191107685802E+00, 0.1816316160365039E-01, 0.2922900104287094E-01, 0.7301815407101296E+00, 0.2485339201737921E+00, 0.2127164214993764E+00, 0.4998108187459615E+00, 0.2524424849785568E+00, 0.3107607890010585E+00, 0.9666552191416621E+00, 0.1543981540828483E+00, 0.9269213106461920E+00, 0.1580490693604223E+01, 0.3640096408786760E+00, 0.3130858727896049E+00, 0.7864169332819952E+00, 0.1077686100636843E+01, 0.7535963003344312E-02, 0.7274051012980143E-01, 0.6279274220262862E+00, 0.8147329327002600E-01, 0.3379430237022286E+00, 0.5539901283655780E+00, 0.6437610281386123E-01, 0.3830459800631597E+00, 0.2847358698029713E-01, 0.1927318305890360E-01, 0.6666392685468185E-01, 0.9510720451940254E-01, 0.2827355178816361E-06, 0.3034592154998885E-04, 0.6208664319525956E-03, 0.6211134452527084E-03, 0.4264931214133004E-11, 0.7232819130731082E-07, 0.4059351646211308E-14, 0.3661024252896476E-07, 0.8437225690290790E-03, 0.5160120970399038E-02},
			tauOut:   4.0593516183260787E-015,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2204217311496068E+01, 0.2307826269503742E+01, 0.1036089580076783E+00, 0.8812934261040264E-01, 0.1963027289351853E+01, 0.1924020963289845E+01, 0.4912301654839819E-01, 0.2668944646890268E-01, 0.1045356293504034E+01, 0.1148025458776649E+01, 0.1293586117415210E+00, 0.1915312145184878E+00, 0.1699791822573021E+01, 0.1591452977947247E+01, 0.8319236989271818E-01, 0.4281642408847453E-01, 0.8190694135595162E+00, 0.7770967120400667E+00, 0.8437225690290790E-03, 0.1684766835109566E-03, 0.1551726617464312E+00, 0.5742252947387723E+00, 0.4192211096758560E+00, 0.1077768810640642E+01, 0.1476266577866037E+01, 0.4753686674121431E+00, 0.7687090018675194E-01, 0.4631342794110950E-01, 0.2864016483502532E+00, 0.4130536065801048E+00, 0.1729653861709652E+00, 0.1112809541240190E+00, 0.2657468090128033E+00, 0.3916210545154161E+00, 0.2371551996266359E+00, 0.4051405202062471E+00, 0.6690199413712319E+00, 0.2993896652859737E+00, 0.3551024412099299E-01, 0.5343038813025977E-01, 0.4504758109776356E+00, 0.4013412006241577E+00, 0.4295777776785939E-02, 0.4280911574482486E-02, 0.3999522974286917E+00, 0.4138345474578556E+00, 0.1816316160365039E-01, 0.3204760309498118E-01, 0.7301815407101296E+00, 0.9108503591145208E+00, 0.2127164214993764E+00, 0.5895442813598011E-01, 0.2524424849785568E+00, 0.1160143275984235E+01, 0.9666552191416621E+00, 0.7723298847804961E+00, 0.9269213106461920E+00, 0.5186010667443678E+00, 0.3640096408786760E+00, 0.5519914319921668E+00, 0.7864169332819952E+00, 0.2419614642931688E+00, 0.7535963003344312E-02, 0.1955698951896722E-01, 0.6279274220262862E+00, 0.9463134562095434E+00, 0.3379430237022286E+00, 0.2298969194226957E-01, 0.6437610281386123E-01, 0.6985999785188474E-01, 0.2847358698029713E-01, 0.2717093011896993E-01, 0.6666392685468185E-01, 0.3949327947122574E-01, 0.2827355178816361E-06, 0.4444831994804014E-08, 0.6208664319525956E-03, 0.6208619913814727E-03, 0.4264931214133004E-11, 0.2788519153273688E-22, 0.4059351646211308E-14, 0.3786532345060857E-28, 0.8437225690290790E-03, 0.1684766835109566E-03, 0.1037537856266618 - 321},
			i0:       1,
			n0:       21,
			pp:       1,
			n0in:     21,
			dmin:     3.7865323450608567E-029,
			dmin1:    6.2086198711654151E-004,
			dmin2:    3.9492996735707858E-002,
			dn:       3.7865323450608567E-029,
			dn1:      6.2086198711654151E-004,
			dn2:      3.9492996735707858E-002,
			tau:      4.0593516183260787E-015,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2204217311496068E+01, 0.2307826269503742E+01, 0.1036089580076783E+00, 0.8812934261040264E-01, 0.1963027289351853E+01, 0.1924020963289845E+01, 0.4912301654839819E-01, 0.2668944646890268E-01, 0.1045356293504034E+01, 0.1148025458776649E+01, 0.1293586117415210E+00, 0.1915312145184878E+00, 0.1699791822573021E+01, 0.1591452977947247E+01, 0.8319236989271818E-01, 0.4281642408847453E-01, 0.8190694135595162E+00, 0.7770967120400667E+00, 0.8437225690290790E-03, 0.1684766835109566E-03, 0.1551726617464312E+00, 0.5742252947387723E+00, 0.4192211096758560E+00, 0.1077768810640642E+01, 0.1476266577866037E+01, 0.4753686674121431E+00, 0.7687090018675194E-01, 0.4631342794110950E-01, 0.2864016483502532E+00, 0.4130536065801048E+00, 0.1729653861709652E+00, 0.1112809541240190E+00, 0.2657468090128033E+00, 0.3916210545154161E+00, 0.2371551996266359E+00, 0.4051405202062471E+00, 0.6690199413712319E+00, 0.2993896652859737E+00, 0.3551024412099299E-01, 0.5343038813025977E-01, 0.4504758109776356E+00, 0.4013412006241577E+00, 0.4295777776785939E-02, 0.4280911574482486E-02, 0.3999522974286917E+00, 0.4138345474578556E+00, 0.1816316160365039E-01, 0.3204760309498118E-01, 0.7301815407101296E+00, 0.9108503591145208E+00, 0.2127164214993764E+00, 0.5895442813598011E-01, 0.2524424849785568E+00, 0.1160143275984235E+01, 0.9666552191416621E+00, 0.7723298847804961E+00, 0.9269213106461920E+00, 0.5186010667443678E+00, 0.3640096408786760E+00, 0.5519914319921668E+00, 0.7864169332819952E+00, 0.2419614642931688E+00, 0.7535963003344312E-02, 0.1955698951896722E-01, 0.6279274220262862E+00, 0.9463134562095434E+00, 0.3379430237022286E+00, 0.2298969194226957E-01, 0.6437610281386123E-01, 0.6985999785188474E-01, 0.2847358698029713E-01, 0.2717093011896993E-01, 0.6666392685468185E-01, 0.3949327947122574E-01, 0.2827355178816361E-06, 0.4444831994804014E-08, 0.6208664319525956E-03, 0.6208619913814727E-03, 0.4264931214133004E-11, 0.2788519153273688E-22, 0.4059351646211308E-14, 0.3786532345060857E-28, 0.8437225690290790E-03, 0.1684766835109566E-03, 0.1037537856266618 - 321},
			tauOut:   3.7865323450608567E-029,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2395955612114145E+01, 0.2307826269503742E+01, 0.7077038564739886E-01, 0.8812934261040264E-01, 0.1879940024111348E+01, 0.1924020963289845E+01, 0.1629847954401656E-01, 0.2668944646890268E-01, 0.1323258193751120E+01, 0.1148025458776649E+01, 0.2303502998543534E+00, 0.1915312145184878E+00, 0.1403919102181368E+01, 0.1591452977947247E+01, 0.2369972908607684E-01, 0.4281642408847453E-01, 0.7535654596375009E+00, 0.7770967120400667E+00, 0.1283811140869274E-03, 0.1684766835109566E-03, 0.1651865724265327E+01, 0.5742252947387723E+00, 0.3101568824672333E+00, 0.1077768810640642E+01, 0.2115252128860193E+00, 0.4753686674121431E+00, 0.9043805314343908E-01, 0.4631342794110950E-01, 0.4338965075606848E+00, 0.4130536065801048E+00, 0.1004386157577793E+00, 0.1112809541240190E+00, 0.6963229589638840E+00, 0.3916210545154161E+00, 0.1741934301847783E+00, 0.4051405202062471E+00, 0.1786266232314551E+00, 0.2993896652859737E+00, 0.1200482645536405E+00, 0.5343038813025977E-01, 0.2855738476449996E+00, 0.4013412006241577E+00, 0.6203611145567293E-02, 0.4280911574482486E-02, 0.4396785394072695E+00, 0.4138345474578556E+00, 0.6639071087521134E-01, 0.3204760309498118E-01, 0.9034140763752896E+00, 0.9108503591145208E+00, 0.7570790092830114E-01, 0.5895442813598011E-01, 0.1856765259836430E+01, 0.1160143275984235E+01, 0.2157144528657351E+00, 0.7723298847804961E+00, 0.8548780458707995E+00, 0.5186010667443678E+00, 0.1562335771835849E+00, 0.5519914319921668E+00, 0.1052848766285511E+00, 0.2419614642931688E+00, 0.1757806338135458E+00, 0.1955698951896722E-01, 0.7935225143382671E+00, 0.9463134562095434E+00, 0.2023962522401480E-02, 0.2298969194226957E-01, 0.9500696544845319E-01, 0.6985999785188474E-01, 0.1129463646814219E-01, 0.2717093011896993E-01, 0.2819864744791555E-01, 0.3949327947122574E-01, 0.9786381594178541E-10, 0.4444831994804014E-08, 0.6208618935176568E-03, 0.6208619913814727E-03, 0.1700670967075909E-47, 0.2788519153273688E-22},
			i0:       1,
			n0:       20,
			pp:       0,
			n0in:     21,
			dmin:     3.7865323450608567E-029,
			dmin1:    6.2086189351765679E-004,
			dmin2:    2.8198643003083550E-002,
			dn:       3.7865323450608567E-029,
			dn1:      6.2086189351765679E-004,
			dn2:      2.8198643003083550E-002,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2395955612114145E+01, 0.2307826269503742E+01, 0.7077038564739886E-01, 0.8812934261040264E-01, 0.1879940024111348E+01, 0.1924020963289845E+01, 0.1629847954401656E-01, 0.2668944646890268E-01, 0.1323258193751120E+01, 0.1148025458776649E+01, 0.2303502998543534E+00, 0.1915312145184878E+00, 0.1403919102181368E+01, 0.1591452977947247E+01, 0.2369972908607684E-01, 0.4281642408847453E-01, 0.7535654596375009E+00, 0.7770967120400667E+00, 0.1283811140869274E-03, 0.1684766835109566E-03, 0.1651865724265327E+01, 0.5742252947387723E+00, 0.3101568824672333E+00, 0.1077768810640642E+01, 0.2115252128860193E+00, 0.4753686674121431E+00, 0.9043805314343908E-01, 0.4631342794110950E-01, 0.4338965075606848E+00, 0.4130536065801048E+00, 0.1004386157577793E+00, 0.1112809541240190E+00, 0.6963229589638840E+00, 0.3916210545154161E+00, 0.1741934301847783E+00, 0.4051405202062471E+00, 0.1786266232314551E+00, 0.2993896652859737E+00, 0.1200482645536405E+00, 0.5343038813025977E-01, 0.2855738476449996E+00, 0.4013412006241577E+00, 0.6203611145567293E-02, 0.4280911574482486E-02, 0.4396785394072695E+00, 0.4138345474578556E+00, 0.6639071087521134E-01, 0.3204760309498118E-01, 0.9034140763752896E+00, 0.9108503591145208E+00, 0.7570790092830114E-01, 0.5895442813598011E-01, 0.1856765259836430E+01, 0.1160143275984235E+01, 0.2157144528657351E+00, 0.7723298847804961E+00, 0.8548780458707995E+00, 0.5186010667443678E+00, 0.1562335771835849E+00, 0.5519914319921668E+00, 0.1052848766285511E+00, 0.2419614642931688E+00, 0.1757806338135458E+00, 0.1955698951896722E-01, 0.7935225143382671E+00, 0.9463134562095434E+00, 0.2023962522401480E-02, 0.2298969194226957E-01, 0.9500696544845319E-01, 0.6985999785188474E-01, 0.1129463646814219E-01, 0.2717093011896993E-01, 0.2819864744791555E-01, 0.3949327947122574E-01, 0.9786381594178541E-10, 0.4444831994804014E-08, 0.6208618935176568E-03, 0.6208619913814727E-03, 0.1700670967075909E-47, 0.2788519153273688E-22},
			tauOut:   0.0000000000000000,
			ttypeOut: -7,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2395955612114145E+01, 0.2466725997761544E+01, 0.7077038564739886E-01, 0.5393549207373363E-01, 0.1879940024111348E+01, 0.1842303011581631E+01, 0.1629847954401656E-01, 0.1170659574821484E-01, 0.1323258193751120E+01, 0.1541901897857258E+01, 0.2303502998543534E+00, 0.2097365510789915E+00, 0.1403919102181368E+01, 0.1217882280188454E+01, 0.2369972908607684E-01, 0.1466422291592108E-01, 0.7535654596375009E+00, 0.7390296178356669E+00, 0.1283811140869274E-03, 0.2869551596920546E-03, 0.1651865724265327E+01, 0.1961735651572868E+01, 0.3101568824672333E+00, 0.3344283443049241E-01, 0.2115252128860193E+00, 0.2685204315989660E+00, 0.9043805314343908E-01, 0.1461369444993732E+00, 0.4338965075606848E+00, 0.3881981788190908E+00, 0.1004386157577793E+00, 0.1801598202532681E+00, 0.6963229589638840E+00, 0.6903565688953941E+00, 0.1741934301847783E+00, 0.4507175802324546E-01, 0.1786266232314551E+00, 0.2536031297618502E+00, 0.1200482645536405E+00, 0.1351822623162481E+00, 0.2855738476449996E+00, 0.1565951964743188E+00, 0.6203611145567293E-02, 0.1741812487831326E-01, 0.4396785394072695E+00, 0.4886511254041676E+00, 0.6639071087521134E-01, 0.1227425859208231E+00, 0.9034140763752896E+00, 0.8563793913827676E+00, 0.7570790092830114E-01, 0.1641466407918003E+00, 0.1856765259836430E+01, 0.1908333071910365E+01, 0.2157144528657351E+00, 0.9663383852973971E-01, 0.8548780458707995E+00, 0.9144777845246447E+00, 0.1562335771835849E+00, 0.1798735100772441E-01, 0.1052848766285511E+00, 0.2630781594343725E+00, 0.1757806338135458E+00, 0.5302070335887964E+00, 0.7935225143382671E+00, 0.2653394432718723E+00, 0.2023962522401480E-02, 0.7246963929058098E-03, 0.9500696544845319E-01, 0.1055769055236896E+00, 0.1129463646814219E-01, 0.3016696409481782E-02, 0.2819864744791555E-01, 0.2518195113629758E-01, 0.9786381594178541E-10, 0.2412835834031154E-11, 0.6208618935176568E-03, 0.6208618911048210E-03, 0.1700670967075909E-47, 0.2869551596920546E-03, 0.7811996215926567E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			n0in:     20,
			dmin:     6.2086189110482101E-004,
			dmin1:    2.5181951038433764E-002,
			dmin2:    8.7297525620826724E-002,
			dn:       6.2086189110482101E-004,
			dn1:      2.5181951038433764E-002,
			dn2:      9.4282269055547374E-002,
			tau:      0.0000000000000000,
			ttype:    -7,
			g:        0.0000000000000000,
			zOut:     []float64{0.2395955612114145E+01, 0.2466725997761544E+01, 0.7077038564739886E-01, 0.5393549207373363E-01, 0.1879940024111348E+01, 0.1842303011581631E+01, 0.1629847954401656E-01, 0.1170659574821484E-01, 0.1323258193751120E+01, 0.1541901897857258E+01, 0.2303502998543534E+00, 0.2097365510789915E+00, 0.1403919102181368E+01, 0.1217882280188454E+01, 0.2369972908607684E-01, 0.1466422291592108E-01, 0.7535654596375009E+00, 0.7390296178356669E+00, 0.1283811140869274E-03, 0.2869551596920546E-03, 0.1651865724265327E+01, 0.1961735651572868E+01, 0.3101568824672333E+00, 0.3344283443049241E-01, 0.2115252128860193E+00, 0.2685204315989660E+00, 0.9043805314343908E-01, 0.1461369444993732E+00, 0.4338965075606848E+00, 0.3881981788190908E+00, 0.1004386157577793E+00, 0.1801598202532681E+00, 0.6963229589638840E+00, 0.6903565688953941E+00, 0.1741934301847783E+00, 0.4507175802324546E-01, 0.1786266232314551E+00, 0.2536031297618502E+00, 0.1200482645536405E+00, 0.1351822623162481E+00, 0.2855738476449996E+00, 0.1565951964743188E+00, 0.6203611145567293E-02, 0.1741812487831326E-01, 0.4396785394072695E+00, 0.4886511254041676E+00, 0.6639071087521134E-01, 0.1227425859208231E+00, 0.9034140763752896E+00, 0.8563793913827676E+00, 0.7570790092830114E-01, 0.1641466407918003E+00, 0.1856765259836430E+01, 0.1908333071910365E+01, 0.2157144528657351E+00, 0.9663383852973971E-01, 0.8548780458707995E+00, 0.9144777845246447E+00, 0.1562335771835849E+00, 0.1798735100772441E-01, 0.1052848766285511E+00, 0.2630781594343725E+00, 0.1757806338135458E+00, 0.5302070335887964E+00, 0.7935225143382671E+00, 0.2653394432718723E+00, 0.2023962522401480E-02, 0.7246963929058098E-03, 0.9500696544845319E-01, 0.1055769055236896E+00, 0.1129463646814219E-01, 0.3016696409481782E-02, 0.2819864744791555E-01, 0.2518195113629758E-01, 0.9786381594178541E-10, 0.2412835834031154E-11, 0.6208618935176568E-03, 0.6208618911048210E-03, 0.1700670967075909E-47, 0.2869551596920546E-03, 0.7811996215926567E-04},
			tauOut:   6.2086189103875732E-004,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2520040627944239E+01, 0.2466725997761544E+01, 0.3943012599746679E-01, 0.5393549207373363E-01, 0.1813958619441340E+01, 0.1842303011581631E+01, 0.9950845630193767E-02, 0.1170659574821484E-01, 0.1741066741415017E+01, 0.1541901897857258E+01, 0.1467114516582776E+00, 0.2097365510789915E+00, 0.1085214189555058E+01, 0.1217882280188454E+01, 0.9986318978978259E-02, 0.1466422291592108E-01, 0.7287093921253419E+00, 0.7390296178356669E+00, 0.7725029665513934E-03, 0.2869551596920546E-03, 0.1993785121145770E+01, 0.1961735651572868E+01, 0.4504038193447841E-02, 0.3344283443049241E-01, 0.4095324760138526E+00, 0.2685204315989660E+00, 0.1385240464077977E+00, 0.1461369444993732E+00, 0.4292130907735224E+00, 0.3881981788190908E+00, 0.2897733504323247E+00, 0.1801598202532681E+00, 0.4450341145952761E+00, 0.6903565688953941E+00, 0.2568418582687495E-01, 0.4507175802324546E-01, 0.3624803443601846E+00, 0.2536031297618502E+00, 0.5840011260368079E-01, 0.1351822623162481E+00, 0.1149923468579125E+00, 0.1565951964743188E+00, 0.7401698075381481E-01, 0.1741812487831326E-01, 0.5367558686801371E+00, 0.4886511254041676E+00, 0.1958324578473516E+00, 0.1227425859208231E+00, 0.8240727124361776E+00, 0.8563793913827676E+00, 0.3801199318200257E+00, 0.1641466407918003E+00, 0.1624226116729040E+01, 0.1908333071910365E+01, 0.5440714051978934E-01, 0.9663383852973971E-01, 0.8774371331215411E+00, 0.9144777845246447E+00, 0.5393069221241477E-02, 0.1798735100772441E-01, 0.7872712619108886E+00, 0.2630781594343725E+00, 0.1786993199393658E+00, 0.5302070335887964E+00, 0.8674395783437358E-01, 0.2653394432718723E+00, 0.8820349511059155E-03, 0.7246963929058098E-03, 0.1070907050910267E+00, 0.1055769055236896E+00, 0.7093640994523618E-03, 0.3016696409481782E-02, 0.2385172514821930E-01, 0.2518195113629758E-01, 0.6280626703238275E-13, 0.2412835834031154E-11, 0.3257377007015450E-14, 0.6208618911048210E-03, 0.7725029665513934E-03, 0.2869551596920546E-03},
			i0:       1,
			n0:       20,
			pp:       0,
			n0in:     20,
			dmin:     3.2573770070154495E-015,
			dmin1:    2.3851725145806461E-002,
			dmin2:    8.6019261441467765E-002,
			dn:       3.2573770070154495E-015,
			dn1:      2.3851725145806461E-002,
			dn2:      0.10407400868154487,
			tau:      6.2086189103875732E-004,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2520040627944239E+01, 0.2466725997761544E+01, 0.3943012599746679E-01, 0.5393549207373363E-01, 0.1813958619441340E+01, 0.1842303011581631E+01, 0.9950845630193767E-02, 0.1170659574821484E-01, 0.1741066741415017E+01, 0.1541901897857258E+01, 0.1467114516582776E+00, 0.2097365510789915E+00, 0.1085214189555058E+01, 0.1217882280188454E+01, 0.9986318978978259E-02, 0.1466422291592108E-01, 0.7287093921253419E+00, 0.7390296178356669E+00, 0.7725029665513934E-03, 0.2869551596920546E-03, 0.1993785121145770E+01, 0.1961735651572868E+01, 0.4504038193447841E-02, 0.3344283443049241E-01, 0.4095324760138526E+00, 0.2685204315989660E+00, 0.1385240464077977E+00, 0.1461369444993732E+00, 0.4292130907735224E+00, 0.3881981788190908E+00, 0.2897733504323247E+00, 0.1801598202532681E+00, 0.4450341145952761E+00, 0.6903565688953941E+00, 0.2568418582687495E-01, 0.4507175802324546E-01, 0.3624803443601846E+00, 0.2536031297618502E+00, 0.5840011260368079E-01, 0.1351822623162481E+00, 0.1149923468579125E+00, 0.1565951964743188E+00, 0.7401698075381481E-01, 0.1741812487831326E-01, 0.5367558686801371E+00, 0.4886511254041676E+00, 0.1958324578473516E+00, 0.1227425859208231E+00, 0.8240727124361776E+00, 0.8563793913827676E+00, 0.3801199318200257E+00, 0.1641466407918003E+00, 0.1624226116729040E+01, 0.1908333071910365E+01, 0.5440714051978934E-01, 0.9663383852973971E-01, 0.8774371331215411E+00, 0.9144777845246447E+00, 0.5393069221241477E-02, 0.1798735100772441E-01, 0.7872712619108886E+00, 0.2630781594343725E+00, 0.1786993199393658E+00, 0.5302070335887964E+00, 0.8674395783437358E-01, 0.2653394432718723E+00, 0.8820349511059155E-03, 0.7246963929058098E-03, 0.1070907050910267E+00, 0.1055769055236896E+00, 0.7093640994523618E-03, 0.3016696409481782E-02, 0.2385172514821930E-01, 0.2518195113629758E-01, 0.6280626703238275E-13, 0.2412835834031154E-11, 0.3257377007015450E-14, 0.6208618911048210E-03, 0.7725029665513934E-03, 0.2869551596920546E-03},
			tauOut:   3.2573770070067200E-015,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2520040627944239E+01, 0.2559470753941703E+01, 0.3943012599746679E-01, 0.2794508075882943E-01, 0.1813958619441340E+01, 0.1795964384312701E+01, 0.9950845630193767E-02, 0.9646675917972324E-02, 0.1741066741415017E+01, 0.1878131517155319E+01, 0.1467114516582776E+00, 0.8477220453173254E-01, 0.1085214189555058E+01, 0.1010428304002300E+01, 0.9986318978978259E-02, 0.7202019583097946E-02, 0.7287093921253419E+00, 0.7222798755087921E+00, 0.7725029665513934E-03, 0.2132421202606784E-02, 0.1993785121145770E+01, 0.1996156738136608E+01, 0.4504038193447841E-02, 0.9240506410060376E-03, 0.4095324760138526E+00, 0.5471324717806409E+00, 0.1385240464077977E+00, 0.1086689918286978E+00, 0.4292130907735224E+00, 0.6103174493771462E+00, 0.2897733504323247E+00, 0.2112982785836522E+00, 0.4450341145952761E+00, 0.2594200218384955E+00, 0.2568418582687495E-01, 0.3588779484774172E-01, 0.3624803443601846E+00, 0.3849926621161204E+00, 0.5840011260368079E-01, 0.1744336104525046E-01, 0.1149923468579125E+00, 0.1715659665664736E+00, 0.7401698075381481E-01, 0.2315671901408358E+00, 0.5367558686801371E+00, 0.5010211363866497E+00, 0.1958324578473516E+00, 0.3221025481782661E+00, 0.8240727124361776E+00, 0.8820900960779340E+00, 0.3801199318200257E+00, 0.6999293195746292E+00, 0.1624226116729040E+01, 0.9787039376741966E+00, 0.5440714051978934E-01, 0.4877761656142095E-01, 0.8774371331215411E+00, 0.8340525857813585E+00, 0.5393069221241477E-02, 0.5090576402208483E-02, 0.7872712619108886E+00, 0.9608800054480426E+00, 0.1786993199393658E+00, 0.1613217694817542E-01, 0.8674395783437358E-01, 0.7149381583730083E-01, 0.8820349511059155E-03, 0.1321201613351008E-02, 0.1070907050910267E+00, 0.1064788675771248E+00, 0.7093640994523618E-03, 0.1589006148839502E-03, 0.2385172514821930E-01, 0.2369282453339490E-01, 0.6280626703238275E-13, 0.8634837515442557E-26, 0.3257377007015450E-14, 0.9466330862652142E-28, 0.7725029665513934E-03, 0.9240506410060376E-03, 0.7811996215926567E-04},
			i0:       1,
			n0:       20,
			pp:       1,
			n0in:     20,
			dmin:     9.4663308626521417E-029,
			dmin1:    2.3692824533332088E-002,
			dmin2:    7.0611780886194908E-002,
			dn:       9.4663308626521417E-029,
			dn1:      2.3692824533332088E-002,
			dn2:      0.10576950347767239,
			tau:      3.2573770070067200E-015,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2520040627944239E+01, 0.2559470753941703E+01, 0.3943012599746679E-01, 0.2794508075882943E-01, 0.1813958619441340E+01, 0.1795964384312701E+01, 0.9950845630193767E-02, 0.9646675917972324E-02, 0.1741066741415017E+01, 0.1878131517155319E+01, 0.1467114516582776E+00, 0.8477220453173254E-01, 0.1085214189555058E+01, 0.1010428304002300E+01, 0.9986318978978259E-02, 0.7202019583097946E-02, 0.7287093921253419E+00, 0.7222798755087921E+00, 0.7725029665513934E-03, 0.2132421202606784E-02, 0.1993785121145770E+01, 0.1996156738136608E+01, 0.4504038193447841E-02, 0.9240506410060376E-03, 0.4095324760138526E+00, 0.5471324717806409E+00, 0.1385240464077977E+00, 0.1086689918286978E+00, 0.4292130907735224E+00, 0.6103174493771462E+00, 0.2897733504323247E+00, 0.2112982785836522E+00, 0.4450341145952761E+00, 0.2594200218384955E+00, 0.2568418582687495E-01, 0.3588779484774172E-01, 0.3624803443601846E+00, 0.3849926621161204E+00, 0.5840011260368079E-01, 0.1744336104525046E-01, 0.1149923468579125E+00, 0.1715659665664736E+00, 0.7401698075381481E-01, 0.2315671901408358E+00, 0.5367558686801371E+00, 0.5010211363866497E+00, 0.1958324578473516E+00, 0.3221025481782661E+00, 0.8240727124361776E+00, 0.8820900960779340E+00, 0.3801199318200257E+00, 0.6999293195746292E+00, 0.1624226116729040E+01, 0.9787039376741966E+00, 0.5440714051978934E-01, 0.4877761656142095E-01, 0.8774371331215411E+00, 0.8340525857813585E+00, 0.5393069221241477E-02, 0.5090576402208483E-02, 0.7872712619108886E+00, 0.9608800054480426E+00, 0.1786993199393658E+00, 0.1613217694817542E-01, 0.8674395783437358E-01, 0.7149381583730083E-01, 0.8820349511059155E-03, 0.1321201613351008E-02, 0.1070907050910267E+00, 0.1064788675771248E+00, 0.7093640994523618E-03, 0.1589006148839502E-03, 0.2385172514821930E-01, 0.2369282453339490E-01, 0.6280626703238275E-13, 0.8634837515442557E-26, 0.3257377007015450E-14, 0.9466330862652142E-28, 0.7725029665513934E-03, 0.9240506410060376E-03, 0.7811996215926567E-04},
			tauOut:   9.4663308626521417E-029,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2587415834700532E+01, 0.2559470753941703E+01, 0.1939710234687058E-01, 0.2794508075882943E-01, 0.1786213957883803E+01, 0.1795964384312701E+01, 0.1014308840067055E-01, 0.9646675917972324E-02, 0.1952760633286381E+01, 0.1878131517155319E+01, 0.4386417535844126E-01, 0.8477220453173254E-01, 0.9737661482269571E+00, 0.1010428304002300E+01, 0.5342015449359675E-02, 0.7202019583097946E-02, 0.7190702812620393E+00, 0.7222798755087921E+00, 0.5919653562455767E-02, 0.2132421202606784E-02, 0.1991161135215158E+01, 0.1996156738136608E+01, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.6555475524103290E+00, 0.5471324717806409E+00, 0.1011712753337584E+00, 0.1086689918286978E+00, 0.7204444526270400E+00, 0.6103174493771462E+00, 0.7608498315828412E-01, 0.2112982785836522E+00, 0.2192228335279531E+00, 0.2594200218384955E+00, 0.6302508481237896E-01, 0.3588779484774172E-01, 0.3394109383489919E+00, 0.3849926621161204E+00, 0.8817297145618809E-02, 0.1744336104525046E-01, 0.3943158595616905E+00, 0.1715659665664736E+00, 0.2942312715577539E+00, 0.2315671901408358E+00, 0.5288924130071619E+00, 0.5010211363866497E+00, 0.5372046576619481E+00, 0.3221025481782661E+00, 0.1044814757990615E+01, 0.8820900960779340E+00, 0.6556411803358774E+00, 0.6999293195746292E+00, 0.3718403738997403E+00, 0.9787039376741966E+00, 0.1094101127175453E+00, 0.4877761656142095E-01, 0.7297330494660218E+00, 0.8340525857813585E+00, 0.6703044469024726E-02, 0.5090576402208483E-02, 0.9703091379271934E+00, 0.9608800054480426E+00, 0.1188642714683105E-02, 0.1613217694817542E-01, 0.7162637473596872E-01, 0.7149381583730083E-01, 0.1964081696850700E-02, 0.1321201613351008E-02, 0.1046736864951580E+00, 0.1064788675771248E+00, 0.3596705640885345E-04, 0.1589006148839502E-03, 0.2365685747698604E-01, 0.2369282453339490E-01, 0.3455244592226135E-52, 0.8634837515442557E-26},
			i0:       1,
			n0:       19,
			pp:       0,
			n0in:     20,
			dmin:     9.4663308626521417E-029,
			dmin1:    2.3656857476986041E-002,
			dmin2:    7.0305173122617720E-002,
			dn:       9.4663308626521417E-029,
			dn1:      2.3656857476986041E-002,
			dn2:      0.10451478588027406,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2587415834700532E+01, 0.2559470753941703E+01, 0.1939710234687058E-01, 0.2794508075882943E-01, 0.1786213957883803E+01, 0.1795964384312701E+01, 0.1014308840067055E-01, 0.9646675917972324E-02, 0.1952760633286381E+01, 0.1878131517155319E+01, 0.4386417535844126E-01, 0.8477220453173254E-01, 0.9737661482269571E+00, 0.1010428304002300E+01, 0.5342015449359675E-02, 0.7202019583097946E-02, 0.7190702812620393E+00, 0.7222798755087921E+00, 0.5919653562455767E-02, 0.2132421202606784E-02, 0.1991161135215158E+01, 0.1996156738136608E+01, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.6555475524103290E+00, 0.5471324717806409E+00, 0.1011712753337584E+00, 0.1086689918286978E+00, 0.7204444526270400E+00, 0.6103174493771462E+00, 0.7608498315828412E-01, 0.2112982785836522E+00, 0.2192228335279531E+00, 0.2594200218384955E+00, 0.6302508481237896E-01, 0.3588779484774172E-01, 0.3394109383489919E+00, 0.3849926621161204E+00, 0.8817297145618809E-02, 0.1744336104525046E-01, 0.3943158595616905E+00, 0.1715659665664736E+00, 0.2942312715577539E+00, 0.2315671901408358E+00, 0.5288924130071619E+00, 0.5010211363866497E+00, 0.5372046576619481E+00, 0.3221025481782661E+00, 0.1044814757990615E+01, 0.8820900960779340E+00, 0.6556411803358774E+00, 0.6999293195746292E+00, 0.3718403738997403E+00, 0.9787039376741966E+00, 0.1094101127175453E+00, 0.4877761656142095E-01, 0.7297330494660218E+00, 0.8340525857813585E+00, 0.6703044469024726E-02, 0.5090576402208483E-02, 0.9703091379271934E+00, 0.9608800054480426E+00, 0.1188642714683105E-02, 0.1613217694817542E-01, 0.7162637473596872E-01, 0.7149381583730083E-01, 0.1964081696850700E-02, 0.1321201613351008E-02, 0.1046736864951580E+00, 0.1064788675771248E+00, 0.3596705640885345E-04, 0.1589006148839502E-03, 0.2365685747698604E-01, 0.2369282453339490E-01, 0.3455244592226135E-52, 0.8634837515442557E-26},
			tauOut:   1.1828428738493020E-002,
			ttypeOut: -9,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2587415834700532E+01, 0.2594984508308910E+01, 0.1939710234687058E-01, 0.1335166928493912E-01, 0.1786213957883803E+01, 0.1771176948261041E+01, 0.1014308840067055E-01, 0.1118297285215907E-01, 0.1952760633286381E+01, 0.1973613407054170E+01, 0.4386417535844126E-01, 0.2164225725832272E-01, 0.9737661482269571E+00, 0.9456374776795010E+00, 0.5342015449359675E-02, 0.4062111160297227E-02, 0.7190702812620393E+00, 0.7090993949257048E+00, 0.5919653562455767E-02, 0.1662247097070905E-01, 0.1991161135215158E+01, 0.1962964146704966E+01, 0.2539111990097616E-03, 0.8479567256479229E-04, 0.6555475524103290E+00, 0.7448056033330296E+00, 0.1011712753337584E+00, 0.9786215859981671E-01, 0.7204444526270400E+00, 0.6868388484470145E+00, 0.7608498315828412E-01, 0.2428454015756268E-01, 0.2192228335279531E+00, 0.2461349494442763E+00, 0.6302508481237896E-01, 0.8690924724014965E-01, 0.3394109383489919E+00, 0.2494905595159681E+00, 0.8817297145618809E-02, 0.1393559784278329E-01, 0.3943158595616905E+00, 0.6627831045381681E+00, 0.2942312715577539E+00, 0.2347927793131975E+00, 0.5288924130071619E+00, 0.8194758626174196E+00, 0.5372046576619481E+00, 0.6849248159595133E+00, 0.1044814757990615E+01, 0.1003702693628486E+01, 0.6556411803358774E+00, 0.2428944977310168E+00, 0.3718403738997403E+00, 0.2265275601477758E+00, 0.1094101127175453E+00, 0.3524523689025369E+00, 0.7297330494660218E+00, 0.3721552962940165E+00, 0.6703044469024726E-02, 0.1747664312451059E-01, 0.9703091379271934E+00, 0.9421927087788731E+00, 0.1188642714683105E-02, 0.9036173567869616E-04, 0.7162637473596872E-01, 0.6167166595864771E-01, 0.1964081696850700E-02, 0.3333583884775863E-02, 0.1046736864951580E+00, 0.8954764092829798E-01, 0.3596705640885345E-04, 0.9501841907954501E-05, 0.2365685747698604E-01, 0.1181892689658507E-01, 0.3455244592226135E-52, 0.8479567256479229E-04, 0.6989818532012803E-03},
			i0:       1,
			n0:       19,
			pp:       1,
			n0in:     19,
			dmin:     1.1818926896585069E-002,
			dmin1:    5.9707584261797009E-002,
			dmin2:    5.9707584261797009E-002,
			dn:       1.1818926896585069E-002,
			dn1:      8.9511673871889130E-002,
			dn2:      5.9707584261797009E-002,
			tau:      1.1828428738493020E-002,
			ttype:    -9,
			g:        0.0000000000000000,
			zOut:     []float64{0.2587415834700532E+01, 0.2594984508308910E+01, 0.1939710234687058E-01, 0.1335166928493912E-01, 0.1786213957883803E+01, 0.1771176948261041E+01, 0.1014308840067055E-01, 0.1118297285215907E-01, 0.1952760633286381E+01, 0.1973613407054170E+01, 0.4386417535844126E-01, 0.2164225725832272E-01, 0.9737661482269571E+00, 0.9456374776795010E+00, 0.5342015449359675E-02, 0.4062111160297227E-02, 0.7190702812620393E+00, 0.7090993949257048E+00, 0.5919653562455767E-02, 0.1662247097070905E-01, 0.1991161135215158E+01, 0.1962964146704966E+01, 0.2539111990097616E-03, 0.8479567256479229E-04, 0.6555475524103290E+00, 0.7448056033330296E+00, 0.1011712753337584E+00, 0.9786215859981671E-01, 0.7204444526270400E+00, 0.6868388484470145E+00, 0.7608498315828412E-01, 0.2428454015756268E-01, 0.2192228335279531E+00, 0.2461349494442763E+00, 0.6302508481237896E-01, 0.8690924724014965E-01, 0.3394109383489919E+00, 0.2494905595159681E+00, 0.8817297145618809E-02, 0.1393559784278329E-01, 0.3943158595616905E+00, 0.6627831045381681E+00, 0.2942312715577539E+00, 0.2347927793131975E+00, 0.5288924130071619E+00, 0.8194758626174196E+00, 0.5372046576619481E+00, 0.6849248159595133E+00, 0.1044814757990615E+01, 0.1003702693628486E+01, 0.6556411803358774E+00, 0.2428944977310168E+00, 0.3718403738997403E+00, 0.2265275601477758E+00, 0.1094101127175453E+00, 0.3524523689025369E+00, 0.7297330494660218E+00, 0.3721552962940165E+00, 0.6703044469024726E-02, 0.1747664312451059E-01, 0.9703091379271934E+00, 0.9421927087788731E+00, 0.1188642714683105E-02, 0.9036173567869616E-04, 0.7162637473596872E-01, 0.6167166595864771E-01, 0.1964081696850700E-02, 0.3333583884775863E-02, 0.1046736864951580E+00, 0.8954764092829798E-01, 0.3596705640885345E-04, 0.9501841907954501E-05, 0.2365685747698604E-01, 0.1181892689658507E-01, 0.3455244592226135E-52, 0.8479567256479229E-04, 0.6989818532012803E-03},
			tauOut:   1.1689473904977585E-002,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2596646703688871E+01, 0.2594984508308910E+01, 0.9107195378059658E-02, 0.1335166928493912E-01, 0.1761563251830163E+01, 0.1771176948261041E+01, 0.1252913577120413E-01, 0.1118297285215907E-01, 0.1971037054636311E+01, 0.1973613407054170E+01, 0.1038322923301279E-01, 0.2164225725832272E-01, 0.9276268857018078E+00, 0.9456374776795010E+00, 0.3105171497598932E-02, 0.4062111160297227E-02, 0.7109272204938374E+00, 0.7090993949257048E+00, 0.4589684232723624E-01, 0.1662247097070905E-01, 0.1905462626145317E+01, 0.1962964146704966E+01, 0.3314486004504485E-04, 0.8479567256479229E-04, 0.8309451431678238E+00, 0.7448056033330296E+00, 0.8089045693556918E-01, 0.9786215859981671E-01, 0.6185434577640304E+00, 0.6868388484470145E+00, 0.9663466631053550E-02, 0.2428454015756268E-01, 0.3116912561483949E+00, 0.2461349494442763E+00, 0.6956575230565126E-01, 0.8690924724014965E-01, 0.1821709311481225E+00, 0.2494905595159681E+00, 0.5070116699532772E-01, 0.1393559784278329E-01, 0.8351852429510604E+00, 0.6627831045381681E+00, 0.2303764547900405E+00, 0.2347927793131975E+00, 0.1262334749881915E+01, 0.8194758626174196E+00, 0.5445947541061251E+00, 0.6849248159595133E+00, 0.6903129633483998E+00, 0.1003702693628486E+01, 0.7970630839299049E-01, 0.2428944977310168E+00, 0.4875841467523446E+00, 0.2265275601477758E+00, 0.2690141110044630E+00, 0.3524523689025369E+00, 0.1089283545090865E+00, 0.3721552962940165E+00, 0.1511669372043137E+00, 0.1747664312451059E-01, 0.7794266594052606E+00, 0.9421927087788731E+00, 0.7149818024536710E-05, 0.9036173567869616E-04, 0.5330862612042146E-01, 0.6167166595864771E-01, 0.5599742376476567E-02, 0.3333583884775863E-02, 0.7226792648875177E-01, 0.8954764092829798E-01, 0.1553961492315709E-05, 0.9501841907954501E-05, 0.1278990301151681E-03, 0.1181892689658507E-01, 0.7149818024536710E-05, 0.8479567256479229E-04},
			i0:       1,
			n0:       19,
			pp:       0,
			n0in:     19,
			dmin:     1.2789903011516807E-004,
			dmin1:    4.9975042235645591E-002,
			dmin2:    4.9975042235645591E-002,
			dn:       1.2789903011516807E-004,
			dn1:      7.2258424646843816E-002,
			dn2:      4.9975042235645591E-002,
			tau:      1.1689473904977585E-002,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2596646703688871E+01, 0.2594984508308910E+01, 0.9107195378059658E-02, 0.1335166928493912E-01, 0.1761563251830163E+01, 0.1771176948261041E+01, 0.1252913577120413E-01, 0.1118297285215907E-01, 0.1971037054636311E+01, 0.1973613407054170E+01, 0.1038322923301279E-01, 0.2164225725832272E-01, 0.9276268857018078E+00, 0.9456374776795010E+00, 0.3105171497598932E-02, 0.4062111160297227E-02, 0.7109272204938374E+00, 0.7090993949257048E+00, 0.4589684232723624E-01, 0.1662247097070905E-01, 0.1905462626145317E+01, 0.1962964146704966E+01, 0.3314486004504485E-04, 0.8479567256479229E-04, 0.8309451431678238E+00, 0.7448056033330296E+00, 0.8089045693556918E-01, 0.9786215859981671E-01, 0.6185434577640304E+00, 0.6868388484470145E+00, 0.9663466631053550E-02, 0.2428454015756268E-01, 0.3116912561483949E+00, 0.2461349494442763E+00, 0.6956575230565126E-01, 0.8690924724014965E-01, 0.1821709311481225E+00, 0.2494905595159681E+00, 0.5070116699532772E-01, 0.1393559784278329E-01, 0.8351852429510604E+00, 0.6627831045381681E+00, 0.2303764547900405E+00, 0.2347927793131975E+00, 0.1262334749881915E+01, 0.8194758626174196E+00, 0.5445947541061251E+00, 0.6849248159595133E+00, 0.6903129633483998E+00, 0.1003702693628486E+01, 0.7970630839299049E-01, 0.2428944977310168E+00, 0.4875841467523446E+00, 0.2265275601477758E+00, 0.2690141110044630E+00, 0.3524523689025369E+00, 0.1089283545090865E+00, 0.3721552962940165E+00, 0.1511669372043137E+00, 0.1747664312451059E-01, 0.7794266594052606E+00, 0.9421927087788731E+00, 0.7149818024536710E-05, 0.9036173567869616E-04, 0.5330862612042146E-01, 0.6167166595864771E-01, 0.5599742376476567E-02, 0.3333583884775863E-02, 0.7226792648875177E-01, 0.8954764092829798E-01, 0.1553961492315709E-05, 0.9501841907954501E-05, 0.1278990301151681E-03, 0.1181892689658507E-01, 0.7149818024536710E-05, 0.8479567256479229E-04},
			tauOut:   1.1689473904977585E-002,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2596646703688871E+01, 0.2605626003858251E+01, 0.9107195378059658E-02, 0.6157023564192275E-02, 0.1761563251830163E+01, 0.1767807468828494E+01, 0.1252913577120413E-01, 0.1396950250695455E-01, 0.1971037054636311E+01, 0.1967322886153689E+01, 0.1038322923301279E-01, 0.4895872794820515E-02, 0.9276268857018078E+00, 0.9257082891959054E+00, 0.3105171497598932E-02, 0.2384715539127593E-02, 0.7109272204938374E+00, 0.7543114520732653E+00, 0.4589684232723624E-01, 0.1159397984377132E+00, 0.1905462626145317E+01, 0.1789428077358968E+01, 0.3314486004504485E-04, 0.1539126429493388E-04, 0.8309451431678238E+00, 0.9116923136304173E+00, 0.8089045693556918E-01, 0.5488064578914756E-01, 0.6185434577640304E+00, 0.5731983833972557E+00, 0.9663466631053550E-02, 0.5254756712901782E-02, 0.3116912561483949E+00, 0.3758743565324636E+00, 0.6956575230565126E-01, 0.3371567560620648E-01, 0.1821709311481225E+00, 0.1990285273285630E+00, 0.5070116699532772E-01, 0.2127577741907859E+00, 0.8351852429510604E+00, 0.8526760283416343E+00, 0.2303764547900405E+00, 0.3410582621885915E+00, 0.1262334749881915E+01, 0.1465743346590768E+01, 0.5445947541061251E+00, 0.2564847518533230E+00, 0.6903129633483998E+00, 0.5134066246793865E+00, 0.7970630839299049E-01, 0.7569737222001199E-01, 0.4875841467523446E+00, 0.6807729903281149E+00, 0.2690141110044630E+00, 0.4304410555024735E-01, 0.1089283545090865E+00, 0.2169232909544721E+00, 0.1511669372043137E+00, 0.5431576312495270E+00, 0.7794266594052606E+00, 0.2361482827650774E+00, 0.7149818024536710E-05, 0.1614015445872399E-05, 0.5330862612042146E-01, 0.5877885927277143E-01, 0.5599742376476567E-02, 0.6884818375619963E-02, 0.7226792648875177E-01, 0.6525676686594341E-01, 0.1553961492315709E-05, 0.3045663725752605E-08, 0.1278990301151681E-03, 0.7757707209639971E-09, 0.7149818024536710E-05, 0.1614015445872399E-05, 0.6989818532012803E-03},
			i0:       1,
			n0:       19,
			pp:       1,
			n0in:     19,
			dmin:     7.7577072096399712E-010,
			dmin1:    5.3179116896294863E-002,
			dmin2:    5.3179116896294863E-002,
			dn:       7.7577072096399712E-010,
			dn1:      6.5255212904451090E-002,
			dn2:      5.3179116896294863E-002,
			tau:      1.2789520868072135E-004,
			ttype:    -15,
			g:        0.0000000000000000,
			zOut:     []float64{0.2596646703688871E+01, 0.2605626003858251E+01, 0.9107195378059658E-02, 0.6157023564192275E-02, 0.1761563251830163E+01, 0.1767807468828494E+01, 0.1252913577120413E-01, 0.1396950250695455E-01, 0.1971037054636311E+01, 0.1967322886153689E+01, 0.1038322923301279E-01, 0.4895872794820515E-02, 0.9276268857018078E+00, 0.9257082891959054E+00, 0.3105171497598932E-02, 0.2384715539127593E-02, 0.7109272204938374E+00, 0.7543114520732653E+00, 0.4589684232723624E-01, 0.1159397984377132E+00, 0.1905462626145317E+01, 0.1789428077358968E+01, 0.3314486004504485E-04, 0.1539126429493388E-04, 0.8309451431678238E+00, 0.9116923136304173E+00, 0.8089045693556918E-01, 0.5488064578914756E-01, 0.6185434577640304E+00, 0.5731983833972557E+00, 0.9663466631053550E-02, 0.5254756712901782E-02, 0.3116912561483949E+00, 0.3758743565324636E+00, 0.6956575230565126E-01, 0.3371567560620648E-01, 0.1821709311481225E+00, 0.1990285273285630E+00, 0.5070116699532772E-01, 0.2127577741907859E+00, 0.8351852429510604E+00, 0.8526760283416343E+00, 0.2303764547900405E+00, 0.3410582621885915E+00, 0.1262334749881915E+01, 0.1465743346590768E+01, 0.5445947541061251E+00, 0.2564847518533230E+00, 0.6903129633483998E+00, 0.5134066246793865E+00, 0.7970630839299049E-01, 0.7569737222001199E-01, 0.4875841467523446E+00, 0.6807729903281149E+00, 0.2690141110044630E+00, 0.4304410555024735E-01, 0.1089283545090865E+00, 0.2169232909544721E+00, 0.1511669372043137E+00, 0.5431576312495270E+00, 0.7794266594052606E+00, 0.2361482827650774E+00, 0.7149818024536710E-05, 0.1614015445872399E-05, 0.5330862612042146E-01, 0.5877885927277143E-01, 0.5599742376476567E-02, 0.6884818375619963E-02, 0.7226792648875177E-01, 0.6525676686594341E-01, 0.1553961492315709E-05, 0.3045663725752605E-08, 0.1278990301151681E-03, 0.7757707209639971E-09, 0.7149818024536710E-05, 0.1614015445872399E-05, 0.6989818532012803E-03},
			tauOut:   1.2789520868072135E-004,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2611783026646672E+01, 0.2605626003858251E+01, 0.4167433562238479E-02, 0.6157023564192275E-02, 0.1777609536997439E+01, 0.1767807468828494E+01, 0.1546038171944878E-01, 0.1396950250695455E-01, 0.1956758376453290E+01, 0.1967322886153689E+01, 0.2316152103168096E-02, 0.4895872794820515E-02, 0.9257768518560943E+00, 0.9257082891959054E+00, 0.1943036529261404E-02, 0.2384715539127593E-02, 0.8683082132059464E+00, 0.7543114520732653E+00, 0.2389312083572061E+00, 0.1159397984377132E+00, 0.1550512259490286E+01, 0.1789428077358968E+01, 0.9049975109102588E-05, 0.1539126429493388E-04, 0.9665639086686850E+00, 0.9116923136304173E+00, 0.3254569838994440E-01, 0.5488064578914756E-01, 0.5459074409444424E+00, 0.5731983833972557E+00, 0.3618064437406363E-02, 0.5254756712901782E-02, 0.4059719669254931E+00, 0.3758743565324636E+00, 0.1652917395900484E-01, 0.3371567560620648E-01, 0.3952571267845734E+00, 0.1990285273285630E+00, 0.4589757947481138E+00, 0.2127577741907859E+00, 0.7347584950063413E+00, 0.8526760283416343E+00, 0.6803648845168142E+00, 0.3410582621885915E+00, 0.1041863213151506E+01, 0.1465743346590768E+01, 0.1263898840735783E+00, 0.2564847518533230E+00, 0.4627141120500496E+00, 0.5134066246793865E+00, 0.1113705527974558E+00, 0.7569737222001199E-01, 0.6124465423051357E+00, 0.6807729903281149E+00, 0.1524585149425051E-01, 0.4304410555024735E-01, 0.7448350699339780E+00, 0.2169232909544721E+00, 0.1722069046798406E+00, 0.5431576312495270E+00, 0.6394299132491200E-01, 0.2361482827650774E+00, 0.1483665133446019E-05, 0.1614015445872399E-05, 0.6566219320748727E-01, 0.5877885927277143E-01, 0.6842308575232983E-02, 0.6884818375619963E-02, 0.5841446056060346E-01, 0.6525676686594341E-01, 0.4044780558898965E-16, 0.3045663725752605E-08, 0.1013559914197710E-18, 0.7757707209639971E-09, 0.1483665133446019E-05, 0.1614015445872399E-05},
			i0:       1,
			n0:       19,
			pp:       0,
			n0in:     19,
			dmin:     1.0135599141977102E-019,
			dmin1:    5.8414457514939733E-002,
			dmin2:    5.8777374831867304E-002,
			dn:       1.0135599141977102E-019,
			dn1:      5.8414457514939733E-002,
			dn2:      5.8777374831867304E-002,
			tau:      7.7577068041483555E-010,
			ttype:    -15,
			g:        0.0000000000000000,
			zOut:     []float64{0.2611783026646672E+01, 0.2605626003858251E+01, 0.4167433562238479E-02, 0.6157023564192275E-02, 0.1777609536997439E+01, 0.1767807468828494E+01, 0.1546038171944878E-01, 0.1396950250695455E-01, 0.1956758376453290E+01, 0.1967322886153689E+01, 0.2316152103168096E-02, 0.4895872794820515E-02, 0.9257768518560943E+00, 0.9257082891959054E+00, 0.1943036529261404E-02, 0.2384715539127593E-02, 0.8683082132059464E+00, 0.7543114520732653E+00, 0.2389312083572061E+00, 0.1159397984377132E+00, 0.1550512259490286E+01, 0.1789428077358968E+01, 0.9049975109102588E-05, 0.1539126429493388E-04, 0.9665639086686850E+00, 0.9116923136304173E+00, 0.3254569838994440E-01, 0.5488064578914756E-01, 0.5459074409444424E+00, 0.5731983833972557E+00, 0.3618064437406363E-02, 0.5254756712901782E-02, 0.4059719669254931E+00, 0.3758743565324636E+00, 0.1652917395900484E-01, 0.3371567560620648E-01, 0.3952571267845734E+00, 0.1990285273285630E+00, 0.4589757947481138E+00, 0.2127577741907859E+00, 0.7347584950063413E+00, 0.8526760283416343E+00, 0.6803648845168142E+00, 0.3410582621885915E+00, 0.1041863213151506E+01, 0.1465743346590768E+01, 0.1263898840735783E+00, 0.2564847518533230E+00, 0.4627141120500496E+00, 0.5134066246793865E+00, 0.1113705527974558E+00, 0.7569737222001199E-01, 0.6124465423051357E+00, 0.6807729903281149E+00, 0.1524585149425051E-01, 0.4304410555024735E-01, 0.7448350699339780E+00, 0.2169232909544721E+00, 0.1722069046798406E+00, 0.5431576312495270E+00, 0.6394299132491200E-01, 0.2361482827650774E+00, 0.1483665133446019E-05, 0.1614015445872399E-05, 0.6566219320748727E-01, 0.5877885927277143E-01, 0.6842308575232983E-02, 0.6884818375619963E-02, 0.5841446056060346E-01, 0.6525676686594341E-01, 0.4044780558898965E-16, 0.3045663725752605E-08, 0.1013559914197710E-18, 0.7757707209639971E-09, 0.1483665133446019E-05, 0.1614015445872399E-05},
			tauOut:   1.0135599141977091E-019,
			ttypeOut: -2,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2611783026646672E+01, 0.2615950460208911E+01, 0.4167433562238479E-02, 0.2831884532112553E-02, 0.1777609536997439E+01, 0.1790238034184775E+01, 0.1546038171944878E-01, 0.1689844079671380E-01, 0.1956758376453290E+01, 0.1942176087759744E+01, 0.2316152103168096E-02, 0.1104039956008399E-02, 0.9257768518560943E+00, 0.9266158484293474E+00, 0.1943036529261404E-02, 0.1820770257466081E-02, 0.8683082132059464E+00, 0.1105418651305687E+01, 0.2389312083572061E+00, 0.3351361651941490E+00, 0.1550512259490286E+01, 0.1215385144271246E+01, 0.9049975109102588E-05, 0.7197207696703830E-05, 0.9665639086686850E+00, 0.9991024098509327E+00, 0.3254569838994440E-01, 0.1778290067827487E-01, 0.5459074409444424E+00, 0.5317426047035739E+00, 0.3618064437406363E-02, 0.2762300261676148E-02, 0.4059719669254931E+00, 0.4197388406228219E+00, 0.1652917395900484E-01, 0.1556509232613395E-01, 0.3952571267845734E+00, 0.8386678292065533E+00, 0.4589757947481138E+00, 0.4021095747913879E+00, 0.7347584950063413E+00, 0.1013013804731768E+01, 0.6803648845168142E+00, 0.6997408538631263E+00, 0.1041863213151506E+01, 0.4685122433619579E+00, 0.1263898840735783E+00, 0.1248257304047288E+00, 0.4627141120500496E+00, 0.4492589344427766E+00, 0.1113705527974558E+00, 0.1518244930621437E+00, 0.6124465423051357E+00, 0.4758679007372426E+00, 0.1524585149425051E-01, 0.2386301922514691E-01, 0.7448350699339780E+00, 0.8931789553886716E+00, 0.1722069046798406E+00, 0.1232835205710967E-01, 0.6394299132491200E-01, 0.5161612293293578E-01, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6566219320748727E-01, 0.7250261437426894E-01, 0.6842308575232983E-02, 0.5512763475647510E-02, 0.5841446056060346E-01, 0.5290169708495600E-01, 0.4044780558898965E-16, 0.7749519698096867E-34, 0.2434478048112329E-01},
			i0:       1,
			n0:       18,
			pp:       1,
			n0in:     19,
			dmin:     1.0135599141977093E-019,
			dmin1:    5.1614639267802333E-002,
			dmin2:    5.1614639267802333E-002,
			dn:       1.0135599141977093E-019,
			dn1:      5.2901697084955956E-002,
			dn2:      6.5660305799035965E-002,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.0000000000000000,
			zOut:     []float64{0.2611783026646672E+01, 0.2615950460208911E+01, 0.4167433562238479E-02, 0.2831884532112553E-02, 0.1777609536997439E+01, 0.1790238034184775E+01, 0.1546038171944878E-01, 0.1689844079671380E-01, 0.1956758376453290E+01, 0.1942176087759744E+01, 0.2316152103168096E-02, 0.1104039956008399E-02, 0.9257768518560943E+00, 0.9266158484293474E+00, 0.1943036529261404E-02, 0.1820770257466081E-02, 0.8683082132059464E+00, 0.1105418651305687E+01, 0.2389312083572061E+00, 0.3351361651941490E+00, 0.1550512259490286E+01, 0.1215385144271246E+01, 0.9049975109102588E-05, 0.7197207696703830E-05, 0.9665639086686850E+00, 0.9991024098509327E+00, 0.3254569838994440E-01, 0.1778290067827487E-01, 0.5459074409444424E+00, 0.5317426047035739E+00, 0.3618064437406363E-02, 0.2762300261676148E-02, 0.4059719669254931E+00, 0.4197388406228219E+00, 0.1652917395900484E-01, 0.1556509232613395E-01, 0.3952571267845734E+00, 0.8386678292065533E+00, 0.4589757947481138E+00, 0.4021095747913879E+00, 0.7347584950063413E+00, 0.1013013804731768E+01, 0.6803648845168142E+00, 0.6997408538631263E+00, 0.1041863213151506E+01, 0.4685122433619579E+00, 0.1263898840735783E+00, 0.1248257304047288E+00, 0.4627141120500496E+00, 0.4492589344427766E+00, 0.1113705527974558E+00, 0.1518244930621437E+00, 0.6124465423051357E+00, 0.4758679007372426E+00, 0.1524585149425051E-01, 0.2386301922514691E-01, 0.7448350699339780E+00, 0.8931789553886716E+00, 0.1722069046798406E+00, 0.1232835205710967E-01, 0.6394299132491200E-01, 0.5161612293293578E-01, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6566219320748727E-01, 0.7250261437426894E-01, 0.6842308575232983E-02, 0.5512763475647510E-02, 0.5841446056060346E-01, 0.5290169708495600E-01, 0.4044780558898965E-16, 0.7749519698096867E-34, 0.2434478048112329E-01},
			tauOut:   1.2903659816950583E-002,
			ttypeOut: -9,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2605878684924073E+01, 0.2615950460208911E+01, 0.1945503997226627E-02, 0.2831884532112553E-02, 0.1792287311167312E+01, 0.1790238034184775E+01, 0.1831165540887852E-01, 0.1689844079671380E-01, 0.1912064812489923E+01, 0.1942176087759744E+01, 0.5350346462390188E-03, 0.1104039956008399E-02, 0.9149979242236238E+00, 0.9266158484293474E+00, 0.2199691768758329E-02, 0.1820770257466081E-02, 0.1425451464914127E+01, 0.1105418651305687E+01, 0.2857477273065492E+00, 0.3351361651941490E+00, 0.9167409543554429E+00, 0.1215385144271246E+01, 0.7843816205451699E-05, 0.7197207696703830E-05, 0.1003973806896052E+01, 0.9991024098509327E+00, 0.9418498631040351E-02, 0.1778290067827487E-01, 0.5121827465172590E+00, 0.5317426047035739E+00, 0.2263732461064059E-02, 0.2762300261676148E-02, 0.4201365406709412E+00, 0.4197388406228219E+00, 0.3107071375346623E-01, 0.1556509232613395E-01, 0.1196803030427524E+01, 0.8386678292065533E+00, 0.3403588894097180E+00, 0.4021095747913879E+00, 0.1359492109368225E+01, 0.1013013804731768E+01, 0.2411467892724848E+00, 0.6997408538631263E+00, 0.3392875246772514E+00, 0.4685122433619579E+00, 0.1652848117124708E+00, 0.1248257304047288E+00, 0.4228949559754990E+00, 0.4492589344427766E+00, 0.1708424320817961E+00, 0.1518244930621437E+00, 0.3159848280636428E+00, 0.4758679007372426E+00, 0.6745243660763246E-01, 0.2386301922514691E-01, 0.8251512110211983E+00, 0.8931789553886716E+00, 0.7711819686391232E-03, 0.1232835205710967E-01, 0.3794316855579739E-01, 0.5161612293293578E-01, 0.3606500256058598E-05, 0.1887408451311279E-05, 0.6510811153270980E-01, 0.7250261437426894E-01, 0.4479235177066975E-02, 0.5512763475647510E-02, 0.3551880209093845E-01, 0.5290169708495600E-01, 0.7843816205451699E-05, 0.7197207696703830E-05},
			i0:       1,
			n0:       18,
			pp:       0,
			n0in:     18,
			dmin:     3.5518802090938446E-002,
			dmin1:    3.7941281147346073E-002,
			dmin2:    3.7941281147346073E-002,
			dn:       3.5518802090938446E-002,
			dn1:      5.9595348057062299E-002,
			dn2:      3.7941281147346073E-002,
			tau:      1.2903659816950583E-002,
			ttype:    -9,
			g:        0.0000000000000000,
			zOut:     []float64{0.2605878684924073E+01, 0.2615950460208911E+01, 0.1945503997226627E-02, 0.2831884532112553E-02, 0.1792287311167312E+01, 0.1790238034184775E+01, 0.1831165540887852E-01, 0.1689844079671380E-01, 0.1912064812489923E+01, 0.1942176087759744E+01, 0.5350346462390188E-03, 0.1104039956008399E-02, 0.9149979242236238E+00, 0.9266158484293474E+00, 0.2199691768758329E-02, 0.1820770257466081E-02, 0.1425451464914127E+01, 0.1105418651305687E+01, 0.2857477273065492E+00, 0.3351361651941490E+00, 0.9167409543554429E+00, 0.1215385144271246E+01, 0.7843816205451699E-05, 0.7197207696703830E-05, 0.1003973806896052E+01, 0.9991024098509327E+00, 0.9418498631040351E-02, 0.1778290067827487E-01, 0.5121827465172590E+00, 0.5317426047035739E+00, 0.2263732461064059E-02, 0.2762300261676148E-02, 0.4201365406709412E+00, 0.4197388406228219E+00, 0.3107071375346623E-01, 0.1556509232613395E-01, 0.1196803030427524E+01, 0.8386678292065533E+00, 0.3403588894097180E+00, 0.4021095747913879E+00, 0.1359492109368225E+01, 0.1013013804731768E+01, 0.2411467892724848E+00, 0.6997408538631263E+00, 0.3392875246772514E+00, 0.4685122433619579E+00, 0.1652848117124708E+00, 0.1248257304047288E+00, 0.4228949559754990E+00, 0.4492589344427766E+00, 0.1708424320817961E+00, 0.1518244930621437E+00, 0.3159848280636428E+00, 0.4758679007372426E+00, 0.6745243660763246E-01, 0.2386301922514691E-01, 0.8251512110211983E+00, 0.8931789553886716E+00, 0.7711819686391232E-03, 0.1232835205710967E-01, 0.3794316855579739E-01, 0.5161612293293578E-01, 0.3606500256058598E-05, 0.1887408451311279E-05, 0.6510811153270980E-01, 0.7250261437426894E-01, 0.4479235177066975E-02, 0.5512763475647510E-02, 0.3551880209093845E-01, 0.5290169708495600E-01, 0.7843816205451699E-05, 0.7197207696703830E-05},
			tauOut:   2.4222106054237202E-002,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.2605878684924073E+01, 0.2583602082867062E+01, 0.1945503997226627E-02, 0.1349628161076996E-02, 0.1792287311167312E+01, 0.1785027232360876E+01, 0.1831165540887852E-01, 0.1961486711855321E-01, 0.1912064812489923E+01, 0.1868762873963372E+01, 0.5350346462390188E-03, 0.2619677421449130E-03, 0.9149979242236238E+00, 0.8927135421960001E+00, 0.2199691768758329E-02, 0.3512385223173503E-02, 0.1425451464914127E+01, 0.1683464700943265E+01, 0.2857477273065492E+00, 0.1556056649653108E+00, 0.9167409543554429E+00, 0.7369210271521004E+00, 0.7843816205451699E-05, 0.1068633642713914E-04, 0.1003973806896052E+01, 0.9891595131364276E+00, 0.9418498631040351E-02, 0.4876860034049891E-02, 0.5121827465172590E+00, 0.4853475128900360E+00, 0.2263732461064059E-02, 0.1959578858316830E-02, 0.4201365406709412E+00, 0.4250255695118534E+00, 0.3107071375346623E-01, 0.8749008776201990E-01, 0.1196803030427524E+01, 0.1425449726020986E+01, 0.3403588894097180E+00, 0.3246099922425689E+00, 0.1359492109368225E+01, 0.1251806800343904E+01, 0.2411467892724848E+00, 0.6536000379104072E-01, 0.3392875246772514E+00, 0.4149902265444443E+00, 0.1652848117124708E+00, 0.1684331550518530E+00, 0.4228949559754990E+00, 0.4010821269512049E+00, 0.1708424320817961E+00, 0.1345949193440586E+00, 0.3159848280636428E+00, 0.2246202392729794E+00, 0.6745243660763246E-01, 0.2477891570824889E+00, 0.8251512110211983E+00, 0.5539111298531113E+00, 0.7711819686391232E-03, 0.5282632149136541E-04, 0.3794316855579739E-01, 0.1367184268032488E-01, 0.3606500256058598E-05, 0.1717489195894037E-04, 0.6510811153270980E-01, 0.4534806576358064E-01, 0.4479235177066975E-02, 0.3508353996892708E-02, 0.3551880209093845E-01, 0.7788342039808532E-02, 0.7843816205451699E-05, 0.1068633642713914E-04, 0.2434478048112329E-01},
			i0:       1,
			n0:       18,
			pp:       1,
			n0in:     18,
			dmin:     7.7883420398085317E-003,
			dmin1:    1.3668236180068825E-002,
			dmin2:    1.3668236180068825E-002,
			dn:       7.7883420398085317E-003,
			dn1:      4.0868830586513666E-002,
			dn2:      1.3668236180068825E-002,
			tau:      2.4222106054237202E-002,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.2605878684924073E+01, 0.2583602082867062E+01, 0.1945503997226627E-02, 0.1349628161076996E-02, 0.1792287311167312E+01, 0.1785027232360876E+01, 0.1831165540887852E-01, 0.1961486711855321E-01, 0.1912064812489923E+01, 0.1868762873963372E+01, 0.5350346462390188E-03, 0.2619677421449130E-03, 0.9149979242236238E+00, 0.8927135421960001E+00, 0.2199691768758329E-02, 0.3512385223173503E-02, 0.1425451464914127E+01, 0.1683464700943265E+01, 0.2857477273065492E+00, 0.1556056649653108E+00, 0.9167409543554429E+00, 0.7369210271521004E+00, 0.7843816205451699E-05, 0.1068633642713914E-04, 0.1003973806896052E+01, 0.9891595131364276E+00, 0.9418498631040351E-02, 0.4876860034049891E-02, 0.5121827465172590E+00, 0.4853475128900360E+00, 0.2263732461064059E-02, 0.1959578858316830E-02, 0.4201365406709412E+00, 0.4250255695118534E+00, 0.3107071375346623E-01, 0.8749008776201990E-01, 0.1196803030427524E+01, 0.1425449726020986E+01, 0.3403588894097180E+00, 0.3246099922425689E+00, 0.1359492109368225E+01, 0.1251806800343904E+01, 0.2411467892724848E+00, 0.6536000379104072E-01, 0.3392875246772514E+00, 0.4149902265444443E+00, 0.1652848117124708E+00, 0.1684331550518530E+00, 0.4228949559754990E+00, 0.4010821269512049E+00, 0.1708424320817961E+00, 0.1345949193440586E+00, 0.3159848280636428E+00, 0.2246202392729794E+00, 0.6745243660763246E-01, 0.2477891570824889E+00, 0.8251512110211983E+00, 0.5539111298531113E+00, 0.7711819686391232E-03, 0.5282632149136541E-04, 0.3794316855579739E-01, 0.1367184268032488E-01, 0.3606500256058598E-05, 0.1717489195894037E-04, 0.6510811153270980E-01, 0.4534806576358064E-01, 0.4479235177066975E-02, 0.3508353996892708E-02, 0.3551880209093845E-01, 0.7788342039808532E-02, 0.7843816205451699E-05, 0.1068633642713914E-04, 0.2434478048112329E-01},
			tauOut:   5.1484099711571517E-003,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},

		{
			z:        []float64{0.1845045664413523E+01, 0.1845045673681623E+01, 0.9284324334305927E-09, 0.1331924177594376E-08, 0.1286110617891388E+01, 0.1286109423357647E+01, 0.8600075793510468E-06, 0.1206062196852843E-05, 0.9228692597002398E+00, 0.9170877606119078E+00, 0.7344017075309068E-02, 0.5782369695935185E-02, 0.1164764979181097E+01, 0.1172109006845551E+01, 0.1416795225784663E-10, 0.1087898982148097E-10, 0.1516899526939413E+01, 0.1516899536394917E+01, 0.1175479719901021E-09, 0.1158688024893155E-08, 0.1569402316257081E+00, 0.1538882419371820E+00, 0.4821181617659422E-02, 0.3052000406097992E-02, 0.2431018491557045E+00, 0.2479152225121892E+00, 0.8755391955186837E-05, 0.7818861198543554E-05, 0.2722314297289909E+00, 0.2722202019375114E+00, 0.1357252989661879E-04, 0.1999378345859679E-04, 0.1847873189363651E+00, 0.1848009020627515E+00, 0.2027258571663375E-18, 0.3534048617228375E-11, 0.4750110334503861E-13, 0.1060007138617788E-07, 0.1416795225784663E-10, 0.1087898982148097E-10},
			i0:       1,
			n0:       10,
			pp:       0,
			n0in:     10,
			dmin:     4.7501103345038606E-014,
			dmin1:    0.15388823121961009,
			dmin2:    0.15388823121961009,
			dn:       4.7501103345038606E-014,
			dn1:      0.18478731893283101,
			dn2:      0.27221143594553232,
			tau:      1.0600023884871808E-008,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1845045664413523E+01, 0.1845045673681623E+01, 0.9284324334305927E-09, 0.1331924177594376E-08, 0.1286110617891388E+01, 0.1286109423357647E+01, 0.8600075793510468E-06, 0.1206062196852843E-05, 0.9228692597002398E+00, 0.9170877606119078E+00, 0.7344017075309068E-02, 0.5782369695935185E-02, 0.1164764979181097E+01, 0.1172109006845551E+01, 0.1416795225784663E-10, 0.1087898982148097E-10, 0.1516899526939413E+01, 0.1516899536394917E+01, 0.1175479719901021E-09, 0.1158688024893155E-08, 0.1569402316257081E+00, 0.1538882419371820E+00, 0.4821181617659422E-02, 0.3052000406097992E-02, 0.2431018491557045E+00, 0.2479152225121892E+00, 0.8755391955186837E-05, 0.7818861198543554E-05, 0.2722314297289909E+00, 0.2722202019375114E+00, 0.1357252989661879E-04, 0.1999378345859679E-04, 0.1847873189363651E+00, 0.1848009020627515E+00, 0.2027258571663375E-18, 0.3534048617228375E-11, 0.4750110334503861E-13, 0.1060007138617788E-07, 0.1416795225784663E-10, 0.1087898982148097E-10},
			tauOut:   4.7501103294055340E-014,
			ttypeOut: -4,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1845045664413523E+01, 0.1845045665341908E+01, 0.9284324334305927E-09, 0.6471746651368383E-09, 0.1286110617891388E+01, 0.1286111477251745E+01, 0.8600075793510468E-06, 0.6171117917307419E-06, 0.9228692597002398E+00, 0.9302126596637096E+00, 0.7344017075309068E-02, 0.9195804644198721E-02, 0.1164764979181097E+01, 0.1155569174551018E+01, 0.1416795225784663E-10, 0.1859807318413278E-10, 0.1516899526939413E+01, 0.1516899527038316E+01, 0.1175479719901021E-09, 0.1216165317638265E-10, 0.1569402316257081E+00, 0.1617614132311584E+00, 0.4821181617659422E-02, 0.7245474325163344E-02, 0.2431018491557045E+00, 0.2358651302224488E+00, 0.8755391955186837E-05, 0.1010532106865606E-04, 0.2722314297289909E+00, 0.2722348969377714E+00, 0.1357252989661879E-04, 0.9212747663842461E-05, 0.1847873189363651E+00, 0.1847781061886537E+00, 0.2027258571663375E-18, 0.5211495068651724E-31, 0.8046649468928653E+00},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     10,
			dmin:     5.0983266199979805E-023,
			dmin1:    0.15694023161349893,
			dmin2:    0.15694023161349893,
			dn:       5.0983266199979805E-023,
			dn1:      0.18477810618865373,
			dn2:      0.27222132440787472,
			tau:      4.7501103294055340E-014,
			ttype:    -4,
			g:        0.0000000000000000,
			zOut:     []float64{0.1845045664413523E+01, 0.1845045665341908E+01, 0.9284324334305927E-09, 0.6471746651368383E-09, 0.1286110617891388E+01, 0.1286111477251745E+01, 0.8600075793510468E-06, 0.6171117917307419E-06, 0.9228692597002398E+00, 0.9302126596637096E+00, 0.7344017075309068E-02, 0.9195804644198721E-02, 0.1164764979181097E+01, 0.1155569174551018E+01, 0.1416795225784663E-10, 0.1859807318413278E-10, 0.1516899526939413E+01, 0.1516899527038316E+01, 0.1175479719901021E-09, 0.1216165317638265E-10, 0.1569402316257081E+00, 0.1617614132311584E+00, 0.4821181617659422E-02, 0.7245474325163344E-02, 0.2431018491557045E+00, 0.2358651302224488E+00, 0.8755391955186837E-05, 0.1010532106865606E-04, 0.2722314297289909E+00, 0.2722348969377714E+00, 0.1357252989661879E-04, 0.9212747663842461E-05, 0.1847873189363651E+00, 0.1847781061886537E+00, 0.2027258571663375E-18, 0.5211495068651724E-31, 0.8046649468928653E+00},
			tauOut:   3.9235057903374733E-002,
			ttypeOut: -9,
			gOut:     0.0000000000000000,
		},
		{
			z:        []float64{0.1805810608085708E+01, 0.1845045665341908E+01, 0.4609225136302542E-09, 0.6471746651368383E-09, 0.1246877035999240E+01, 0.1286111477251745E+01, 0.4603863769418564E-06, 0.6171117917307419E-06, 0.9001729460181566E+00, 0.9302126596637096E+00, 0.1180482975969687E-01, 0.9195804644198721E-02, 0.1104529286906545E+01, 0.1155569174551018E+01, 0.2554156666668993E-10, 0.1859807318413278E-10, 0.1477664469121561E+01, 0.1516899527038316E+01, 0.1331348385339719E-11, 0.1216165317638265E-10, 0.1297718296516156E+00, 0.1617614132311584E+00, 0.1316891924708087E-01, 0.7245474325163344E-02, 0.1834712583930619E+00, 0.2358651302224488E+00, 0.1499428882618218E-04, 0.1010532106865606E-04, 0.2329940574932343E+00, 0.2722348969377714E+00, 0.7306255294378852E-05, 0.9212747663842461E-05, 0.1455357420299846E+00, 0.1847781061886537E+00, 0.1331348385339719E-11, 0.1216165317638265E-10},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     0.12252635532645229,
			dmin1:    0.12252635532645229,
			dmin2:    0.12252635532645229,
			dn:       0.14553574202998462,
			dn1:      0.23298484474557046,
			dn2:      0.18346115307199323,
			tau:      3.9235057903374733E-002,
			ttype:    -9,
			g:        0.0000000000000000,
			zOut:     []float64{0.1805810608085708E+01, 0.1845045665341908E+01, 0.4609225136302542E-09, 0.6471746651368383E-09, 0.1246877035999240E+01, 0.1286111477251745E+01, 0.4603863769418564E-06, 0.6171117917307419E-06, 0.9001729460181566E+00, 0.9302126596637096E+00, 0.1180482975969687E-01, 0.9195804644198721E-02, 0.1104529286906545E+01, 0.1155569174551018E+01, 0.2554156666668993E-10, 0.1859807318413278E-10, 0.1477664469121561E+01, 0.1516899527038316E+01, 0.1331348385339719E-11, 0.1216165317638265E-10, 0.1297718296516156E+00, 0.1617614132311584E+00, 0.1316891924708087E-01, 0.7245474325163344E-02, 0.1834712583930619E+00, 0.2358651302224488E+00, 0.1499428882618218E-04, 0.1010532106865606E-04, 0.2329940574932343E+00, 0.2722348969377714E+00, 0.7306255294378852E-05, 0.9212747663842461E-05, 0.1455357420299846E+00, 0.1847781061886537E+00, 0.1331348385339719E-11, 0.1216165317638265E-10},
			tauOut:   3.0631588831613071E-002,
			ttypeOut: -6,
			gOut:     0.25000000000000000,
		},
		{
			z:        []float64{0.1805810608085708E+01, 0.1775179019715018E+01, 0.4609225136302542E-09, 0.3237497126982006E-09, 0.1246877035999240E+01, 0.1216245907230254E+01, 0.4603863769418564E-06, 0.3407430674789675E-06, 0.9001729460181566E+00, 0.8813458462031728E+00, 0.1180482975969687E-01, 0.1479416990810367E-01, 0.1104529286906545E+01, 0.1059103528192370E+01, 0.2554156666668993E-10, 0.3563567162644001E-10, 0.1477664469121561E+01, 0.1447032880255644E+01, 0.1331348385339719E-11, 0.1193970905752580E-12, 0.1297718296516156E+00, 0.1123091600669640E+00, 0.1316891924708087E-01, 0.2151309995104528E-01, 0.1834712583930619E+00, 0.1313415638992297E+00, 0.1499428882618218E-04, 0.2659919745982362E-04, 0.2329940574932343E+00, 0.2023431757194558E+00, 0.7306255294378852E-05, 0.5255039029347846E-05, 0.1455357420299846E+00, 0.1148988981593422E+00, 0.1331348385339719E-11, 0.1193970905752580E-12, 0.8046649468928653E+00},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     9,
			dmin:     9.9140240819883152E-002,
			dmin1:    9.9140240819883152E-002,
			dmin2:    9.9140240819883152E-002,
			dn:       0.11489889815934221,
			dn1:      0.20233586946416143,
			dn2:      0.13132656961040354,
			tau:      3.0631588831613071E-002,
			ttype:    -6,
			g:        0.25000000000000000,
			zOut:     []float64{0.1805810608085708E+01, 0.1775179019715018E+01, 0.4609225136302542E-09, 0.3237497126982006E-09, 0.1246877035999240E+01, 0.1216245907230254E+01, 0.4603863769418564E-06, 0.3407430674789675E-06, 0.9001729460181566E+00, 0.8813458462031728E+00, 0.1180482975969687E-01, 0.1479416990810367E-01, 0.1104529286906545E+01, 0.1059103528192370E+01, 0.2554156666668993E-10, 0.3563567162644001E-10, 0.1477664469121561E+01, 0.1447032880255644E+01, 0.1331348385339719E-11, 0.1193970905752580E-12, 0.1297718296516156E+00, 0.1123091600669640E+00, 0.1316891924708087E-01, 0.2151309995104528E-01, 0.1834712583930619E+00, 0.1313415638992297E+00, 0.1499428882618218E-04, 0.2659919745982362E-04, 0.2329940574932343E+00, 0.2023431757194558E+00, 0.7306255294378852E-05, 0.5255039029347846E-05, 0.1455357420299846E+00, 0.1148988981593422E+00, 0.1331348385339719E-11, 0.1193970905752580E-12, 0.8046649468928653E+00},
			tauOut:   4.9545335349736611E-002,
			ttypeOut: -6,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1725633684689031E+01, 0.1775179019715018E+01, 0.2281824158451768E-09, 0.3237497126982006E-09, 0.1166700912395402E+01, 0.1216245907230254E+01, 0.2574031475886406E-06, 0.3407430674789675E-06, 0.8465944233583922E+00, 0.8813458462031728E+00, 0.1850774953630535E-01, 0.1479416990810367E-01, 0.9910504433419636E+00, 0.1059103528192370E+01, 0.5203164874187830E-10, 0.3563567162644001E-10, 0.1397487544853995E+01, 0.1447032880255644E+01, 0.9595353465813834E-14, 0.1193970905752580E-12, 0.8427692466826309E-01, 0.1123091600669640E+00, 0.3352713928531350E-01, 0.2151309995104528E-01, 0.4829568846163943E-01, 0.1313415638992297E+00, 0.1114419580100731E-03, 0.2659919745982362E-04, 0.1526916534507385E+00, 0.2023431757194558E+00, 0.3954362799871079E-05, 0.5255039029347846E-05, 0.6534960844680572E-01, 0.1148988981593422E+00, 0.9595353465813834E-14, 0.1193970905752580E-12},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     4.8269089264179610E-002,
			dmin1:    4.8269089264179610E-002,
			dmin2:    4.8269089264179610E-002,
			dn:       6.5349608446805721E-002,
			dn1:      0.15268639841170917,
			dn2:      4.8269089264179610E-002,
			tau:      4.9545335349736611E-002,
			ttype:    -6,
			g:        0.49975000000000003,
			zOut:     []float64{0.1725633684689031E+01, 0.1775179019715018E+01, 0.2281824158451768E-09, 0.3237497126982006E-09, 0.1166700912395402E+01, 0.1216245907230254E+01, 0.2574031475886406E-06, 0.3407430674789675E-06, 0.8465944233583922E+00, 0.8813458462031728E+00, 0.1850774953630535E-01, 0.1479416990810367E-01, 0.9910504433419636E+00, 0.1059103528192370E+01, 0.5203164874187830E-10, 0.3563567162644001E-10, 0.1397487544853995E+01, 0.1447032880255644E+01, 0.9595353465813834E-14, 0.1193970905752580E-12, 0.8427692466826309E-01, 0.1123091600669640E+00, 0.3352713928531350E-01, 0.2151309995104528E-01, 0.4829568846163943E-01, 0.1313415638992297E+00, 0.1114419580100731E-03, 0.2659919745982362E-04, 0.1526916534507385E+00, 0.2023431757194558E+00, 0.3954362799871079E-05, 0.5255039029347846E-05, 0.6534960844680572E-01, 0.1148988981593422E+00, 0.9595353465813834E-14, 0.1193970905752580E-12},
			tauOut:   1.2037453975339630E-002,
			ttypeOut: -5,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1725633684689031E+01, 0.1713596230941874E+01, 0.2281824158451768E-09, 0.1553578538234923E-09, 0.1166700912395402E+01, 0.1154663715667853E+01, 0.2574031475886406E-06, 0.1887268702969492E-06, 0.8465944233583922E+00, 0.8530645301924875E+00, 0.1850774953630535E-01, 0.2150143715280096E-01, 0.9910504433419636E+00, 0.9575115522658546E+00, 0.5203164874187830E-10, 0.7594016059954953E-10, 0.1397487544853995E+01, 0.1385450090802725E+01, 0.9595353465813834E-14, 0.5836853211617396E-15, 0.8427692466826309E-01, 0.1057666099782364E+00, 0.3352713928531350E-01, 0.1530933320323569E-01, 0.4829568846163943E-01, 0.2106034324107419E-01, 0.1114419580100731E-03, 0.8079762346493418E-03, 0.1526916534507385E+00, 0.1398501776035494E+00, 0.3954362799871079E-05, 0.1847806452993955E-05, 0.6534960844680572E-01, 0.5331030666501309E-01, 0.9595353465813834E-14, 0.5836853211617396E-15, 0.8046649468928653E+00},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     9,
			dmin:     2.0948901283064113E-002,
			dmin1:    2.0948901283064113E-002,
			dmin2:    2.0948901283064113E-002,
			dn:       5.3310306665013088E-002,
			dn1:      0.13984622324074955,
			dn2:      2.0948901283064113E-002,
			tau:      1.2037453975339630E-002,
			ttype:    -5,
			g:        0.49975000000000003,
			zOut:     []float64{0.1725633684689031E+01, 0.1713596230941874E+01, 0.2281824158451768E-09, 0.1553578538234923E-09, 0.1166700912395402E+01, 0.1154663715667853E+01, 0.2574031475886406E-06, 0.1887268702969492E-06, 0.8465944233583922E+00, 0.8530645301924875E+00, 0.1850774953630535E-01, 0.2150143715280096E-01, 0.9910504433419636E+00, 0.9575115522658546E+00, 0.5203164874187830E-10, 0.7594016059954953E-10, 0.1397487544853995E+01, 0.1385450090802725E+01, 0.9595353465813834E-14, 0.5836853211617396E-15, 0.8427692466826309E-01, 0.1057666099782364E+00, 0.3352713928531350E-01, 0.1530933320323569E-01, 0.4829568846163943E-01, 0.2106034324107419E-01, 0.1114419580100731E-03, 0.8079762346493418E-03, 0.1526916534507385E+00, 0.1398501776035494E+00, 0.3954362799871079E-05, 0.1847806452993955E-05, 0.6534960844680572E-01, 0.5331030666501309E-01, 0.9595353465813834E-14, 0.5836853211617396E-15, 0.8046649468928653E+00},
			tauOut:   1.1070386405955311E-002,
			ttypeOut: -5,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1702525844691276E+01, 0.1713596230941874E+01, 0.1053646717395619E-09, 0.1553578538234923E-09, 0.1143593517883403E+01, 0.1154663715667853E+01, 0.1407809649380857E-06, 0.1887268702969492E-06, 0.8634954401583683E+00, 0.8530645301924875E+00, 0.2384248197112572E-01, 0.2150143715280096E-01, 0.9225986839647138E+00, 0.9575115522658546E+00, 0.1140379931457212E-09, 0.7594016059954953E-10, 0.1374379704282732E+01, 0.1385450090802725E+01, 0.4491802194179927E-16, 0.5836853211617396E-15, 0.1100055567755167E+00, 0.1057666099782364E+00, 0.2930941140637693E-02, 0.1530933320323569E-01, 0.7866991929130526E-02, 0.2106034324107419E-01, 0.1436325611276508E-01, 0.8079762346493418E-03, 0.1144183828912820E+00, 0.1398501776035494E+00, 0.8609379557504958E-06, 0.1847806452993955E-05, 0.4223905932110202E-01, 0.5331030666501309E-01, 0.4491802194179927E-16, 0.5836853211617396E-15},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     7.0590156944811844E-003,
			dmin1:    7.0590156944811844E-003,
			dmin2:    7.0590156944811844E-003,
			dn:       4.2239059321102022E-002,
			dn1:      0.11441653508482905,
			dn2:      7.0590156944811844E-003,
			tau:      1.1070386405955311E-002,
			ttype:    -5,
			g:        0.49975000000000003,
			zOut:     []float64{0.1702525844691276E+01, 0.1713596230941874E+01, 0.1053646717395619E-09, 0.1553578538234923E-09, 0.1143593517883403E+01, 0.1154663715667853E+01, 0.1407809649380857E-06, 0.1887268702969492E-06, 0.8634954401583683E+00, 0.8530645301924875E+00, 0.2384248197112572E-01, 0.2150143715280096E-01, 0.9225986839647138E+00, 0.9575115522658546E+00, 0.1140379931457212E-09, 0.7594016059954953E-10, 0.1374379704282732E+01, 0.1385450090802725E+01, 0.4491802194179927E-16, 0.5836853211617396E-15, 0.1100055567755167E+00, 0.1057666099782364E+00, 0.2930941140637693E-02, 0.1530933320323569E-01, 0.7866991929130526E-02, 0.2106034324107419E-01, 0.1436325611276508E-01, 0.8079762346493418E-03, 0.1144183828912820E+00, 0.1398501776035494E+00, 0.8609379557504958E-06, 0.1847806452993955E-05, 0.4223905932110202E-01, 0.5331030666501309E-01, 0.4491802194179927E-16, 0.5836853211617396E-15},
			tauOut:   5.5670727120955849E-003,
			ttypeOut: -5,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1702525844691276E+01, 0.1696958772084545E+01, 0.1053646717395619E-09, 0.7100605954454643E-10, 0.1143593517883403E+01, 0.1138026585881266E+01, 0.1407809649380857E-06, 0.1068197551738174E-06, 0.8634954401583683E+00, 0.8817707425976433E+00, 0.2384248197112572E-01, 0.2494644177488929E-01, 0.9225986839647138E+00, 0.8920851695917670E+00, 0.1140379931457212E-09, 0.1756911880603681E-09, 0.1374379704282732E+01, 0.1368812631394945E+01, 0.4491802194179927E-16, 0.3609867340226789E-17, 0.1100055567755167E+00, 0.1073694252040588E+00, 0.2930941140637693E-02, 0.2147509894397918E-03, 0.7866991929130526E-02, 0.1644842434036023E-01, 0.1436325611276508E-01, 0.9991355423895336E-01, 0.1144183828912820E+00, 0.8938616878188867E-02, 0.8609379557504958E-06, 0.4068326216494222E-05, 0.4223905932110202E-01, 0.3666791828278994E-01, 0.4491802194179927E-16, 0.3609867340226789E-17, 0.8046649468928653E+00},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     9,
			dmin:     2.0851682275951503E-003,
			dmin1:    2.0851682275951503E-003,
			dmin2:    2.0851682275951503E-003,
			dn:       3.6667918282789938E-002,
			dn1:      8.9377559402331157E-003,
			dn2:      2.0851682275951503E-003,
			tau:      5.5670727120955849E-003,
			ttype:    -5,
			g:        0.49975000000000003,
			zOut:     []float64{0.1702525844691276E+01, 0.1696958772084545E+01, 0.1053646717395619E-09, 0.7100605954454643E-10, 0.1143593517883403E+01, 0.1138026585881266E+01, 0.1407809649380857E-06, 0.1068197551738174E-06, 0.8634954401583683E+00, 0.8817707425976433E+00, 0.2384248197112572E-01, 0.2494644177488929E-01, 0.9225986839647138E+00, 0.8920851695917670E+00, 0.1140379931457212E-09, 0.1756911880603681E-09, 0.1374379704282732E+01, 0.1368812631394945E+01, 0.4491802194179927E-16, 0.3609867340226789E-17, 0.1100055567755167E+00, 0.1073694252040588E+00, 0.2930941140637693E-02, 0.2147509894397918E-03, 0.7866991929130526E-02, 0.1644842434036023E-01, 0.1436325611276508E-01, 0.9991355423895336E-01, 0.1144183828912820E+00, 0.8938616878188867E-02, 0.8609379557504958E-06, 0.4068326216494222E-05, 0.4223905932110202E-01, 0.3666791828278994E-01, 0.4491802194179927E-16, 0.3609867340226789E-17, 0.8046649468928653E+00},
			tauOut:   1.1659821590613959E-003,
			ttypeOut: -5,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1695792789996490E+01, 0.1696958772084545E+01, 0.4765133098633433E-10, 0.7100605954454643E-10, 0.1136860710494309E+01, 0.1138026585881266E+01, 0.8285142935651389E-07, 0.1068197551738174E-06, 0.9055511193620417E+00, 0.8817707425976433E+00, 0.2457547703893446E-01, 0.2494644177488929E-01, 0.8663437105694624E+00, 0.8920851695917670E+00, 0.2775899617066991E-09, 0.1756911880603681E-09, 0.1367646648958294E+01, 0.1368812631394945E+01, 0.2833987723936393E-18, 0.3609867340226789E-17, 0.1064181940344372E+00, 0.1073694252040588E+00, 0.3319277717374952E-04, 0.2147509894397918E-03, 0.1151628036430784E+00, 0.1644842434036023E-01, 0.7755012504281143E-02, 0.9991355423895336E-01, 0.2169054106282214E-04, 0.8938616878188867E-02, 0.6877516463147774E-02, 0.4068326216494222E-05, 0.2862441966058077E-01, 0.3666791828278994E-01, 0.2833987723936393E-18, 0.3609867340226789E-17},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     1.7622214846327918E-005,
			dmin1:    1.7622214846327918E-005,
			dmin2:    1.5249249404125084E-002,
			dn:       2.8624419660580765E-002,
			dn1:      1.7622214846327918E-005,
			dn2:      1.5249249404125084E-002,
			tau:      1.1659821590613959E-003,
			ttype:    -5,
			g:        0.49975000000000003,
			zOut:     []float64{0.1695792789996490E+01, 0.1696958772084545E+01, 0.4765133098633433E-10, 0.7100605954454643E-10, 0.1136860710494309E+01, 0.1138026585881266E+01, 0.8285142935651389E-07, 0.1068197551738174E-06, 0.9055511193620417E+00, 0.8817707425976433E+00, 0.2457547703893446E-01, 0.2494644177488929E-01, 0.8663437105694624E+00, 0.8920851695917670E+00, 0.2775899617066991E-09, 0.1756911880603681E-09, 0.1367646648958294E+01, 0.1368812631394945E+01, 0.2833987723936393E-18, 0.3609867340226789E-17, 0.1064181940344372E+00, 0.1073694252040588E+00, 0.3319277717374952E-04, 0.2147509894397918E-03, 0.1151628036430784E+00, 0.1644842434036023E-01, 0.7755012504281143E-02, 0.9991355423895336E-01, 0.2169054106282214E-04, 0.8938616878188867E-02, 0.6877516463147774E-02, 0.4068326216494222E-05, 0.2862441966058077E-01, 0.3666791828278994E-01, 0.2833987723936393E-18, 0.3609867340226789E-17},
			tauOut:   1.2076215031173109E-005,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1695792789996490E+01, 0.1695780713829110E+01, 0.4765133098633433E-10, 0.3194571418305606E-10, 0.1136860710494309E+01, 0.1136848717098761E+01, 0.8285142935651389E-07, 0.6599488873594650E-07, 0.9055511193620417E+00, 0.9301144541910563E+00, 0.2457547703893446E-01, 0.2289052693567938E-01, 0.8663437105694624E+00, 0.8434411076963417E+00, 0.2775899617066991E-09, 0.4501143914475995E-09, 0.1367646648958294E+01, 0.1367634572293148E+01, 0.2833987723936393E-18, 0.2205178646452290E-19, 0.1064181940344372E+00, 0.1064393105965798E+00, 0.3319277717374952E-04, 0.3591317210346347E-04, 0.1151628036430784E+00, 0.1228698267602250E+00, 0.7755012504281143E-02, 0.1369013219942635E-05, 0.2169054106282214E-04, 0.6885761775959480E-02, 0.6877516463147774E-02, 0.2859014352646020E-01, 0.2862441966058077E-01, 0.2219991908939190E-04, 0.2833987723936393E-18, 0.2205178646452290E-19, 0.8046649468928653E+00},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     9,
			dmin:     8.2453128117063980E-006,
			dmin1:    8.2453128117063980E-006,
			dmin2:    0.10640611781940604,
			dn:       2.2199919089391903E-005,
			dn1:      8.2453128117063980E-006,
			dn2:      0.11511481425594380,
			tau:      1.2076215031173109E-005,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.1695792789996490E+01, 0.1695780713829110E+01, 0.4765133098633433E-10, 0.3194571418305606E-10, 0.1136860710494309E+01, 0.1136848717098761E+01, 0.8285142935651389E-07, 0.6599488873594650E-07, 0.9055511193620417E+00, 0.9301144541910563E+00, 0.2457547703893446E-01, 0.2289052693567938E-01, 0.8663437105694624E+00, 0.8434411076963417E+00, 0.2775899617066991E-09, 0.4501143914475995E-09, 0.1367646648958294E+01, 0.1367634572293148E+01, 0.2833987723936393E-18, 0.2205178646452290E-19, 0.1064181940344372E+00, 0.1064393105965798E+00, 0.3319277717374952E-04, 0.3591317210346347E-04, 0.1151628036430784E+00, 0.1228698267602250E+00, 0.7755012504281143E-02, 0.1369013219942635E-05, 0.2169054106282214E-04, 0.6885761775959480E-02, 0.6877516463147774E-02, 0.2859014352646020E-01, 0.2862441966058077E-01, 0.2219991908939190E-04, 0.2833987723936393E-18, 0.2205178646452290E-19, 0.8046649468928653E+00},
			tauOut:   3.2770273787704380E-006,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1695777436833677E+01, 0.1695780713829110E+01, 0.2141639781080129E-10, 0.3194571418305606E-10, 0.1136845506044855E+01, 0.1136848717098761E+01, 0.5399396803668456E-07, 0.6599488873594650E-07, 0.9530016501053890E+00, 0.9301144541910563E+00, 0.2025894854667596E-01, 0.2289052693567938E-01, 0.8231788825724013E+00, 0.8434411076963417E+00, 0.7478228806194929E-09, 0.4501143914475995E-09, 0.1367631294517947E+01, 0.1367634572293148E+01, 0.1716235185693175E-20, 0.2205178646452290E-19, 0.1064719467413045E+00, 0.1064393105965798E+00, 0.4144420544393849E-04, 0.3591317210346347E-04, 0.1228264745406222E+00, 0.1228698267602250E+00, 0.7674810284932935E-07, 0.1369013219942635E-05, 0.3547255152693805E-01, 0.6885761775959480E-02, 0.1789267604726782E-04, 0.2859014352646020E-01, 0.1030215663353646E-05, 0.2219991908939190E-04, 0.1716235185693175E-20, 0.2205178646452290E-19},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     1.0302156633536465E-006,
			dmin1:    6.8824080004778604E-003,
			dmin2:    0.10643603356920101,
			dn:       1.0302156633536465E-006,
			dn1:      6.8824080004778604E-003,
			dn2:      0.12282510552740224,
			tau:      3.2770273787704380E-006,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.1695777436833677E+01, 0.1695780713829110E+01, 0.2141639781080129E-10, 0.3194571418305606E-10, 0.1136845506044855E+01, 0.1136848717098761E+01, 0.5399396803668456E-07, 0.6599488873594650E-07, 0.9530016501053890E+00, 0.9301144541910563E+00, 0.2025894854667596E-01, 0.2289052693567938E-01, 0.8231788825724013E+00, 0.8434411076963417E+00, 0.7478228806194929E-09, 0.4501143914475995E-09, 0.1367631294517947E+01, 0.1367634572293148E+01, 0.1716235185693175E-20, 0.2205178646452290E-19, 0.1064719467413045E+00, 0.1064393105965798E+00, 0.4144420544393849E-04, 0.3591317210346347E-04, 0.1228264745406222E+00, 0.1228698267602250E+00, 0.7674810284932935E-07, 0.1369013219942635E-05, 0.3547255152693805E-01, 0.6885761775959480E-02, 0.1789267604726782E-04, 0.2859014352646020E-01, 0.1030215663353646E-05, 0.2219991908939190E-04, 0.1716235185693175E-20, 0.2205178646452290E-19},
			tauOut:   1.0296962592568820E-006,
			ttypeOut: -2,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1695777436833677E+01, 0.1695776407158834E+01, 0.2141639781080129E-10, 0.1435751523850387E-10, 0.1136845506044855E+01, 0.1136844530328206E+01, 0.5399396803668456E-07, 0.4526242530264241E-07, 0.9530016501053890E+00, 0.9732595236933804E+00, 0.2025894854667596E-01, 0.1713493494875720E-01, 0.8231788825724013E+00, 0.8060429186752077E+00, 0.7478228806194929E-09, 0.1268848035998799E-08, 0.1367631294517947E+01, 0.1367630263552839E+01, 0.1716235185693175E-20, 0.1336113320657127E-21, 0.1064719467413045E+00, 0.1065123612504892E+00, 0.4144420544393849E-04, 0.4779206455525695E-04, 0.1228264745406222E+00, 0.1227777295279105E+00, 0.7674810284932935E-07, 0.2217381803186623E-07, 0.3547255152693805E-01, 0.3548939233290803E-01, 0.1789267604726782E-04, 0.5194035150079302E-09, 0.1030215663353646E-05, 0.5817566277343568E-15, 0.1716235185693175E-20, 0.1336113320657127E-21, 0.8046649468928653E+00},
			i0:       1,
			n0:       9,
			pp:       1,
			n0in:     9,
			dmin:     5.8175662773435680E-016,
			dmin1:    3.5471499656860764E-002,
			dmin2:    0.10647091704504522,
			dn:       5.8175662773435680E-016,
			dn1:      3.5471499656860764E-002,
			dn2:      0.12277765277980766,
			tau:      1.0296962592568820E-006,
			ttype:    -2,
			g:        0.49975000000000003,
			zOut:     []float64{0.1695777436833677E+01, 0.1695776407158834E+01, 0.2141639781080129E-10, 0.1435751523850387E-10, 0.1136845506044855E+01, 0.1136844530328206E+01, 0.5399396803668456E-07, 0.4526242530264241E-07, 0.9530016501053890E+00, 0.9732595236933804E+00, 0.2025894854667596E-01, 0.1713493494875720E-01, 0.8231788825724013E+00, 0.8060429186752077E+00, 0.7478228806194929E-09, 0.1268848035998799E-08, 0.1367631294517947E+01, 0.1367630263552839E+01, 0.1716235185693175E-20, 0.1336113320657127E-21, 0.1064719467413045E+00, 0.1065123612504892E+00, 0.4144420544393849E-04, 0.4779206455525695E-04, 0.1228264745406222E+00, 0.1227777295279105E+00, 0.7674810284932935E-07, 0.2217381803186623E-07, 0.3547255152693805E-01, 0.3548939233290803E-01, 0.1789267604726782E-04, 0.5194035150079302E-09, 0.1030215663353646E-05, 0.5817566277343568E-15, 0.1716235185693175E-20, 0.1336113320657127E-21, 0.8046649468928653E+00},
			tauOut:   5.8175661922007790E-016,
			ttypeOut: -2,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1695776407173191E+01, 0.1695776407158834E+01, 0.9625244577618415E-11, 0.1435751523850387E-10, 0.1136844575581005E+01, 0.1136844530328206E+01, 0.3874943632355666E-07, 0.4526242530264241E-07, 0.9903944198927007E+00, 0.9732595236933804E+00, 0.1394544708652782E-01, 0.1713493494875720E-01, 0.7920974728575274E+00, 0.8060429186752077E+00, 0.2190784636165185E-08, 0.1268848035998799E-08, 0.1367630261362054E+01, 0.1367630263552839E+01, 0.1040577915698430E-22, 0.1336113320657127E-21, 0.1065601533150438E+00, 0.1065123612504892E+00, 0.5506562249584694E-04, 0.4779206455525695E-04, 0.1227226860792321E+00, 0.1227777295279105E+00, 0.6412305277798032E-08, 0.2217381803186623E-07, 0.3548938644000568E-01, 0.3548939233290803E-01, 0.8514276171981532E-23, 0.5194035150079302E-09, 0.2761013168273541E-29, 0.5817566277343568E-15, 0.1040577915698430E-22, 0.1336113320657127E-21},
			i0:       1,
			n0:       9,
			pp:       0,
			n0in:     9,
			dmin:     2.7610131682735413E-030,
			dmin1:    3.5489385920602169E-002,
			dmin2:    0.10651236125048857,
			dn:       2.7610131682735413E-030,
			dn1:      3.5489385920602169E-002,
			dn2:      0.12272266390541409,
			tau:      5.8175661922007790E-016,
			ttype:    -2,
			g:        0.49975000000000003,
			zOut:     []float64{0.1695776407173191E+01, 0.1695776407158834E+01, 0.9625244577618415E-11, 0.1435751523850387E-10, 0.1136844575581005E+01, 0.1136844530328206E+01, 0.3874943632355666E-07, 0.4526242530264241E-07, 0.9903944198927007E+00, 0.9732595236933804E+00, 0.1394544708652782E-01, 0.1713493494875720E-01, 0.7920974728575274E+00, 0.8060429186752077E+00, 0.2190784636165185E-08, 0.1268848035998799E-08, 0.1367630261362054E+01, 0.1367630263552839E+01, 0.1040577915698430E-22, 0.1336113320657127E-21, 0.1065601533150438E+00, 0.1065123612504892E+00, 0.5506562249584694E-04, 0.4779206455525695E-04, 0.1227226860792321E+00, 0.1227777295279105E+00, 0.6412305277798032E-08, 0.2217381803186623E-07, 0.3548938644000568E-01, 0.3548939233290803E-01, 0.8514276171981532E-23, 0.5194035150079302E-09, 0.2761013168273541E-29, 0.5817566277343568E-15, 0.1040577915698430E-22, 0.1336113320657127E-21},
			tauOut:   2.7610131682735413E-030,
			ttypeOut: -2,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1695776407173191E+01, 0.1695776407182817E+01, 0.9625244577618415E-11, 0.6452741670633652E-11, 0.1136844575581005E+01, 0.1136844614323989E+01, 0.3874943632355666E-07, 0.3375767015588020E-07, 0.9903944198927007E+00, 0.1004339833221559E+01, 0.1394544708652782E-01, 0.1099842207759001E-01, 0.7920974728575274E+00, 0.7810990529707220E+00, 0.2190784636165185E-08, 0.3835855840755795E-08, 0.1367630261362054E+01, 0.1367630257526198E+01, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.1065601533150438E+00, 0.1066152189375397E+00, 0.5506562249584694E-04, 0.6338495733216447E-04, 0.1227226860792321E+00, 0.1226593075342052E+00, 0.6412305277798032E-08, 0.1855291575909163E-08, 0.3548938644000568E-01, 0.3548938458471410E-01, 0.8514276171981532E-23, 0.6623960630549781E-51, 0.9539342071687115E+00},
			i0:       1,
			n0:       8,
			pp:       1,
			n0in:     9,
			dmin:     2.7610131682735413E-030,
			dmin1:    3.5489384584714102E-002,
			dmin2:    0.10656015331504384,
			dn:       2.7610131682735413E-030,
			dn1:      3.5489384584714102E-002,
			dn2:      0.12265930112189996,
			tau:      0.0000000000000000,
			ttype:    -2,
			g:        0.49975000000000003,
			zOut:     []float64{0.1695776407173191E+01, 0.1695776407182817E+01, 0.9625244577618415E-11, 0.6452741670633652E-11, 0.1136844575581005E+01, 0.1136844614323989E+01, 0.3874943632355666E-07, 0.3375767015588020E-07, 0.9903944198927007E+00, 0.1004339833221559E+01, 0.1394544708652782E-01, 0.1099842207759001E-01, 0.7920974728575274E+00, 0.7810990529707220E+00, 0.2190784636165185E-08, 0.3835855840755795E-08, 0.1367630261362054E+01, 0.1367630257526198E+01, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.1065601533150438E+00, 0.1066152189375397E+00, 0.5506562249584694E-04, 0.6338495733216447E-04, 0.1227226860792321E+00, 0.1226593075342052E+00, 0.6412305277798032E-08, 0.1855291575909163E-08, 0.3548938644000568E-01, 0.3548938458471410E-01, 0.8514276171981532E-23, 0.6623960630549781E-51, 0.9539342071687115E+00},
			tauOut:   1.7744692292357051E-002,
			ttypeOut: -9,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1678031714896912E+01, 0.1695776407182817E+01, 0.4371648372768990E-11, 0.6452741670633652E-11, 0.1119099955784930E+01, 0.1136844614323989E+01, 0.3029592900888367E-07, 0.3375767015588020E-07, 0.9975935327108624E+00, 0.1004339833221559E+01, 0.8611580555893368E-02, 0.1099842207759001E-01, 0.7547427839583274E+00, 0.7810990529707220E+00, 0.6950755439903452E-08, 0.3835855840755795E-08, 0.1349885558283086E+01, 0.1367630257526198E+01, 0.6403582257922030E-25, 0.8107757314001177E-24, 0.8893391160251481E-01, 0.1066152189375397E+00, 0.8742171388117141E-04, 0.6338495733216447E-04, 0.1048271953832586E+00, 0.1226593075342052E+00, 0.6281113981299557E-09, 0.1855291575909163E-08, 0.1774469166424565E-01, 0.3548938458471410E-01, 0.6403582257922030E-25, 0.8107757314001177E-24},
			i0:       1,
			n0:       8,
			pp:       0,
			n0in:     8,
			dmin:     1.7744691664245655E-002,
			dmin1:    8.8870526645182649E-002,
			dmin2:    8.8870526645182649E-002,
			dn:       1.7744691664245655E-002,
			dn1:      0.10482719352796703,
			dn2:      8.8870526645182649E-002,
			tau:      1.7744692292357051E-002,
			ttype:    -9,
			g:        0.49975000000000003,
			zOut:     []float64{0.1678031714896912E+01, 0.1695776407182817E+01, 0.4371648372768990E-11, 0.6452741670633652E-11, 0.1119099955784930E+01, 0.1136844614323989E+01, 0.3029592900888367E-07, 0.3375767015588020E-07, 0.9975935327108624E+00, 0.1004339833221559E+01, 0.8611580555893368E-02, 0.1099842207759001E-01, 0.7547427839583274E+00, 0.7810990529707220E+00, 0.6950755439903452E-08, 0.3835855840755795E-08, 0.1349885558283086E+01, 0.1367630257526198E+01, 0.6403582257922030E-25, 0.8107757314001177E-24, 0.8893391160251481E-01, 0.1066152189375397E+00, 0.8742171388117141E-04, 0.6338495733216447E-04, 0.1048271953832586E+00, 0.1226593075342052E+00, 0.6281113981299557E-09, 0.1855291575909163E-08, 0.1774469166424565E-01, 0.3548938458471410E-01, 0.6403582257922030E-25, 0.8107757314001177E-24},
			tauOut:   1.7743283373674822E-002,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1678031714896912E+01, 0.1660288431527609E+01, 0.4371648372768990E-11, 0.2946663608425970E-11, 0.1119099955784930E+01, 0.1101356702704238E+01, 0.3029592900888367E-07, 0.2744162973950316E-07, 0.9975935327108624E+00, 0.9884618024514511E+00, 0.8611580555893368E-02, 0.6575396506892930E-02, 0.7547427839583274E+00, 0.7304241110285150E+00, 0.6950755439903452E-08, 0.1284558415558242E-07, 0.1349885558283086E+01, 0.1332142262063827E+01, 0.6403582257922030E-25, 0.4275036042946169E-26, 0.8893391160251481E-01, 0.7127804994272115E-01, 0.8742171388117141E-04, 0.1285693574547170E-03, 0.1048271953832586E+00, 0.8695534328024046E-01, 0.6281113981299557E-09, 0.1281766326273249E-09, 0.1774469166424565E-01, 0.1408162394200135E-05, 0.6403582257922030E-25, 0.4275036042946169E-26, 0.9539342071687115E+00},
			i0:       1,
			n0:       8,
			pp:       1,
			n0in:     8,
			dmin:     1.4081623942001353E-006,
			dmin1:    7.1190628228839981E-002,
			dmin2:    7.1190628228839981E-002,
			dn:       1.4081623942001353E-006,
			dn1:      8.6955342652129064E-002,
			dn2:      7.1190628228839981E-002,
			tau:      1.7743283373674822E-002,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.1678031714896912E+01, 0.1660288431527609E+01, 0.4371648372768990E-11, 0.2946663608425970E-11, 0.1119099955784930E+01, 0.1101356702704238E+01, 0.3029592900888367E-07, 0.2744162973950316E-07, 0.9975935327108624E+00, 0.9884618024514511E+00, 0.8611580555893368E-02, 0.6575396506892930E-02, 0.7547427839583274E+00, 0.7304241110285150E+00, 0.6950755439903452E-08, 0.1284558415558242E-07, 0.1349885558283086E+01, 0.1332142262063827E+01, 0.6403582257922030E-25, 0.4275036042946169E-26, 0.8893391160251481E-01, 0.7127804994272115E-01, 0.8742171388117141E-04, 0.1285693574547170E-03, 0.1048271953832586E+00, 0.8695534328024046E-01, 0.6281113981299557E-09, 0.1281766326273249E-09, 0.1774469166424565E-01, 0.1408162394200135E-05, 0.6403582257922030E-25, 0.4275036042946169E-26, 0.9539342071687115E+00},
			tauOut:   1.4081069428512511E-006,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1660287023423613E+01, 0.1660288431527609E+01, 0.1954678721190348E-11, 0.2946663608425970E-11, 0.1101355322036970E+01, 0.1101356702704238E+01, 0.2462874809952034E-07, 0.2744162973950316E-07, 0.9950357662226532E+00, 0.9884618024514511E+00, 0.4826789459478154E-02, 0.6575396506892930E-02, 0.7255959263076781E+00, 0.7304241110285150E+00, 0.2358357443050565E-07, 0.1284558415558242E-07, 0.1332140830373310E+01, 0.1332142262063827E+01, -0.9894235909416862E+00, 0.4275036042946169E-26, 0.7140521119323301E-01, 0.7127804994272115E-01, 0.1565683011922014E-03, 0.1285693574547170E-03, 0.8679736700028205E-01, 0.8695534328024046E-01, 0.2079481441878513E-14, 0.1281766326273249E-09, 0.5544926940271698E-10, 0.1408162394200135E-05, 0.1565683011922014E-03, 0.1285693574547170E-03},
			i0:       6,
			n0:       8,
			pp:       0,
			n0in:     8,
			dmin:     5.5449269402716976E-011,
			dmin1:    7.1276641835778295E-002,
			dmin2:    7.1276641835778295E-002,
			dn:       5.5449269402716976E-011,
			dn1:      8.6797366872105416E-002,
			dn2:      7.1276641835778295E-002,
			tau:      1.4081069428512511E-006,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.1660287023423613E+01, 0.1660288431527609E+01, 0.1954678721190348E-11, 0.2946663608425970E-11, 0.1101355322036970E+01, 0.1101356702704238E+01, 0.2462874809952034E-07, 0.2744162973950316E-07, 0.9950357662226532E+00, 0.9884618024514511E+00, 0.4826789459478154E-02, 0.6575396506892930E-02, 0.7255959263076781E+00, 0.7304241110285150E+00, 0.2358357443050565E-07, 0.1284558415558242E-07, 0.1332140830373310E+01, 0.1332142262063827E+01, -0.9894235909416862E+00, 0.4275036042946169E-26, 0.7140521119323301E-01, 0.7127804994272115E-01, 0.1565683011922014E-03, 0.1285693574547170E-03, 0.8679736700028205E-01, 0.8695534328024046E-01, 0.2079481441878513E-14, 0.1281766326273249E-09, 0.5544926940271698E-10, 0.1408162394200135E-05, 0.1565683011922014E-03, 0.1285693574547170E-03},
			tauOut:   5.5449260598506680E-011,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1660287023423613E+01, 0.1660288431527609E+01, 0.1954678721190348E-11, 0.2946663608425970E-11, 0.1101355322036970E+01, 0.1101356702704238E+01, 0.2462874809952034E-07, 0.2744162973950316E-07, 0.9950357662226532E+00, 0.9884618024514511E+00, 0.4826789459478154E-02, 0.6575396506892930E-02, 0.7255959263076781E+00, 0.7304241110285150E+00, 0.2358357443050565E-07, 0.1284558415558242E-07, 0.1332140830373310E+01, 0.1332142262063827E+01, -0.9894235909416862E+00, 0.4275036042946169E-26, 0.7140521119323301E-01, 0.7156177943897596E-01, 0.1565683011922014E-03, 0.1899018778701386E-03, 0.8679736700028205E-01, 0.8660746506696473E-01, 0.2079481441878513E-14, 0.1331360138522907E-23, 0.5544926940271698E-10, 0.8804208964992894E-17, 0.1565683011922014E-03, 0.8679736700028205E-01, 0.9539342071687115E+00},
			i0:       6,
			n0:       8,
			pp:       1,
			n0in:     8,
			dmin:     8.8042089649928937E-018,
			dmin1:    7.1405211137783753E-002,
			dmin2:    7.1405211137783753E-002,
			dn:       8.8042089649928937E-018,
			dn1:      8.6607465066962652E-002,
			dn2:      7.1405211137783753E-002,
			tau:      5.5449260598506680E-011,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.1660287023423613E+01, 0.1660288431527609E+01, 0.1954678721190348E-11, 0.2946663608425970E-11, 0.1101355322036970E+01, 0.1101356702704238E+01, 0.2462874809952034E-07, 0.2744162973950316E-07, 0.9950357662226532E+00, 0.9884618024514511E+00, 0.4826789459478154E-02, 0.6575396506892930E-02, 0.7255959263076781E+00, 0.7304241110285150E+00, 0.2358357443050565E-07, 0.1284558415558242E-07, 0.1332140830373310E+01, 0.1332142262063827E+01, -0.9894235909416862E+00, 0.4275036042946169E-26, 0.7140521119323301E-01, 0.7156177943897596E-01, 0.1565683011922014E-03, 0.1899018778701386E-03, 0.8679736700028205E-01, 0.8660746506696473E-01, 0.2079481441878513E-14, 0.1331360138522907E-23, 0.5544926940271698E-10, 0.8804208964992894E-17, 0.1565683011922014E-03, 0.8679736700028205E-01, 0.9539342071687115E+00},
			tauOut:   8.8042089649574750E-018,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1660287023423613E+01, 0.1660288431527609E+01, 0.1954678721190348E-11, 0.2946663608425970E-11, 0.1101355322036970E+01, 0.1101356702704238E+01, 0.2462874809952034E-07, 0.2744162973950316E-07, 0.9950357662226532E+00, 0.9884618024514511E+00, 0.4826789459478154E-02, 0.6575396506892930E-02, 0.7255959263076781E+00, 0.7304241110285150E+00, 0.2358357443050565E-07, 0.1284558415558242E-07, 0.1332140830373310E+01, 0.1332142262063827E+01, -0.9894235909416862E+00, 0.4275036042946169E-26},
			i0:       1,
			n0:       5,
			pp:       0,
			n0in:     5,
			dmin:     -0.60723548073666500,
			dmin1:    7.1561779438975959E-002,
			dmin2:    7.1561779438975959E-002,
			dn:       8.8042089649928937E-018,
			dn1:      8.6378245068883266E-002,
			dn2:      7.1561779438975959E-002,
			tau:      0.0000000000000000,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.1660287023423613E+01, 0.1660288431527609E+01, 0.1954678721190348E-11, 0.2946663608425970E-11, 0.1101355322036970E+01, 0.1101356702704238E+01, 0.2462874809952034E-07, 0.2744162973950316E-07, 0.9950357662226532E+00, 0.9884618024514511E+00, 0.4826789459478154E-02, 0.6575396506892930E-02, 0.7255959263076781E+00, 0.7304241110285150E+00, 0.2358357443050565E-07, 0.1284558415558242E-07, 0.1332140830373310E+01, 0.1332142262063827E+01, -0.9894235909416862E+00, 0.4275036042946169E-26},
			tauOut:   0.60723548073666500,
			ttypeOut: -1,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.1660287023423613E+01, 0.1053051542688903E+01, 0.1954678721190348E-11, 0.2044340400431280E-11, 0.1101355322036970E+01, 0.4941198659270088E+00, 0.2462874809952034E-07, 0.4959623550114665E-07, 0.9950357662226532E+00, 0.3926270253492309E+00, 0.4826789459478154E-02, 0.8920167341580701E-02, 0.7255959263076781E+00, 0.1094403018130068E+00, 0.2358357443050565E-07, 0.2870664819501689E-06, 0.1332140830373310E+01, 0.7249050625701626E+00, -0.9894235909416862E+00, 0.2044340400431280E-11, 0.1077066053646038E+01},
			i0:       1,
			n0:       5,
			pp:       1,
			n0in:     5,
			dmin:     0.10944027822943236,
			dmin1:    0.10944027822943236,
			dmin2:    0.38780023588975276,
			dn:       0.72490506257016263,
			dn1:      0.10944027822943236,
			dn2:      0.38780023588975276,
			tau:      0.60723548073666500,
			ttype:    -1,
			g:        0.49975000000000003,
			zOut:     []float64{0.1660287023423613E+01, 0.1053051542688903E+01, 0.1954678721190348E-11, 0.2044340400431280E-11, 0.1101355322036970E+01, 0.4941198659270088E+00, 0.2462874809952034E-07, 0.4959623550114665E-07, 0.9950357662226532E+00, 0.3926270253492309E+00, 0.4826789459478154E-02, 0.8920167341580701E-02, 0.7255959263076781E+00, 0.1094403018130068E+00, 0.2358357443050565E-07, 0.2870664819501689E-06, 0.1332140830373310E+01, 0.7249050625701626E+00, -0.9894235909416862E+00, 0.2044340400431280E-11, 0.1077066053646038E+01},
			tauOut:   9.0381042476589277E-002,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.9626705002143580E+00, 0.1053051542688903E+01, 0.1049319787347115E-11, 0.2044340400431280E-11, 0.4037388730456057E+00, 0.4941198659270088E+00, 0.4823122991958101E-07, 0.4959623550114665E-07, 0.3111661019829924E+00, 0.3926270253492309E+00, 0.3137314122148422E-02, 0.8920167341580701E-02, 0.1592223228075105E-01, 0.1094403018130068E+00, 0.1306952080528672E-04, 0.2870664819501689E-06, 0.6345109505727681E+00, 0.7249050625701626E+00, 0.1049319787347115E-11, 0.2044340400431280E-11},
			i0:       1,
			n0:       5,
			pp:       0,
			n0in:     5,
			dmin:     1.5921945214269095E-002,
			dmin1:    1.5921945214269095E-002,
			dmin2:    0.30224593464141175,
			dn:       0.63451095057276807,
			dn1:      1.5921945214269095E-002,
			dn2:      0.30224593464141175,
			tau:      9.0381042476589277E-002,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.9626705002143580E+00, 0.1053051542688903E+01, 0.1049319787347115E-11, 0.2044340400431280E-11, 0.4037388730456057E+00, 0.4941198659270088E+00, 0.4823122991958101E-07, 0.4959623550114665E-07, 0.3111661019829924E+00, 0.3926270253492309E+00, 0.3137314122148422E-02, 0.8920167341580701E-02, 0.1592223228075105E-01, 0.1094403018130068E+00, 0.1306952080528672E-04, 0.2870664819501689E-06, 0.6345109505727681E+00, 0.7249050625701626E+00, 0.1049319787347115E-11, 0.2044340400431280E-11},
			tauOut:   1.4134050686447828E-002,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.9626705002143580E+00, 0.9485364495289594E+00, 0.1049319787347115E-11, 0.4466366986934060E-12, 0.4037388730456057E+00, 0.3896048705899411E+00, 0.4823122991958101E-07, 0.3852088344069326E-07, 0.3111661019829924E+00, 0.3001693268978096E+00, 0.3137314122148422E-02, 0.1664162181618706E-03, 0.1592223228075105E-01, 0.1634834896946635E-02, 0.1306952080528672E-04, 0.5072533064458891E-02, 0.6345109505727681E+00, 0.6153043668218613E+00, 0.1049319787347115E-11, 0.4466366986934060E-12, 0.1077066053646038E+01},
			i0:       1,
			n0:       5,
			pp:       1,
			n0in:     5,
			dmin:     1.6217653761413479E-003,
			dmin1:    1.6217653761413479E-003,
			dmin2:    0.29703201277566116,
			dn:       0.61530436682186129,
			dn1:      1.6217653761413479E-003,
			dn2:      0.29703201277566116,
			tau:      1.4134050686447828E-002,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.9626705002143580E+00, 0.9485364495289594E+00, 0.1049319787347115E-11, 0.4466366986934060E-12, 0.4037388730456057E+00, 0.3896048705899411E+00, 0.4823122991958101E-07, 0.3852088344069326E-07, 0.3111661019829924E+00, 0.3001693268978096E+00, 0.3137314122148422E-02, 0.1664162181618706E-03, 0.1592223228075105E-01, 0.1634834896946635E-02, 0.1306952080528672E-04, 0.5072533064458891E-02, 0.6345109505727681E+00, 0.6153043668218613E+00, 0.1049319787347115E-11, 0.4466366986934060E-12, 0.1077066053646038E+01},
			tauOut:   1.5809617416939919E-003,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.9469554877877121E+00, 0.9485364495289594E+00, 0.1837592531426069E-12, 0.4466366986934060E-12, 0.3880239473689468E+00, 0.3896048705899411E+00, 0.2979915990315817E-07, 0.3852088344069326E-07, 0.2987547515751175E+00, 0.3001693268978096E+00, 0.9106567826436868E-06, 0.1664162181618706E-03, 0.5125495562928890E-02, 0.1634834896946635E-02, 0.6089463364253304E+00, 0.5072533064458891E-02, 0.4777068654836977E-02, 0.6153043668218613E+00, 0.1837592531426069E-12, 0.4466366986934060E-12},
			i0:       1,
			n0:       5,
			pp:       0,
			n0in:     5,
			dmin:     5.2962498469998932E-005,
			dmin1:    5.2962498469998932E-005,
			dmin2:    0.29858833535695567,
			dn:       4.7770686548369769E-003,
			dn1:      5.2962498469998932E-005,
			dn2:      0.29858833535695567,
			tau:      1.5809617416939919E-003,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.9469554877877121E+00, 0.9485364495289594E+00, 0.1837592531426069E-12, 0.4466366986934060E-12, 0.3880239473689468E+00, 0.3896048705899411E+00, 0.2979915990315817E-07, 0.3852088344069326E-07, 0.2987547515751175E+00, 0.3001693268978096E+00, 0.9106567826436868E-06, 0.1664162181618706E-03, 0.5125495562928890E-02, 0.1634834896946635E-02, 0.6089463364253304E+00, 0.5072533064458891E-02, 0.4777068654836977E-02, 0.6153043668218613E+00, 0.1837592531426069E-12, 0.4466366986934060E-12},
			tauOut:   4.7621670852039755E-005,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.9469554877877121E+00, 0.9469159823779022E+00, 0.1837592531426069E-12, 0.7530022947855128E-13, 0.3880239473689468E+00, 0.3879844717580378E+00, 0.2979915990315817E-07, 0.2294586835827609E-07, 0.2987547515751175E+00, 0.2987161338760382E+00, 0.9106567826436868E-06, 0.1562542751952015E-07, 0.5125495562928890E-02, 0.6140323109528382E+00, 0.6089463364253304E+00, 0.4737500623869755E-02, 0.4777068654836977E-02, 0.6262097360996939E-07, 0.1837592531426069E-12, 0.7530022947855128E-13, 0.1077066053646038E+01},
			i0:       1,
			n0:       5,
			pp:       1,
			n0in:     5,
			dmin:     6.2620973609969386E-008,
			dmin1:    5.0859745275077589E-003,
			dmin2:    0.29871522321925559,
			dn:       6.2620973609969386E-008,
			dn1:      5.0859745275077589E-003,
			dn2:      0.29871522321925559,
			tau:      3.9505409993611756E-005,
			ttype:    -15,
			g:        0.49975000000000003,
			zOut:     []float64{0.9469554877877121E+00, 0.9469159823779022E+00, 0.1837592531426069E-12, 0.7530022947855128E-13, 0.3880239473689468E+00, 0.3879844717580378E+00, 0.2979915990315817E-07, 0.2294586835827609E-07, 0.2987547515751175E+00, 0.2987161338760382E+00, 0.9106567826436868E-06, 0.1562542751952015E-07, 0.5125495562928890E-02, 0.6140323109528382E+00, 0.6089463364253304E+00, 0.4737500623869755E-02, 0.4777068654836977E-02, 0.6262097360996939E-07, 0.1837592531426069E-12, 0.7530022947855128E-13, 0.1077066053646038E+01},
			tauOut:   6.2141437994562936E-008,
			ttypeOut: -2,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.9469159202365395E+00, 0.9469159823779022E+00, 0.3085312975855001E-13, 0.7530022947855128E-13, 0.3879844325624373E+00, 0.3879844717580378E+00, 0.1766643326162244E-07, 0.2294586835827609E-07, 0.2987160696935945E+00, 0.2987161338760382E+00, 0.3211918722443868E-07, 0.1562542751952015E-07, 0.6187697173160828E+00, 0.6140323109528382E+00, 0.4794463808464251E-09, 0.4737500623869755E-02, 0.8923456002485611E-13, 0.6262097360996939E-07, 0.3085312975855001E-13, 0.7530022947855128E-13},
			i0:       1,
			n0:       5,
			pp:       0,
			n0in:     5,
			dmin:     8.9234560024856112E-014,
			dmin1:    0.29871605406816698,
			dmin2:    0.29871605406816698,
			dn:       8.9234560024856112E-014,
			dn1:      0.61403221669221297,
			dn2:      0.29871605406816698,
			tau:      6.2141437994562936E-008,
			ttype:    -2,
			g:        0.49975000000000003,
			zOut:     []float64{0.9469159202365395E+00, 0.9469159823779022E+00, 0.3085312975855001E-13, 0.7530022947855128E-13, 0.3879844325624373E+00, 0.3879844717580378E+00, 0.1766643326162244E-07, 0.2294586835827609E-07, 0.2987160696935945E+00, 0.2987161338760382E+00, 0.3211918722443868E-07, 0.1562542751952015E-07, 0.6187697173160828E+00, 0.6140323109528382E+00, 0.4794463808464251E-09, 0.4737500623869755E-02, 0.8923456002485611E-13, 0.6262097360996939E-07, 0.3085312975855001E-13, 0.7530022947855128E-13},
			tauOut:   8.9232014685788963E-014,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.9469159202365395E+00, 0.9469159202364811E+00, 0.3085312975855001E-13, 0.1264160184270297E-13, 0.3879844325624373E+00, 0.3879844502287687E+00, 0.1766643326162244E-07, 0.1360169848638109E-07, 0.2987160696935945E+00, 0.2987160882109940E+00, 0.3211918722443868E-07, 0.6653267495003571E-07, 0.6187697173160828E+00, 0.6187696512627651E+00, 0.4794463808464251E-09, 0.6914234847011291E-22, 0.8923456002485611E-13, 0.2545269924803487E-17, 0.3085312975855001E-13, 0.1264160184270297E-13, 0.1077066053646038E+01},
			i0:       1,
			n0:       5,
			pp:       1,
			n0in:     5,
			dmin:     2.5452699248034865E-018,
			dmin1:    0.29871605609180679,
			dmin2:    0.29871605609180679,
			dn:       2.5452699248034865E-018,
			dn1:      0.61876965078331869,
			dn2:      0.29871605609180679,
			tau:      8.9232014685788963E-014,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.9469159202365395E+00, 0.9469159202364811E+00, 0.3085312975855001E-13, 0.1264160184270297E-13, 0.3879844325624373E+00, 0.3879844502287687E+00, 0.1766643326162244E-07, 0.1360169848638109E-07, 0.2987160696935945E+00, 0.2987160882109940E+00, 0.3211918722443868E-07, 0.6653267495003571E-07, 0.6187697173160828E+00, 0.6187696512627651E+00, 0.4794463808464251E-09, 0.6914234847011291E-22, 0.8923456002485611E-13, 0.2545269924803487E-17, 0.3085312975855001E-13, 0.1264160184270297E-13, 0.1077066053646038E+01},
			tauOut:   2.5452699247759164E-018,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.9469159202364937E+00, 0.9469159202364811E+00, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.3879844638304620E+00, 0.3879844502287687E+00, 0.1047218779010850E-07, 0.1360169848638109E-07, 0.2987161442714811E+00, 0.2987160882109940E+00, 0.1378177941363543E-06, 0.6653267495003571E-07, 0.6187695134449710E+00, 0.6187696512627651E+00, 0.2844127518685707E-39, 0.6914234847011291E-22},
			i0:       1,
			n0:       4,
			pp:       0,
			n0in:     5,
			dmin:     2.5452699248034865E-018,
			dmin1:    0.29871607773880621,
			dmin2:    0.29871607773880621,
			dn:       2.5452699248034865E-018,
			dn1:      0.61876951344497100,
			dn2:      0.29871607773880621,
			tau:      0.0000000000000000,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.9469159202364937E+00, 0.9469159202364811E+00, 0.5179704803914515E-14, 0.1264160184270297E-13, 0.3879844638304620E+00, 0.3879844502287687E+00, 0.1047218779010850E-07, 0.1360169848638109E-07, 0.2987161442714811E+00, 0.2987160882109940E+00, 0.1378177941363543E-06, 0.6653267495003571E-07, 0.6187695134449710E+00, 0.6187696512627651E+00, 0.2844127518685707E-39, 0.6914234847011291E-22},
			tauOut:   7.4679019434701552E-002,
			ttypeOut: -9,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.9469159202364937E+00, 0.8722369008017974E+00, 0.5179704803914515E-14, 0.2304012808102351E-14, 0.3879844638304620E+00, 0.3133054548679459E+00, 0.1047218779010850E-07, 0.9984542273822185E-08, 0.2987161442714811E+00, 0.2240372526700314E+00, 0.1378177941363543E-06, 0.3806395963416417E-06, 0.6187695134449710E+00, 0.5440901133706730E+00, 0.2844127518685707E-39, 0.2304012808102351E-14, 0.1702794694134603E+01},
			i0:       1,
			n0:       4,
			pp:       1,
			n0in:     4,
			dmin:     0.22403711485223726,
			dmin1:    0.22403711485223726,
			dmin2:    0.31330544439575814,
			dn:       0.54409011337067303,
			dn1:      0.22403711485223726,
			dn2:      0.31330544439575814,
			tau:      7.4679019434701552E-002,
			ttype:    -9,
			g:        0.49975000000000003,
			zOut:     []float64{0.9469159202364937E+00, 0.8722369008017974E+00, 0.5179704803914515E-14, 0.2304012808102351E-14, 0.3879844638304620E+00, 0.3133054548679459E+00, 0.1047218779010850E-07, 0.9984542273822185E-08, 0.2987161442714811E+00, 0.2240372526700314E+00, 0.1378177941363543E-06, 0.3806395963416417E-06, 0.6187695134449710E+00, 0.5440901133706730E+00, 0.2844127518685707E-39, 0.2304012808102351E-14, 0.1702794694134603E+01},
			tauOut:   0.22392121955733330,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.6483156812444665E+00, 0.8722369008017974E+00, 0.1113438717814820E-14, 0.2304012808102351E-14, 0.8938424529515379E-01, 0.3133054548679459E+00, 0.2502576838690600E-07, 0.9984542273822185E-08, 0.1163887265260836E-03, 0.2240372526700314E+00, 0.1779401212715201E-02, 0.3806395963416417E-06, 0.3183894926006245E+00, 0.5440901133706730E+00, 0.1113438717814820E-14, 0.2304012808102351E-14},
			i0:       1,
			n0:       4,
			pp:       0,
			n0in:     4,
			dmin:     1.1600808692974196E-004,
			dmin1:    1.1600808692974196E-004,
			dmin2:    8.9384235310611515E-002,
			dn:       0.31838949260062449,
			dn1:      1.1600808692974196E-004,
			dn2:      8.9384235310611515E-002,
			tau:      0.22392121955733330,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.6483156812444665E+00, 0.8722369008017974E+00, 0.1113438717814820E-14, 0.2304012808102351E-14, 0.8938424529515379E-01, 0.3133054548679459E+00, 0.2502576838690600E-07, 0.9984542273822185E-08, 0.1163887265260836E-03, 0.2240372526700314E+00, 0.1779401212715201E-02, 0.3806395963416417E-06, 0.3183894926006245E+00, 0.5440901133706730E+00, 0.1113438717814820E-14, 0.2304012808102351E-14},
			tauOut:   1.1589031543524988E-004,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.6483156812444665E+00, 0.6482265310101248E+00, 0.1113438717814820E-14, 0.1535325610927932E-15, 0.8938424529515379E-01, 0.8929512008657929E-01, 0.2502576838690600E-07, 0.3261899765703413E-10, 0.1163887265260836E-03, 0.1806639672279546E-02, 0.1779401212715201E-02, 0.3135891777105103E+00, 0.3183894926006245E+00, 0.4711164655771408E-02, 0.1113438717814820E-14, 0.1535325610927932E-15, 0.1702794694134603E+01},
			i0:       1,
			n0:       4,
			pp:       1,
			n0in:     4,
			dmin:     2.7238459564345061E-005,
			dmin1:    2.7238459564345061E-005,
			dmin2:    8.9295095060810903E-002,
			dn:       4.7111646557714080E-003,
			dn1:      2.7238459564345061E-005,
			dn2:      8.9295095060810903E-002,
			tau:      8.9150234342740890E-005,
			ttype:    -15,
			g:        0.49975000000000003,
			zOut:     []float64{0.6483156812444665E+00, 0.6482265310101248E+00, 0.1113438717814820E-14, 0.1535325610927932E-15, 0.8938424529515379E-01, 0.8929512008657929E-01, 0.2502576838690600E-07, 0.3261899765703413E-10, 0.1163887265260836E-03, 0.1806639672279546E-02, 0.1779401212715201E-02, 0.3135891777105103E+00, 0.3183894926006245E+00, 0.4711164655771408E-02, 0.1113438717814820E-14, 0.1535325610927932E-15, 0.1702794694134603E+01},
			tauOut:   2.5005145741075717E-005,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.6482015258643838E+00, 0.6482265310101248E+00, 0.2115037983241246E-16, 0.1535325610927932E-15, 0.8927011497345719E-01, 0.8929512008657929E-01, 0.6601400172354812E-12, 0.3261899765703413E-10, 0.3153708122363886E+00, 0.1806639672279546E-02, 0.4684549720964035E-02, 0.3135891777105103E+00, 0.1609789066298190E-05, 0.4711164655771408E-02, 0.2115037983241246E-16, 0.1535325610927932E-15},
			i0:       1,
			n0:       4,
			pp:       0,
			n0in:     4,
			dmin:     1.6097890662981897E-006,
			dmin1:    1.7816345258783307E-003,
			dmin2:    8.9270114940838197E-002,
			dn:       1.6097890662981897E-006,
			dn1:      1.7816345258783307E-003,
			dn2:      8.9270114940838197E-002,
			tau:      2.5005145741075717E-005,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.6482015258643838E+00, 0.6482265310101248E+00, 0.2115037983241246E-16, 0.1535325610927932E-15, 0.8927011497345719E-01, 0.8929512008657929E-01, 0.6601400172354812E-12, 0.3261899765703413E-10, 0.3153708122363886E+00, 0.1806639672279546E-02, 0.4684549720964035E-02, 0.3135891777105103E+00, 0.1609789066298190E-05, 0.4711164655771408E-02, 0.2115037983241246E-16, 0.1535325610927932E-15},
			tauOut:   1.5862205428611591E-006,
			ttypeOut: -2,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.6482015258643838E+00, 0.6481999396438409E+00, 0.2115037983241246E-16, 0.2912830939801045E-17, 0.8927011497345719E-01, 0.8926852875357447E-01, 0.6601400172354812E-12, 0.2332164496627946E-11, 0.3153708122363886E+00, 0.3200537757344777E+00, 0.4684549720964035E-02, 0.2356209328895529E-07, 0.1609789066298190E-05, 0.6430148075154844E-11, 0.2115037983241246E-16, 0.2912830939801045E-17, 0.1702794694134603E+01},
			i0:       1,
			n0:       4,
			pp:       1,
			n0in:     4,
			dmin:     6.4301480751548441E-012,
			dmin1:    8.9268528752914328E-002,
			dmin2:    8.9268528752914328E-002,
			dn:       6.4301480751548441E-012,
			dn1:      0.31536922601351364,
			dn2:      8.9268528752914328E-002,
			tau:      1.5862205428611591E-006,
			ttype:    -2,
			g:        0.49975000000000003,
			zOut:     []float64{0.6482015258643838E+00, 0.6481999396438409E+00, 0.2115037983241246E-16, 0.2912830939801045E-17, 0.8927011497345719E-01, 0.8926852875357447E-01, 0.6601400172354812E-12, 0.2332164496627946E-11, 0.3153708122363886E+00, 0.3200537757344777E+00, 0.4684549720964035E-02, 0.2356209328895529E-07, 0.1609789066298190E-05, 0.6430148075154844E-11, 0.2115037983241246E-16, 0.2912830939801045E-17, 0.1702794694134603E+01},
			tauOut:   6.4283598105616478E-012,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.6481999396374125E+00, 0.6481999396438409E+00, 0.4011480356653261E-18, 0.2912830939801045E-17, 0.8926852874947827E-01, 0.8926852875357447E-01, 0.8361491594360282E-11, 0.2332164496627946E-11, 0.3200537992817811E+00, 0.3200537757344777E+00, 0.4733821287189426E-18, 0.2356209328895529E-07, 0.1787791211067534E-14, 0.6430148075154844E-11, 0.4011480356653261E-18, 0.2912830939801045E-17},
			i0:       1,
			n0:       4,
			pp:       0,
			n0in:     4,
			dmin:     1.7877912110675335E-015,
			dmin1:    8.9268528747146109E-002,
			dmin2:    8.9268528747146109E-002,
			dn:       1.7877912110675335E-015,
			dn1:      0.32005377571968785,
			dn2:      8.9268528747146109E-002,
			tau:      6.4283598105616478E-012,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.6481999396374125E+00, 0.6481999396438409E+00, 0.4011480356653261E-18, 0.2912830939801045E-17, 0.8926852874947827E-01, 0.8926852875357447E-01, 0.8361491594360282E-11, 0.2332164496627946E-11, 0.3200537992817811E+00, 0.3200537757344777E+00, 0.4733821287189426E-18, 0.2356209328895529E-07, 0.1787791211067534E-14, 0.6430148075154844E-11, 0.4011480356653261E-18, 0.2912830939801045E-17},
			tauOut:   1.7877912088395813E-015,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.6481999396374125E+00, 0.6481999396374107E+00, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.8926852874947827E-01, 0.8926852875783797E-01, 0.8361491594360282E-11, 0.2997839428604580E-10, 0.3200537992817811E+00, 0.3200537992518009E+00, 0.4733821287189426E-18, 0.2644269217171004E-32, 0.2001510674733695E+01},
			i0:       1,
			n0:       3,
			pp:       1,
			n0in:     4,
			dmin:     2.2279522444840209E-024,
			dmin1:    8.9268528749476481E-002,
			dmin2:    8.9268528749476481E-002,
			dn:       2.2279522444840209E-024,
			dn1:      0.32005379925180094,
			dn2:      8.9268528749476481E-002,
			tau:      1.7877912088395813E-015,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.6481999396374125E+00, 0.6481999396374107E+00, 0.4011480356653261E-18, 0.5524513774965514E-19, 0.8926852874947827E-01, 0.8926852875783797E-01, 0.8361491594360282E-11, 0.2997839428604580E-10, 0.3200537992817811E+00, 0.3200537992518009E+00, 0.4733821287189426E-18, 0.2644269217171004E-32, 0.2001510674733695E+01},
			tauOut:   2.2317132187369120E-002,
			ttypeOut: -9,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.6258828074500417E+00, 0.6481999396374107E+00, 0.7879513719234823E-20, 0.5524513774965514E-19, 0.6695139660044724E-01, 0.8926852875783797E-01, 0.1433084218388560E-09, 0.2997839428604580E-10, 0.2977366669211234E+00, 0.3200537992518009E+00, 0.8926852875783797E-01, 0.2644269217171004E-32},
			i0:       1,
			n0:       3,
			pp:       0,
			n0in:     3,
			dmin:     6.6951396570468849E-002,
			dmin1:    6.6951396570468849E-002,
			dmin2:    0.62588280745004166,
			dn:       0.29773666692112338,
			dn1:      6.6951396570468849E-002,
			dn2:      0.62588280745004166,
			tau:      2.2317132187369120E-002,
			ttype:    -9,
			g:        0.49975000000000003,
			zOut:     []float64{0.6258828074500417E+00, 0.6481999396374107E+00, 0.7879513719234823E-20, 0.5524513774965514E-19, 0.6695139660044724E-01, 0.8926852875783797E-01, 0.1433084218388560E-09, 0.2997839428604580E-10, 0.2977366669211234E+00, 0.3200537992518009E+00, 0.8926852875783797E-01, 0.2644269217171004E-32},
			tauOut:   6.6950732596142107E-002,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.6258828074500417E+00, 0.5589320748538995E+00, 0.7879513719234823E-20, 0.9438435755776795E-21, 0.6695139660044724E-01, 0.6641476135588615E-06, 0.1433084218388560E-09, 0.6424501268835132E-04, 0.2977366669211234E+00, 0.2307216893122929E+00, 0.8926852875783797E-01, 0.6695139660044724E-01, 0.2001510674733695E+01},
			i0:       1,
			n0:       3,
			pp:       1,
			n0in:     3,
			dmin:     6.6400430513702258E-007,
			dmin1:    6.6400430513702258E-007,
			dmin2:    0.55893207485389951,
			dn:       0.23072168931229292,
			dn1:      6.6400430513702258E-007,
			dn2:      0.55893207485389951,
			tau:      6.6950732596142107E-002,
			ttype:    -4,
			g:        0.49975000000000003,
			zOut:     []float64{0.6258828074500417E+00, 0.5589320748538995E+00, 0.7879513719234823E-20, 0.9438435755776795E-21, 0.6695139660044724E-01, 0.6641476135588615E-06, 0.1433084218388560E-09, 0.6424501268835132E-04, 0.2977366669211234E+00, 0.2307216893122929E+00, 0.8926852875783797E-01, 0.6695139660044724E-01, 0.2001510674733695E+01},
			tauOut:   6.6398937736467640E-007,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
		{
			z:        []float64{0.5589315065851642E+00, 0.5589320748538995E+00, 0.1121517486324177E-26, 0.9438435755776795E-21, 0.6434089156657428E-04, 0.6641476135588615E-06, 0.2303778747300831E+00, 0.6424501268835132E-04, 0.3432463134744483E-03, 0.2307216893122929E+00, 0.6641476135588615E-06, 0.6695139660044724E-01},
			i0:       1,
			n0:       3,
			pp:       0,
			n0in:     3,
			dmin:     9.5878878222950548E-008,
			dmin1:    9.5878878222950548E-008,
			dmin2:    0.55893150658516422,
			dn:       3.4324631347444829E-004,
			dn1:      9.5878878222950548E-008,
			dn2:      0.55893150658516422,
			tau:      5.6826873533591094E-007,
			ttype:    -15,
			g:        0.49975000000000003,
			zOut:     []float64{0.5589315065851642E+00, 0.5589320748538995E+00, 0.1121517486324177E-26, 0.9438435755776795E-21, 0.6434089156657428E-04, 0.6641476135588615E-06, 0.2303778747300831E+00, 0.6424501268835132E-04, 0.3432463134744483E-03, 0.2307216893122929E+00, 0.6641476135588615E-06, 0.6695139660044724E-01},
			tauOut:   9.4211902206835373E-008,
			ttypeOut: -4,
			gOut:     0.49975000000000003,
		},
	} {
		z := make([]float64, len(test.z))
		copy(z, test.z)

		// Print for fortran input
		//printDlasq4FortranInput(test)

		i0 := test.i0 - 1 // zero index
		n0 := test.n0 - 1 // zero index
		n0in := test.n0in - 1

		tauOut, ttypeOut, gOut := impl.Dlasq4(i0, n0, z, test.pp, n0in, test.dmin, test.dmin1, test.dmin2, test.dn, test.dn1, test.dn2, test.tau, test.ttype, test.g)

		if !floats.EqualApprox(z, test.zOut, dTol) {
			t.Errorf("Z mismatch")
		}
		if !floats.EqualWithinAbsOrRel(tauOut, test.tauOut, dTol, dTol) {
			t.Errorf("Case %d, wrong tau. Want %v, got %v", cas, test.tauOut, tauOut)
		}
		if ttypeOut != test.ttypeOut {
			t.Errorf("Wrong ttype. Want %v, got %v", test.ttypeOut, ttypeOut)
		}
		if !floats.EqualWithinAbsOrRel(gOut, test.gOut, dTol, dTol) {
			t.Errorf("Wrong g. Want %v, got %v", test.gOut, gOut)
		}
	}
}
