extern crate cc;
extern crate cmake;
#[cfg(windows)]
extern crate glob;

use std::env;

fn main() {
    let dst = cmake::Config::new("wabt")
        // Turn off building tests and tools. This not only speeds up the build but
        // also prevents building executables that for some reason are put
        // into the `wabt` directory which now is deemed as an error.
        //
        // Since that is all targets available, also don't specify target
        // (by default it is set to `--target install`).
        // Otherwise, there will be an error message "No rule to make target `install'".
        .define("BUILD_TESTS", "OFF")
        .define("BUILD_TOOLS", "OFF")
        .no_build_target(true)
        .build();

    let mut out_build_dir = dst;
    out_build_dir.push("build");

    println!("cargo:rustc-link-search=native={}", out_build_dir.display());

    // help cargo find wabt.lib when targeting windows
    #[cfg(windows)]
    {
        let pattern = format!("{}/*/wabt.lib", out_build_dir.display());
        for entry in glob::glob(&pattern).unwrap() {
            if let Ok(path) = entry {
                let out_lib_dir = path.parent().unwrap().to_path_buf();
                println!("cargo:rustc-link-search=native={}", out_lib_dir.display());
                break;
            }
        }
    }

    println!("cargo:rustc-link-lib=static=wabt");

    // We need to link against C++ std lib
    if let Some(cpp_stdlib) = get_cpp_stdlib() {
        println!("cargo:rustc-link-lib={}", cpp_stdlib);
    }

    let mut cfg = cc::Build::new();
    cfg.file("wabt/src/emscripten-helpers.cc")
        .file("wabt_shim.cc")
        .include("wabt")
        // This is needed for config.h generated by cmake.
        .include(out_build_dir)
        // We link to stdlib above.
        .cpp_link_stdlib(None)
        .warnings(false)
        .cpp(true)
        .flag("-std=c++11")
        .compile("wabt_shim");
}

// See https://github.com/alexcrichton/gcc-rs/blob/88ac58e25/src/lib.rs#L1197
fn get_cpp_stdlib() -> Option<String> {
    env::var("TARGET").ok().and_then(|target| {
        if target.contains("msvc") {
            None
        } else if target.contains("darwin") {
            Some("c++".to_string())
        } else if target.contains("freebsd") {
            Some("c++".to_string())
        } else if target.contains("musl") {
            Some("static=stdc++".to_string())
        } else {
            Some("stdc++".to_string())
        }
    })
}
