# RFC 3603 - 2020-08-27 - Collecting metrics from MongoDB

This RFC is to introduce a new metrics source to consume metrics from MongoDB database servers. The high level plan is to implement one source that collects metrics from MongoDB server instances.

Background reading on MongoDB monitoring:

- https://docs.mongodb.com/manual/administration/monitoring/

## Scope

This RFC will cover:

- A new source for MongoDB server metrics.

This RFC will not cover:

- Other databases.

## Motivation

Users want to collect, transform, and forward metrics to better observe how their MongoDB databases are performing.

## Internal Proposal

Build a single source called `mongodb_metrics` (name to be confirmed) to collect MongoDB metrics. We support all non-EOL'ed MongoDB versions.

The recommended implementation is to use the Rust MongoDB client to connect the target database server by the address specified in configuration.

Proposed client:

- https://github.com/mongodb/mongo-rust-driver
- https://crates.io/crates/mongodb

We'd call the following on the server from the client upon connection:

- [`serverStatus`](https://docs.mongodb.com/manual/reference/command/serverStatus/)
- [`printShardingStatus`](https://docs.mongodb.com/manual/reference/method/db.printShardingStatus/)
- [`shardConnPoolStats`](https://docs.mongodb.com/manual/reference/command/shardConnPoolStats/)
- [`dbStats`](https://docs.mongodb.com/manual/reference/command/dbStats/) on each DB
- [`collStats`](https://docs.mongodb.com/manual/reference/command/collStats/) per collection
- [`connPoolStats`](https://docs.mongodb.com/manual/reference/command/connPoolStats/) per pool
- [`Top`](https://docs.mongodb.com/manual/reference/command/top/)
- [`indexStats`](https://docs.mongodb.com/manual/reference/operator/aggregation/indexStats/#pipe._S_indexStats)
- [`replSetGetStatus`](https://docs.mongodb.com/manual/reference/command/replSetGetStatus/)

This should result in the following metrics being collected:

- `mongodb_up` -> Used as an uptime metric with 1 for successful collection and 0 for failed collection (gauge)
- `mongodb_asserts_total` (counter)
- `mongodb_connections` (gauge)
- `mongodb_connections_metrics_created_total` (counter)
- `mongodb_extra_info_heap_usage_bytes` (gauge)
- `mongodb_extra_info_page_faults` (gauge)
- `mongodb_instance_local_time` (gauge)
- `mongodb_instance_uptime_estimate_seconds_total` (counter)
- `mongodb_instance_uptime_seconds_total` (counter)
- `mongodb_memory` tagged with type (gauge)
- `mongodb_mongod_global_lock_client` tagged with type (gauge)
- `mongodb_mongod_global_lock_current_queue` tagged with type (gauge)
- `mongodb_mongod_global_lock_ratio` (gauge)
- `mongodb_mongod_global_lock_total` (counter)
- `mongodb_mongod_locks_time_acquiring_global_seconds_total` tagged with database (counter)
- `mongodb_mongod_locks_time_locked_global_seconds_total` tagged with database and type (counter)
- `mongodb_mongod_locks_time_locked_local_seconds_total` tagged with database and type (counter)
- `mongodb_mongod_metrics_cursor_open` tagged with state (gauge)
- `mongodb_mongod_metrics_cursor_timed_out_total` (counter)
- `mongodb_mongod_metrics_document_total` tagged with state (counter)
- `mongodb_mongod_metrics_get_last_error_wtime_num` (gauge)
- `mongodb_mongod_metrics_get_last_error_wtime_seconds_total` (counter)
- `mongodb_mongod_metrics_get_last_error_wtimeouts_total` (counter)
- `mongodb_mongod_metrics_operation_total` tagged with type (counter)
- `mongodb_mongod_metrics_query_executor_total` tagged with state (counter)
- `mongodb_mongod_metrics_record_moves_total` (counter)
- `mongodb_mongod_metrics_repl_apply_batches_num_total` (counter)
- `mongodb_mongod_metrics_repl_apply_batches_seconds_total` (counter)
- `mongodb_mongod_metrics_repl_apply_ops_total` (counter)
- `mongodb_mongod_metrics_repl_buffer_count` (gauge)
- `mongodb_mongod_metrics_repl_buffer_max_size_bytes_total` (counter)
- `mongodb_mongod_metrics_repl_buffer_size_bytes` (gauge)
- `mongodb_mongod_metrics_repl_executor_event_waiters` (gauge)
- `mongodb_mongod_metrics_repl_executor_queue` tagged with type (gauge)
- `mongodb_mongod_metrics_repl_executor_unsignaled_events` (gauge)
- `mongodb_mongod_metrics_repl_network_bytes_total` (counter)
- `mongodb_mongod_metrics_repl_network_getmores_num_total` (counter)
- `mongodb_mongod_metrics_repl_network_getmores_seconds_total` (counter)
- `mongodb_mongod_metrics_repl_network_ops_total` (counter)
- `mongodb_mongod_metrics_repl_network_readers_created_total` (counter)
- `mongodb_mongod_metrics_repl_oplog_insert_bytes_total` (counter)
- `mongodb_mongod_metrics_repl_oplog_insert_num_total` (counter)
- `mongodb_mongod_metrics_repl_oplog_insert_seconds_total` (counter)
- `mongodb_mongod_metrics_ttl_deleted_documents_total` (counter)
- `mongodb_mongod_metrics_ttl_passes_total` (counter)
- `mongodb_mongod_op_latencies_histogram` tagged with micros and type (gauge)
- `mongodb_mongod_op_latencies_latency` tagged with type (gauge)
- `mongodb_mongod_op_latencies_ops_total` tagged with type (gauge)
- `mongodb_mongod_storage_engine` (gauge)
- `mongodb_mongod_wiredtiger_blockmanager_blocks_total` tagged with type (counter)
- `mongodb_mongod_wiredtiger_blockmanager_bytes_total` tagged with type (counter)
- `mongodb_mongod_wiredtiger_cache_bytes` tagged with type (gauge)
- `mongodb_mongod_wiredtiger_cache_bytes_total` tagged with type (counter)
- `mongodb_mongod_wiredtiger_cache_evicted_total` tagged with type (counter)
- `mongodb_mongod_wiredtiger_cache_max_bytes` (gauge)
- `mongodb_mongod_wiredtiger_cache_overhead_percent` (gauge)
- `mongodb_mongod_wiredtiger_cache_pages` tagged with type (gauge)
- `mongodb_mongod_wiredtiger_cache_pages_total` tagged with type (counter)
- `mongodb_mongod_wiredtiger_concurrent_transactions_available_tickets` tagged with type (gauge)
- `mongodb_mongod_wiredtiger_concurrent_transactions_out_tickets` tagged with type (gauge)
- `mongodb_mongod_wiredtiger_concurrent_transactions_total_tickets` tagged with type (gauge)
- `mongodb_mongod_wiredtiger_log_bytes_total` tagged with type (counter)
- `mongodb_mongod_wiredtiger_log_operations_total` tagged with type (counter)
- `mongodb_mongod_wiredtiger_log_records_scanned_total` (counter)
- `mongodb_mongod_wiredtiger_log_records_total` tagged with type (counter)
- `mongodb_mongod_wiredtiger_session_open_cursors` (gauge)
- `mongodb_mongod_wiredtiger_session_open_sessions` (gauge)
- `mongodb_mongod_wiredtiger_transactions_checkpoint_seconds` tagged with type (gauge)
- `mongodb_mongod_wiredtiger_transactions_checkpoint_seconds_total` (counter)
- `mongodb_mongod_wiredtiger_transactions_running_checkpoints` (gauge)
- `mongodb_mongod_wiredtiger_transactions_total` tagged with type (counter)
- `mongodb_network_bytes_total` tagged with state (counter)
- `mongodb_network_metrics_num_requests_total` (counter)
- `mongodb_op_counters_repl_total` tagged with type (counter)
- `mongodb_op_counters_total` tagged with type (counter)

All metrics will also be tagged with the `endpoint` (stripped of user name & password if authentication is used) and `host`.

## Doc-level Proposal

The following additional source configuration will be added:

```toml
[sources.my_source_id]
  type = "mongodb_metrics" # required
  endpoint = "mongodb://mongodb_exporter:s3cr3tpassw0rd@localhost:27017" # required - address of the MongoDBserver.
  scrape_interval_secs = 15 # optional, default, seconds
  namespace = "mongodb" # optional, default is "mongodb", namespace to attach to metrics.
```

We will also expose the TLS settings and support `tls` in the `endpoint` URL.

- We'd also add a guide for doing this with MongoDB Authorization.

## Rationale

MongoDB is a commonly adopted modern database. Users frequently want to monitor its state and performance.

Additionally, as part of Vector's vision to be the "one tool" for ingesting and shipping observability data, it makes sense to add as many sources as possible to reduce the likelihood that a user will not be able to ingest metrics from their tools.

## Prior Art

- https://github.com/dcu/mongodb_exporter
- https://github.com/percona/mongodb_exporter
- https://github.com/influxdata/telegraf/tree/master/plugins/inputs/mongodb
- https://collectd.org/wiki/index.php/Plugin:MongoDB
- https://docs.datadoghq.com/integrations/mongo/?tab=host

### mongodb_exporter output

```text
# HELP go_gc_duration_seconds A summary of the pause duration of garbage collection cycles.
# TYPE go_gc_duration_seconds summary
go_gc_duration_seconds{quantile="0"} 0
go_gc_duration_seconds{quantile="0.25"} 0
go_gc_duration_seconds{quantile="0.5"} 0
go_gc_duration_seconds{quantile="0.75"} 0
go_gc_duration_seconds{quantile="1"} 0
go_gc_duration_seconds_sum 0
go_gc_duration_seconds_count 0
# HELP go_goroutines Number of goroutines that currently exist.
# TYPE go_goroutines gauge
go_goroutines 9
# HELP go_info Information about the Go environment.
# TYPE go_info gauge
go_info{version="go1.15"} 1
# HELP go_memstats_alloc_bytes Number of bytes allocated and still in use.
# TYPE go_memstats_alloc_bytes gauge
go_memstats_alloc_bytes 2.585e+06
# HELP go_memstats_alloc_bytes_total Total number of bytes allocated, even if freed.
# TYPE go_memstats_alloc_bytes_total counter
go_memstats_alloc_bytes_total 2.585e+06
# HELP go_memstats_buck_hash_sys_bytes Number of bytes used by the profiling bucket hash table.
# TYPE go_memstats_buck_hash_sys_bytes gauge
go_memstats_buck_hash_sys_bytes 1.445082e+06
# HELP go_memstats_frees_total Total number of frees.
# TYPE go_memstats_frees_total counter
go_memstats_frees_total 2401
# HELP go_memstats_gc_cpu_fraction The fraction of this program's available CPU time used by the GC since the program started.
# TYPE go_memstats_gc_cpu_fraction gauge
go_memstats_gc_cpu_fraction 0
# HELP go_memstats_gc_sys_bytes Number of bytes used for garbage collection system metadata.
# TYPE go_memstats_gc_sys_bytes gauge
go_memstats_gc_sys_bytes 4.171848e+06
# HELP go_memstats_heap_alloc_bytes Number of heap bytes allocated and still in use.
# TYPE go_memstats_heap_alloc_bytes gauge
go_memstats_heap_alloc_bytes 2.585e+06
# HELP go_memstats_heap_idle_bytes Number of heap bytes waiting to be used.
# TYPE go_memstats_heap_idle_bytes gauge
go_memstats_heap_idle_bytes 6.2349312e+07
# HELP go_memstats_heap_inuse_bytes Number of heap bytes that are in use.
# TYPE go_memstats_heap_inuse_bytes gauge
go_memstats_heap_inuse_bytes 4.13696e+06
# HELP go_memstats_heap_objects Number of allocated objects.
# TYPE go_memstats_heap_objects gauge
go_memstats_heap_objects 17419
# HELP go_memstats_heap_released_bytes Number of heap bytes released to OS.
# TYPE go_memstats_heap_released_bytes gauge
go_memstats_heap_released_bytes 6.2316544e+07
# HELP go_memstats_heap_sys_bytes Number of heap bytes obtained from system.
# TYPE go_memstats_heap_sys_bytes gauge
go_memstats_heap_sys_bytes 6.6486272e+07
# HELP go_memstats_last_gc_time_seconds Number of seconds since 1970 of last garbage collection.
# TYPE go_memstats_last_gc_time_seconds gauge
go_memstats_last_gc_time_seconds 0
# HELP go_memstats_lookups_total Total number of pointer lookups.
# TYPE go_memstats_lookups_total counter
go_memstats_lookups_total 0
# HELP go_memstats_mallocs_total Total number of mallocs.
# TYPE go_memstats_mallocs_total counter
go_memstats_mallocs_total 19820
# HELP go_memstats_mcache_inuse_bytes Number of bytes in use by mcache structures.
# TYPE go_memstats_mcache_inuse_bytes gauge
go_memstats_mcache_inuse_bytes 13888
# HELP go_memstats_mcache_sys_bytes Number of bytes used for mcache structures obtained from system.
# TYPE go_memstats_mcache_sys_bytes gauge
go_memstats_mcache_sys_bytes 16384
# HELP go_memstats_mspan_inuse_bytes Number of bytes in use by mspan structures.
# TYPE go_memstats_mspan_inuse_bytes gauge
go_memstats_mspan_inuse_bytes 72216
# HELP go_memstats_mspan_sys_bytes Number of bytes used for mspan structures obtained from system.
# TYPE go_memstats_mspan_sys_bytes gauge
go_memstats_mspan_sys_bytes 81920
# HELP go_memstats_next_gc_bytes Number of heap bytes when next garbage collection will take place.
# TYPE go_memstats_next_gc_bytes gauge
go_memstats_next_gc_bytes 4.473924e+06
# HELP go_memstats_other_sys_bytes Number of bytes used for other system allocations.
# TYPE go_memstats_other_sys_bytes gauge
go_memstats_other_sys_bytes 1.052638e+06
# HELP go_memstats_stack_inuse_bytes Number of bytes in use by the stack allocator.
# TYPE go_memstats_stack_inuse_bytes gauge
go_memstats_stack_inuse_bytes 622592
# HELP go_memstats_stack_sys_bytes Number of bytes obtained from system for stack allocator.
# TYPE go_memstats_stack_sys_bytes gauge
go_memstats_stack_sys_bytes 622592
# HELP go_memstats_sys_bytes Number of bytes obtained from system.
# TYPE go_memstats_sys_bytes gauge
go_memstats_sys_bytes 7.3876736e+07
# HELP go_threads Number of OS threads created.
# TYPE go_threads gauge
go_threads 8
# HELP mongodb_asserts_total The asserts document reports the number of asserts on the database. While assert errors are typically uncommon, if there are non-zero values for the asserts, you should check the log file for the mongod process for more information. In many cases these errors are trivial, but are worth investigating.
# TYPE mongodb_asserts_total counter
mongodb_asserts_total{type="msg"} 0
mongodb_asserts_total{type="regular"} 0
mongodb_asserts_total{type="rollovers"} 0
mongodb_asserts_total{type="user"} 4
mongodb_asserts_total{type="warning"} 0
# HELP mongodb_connections The connections sub document data regarding the current status of incoming connections and availability of the database server. Use these values to assess the current load and capacity requirements of the server
# TYPE mongodb_connections gauge
mongodb_connections{state="active"} 2
mongodb_connections{state="available"} 201
mongodb_connections{state="current"} 3
# HELP mongodb_connections_metrics_created_total totalCreated provides a count of all incoming connections created to the server. This number includes connections that have since closed
# TYPE mongodb_connections_metrics_created_total counter
mongodb_connections_metrics_created_total 3
# HELP mongodb_exporter_build_info A metric with a constant '1' value labeled by version, revision, branch, and goversion from which mongodb_exporter was built.
# TYPE mongodb_exporter_build_info gauge
mongodb_exporter_build_info{branch="",goversion="go1.15",revision="",version=""} 1
# HELP mongodb_exporter_last_scrape_duration_seconds Duration of the last scrape of metrics from MongoDB.
# TYPE mongodb_exporter_last_scrape_duration_seconds gauge
mongodb_exporter_last_scrape_duration_seconds 0.003769113
# HELP mongodb_exporter_last_scrape_error Whether the last scrape of metrics from MongoDB resulted in an error (1 for error, 0 for success).
# TYPE mongodb_exporter_last_scrape_error gauge
mongodb_exporter_last_scrape_error 0
# HELP mongodb_exporter_scrape_errors_total Total number of times an error occurred scraping a MongoDB.
# TYPE mongodb_exporter_scrape_errors_total counter
mongodb_exporter_scrape_errors_total 0
# HELP mongodb_exporter_scrapes_total Total number of times MongoDB was scraped for metrics.
# TYPE mongodb_exporter_scrapes_total counter
mongodb_exporter_scrapes_total 2
# HELP mongodb_extra_info_heap_usage_bytes The heap_usage_bytes field is only available on Unix/Linux systems, and reports the total size in bytes of heap space used by the database process
# TYPE mongodb_extra_info_heap_usage_bytes gauge
mongodb_extra_info_heap_usage_bytes 0
# HELP mongodb_extra_info_page_faults_total The page_faults Reports the total number of page faults that require disk operations. Page faults refer to operations that require the database server to access data which isn’t available in active memory. The page_faults counter may increase dramatically during moments of poor performance and may correlate with limited memory environments and larger data sets. Limited and sporadic page faults do not necessarily indicate an issue
# TYPE mongodb_extra_info_page_faults_total gauge
mongodb_extra_info_page_faults_total 912
# HELP mongodb_instance_local_time The localTime value is the current time, according to the server, in UTC specified in an ISODate format.
# TYPE mongodb_instance_local_time gauge
mongodb_instance_local_time 1.600833548e+09
# HELP mongodb_instance_uptime_estimate_seconds uptimeEstimate provides the uptime as calculated from MongoDB's internal course-grained time keeping system.
# TYPE mongodb_instance_uptime_estimate_seconds counter
mongodb_instance_uptime_estimate_seconds 35
# HELP mongodb_instance_uptime_seconds The value of the uptime field corresponds to the number of seconds that the mongos or mongod process has been active.
# TYPE mongodb_instance_uptime_seconds counter
mongodb_instance_uptime_seconds 35
# HELP mongodb_memory The mem data structure holds information regarding the target system architecture of mongod and current memory use
# TYPE mongodb_memory gauge
mongodb_memory{type="mapped"} 0
mongodb_memory{type="mapped_with_journal"} 0
mongodb_memory{type="resident"} 36
mongodb_memory{type="virtual"} 6385
# HELP mongodb_mongod_global_lock_client The activeClients data structure provides more granular information about the number of connected clients and the operation types (e.g. read or write) performed by these clients
# TYPE mongodb_mongod_global_lock_client gauge
mongodb_mongod_global_lock_client{type="reader"} 0
mongodb_mongod_global_lock_client{type="writer"} 0
# HELP mongodb_mongod_global_lock_current_queue The currentQueue data structure value provides more granular information concerning the number of operations queued because of a lock
# TYPE mongodb_mongod_global_lock_current_queue gauge
mongodb_mongod_global_lock_current_queue{type="reader"} 0
mongodb_mongod_global_lock_current_queue{type="writer"} 0
# HELP mongodb_mongod_global_lock_ratio The value of ratio displays the relationship between lockTime and totalTime. Low values indicate that operations have held the globalLock frequently for shorter periods of time. High values indicate that operations have held globalLock infrequently for longer periods of time
# TYPE mongodb_mongod_global_lock_ratio gauge
mongodb_mongod_global_lock_ratio 0
# HELP mongodb_mongod_global_lock_total The value of totalTime represents the time, in microseconds, since the database last started and creation of the globalLock. This is roughly equivalent to total server uptime
# TYPE mongodb_mongod_global_lock_total counter
mongodb_mongod_global_lock_total 0
# HELP mongodb_mongod_locks_time_acquiring_global_microseconds_total amount of time in microseconds that any database has spent waiting for the global lock
# TYPE mongodb_mongod_locks_time_acquiring_global_microseconds_total counter
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="Collection",type="read"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="Collection",type="write"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="Database",type="read"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="Database",type="write"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="Global",type="read"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="Global",type="write"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="Mutex",type="read"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="Mutex",type="write"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="ParallelBatchWriterMode",type="read"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="ParallelBatchWriterMode",type="write"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="ReplicationStateTransition",type="read"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="ReplicationStateTransition",type="write"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="oplog",type="read"} 0
mongodb_mongod_locks_time_acquiring_global_microseconds_total{database="oplog",type="write"} 0
# HELP mongodb_mongod_locks_time_locked_global_microseconds_total amount of time in microseconds that any database has held the global lock
# TYPE mongodb_mongod_locks_time_locked_global_microseconds_total counter
mongodb_mongod_locks_time_locked_global_microseconds_total{database="Collection",type="read"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="Collection",type="write"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="Database",type="read"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="Database",type="write"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="Global",type="read"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="Global",type="write"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="Mutex",type="read"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="Mutex",type="write"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="ParallelBatchWriterMode",type="read"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="ParallelBatchWriterMode",type="write"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="ReplicationStateTransition",type="read"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="ReplicationStateTransition",type="write"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="oplog",type="read"} 0
mongodb_mongod_locks_time_locked_global_microseconds_total{database="oplog",type="write"} 0
# HELP mongodb_mongod_locks_time_locked_local_microseconds_total amount of time in microseconds that any database has held the local lock
# TYPE mongodb_mongod_locks_time_locked_local_microseconds_total counter
mongodb_mongod_locks_time_locked_local_microseconds_total{database="Collection",type="read"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="Collection",type="write"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="Database",type="read"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="Database",type="write"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="Global",type="read"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="Global",type="write"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="Mutex",type="read"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="Mutex",type="write"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="ParallelBatchWriterMode",type="read"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="ParallelBatchWriterMode",type="write"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="ReplicationStateTransition",type="read"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="ReplicationStateTransition",type="write"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="oplog",type="read"} 0
mongodb_mongod_locks_time_locked_local_microseconds_total{database="oplog",type="write"} 0
# HELP mongodb_mongod_metrics_cursor_open The open is an embedded document that contains data regarding open cursors
# TYPE mongodb_mongod_metrics_cursor_open gauge
mongodb_mongod_metrics_cursor_open{state="noTimeout"} 0
mongodb_mongod_metrics_cursor_open{state="pinned"} 0
mongodb_mongod_metrics_cursor_open{state="total"} 0
# HELP mongodb_mongod_metrics_cursor_timed_out_total timedOut provides the total number of cursors that have timed out since the server process started. If this number is large or growing at a regular rate, this may indicate an application error
# TYPE mongodb_mongod_metrics_cursor_timed_out_total counter
mongodb_mongod_metrics_cursor_timed_out_total 0
# HELP mongodb_mongod_metrics_document_total The document holds a document of that reflect document access and modification patterns and data use. Compare these values to the data in the opcounters document, which track total number of operations
# TYPE mongodb_mongod_metrics_document_total counter
mongodb_mongod_metrics_document_total{state="deleted"} 0
mongodb_mongod_metrics_document_total{state="inserted"} 0
mongodb_mongod_metrics_document_total{state="returned"} 0
mongodb_mongod_metrics_document_total{state="updated"} 0
# HELP mongodb_mongod_metrics_get_last_error_wtime_num_total num reports the total number of getLastError operations with a specified write concern (i.e. w) that wait for one or more members of a replica set to acknowledge the write operation (i.e. a w value greater than 1.)
# TYPE mongodb_mongod_metrics_get_last_error_wtime_num_total gauge
mongodb_mongod_metrics_get_last_error_wtime_num_total 0
# HELP mongodb_mongod_metrics_get_last_error_wtime_total_milliseconds total_millis reports the total amount of time in milliseconds that the mongod has spent performing getLastError operations with write concern (i.e. w) that wait for one or more members of a replica set to acknowledge the write operation (i.e. a w value greater than 1.)
# TYPE mongodb_mongod_metrics_get_last_error_wtime_total_milliseconds counter
mongodb_mongod_metrics_get_last_error_wtime_total_milliseconds 0
# HELP mongodb_mongod_metrics_get_last_error_wtimeouts_total wtimeouts reports the number of times that write concern operations have timed out as a result of the wtimeout threshold to getLastError.
# TYPE mongodb_mongod_metrics_get_last_error_wtimeouts_total counter
mongodb_mongod_metrics_get_last_error_wtimeouts_total 0
# HELP mongodb_mongod_metrics_operation_total operation is a sub-document that holds counters for several types of update and query operations that MongoDB handles using special operation types
# TYPE mongodb_mongod_metrics_operation_total counter
mongodb_mongod_metrics_operation_total{type="fastmod"} 0
mongodb_mongod_metrics_operation_total{type="idhack"} 0
mongodb_mongod_metrics_operation_total{type="scan_and_order"} 0
# HELP mongodb_mongod_metrics_query_executor_total queryExecutor is a document that reports data from the query execution system
# TYPE mongodb_mongod_metrics_query_executor_total counter
mongodb_mongod_metrics_query_executor_total{state="scanned"} 0
mongodb_mongod_metrics_query_executor_total{state="scanned_objects"} 0
# HELP mongodb_mongod_metrics_record_moves_total moves reports the total number of times documents move within the on-disk representation of the MongoDB data set. Documents move as a result of operations that increase the size of the document beyond their allocated record size
# TYPE mongodb_mongod_metrics_record_moves_total counter
mongodb_mongod_metrics_record_moves_total 0
# HELP mongodb_mongod_metrics_repl_apply_batches_num_total num reports the total number of batches applied across all databases
# TYPE mongodb_mongod_metrics_repl_apply_batches_num_total counter
mongodb_mongod_metrics_repl_apply_batches_num_total 0
# HELP mongodb_mongod_metrics_repl_apply_batches_total_milliseconds total_millis reports the total amount of time the mongod has spent applying operations from the oplog
# TYPE mongodb_mongod_metrics_repl_apply_batches_total_milliseconds counter
mongodb_mongod_metrics_repl_apply_batches_total_milliseconds 0
# HELP mongodb_mongod_metrics_repl_apply_ops_total ops reports the total number of oplog operations applied
# TYPE mongodb_mongod_metrics_repl_apply_ops_total counter
mongodb_mongod_metrics_repl_apply_ops_total 0
# HELP mongodb_mongod_metrics_repl_buffer_count count reports the current number of operations in the oplog buffer
# TYPE mongodb_mongod_metrics_repl_buffer_count gauge
mongodb_mongod_metrics_repl_buffer_count 0
# HELP mongodb_mongod_metrics_repl_buffer_max_size_bytes maxSizeBytes reports the maximum size of the buffer. This value is a constant setting in the mongod, and is not configurable
# TYPE mongodb_mongod_metrics_repl_buffer_max_size_bytes counter
mongodb_mongod_metrics_repl_buffer_max_size_bytes 0
# HELP mongodb_mongod_metrics_repl_buffer_size_bytes sizeBytes reports the current size of the contents of the oplog buffer
# TYPE mongodb_mongod_metrics_repl_buffer_size_bytes gauge
mongodb_mongod_metrics_repl_buffer_size_bytes 0
# HELP mongodb_mongod_metrics_repl_executor_event_waiters number of event waiters in the replication executor
# TYPE mongodb_mongod_metrics_repl_executor_event_waiters gauge
mongodb_mongod_metrics_repl_executor_event_waiters 0
# HELP mongodb_mongod_metrics_repl_executor_queue number of queued operations in the replication executor
# TYPE mongodb_mongod_metrics_repl_executor_queue gauge
mongodb_mongod_metrics_repl_executor_queue{type="networkInProgress"} 0
mongodb_mongod_metrics_repl_executor_queue{type="sleepers"} 0
# HELP mongodb_mongod_metrics_repl_executor_unsignaled_events number of unsignaled events in the replication executor
# TYPE mongodb_mongod_metrics_repl_executor_unsignaled_events gauge
mongodb_mongod_metrics_repl_executor_unsignaled_events 0
# HELP mongodb_mongod_metrics_repl_network_bytes_total bytes reports the total amount of data read from the replication sync source
# TYPE mongodb_mongod_metrics_repl_network_bytes_total counter
mongodb_mongod_metrics_repl_network_bytes_total 0
# HELP mongodb_mongod_metrics_repl_network_getmores_num_total num reports the total number of getmore operations, which are operations that request an additional set of operations from the replication sync source.
# TYPE mongodb_mongod_metrics_repl_network_getmores_num_total counter
mongodb_mongod_metrics_repl_network_getmores_num_total 0
# HELP mongodb_mongod_metrics_repl_network_getmores_total_milliseconds total_millis reports the total amount of time required to collect data from getmore operations
# TYPE mongodb_mongod_metrics_repl_network_getmores_total_milliseconds counter
mongodb_mongod_metrics_repl_network_getmores_total_milliseconds 0
# HELP mongodb_mongod_metrics_repl_network_ops_total ops reports the total number of operations read from the replication source.
# TYPE mongodb_mongod_metrics_repl_network_ops_total counter
mongodb_mongod_metrics_repl_network_ops_total 0
# HELP mongodb_mongod_metrics_repl_network_readers_created_total readersCreated reports the total number of oplog query processes created. MongoDB will create a new oplog query any time an error occurs in the connection, including a timeout, or a network operation. Furthermore, readersCreated will increment every time MongoDB selects a new source fore replication.
# TYPE mongodb_mongod_metrics_repl_network_readers_created_total counter
mongodb_mongod_metrics_repl_network_readers_created_total 0
# HELP mongodb_mongod_metrics_repl_oplog_insert_bytes_total insertBytes the total size of documents inserted into the oplog.
# TYPE mongodb_mongod_metrics_repl_oplog_insert_bytes_total counter
mongodb_mongod_metrics_repl_oplog_insert_bytes_total 0
# HELP mongodb_mongod_metrics_repl_oplog_insert_num_total num reports the total number of items inserted into the oplog.
# TYPE mongodb_mongod_metrics_repl_oplog_insert_num_total counter
mongodb_mongod_metrics_repl_oplog_insert_num_total 0
# HELP mongodb_mongod_metrics_repl_oplog_insert_total_milliseconds total_millis reports the total amount of time spent for the mongod to insert data into the oplog.
# TYPE mongodb_mongod_metrics_repl_oplog_insert_total_milliseconds counter
mongodb_mongod_metrics_repl_oplog_insert_total_milliseconds 0
# HELP mongodb_mongod_metrics_ttl_deleted_documents_total deletedDocuments reports the total number of documents deleted from collections with a ttl index.
# TYPE mongodb_mongod_metrics_ttl_deleted_documents_total counter
mongodb_mongod_metrics_ttl_deleted_documents_total 0
# HELP mongodb_mongod_metrics_ttl_passes_total passes reports the number of times the background process removes documents from collections with a ttl index
# TYPE mongodb_mongod_metrics_ttl_passes_total counter
mongodb_mongod_metrics_ttl_passes_total 0
# HELP mongodb_mongod_op_latencies_histogram op latencies histogram statistics of mongod
# TYPE mongodb_mongod_op_latencies_histogram gauge
mongodb_mongod_op_latencies_histogram{micros="128",type="command"} 2
mongodb_mongod_op_latencies_histogram{micros="16",type="command"} 4
mongodb_mongod_op_latencies_histogram{micros="256",type="command"} 1
mongodb_mongod_op_latencies_histogram{micros="32",type="command"} 3
# HELP mongodb_mongod_op_latencies_latency_total op latencies statistics in microseconds of mongod
# TYPE mongodb_mongod_op_latencies_latency_total gauge
mongodb_mongod_op_latencies_latency_total{type="command"} 1057
mongodb_mongod_op_latencies_latency_total{type="read"} 0
mongodb_mongod_op_latencies_latency_total{type="write"} 0
# HELP mongodb_mongod_op_latencies_ops_total op latencies ops total statistics of mongod
# TYPE mongodb_mongod_op_latencies_ops_total gauge
mongodb_mongod_op_latencies_ops_total{type="command"} 10
mongodb_mongod_op_latencies_ops_total{type="read"} 0
mongodb_mongod_op_latencies_ops_total{type="write"} 0
# HELP mongodb_mongod_storage_engine The storage engine used by the MongoDB instance
# TYPE mongodb_mongod_storage_engine counter
mongodb_mongod_storage_engine{engine="wiredTiger"} 1
# HELP mongodb_mongod_wiredtiger_blockmanager_blocks_total The total number of blocks read by the WiredTiger BlockManager
# TYPE mongodb_mongod_wiredtiger_blockmanager_blocks_total counter
mongodb_mongod_wiredtiger_blockmanager_blocks_total{type="pre_loaded"} 0
mongodb_mongod_wiredtiger_blockmanager_blocks_total{type="read"} 0
mongodb_mongod_wiredtiger_blockmanager_blocks_total{type="read_mapped"} 0
mongodb_mongod_wiredtiger_blockmanager_blocks_total{type="written"} 0
# HELP mongodb_mongod_wiredtiger_blockmanager_bytes_total The total number of bytes read by the WiredTiger BlockManager
# TYPE mongodb_mongod_wiredtiger_blockmanager_bytes_total counter
mongodb_mongod_wiredtiger_blockmanager_bytes_total{type="read"} 0
mongodb_mongod_wiredtiger_blockmanager_bytes_total{type="read_mapped"} 0
mongodb_mongod_wiredtiger_blockmanager_bytes_total{type="written"} 0
# HELP mongodb_mongod_wiredtiger_cache_bytes The current size of data in the WiredTiger Cache in bytes
# TYPE mongodb_mongod_wiredtiger_cache_bytes gauge
mongodb_mongod_wiredtiger_cache_bytes{type="dirty"} 27241
mongodb_mongod_wiredtiger_cache_bytes{type="internal_pages"} 1912
mongodb_mongod_wiredtiger_cache_bytes{type="leaf_pages"} 27242
mongodb_mongod_wiredtiger_cache_bytes{type="total"} 29154
# HELP mongodb_mongod_wiredtiger_cache_bytes_total The total number of bytes read into/from the WiredTiger Cache
# TYPE mongodb_mongod_wiredtiger_cache_bytes_total counter
mongodb_mongod_wiredtiger_cache_bytes_total{type="read"} 0
mongodb_mongod_wiredtiger_cache_bytes_total{type="written"} 0
# HELP mongodb_mongod_wiredtiger_cache_evicted_total The total number of pages evicted from the WiredTiger Cache
# TYPE mongodb_mongod_wiredtiger_cache_evicted_total counter
mongodb_mongod_wiredtiger_cache_evicted_total{type="modified"} 0
mongodb_mongod_wiredtiger_cache_evicted_total{type="unmodified"} 0
# HELP mongodb_mongod_wiredtiger_cache_max_bytes The maximum size of data in the WiredTiger Cache in bytes
# TYPE mongodb_mongod_wiredtiger_cache_max_bytes gauge
mongodb_mongod_wiredtiger_cache_max_bytes 1.6642998272e+10
# HELP mongodb_mongod_wiredtiger_cache_overhead_percent The percentage overhead of the WiredTiger Cache
# TYPE mongodb_mongod_wiredtiger_cache_overhead_percent gauge
mongodb_mongod_wiredtiger_cache_overhead_percent 8
# HELP mongodb_mongod_wiredtiger_cache_pages The current number of pages in the WiredTiger Cache
# TYPE mongodb_mongod_wiredtiger_cache_pages gauge
mongodb_mongod_wiredtiger_cache_pages{type="dirty"} 7
mongodb_mongod_wiredtiger_cache_pages{type="total"} 18
# HELP mongodb_mongod_wiredtiger_cache_pages_total The total number of pages read into/from the WiredTiger Cache
# TYPE mongodb_mongod_wiredtiger_cache_pages_total counter
mongodb_mongod_wiredtiger_cache_pages_total{type="read"} 0
mongodb_mongod_wiredtiger_cache_pages_total{type="written"} 0
# HELP mongodb_mongod_wiredtiger_concurrent_transactions_available_tickets The number of tickets that are available in WiredTiger
# TYPE mongodb_mongod_wiredtiger_concurrent_transactions_available_tickets gauge
mongodb_mongod_wiredtiger_concurrent_transactions_available_tickets{type="read"} 127
mongodb_mongod_wiredtiger_concurrent_transactions_available_tickets{type="write"} 128
# HELP mongodb_mongod_wiredtiger_concurrent_transactions_out_tickets The number of tickets that are currently in use (out) in WiredTiger
# TYPE mongodb_mongod_wiredtiger_concurrent_transactions_out_tickets gauge
mongodb_mongod_wiredtiger_concurrent_transactions_out_tickets{type="read"} 1
mongodb_mongod_wiredtiger_concurrent_transactions_out_tickets{type="write"} 0
# HELP mongodb_mongod_wiredtiger_concurrent_transactions_total_tickets The total number of tickets that are available in WiredTiger
# TYPE mongodb_mongod_wiredtiger_concurrent_transactions_total_tickets gauge
mongodb_mongod_wiredtiger_concurrent_transactions_total_tickets{type="read"} 128
mongodb_mongod_wiredtiger_concurrent_transactions_total_tickets{type="write"} 128
# HELP mongodb_mongod_wiredtiger_log_bytes_total The total number of bytes written to the WiredTiger log
# TYPE mongodb_mongod_wiredtiger_log_bytes_total counter
mongodb_mongod_wiredtiger_log_bytes_total{type="payload"} 13165
mongodb_mongod_wiredtiger_log_bytes_total{type="written"} 17792
# HELP mongodb_mongod_wiredtiger_log_operations_total The total number of WiredTiger log operations
# TYPE mongodb_mongod_wiredtiger_log_operations_total counter
mongodb_mongod_wiredtiger_log_operations_total{type="flush"} 327
mongodb_mongod_wiredtiger_log_operations_total{type="read"} 0
mongodb_mongod_wiredtiger_log_operations_total{type="scan"} 0
mongodb_mongod_wiredtiger_log_operations_total{type="scan_double"} 0
mongodb_mongod_wiredtiger_log_operations_total{type="sync"} 13
mongodb_mongod_wiredtiger_log_operations_total{type="sync_dir"} 1
mongodb_mongod_wiredtiger_log_operations_total{type="write"} 44
# HELP mongodb_mongod_wiredtiger_log_records_scanned_total The total number of records scanned by log scan in the WiredTiger log
# TYPE mongodb_mongod_wiredtiger_log_records_scanned_total counter
mongodb_mongod_wiredtiger_log_records_scanned_total 0
# HELP mongodb_mongod_wiredtiger_log_records_total The total number of compressed/uncompressed records written to the WiredTiger log
# TYPE mongodb_mongod_wiredtiger_log_records_total counter
mongodb_mongod_wiredtiger_log_records_total{type="compressed"} 13
mongodb_mongod_wiredtiger_log_records_total{type="uncompressed"} 16
# HELP mongodb_mongod_wiredtiger_session_open_cursors_total The total number of cursors opened in WiredTiger
# TYPE mongodb_mongod_wiredtiger_session_open_cursors_total gauge
mongodb_mongod_wiredtiger_session_open_cursors_total 0
# HELP mongodb_mongod_wiredtiger_session_open_sessions_total The total number of sessions opened in WiredTiger
# TYPE mongodb_mongod_wiredtiger_session_open_sessions_total gauge
mongodb_mongod_wiredtiger_session_open_sessions_total 15
# HELP mongodb_mongod_wiredtiger_transactions_checkpoint_milliseconds The time in milliseconds transactions have checkpointed in WiredTiger
# TYPE mongodb_mongod_wiredtiger_transactions_checkpoint_milliseconds gauge
mongodb_mongod_wiredtiger_transactions_checkpoint_milliseconds{type="max"} 0
mongodb_mongod_wiredtiger_transactions_checkpoint_milliseconds{type="min"} 0
# HELP mongodb_mongod_wiredtiger_transactions_checkpoint_milliseconds_total The total time in milliseconds transactions have checkpointed in WiredTiger
# TYPE mongodb_mongod_wiredtiger_transactions_checkpoint_milliseconds_total counter
mongodb_mongod_wiredtiger_transactions_checkpoint_milliseconds_total 0
# HELP mongodb_mongod_wiredtiger_transactions_running_checkpoints The number of currently running checkpoints in WiredTiger
# TYPE mongodb_mongod_wiredtiger_transactions_running_checkpoints gauge
mongodb_mongod_wiredtiger_transactions_running_checkpoints 0
# HELP mongodb_mongod_wiredtiger_transactions_total The total number of transactions WiredTiger has handled
# TYPE mongodb_mongod_wiredtiger_transactions_total counter
mongodb_mongod_wiredtiger_transactions_total{type="begins"} 12
mongodb_mongod_wiredtiger_transactions_total{type="checkpoints"} 0
mongodb_mongod_wiredtiger_transactions_total{type="committed"} 6
mongodb_mongod_wiredtiger_transactions_total{type="rolledback"} 6
# HELP mongodb_network_bytes_total The network data structure contains data regarding MongoDB’s network use
# TYPE mongodb_network_bytes_total counter
mongodb_network_bytes_total{state="in_bytes"} 1981
mongodb_network_bytes_total{state="out_bytes"} 47869
# HELP mongodb_network_metrics_num_requests_total The numRequests field is a counter of the total number of distinct requests that the server has received. Use this value to provide context for the bytesIn and bytesOut values to ensure that MongoDB’s network utilization is consistent with expectations and application use
# TYPE mongodb_network_metrics_num_requests_total counter
mongodb_network_metrics_num_requests_total 14
# HELP mongodb_op_counters_repl_total The opcountersRepl data structure, similar to the opcounters data structure, provides an overview of database replication operations by type and makes it possible to analyze the load on the replica in more granular manner. These values only appear when the current host has replication enabled
# TYPE mongodb_op_counters_repl_total counter
mongodb_op_counters_repl_total{type="command"} 0
mongodb_op_counters_repl_total{type="delete"} 0
mongodb_op_counters_repl_total{type="getmore"} 0
mongodb_op_counters_repl_total{type="insert"} 0
mongodb_op_counters_repl_total{type="query"} 0
mongodb_op_counters_repl_total{type="update"} 0
# HELP mongodb_op_counters_total The opcounters data structure provides an overview of database operations by type and makes it possible to analyze the load on the database in more granular manner. These numbers will grow over time and in response to database use. Analyze these values over time to track database utilization
# TYPE mongodb_op_counters_total counter
mongodb_op_counters_total{type="command"} 17
mongodb_op_counters_total{type="delete"} 0
mongodb_op_counters_total{type="getmore"} 0
mongodb_op_counters_total{type="insert"} 0
mongodb_op_counters_total{type="query"} 1
mongodb_op_counters_total{type="update"} 0
# HELP mongodb_up Whether MongoDB is up.
# TYPE mongodb_up gauge
mongodb_up 1
# HELP mongodb_version_info Software version information for mongodb process.
# TYPE mongodb_version_info gauge
mongodb_version_info{mongodb="4.4.0"} 1
# HELP promhttp_metric_handler_requests_in_flight Current number of scrapes being served.
# TYPE promhttp_metric_handler_requests_in_flight gauge
promhttp_metric_handler_requests_in_flight 1
# HELP promhttp_metric_handler_requests_total Total number of scrapes by HTTP status code.
# TYPE promhttp_metric_handler_requests_total counter
promhttp_metric_handler_requests_total{code="200"} 0
promhttp_metric_handler_requests_total{code="500"} 0
promhttp_metric_handler_requests_total{code="503"} 0
```

### telegraf mongodb output

In Prometheus format.

```text
# HELP go_gc_duration_seconds A summary of the pause duration of garbage collection cycles.
# TYPE go_gc_duration_seconds summary
go_gc_duration_seconds{quantile="0"} 2.8372e-05
go_gc_duration_seconds{quantile="0.25"} 2.8372e-05
go_gc_duration_seconds{quantile="0.5"} 0.00016285
go_gc_duration_seconds{quantile="0.75"} 0.00016426
go_gc_duration_seconds{quantile="1"} 0.00016426
go_gc_duration_seconds_sum 0.000355482
go_gc_duration_seconds_count 3
# HELP go_goroutines Number of goroutines that currently exist.
# TYPE go_goroutines gauge
go_goroutines 23
# HELP go_info Information about the Go environment.
# TYPE go_info gauge
go_info{version="go1.14.5"} 1
# HELP go_memstats_alloc_bytes Number of bytes allocated and still in use.
# TYPE go_memstats_alloc_bytes gauge
go_memstats_alloc_bytes 7.146072e+06
# HELP go_memstats_alloc_bytes_total Total number of bytes allocated, even if freed.
# TYPE go_memstats_alloc_bytes_total counter
go_memstats_alloc_bytes_total 1.0233104e+07
# HELP go_memstats_buck_hash_sys_bytes Number of bytes used by the profiling bucket hash table.
# TYPE go_memstats_buck_hash_sys_bytes gauge
go_memstats_buck_hash_sys_bytes 1.448715e+06
# HELP go_memstats_frees_total Total number of frees.
# TYPE go_memstats_frees_total counter
go_memstats_frees_total 32884
# HELP go_memstats_gc_cpu_fraction The fraction of this program's available CPU time used by the GC since the program started.
# TYPE go_memstats_gc_cpu_fraction gauge
go_memstats_gc_cpu_fraction 0.005923686998994504
# HELP go_memstats_gc_sys_bytes Number of bytes used for garbage collection system metadata.
# TYPE go_memstats_gc_sys_bytes gauge
go_memstats_gc_sys_bytes 3.57812e+06
# HELP go_memstats_heap_alloc_bytes Number of heap bytes allocated and still in use.
# TYPE go_memstats_heap_alloc_bytes gauge
go_memstats_heap_alloc_bytes 7.146072e+06
# HELP go_memstats_heap_idle_bytes Number of heap bytes waiting to be used.
# TYPE go_memstats_heap_idle_bytes gauge
go_memstats_heap_idle_bytes 5.7524224e+07
# HELP go_memstats_heap_inuse_bytes Number of heap bytes that are in use.
# TYPE go_memstats_heap_inuse_bytes gauge
go_memstats_heap_inuse_bytes 8.830976e+06
# HELP go_memstats_heap_objects Number of allocated objects.
# TYPE go_memstats_heap_objects gauge
go_memstats_heap_objects 28822
# HELP go_memstats_heap_released_bytes Number of heap bytes released to OS.
# TYPE go_memstats_heap_released_bytes gauge
go_memstats_heap_released_bytes 5.738496e+07
# HELP go_memstats_heap_sys_bytes Number of heap bytes obtained from system.
# TYPE go_memstats_heap_sys_bytes gauge
go_memstats_heap_sys_bytes 6.63552e+07
# HELP go_memstats_last_gc_time_seconds Number of seconds since 1970 of last garbage collection.
# TYPE go_memstats_last_gc_time_seconds gauge
go_memstats_last_gc_time_seconds 1.599788641219671e+09
# HELP go_memstats_lookups_total Total number of pointer lookups.
# TYPE go_memstats_lookups_total counter
go_memstats_lookups_total 0
# HELP go_memstats_mallocs_total Total number of mallocs.
# TYPE go_memstats_mallocs_total counter
go_memstats_mallocs_total 61706
# HELP go_memstats_mcache_inuse_bytes Number of bytes in use by mcache structures.
# TYPE go_memstats_mcache_inuse_bytes gauge
go_memstats_mcache_inuse_bytes 13888
# HELP go_memstats_mcache_sys_bytes Number of bytes used for mcache structures obtained from system.
# TYPE go_memstats_mcache_sys_bytes gauge
go_memstats_mcache_sys_bytes 16384
# HELP go_memstats_mspan_inuse_bytes Number of bytes in use by mspan structures.
# TYPE go_memstats_mspan_inuse_bytes gauge
go_memstats_mspan_inuse_bytes 124712
# HELP go_memstats_mspan_sys_bytes Number of bytes used for mspan structures obtained from system.
# TYPE go_memstats_mspan_sys_bytes gauge
go_memstats_mspan_sys_bytes 131072
# HELP go_memstats_next_gc_bytes Number of heap bytes when next garbage collection will take place.
# TYPE go_memstats_next_gc_bytes gauge
go_memstats_next_gc_bytes 9.927312e+06
# HELP go_memstats_other_sys_bytes Number of bytes used for other system allocations.
# TYPE go_memstats_other_sys_bytes gauge
go_memstats_other_sys_bytes 1.985773e+06
# HELP go_memstats_stack_inuse_bytes Number of bytes in use by the stack allocator.
# TYPE go_memstats_stack_inuse_bytes gauge
go_memstats_stack_inuse_bytes 753664
# HELP go_memstats_stack_sys_bytes Number of bytes obtained from system for stack allocator.
# TYPE go_memstats_stack_sys_bytes gauge
go_memstats_stack_sys_bytes 753664
# HELP go_memstats_sys_bytes Number of bytes obtained from system.
# TYPE go_memstats_sys_bytes gauge
go_memstats_sys_bytes 7.4268928e+07
# HELP go_threads Number of OS threads created.
# TYPE go_threads gauge
go_threads 14
# HELP mongodb_active_reads Telegraf collected metric
# TYPE mongodb_active_reads untyped
mongodb_active_reads{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 1
# HELP mongodb_active_writes Telegraf collected metric
# TYPE mongodb_active_writes untyped
mongodb_active_writes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_aggregate_command_failed Telegraf collected metric
# TYPE mongodb_aggregate_command_failed untyped
mongodb_aggregate_command_failed{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_aggregate_command_total Telegraf collected metric
# TYPE mongodb_aggregate_command_total untyped
mongodb_aggregate_command_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_assert_msg Telegraf collected metric
# TYPE mongodb_assert_msg untyped
mongodb_assert_msg{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_assert_regular Telegraf collected metric
# TYPE mongodb_assert_regular untyped
mongodb_assert_regular{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_assert_rollovers Telegraf collected metric
# TYPE mongodb_assert_rollovers untyped
mongodb_assert_rollovers{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_assert_user Telegraf collected metric
# TYPE mongodb_assert_user untyped
mongodb_assert_user{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 7
# HELP mongodb_assert_warning Telegraf collected metric
# TYPE mongodb_assert_warning untyped
mongodb_assert_warning{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_available_reads Telegraf collected metric
# TYPE mongodb_available_reads untyped
mongodb_available_reads{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 127
# HELP mongodb_available_writes Telegraf collected metric
# TYPE mongodb_available_writes untyped
mongodb_available_writes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 128
# HELP mongodb_commands Telegraf collected metric
# TYPE mongodb_commands untyped
mongodb_commands{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 20
# HELP mongodb_commands_per_sec Telegraf collected metric
# TYPE mongodb_commands_per_sec untyped
mongodb_commands_per_sec{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_connections_available Telegraf collected metric
# TYPE mongodb_connections_available untyped
mongodb_connections_available{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 838859
# HELP mongodb_connections_current Telegraf collected metric
# TYPE mongodb_connections_current untyped
mongodb_connections_current{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 1
# HELP mongodb_connections_total_created Telegraf collected metric
# TYPE mongodb_connections_total_created untyped
mongodb_connections_total_created{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 1
# HELP mongodb_count_command_failed Telegraf collected metric
# TYPE mongodb_count_command_failed untyped
mongodb_count_command_failed{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_count_command_total Telegraf collected metric
# TYPE mongodb_count_command_total untyped
mongodb_count_command_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 3
# HELP mongodb_cursor_no_timeout Telegraf collected metric
# TYPE mongodb_cursor_no_timeout untyped
mongodb_cursor_no_timeout{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_cursor_no_timeout_count Telegraf collected metric
# TYPE mongodb_cursor_no_timeout_count untyped
mongodb_cursor_no_timeout_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_cursor_pinned Telegraf collected metric
# TYPE mongodb_cursor_pinned untyped
mongodb_cursor_pinned{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_cursor_pinned_count Telegraf collected metric
# TYPE mongodb_cursor_pinned_count untyped
mongodb_cursor_pinned_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_cursor_timed_out Telegraf collected metric
# TYPE mongodb_cursor_timed_out untyped
mongodb_cursor_timed_out{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_cursor_timed_out_count Telegraf collected metric
# TYPE mongodb_cursor_timed_out_count untyped
mongodb_cursor_timed_out_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_cursor_total Telegraf collected metric
# TYPE mongodb_cursor_total untyped
mongodb_cursor_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_cursor_total_count Telegraf collected metric
# TYPE mongodb_cursor_total_count untyped
mongodb_cursor_total_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_delete_command_failed Telegraf collected metric
# TYPE mongodb_delete_command_failed untyped
mongodb_delete_command_failed{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_delete_command_total Telegraf collected metric
# TYPE mongodb_delete_command_total untyped
mongodb_delete_command_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_deletes Telegraf collected metric
# TYPE mongodb_deletes untyped
mongodb_deletes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_deletes_per_sec Telegraf collected metric
# TYPE mongodb_deletes_per_sec untyped
mongodb_deletes_per_sec{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_distinct_command_failed Telegraf collected metric
# TYPE mongodb_distinct_command_failed untyped
mongodb_distinct_command_failed{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_distinct_command_total Telegraf collected metric
# TYPE mongodb_distinct_command_total untyped
mongodb_distinct_command_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_document_deleted Telegraf collected metric
# TYPE mongodb_document_deleted untyped
mongodb_document_deleted{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_document_inserted Telegraf collected metric
# TYPE mongodb_document_inserted untyped
mongodb_document_inserted{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_document_returned Telegraf collected metric
# TYPE mongodb_document_returned untyped
mongodb_document_returned{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_document_updated Telegraf collected metric
# TYPE mongodb_document_updated untyped
mongodb_document_updated{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_find_and_modify_command_failed Telegraf collected metric
# TYPE mongodb_find_and_modify_command_failed untyped
mongodb_find_and_modify_command_failed{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_find_and_modify_command_total Telegraf collected metric
# TYPE mongodb_find_and_modify_command_total untyped
mongodb_find_and_modify_command_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_find_command_failed Telegraf collected metric
# TYPE mongodb_find_command_failed untyped
mongodb_find_command_failed{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_find_command_total Telegraf collected metric
# TYPE mongodb_find_command_total untyped
mongodb_find_command_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 1
# HELP mongodb_flushes Telegraf collected metric
# TYPE mongodb_flushes untyped
mongodb_flushes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_flushes_per_sec Telegraf collected metric
# TYPE mongodb_flushes_per_sec untyped
mongodb_flushes_per_sec{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_flushes_total_time_ns Telegraf collected metric
# TYPE mongodb_flushes_total_time_ns untyped
mongodb_flushes_total_time_ns{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_get_more_command_failed Telegraf collected metric
# TYPE mongodb_get_more_command_failed untyped
mongodb_get_more_command_failed{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_get_more_command_total Telegraf collected metric
# TYPE mongodb_get_more_command_total untyped
mongodb_get_more_command_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_getmores Telegraf collected metric
# TYPE mongodb_getmores untyped
mongodb_getmores{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_getmores_per_sec Telegraf collected metric
# TYPE mongodb_getmores_per_sec untyped
mongodb_getmores_per_sec{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_insert_command_failed Telegraf collected metric
# TYPE mongodb_insert_command_failed untyped
mongodb_insert_command_failed{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_insert_command_total Telegraf collected metric
# TYPE mongodb_insert_command_total untyped
mongodb_insert_command_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_inserts Telegraf collected metric
# TYPE mongodb_inserts untyped
mongodb_inserts{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_inserts_per_sec Telegraf collected metric
# TYPE mongodb_inserts_per_sec untyped
mongodb_inserts_per_sec{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_jumbo_chunks Telegraf collected metric
# TYPE mongodb_jumbo_chunks untyped
mongodb_jumbo_chunks{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_latency_commands Telegraf collected metric
# TYPE mongodb_latency_commands untyped
mongodb_latency_commands{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 3679
# HELP mongodb_latency_commands_count Telegraf collected metric
# TYPE mongodb_latency_commands_count untyped
mongodb_latency_commands_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 13
# HELP mongodb_latency_reads Telegraf collected metric
# TYPE mongodb_latency_reads untyped
mongodb_latency_reads{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 368
# HELP mongodb_latency_reads_count Telegraf collected metric
# TYPE mongodb_latency_reads_count untyped
mongodb_latency_reads_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 3
# HELP mongodb_latency_writes Telegraf collected metric
# TYPE mongodb_latency_writes untyped
mongodb_latency_writes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_latency_writes_count Telegraf collected metric
# TYPE mongodb_latency_writes_count untyped
mongodb_latency_writes_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_net_in_bytes Telegraf collected metric
# TYPE mongodb_net_in_bytes untyped
mongodb_net_in_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 29
# HELP mongodb_net_in_bytes_count Telegraf collected metric
# TYPE mongodb_net_in_bytes_count untyped
mongodb_net_in_bytes_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 1191
# HELP mongodb_net_out_bytes Telegraf collected metric
# TYPE mongodb_net_out_bytes untyped
mongodb_net_out_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 4208
# HELP mongodb_net_out_bytes_count Telegraf collected metric
# TYPE mongodb_net_out_bytes_count untyped
mongodb_net_out_bytes_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 126749
# HELP mongodb_open_connections Telegraf collected metric
# TYPE mongodb_open_connections untyped
mongodb_open_connections{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 1
# HELP mongodb_operation_scan_and_order Telegraf collected metric
# TYPE mongodb_operation_scan_and_order untyped
mongodb_operation_scan_and_order{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_operation_write_conflicts Telegraf collected metric
# TYPE mongodb_operation_write_conflicts untyped
mongodb_operation_write_conflicts{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_page_faults Telegraf collected metric
# TYPE mongodb_page_faults untyped
mongodb_page_faults{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_percent_cache_dirty Telegraf collected metric
# TYPE mongodb_percent_cache_dirty untyped
mongodb_percent_cache_dirty{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_percent_cache_used Telegraf collected metric
# TYPE mongodb_percent_cache_used untyped
mongodb_percent_cache_used{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_queries Telegraf collected metric
# TYPE mongodb_queries untyped
mongodb_queries{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 1
# HELP mongodb_queries_per_sec Telegraf collected metric
# TYPE mongodb_queries_per_sec untyped
mongodb_queries_per_sec{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_queued_reads Telegraf collected metric
# TYPE mongodb_queued_reads untyped
mongodb_queued_reads{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_queued_writes Telegraf collected metric
# TYPE mongodb_queued_writes untyped
mongodb_queued_writes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_resident_megabytes Telegraf collected metric
# TYPE mongodb_resident_megabytes untyped
mongodb_resident_megabytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 97
# HELP mongodb_storage_freelist_search_bucket_exhausted Telegraf collected metric
# TYPE mongodb_storage_freelist_search_bucket_exhausted untyped
mongodb_storage_freelist_search_bucket_exhausted{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_storage_freelist_search_requests Telegraf collected metric
# TYPE mongodb_storage_freelist_search_requests untyped
mongodb_storage_freelist_search_requests{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_storage_freelist_search_scanned Telegraf collected metric
# TYPE mongodb_storage_freelist_search_scanned untyped
mongodb_storage_freelist_search_scanned{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_tcmalloc_central_cache_free_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_central_cache_free_bytes untyped
mongodb_tcmalloc_central_cache_free_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 230960
# HELP mongodb_tcmalloc_current_allocated_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_current_allocated_bytes untyped
mongodb_tcmalloc_current_allocated_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 8.455176e+07
# HELP mongodb_tcmalloc_current_total_thread_cache_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_current_total_thread_cache_bytes untyped
mongodb_tcmalloc_current_total_thread_cache_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 682240
# HELP mongodb_tcmalloc_heap_size Telegraf collected metric
# TYPE mongodb_tcmalloc_heap_size untyped
mongodb_tcmalloc_heap_size{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 8.904704e+07
# HELP mongodb_tcmalloc_max_total_thread_cache_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_max_total_thread_cache_bytes untyped
mongodb_tcmalloc_max_total_thread_cache_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 2.60046848e+08
# HELP mongodb_tcmalloc_pageheap_commit_count Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_commit_count untyped
mongodb_tcmalloc_pageheap_commit_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 46
# HELP mongodb_tcmalloc_pageheap_committed_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_committed_bytes untyped
mongodb_tcmalloc_pageheap_committed_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 8.904704e+07
# HELP mongodb_tcmalloc_pageheap_decommit_count Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_decommit_count untyped
mongodb_tcmalloc_pageheap_decommit_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_tcmalloc_pageheap_free_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_free_bytes untyped
mongodb_tcmalloc_pageheap_free_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 3.52256e+06
# HELP mongodb_tcmalloc_pageheap_reserve_count Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_reserve_count untyped
mongodb_tcmalloc_pageheap_reserve_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 46
# HELP mongodb_tcmalloc_pageheap_scavenge_count Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_scavenge_count untyped
mongodb_tcmalloc_pageheap_scavenge_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_tcmalloc_pageheap_total_commit_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_total_commit_bytes untyped
mongodb_tcmalloc_pageheap_total_commit_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 8.904704e+07
# HELP mongodb_tcmalloc_pageheap_total_decommit_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_total_decommit_bytes untyped
mongodb_tcmalloc_pageheap_total_decommit_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_tcmalloc_pageheap_total_reserve_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_total_reserve_bytes untyped
mongodb_tcmalloc_pageheap_total_reserve_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 8.904704e+07
# HELP mongodb_tcmalloc_pageheap_unmapped_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_pageheap_unmapped_bytes untyped
mongodb_tcmalloc_pageheap_unmapped_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_tcmalloc_spinlock_total_delay_ns Telegraf collected metric
# TYPE mongodb_tcmalloc_spinlock_total_delay_ns untyped
mongodb_tcmalloc_spinlock_total_delay_ns{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_tcmalloc_thread_cache_free_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_thread_cache_free_bytes untyped
mongodb_tcmalloc_thread_cache_free_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 682240
# HELP mongodb_tcmalloc_total_free_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_total_free_bytes untyped
mongodb_tcmalloc_total_free_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 972720
# HELP mongodb_tcmalloc_transfer_cache_free_bytes Telegraf collected metric
# TYPE mongodb_tcmalloc_transfer_cache_free_bytes untyped
mongodb_tcmalloc_transfer_cache_free_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 59520
# HELP mongodb_total_available Telegraf collected metric
# TYPE mongodb_total_available untyped
mongodb_total_available{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_total_created Telegraf collected metric
# TYPE mongodb_total_created untyped
mongodb_total_created{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_total_docs_scanned Telegraf collected metric
# TYPE mongodb_total_docs_scanned untyped
mongodb_total_docs_scanned{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_total_in_use Telegraf collected metric
# TYPE mongodb_total_in_use untyped
mongodb_total_in_use{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_total_keys_scanned Telegraf collected metric
# TYPE mongodb_total_keys_scanned untyped
mongodb_total_keys_scanned{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_total_refreshing Telegraf collected metric
# TYPE mongodb_total_refreshing untyped
mongodb_total_refreshing{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_total_tickets_reads Telegraf collected metric
# TYPE mongodb_total_tickets_reads untyped
mongodb_total_tickets_reads{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 128
# HELP mongodb_total_tickets_writes Telegraf collected metric
# TYPE mongodb_total_tickets_writes untyped
mongodb_total_tickets_writes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 128
# HELP mongodb_ttl_deletes Telegraf collected metric
# TYPE mongodb_ttl_deletes untyped
mongodb_ttl_deletes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_ttl_deletes_per_sec Telegraf collected metric
# TYPE mongodb_ttl_deletes_per_sec untyped
mongodb_ttl_deletes_per_sec{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_ttl_passes Telegraf collected metric
# TYPE mongodb_ttl_passes untyped
mongodb_ttl_passes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_ttl_passes_per_sec Telegraf collected metric
# TYPE mongodb_ttl_passes_per_sec untyped
mongodb_ttl_passes_per_sec{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_update_command_failed Telegraf collected metric
# TYPE mongodb_update_command_failed untyped
mongodb_update_command_failed{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_update_command_total Telegraf collected metric
# TYPE mongodb_update_command_total untyped
mongodb_update_command_total{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_updates Telegraf collected metric
# TYPE mongodb_updates untyped
mongodb_updates{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_updates_per_sec Telegraf collected metric
# TYPE mongodb_updates_per_sec untyped
mongodb_updates_per_sec{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_uptime_ns Telegraf collected metric
# TYPE mongodb_uptime_ns untyped
mongodb_uptime_ns{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 4.7167e+10
# HELP mongodb_vsize_megabytes Telegraf collected metric
# TYPE mongodb_vsize_megabytes untyped
mongodb_vsize_megabytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 1547
# HELP mongodb_wtcache_app_threads_page_read_count Telegraf collected metric
# TYPE mongodb_wtcache_app_threads_page_read_count untyped
mongodb_wtcache_app_threads_page_read_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_app_threads_page_read_time Telegraf collected metric
# TYPE mongodb_wtcache_app_threads_page_read_time untyped
mongodb_wtcache_app_threads_page_read_time{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_app_threads_page_write_count Telegraf collected metric
# TYPE mongodb_wtcache_app_threads_page_write_count untyped
mongodb_wtcache_app_threads_page_write_count{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_bytes_read_into Telegraf collected metric
# TYPE mongodb_wtcache_bytes_read_into untyped
mongodb_wtcache_bytes_read_into{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_bytes_written_from Telegraf collected metric
# TYPE mongodb_wtcache_bytes_written_from untyped
mongodb_wtcache_bytes_written_from{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_current_bytes Telegraf collected metric
# TYPE mongodb_wtcache_current_bytes untyped
mongodb_wtcache_current_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 28138
# HELP mongodb_wtcache_internal_pages_evicted Telegraf collected metric
# TYPE mongodb_wtcache_internal_pages_evicted untyped
mongodb_wtcache_internal_pages_evicted{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_max_bytes_configured Telegraf collected metric
# TYPE mongodb_wtcache_max_bytes_configured untyped
mongodb_wtcache_max_bytes_configured{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 5.06462208e+08
# HELP mongodb_wtcache_modified_pages_evicted Telegraf collected metric
# TYPE mongodb_wtcache_modified_pages_evicted untyped
mongodb_wtcache_modified_pages_evicted{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_pages_evicted_by_app_thread Telegraf collected metric
# TYPE mongodb_wtcache_pages_evicted_by_app_thread untyped
mongodb_wtcache_pages_evicted_by_app_thread{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_pages_queued_for_eviction Telegraf collected metric
# TYPE mongodb_wtcache_pages_queued_for_eviction untyped
mongodb_wtcache_pages_queued_for_eviction{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_pages_read_into Telegraf collected metric
# TYPE mongodb_wtcache_pages_read_into untyped
mongodb_wtcache_pages_read_into{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_pages_requested_from Telegraf collected metric
# TYPE mongodb_wtcache_pages_requested_from untyped
mongodb_wtcache_pages_requested_from{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 225
# HELP mongodb_wtcache_server_evicting_pages Telegraf collected metric
# TYPE mongodb_wtcache_server_evicting_pages untyped
mongodb_wtcache_server_evicting_pages{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_tracked_dirty_bytes Telegraf collected metric
# TYPE mongodb_wtcache_tracked_dirty_bytes untyped
mongodb_wtcache_tracked_dirty_bytes{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 26225
# HELP mongodb_wtcache_unmodified_pages_evicted Telegraf collected metric
# TYPE mongodb_wtcache_unmodified_pages_evicted untyped
mongodb_wtcache_unmodified_pages_evicted{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
# HELP mongodb_wtcache_worker_thread_evictingpages Telegraf collected metric
# TYPE mongodb_wtcache_worker_thread_evictingpages untyped
mongodb_wtcache_worker_thread_evictingpages{host="mba.local",hostname="127.0.0.1:27017",version="4.4.0"} 0
```

## Drawbacks

- Additional maintenance and integration testing burden of a new source

## Alternatives

### Having users run telegraf or Prom node exporter and using Vector's Prometheus source to scrape it

We could not add the source directly to Vector and instead instruct users to run Telegraf's `mongodb` input or Prometheus' `mongodb_exporter` and point Vector at the resulting data. This would leverage the already supported inputs from those projects.

I decided against this as it would be in contrast with one of the listed
principles of Vector:

> One Tool. All Data. - One simple tool gets your logs, metrics, and traces
> (coming soon) from A to B.

[Vector
principles](https://vector.dev/docs/about/what-is-vector/#who-should-use-vector)

On the same page, it is mentioned that Vector should be a replacement for
Telegraf.

> You SHOULD use Vector to replace Logstash, Fluent*, Telegraf, Beats, or
> similar tools.

If users are already running Telegraf or the Prometheus MongoDB Exporter though, they could opt for this path.

## Outstanding Questions

- We need to work out if we need to classify the type of Mongo instance (mongos, replicaset member, or standalone mongod) on connection and use that to return the appropriate metrics for that type of node.

## Plan Of Attack

Incremental steps that execute this change. Generally this is in the form of:

- [ ] Submit a PR with the initial source implementation

## Future Work
