if (!window || !window.navigator || !window.navigator.userAgent) {
    Object.defineProperty(window, 'navigator', { value: { userAgent: 'node' }, writable: true });
    Object.defineProperty(document, 'getElementById', { value: (val) => document.querySelector('#' + val), writable: true });
}
import * as React from 'react';
import * as PropTypes from 'prop-types';
import { getCurrentTab, inBrowser } from '@kui-shell/core';
import { Modal, TextInput, TextArea } from 'carbon-components-react';
import { Copy24 } from '@carbon/icons-react';
import strings from '../util/i18n';
import HTTPClient from '../controller/HTTPClient';
import { SAVE_SEARCH, SEARCH_RELATED_QUERY, SAVED_SEARCH_QUERY } from '../definitions/search-queries';
import { convertStringToQuery } from '../util/search-helper';
export default class ResourceModal extends React.PureComponent {
    constructor(props) {
        super(props);
        this.originalInput = false;
        this.state = {
            description: '',
            name: '',
            errors: null,
            open: true,
        };
        this.handleDescriptionChange = this.handleDescriptionChange.bind(this);
        this.handleNameChange = this.handleNameChange.bind(this);
    }
    handleDelete() {
        const { item } = this.props;
        switch (item['kind']) {
            case 'savedSearches':
                getCurrentTab().REPL.pexec(`search -delete="save" ${item['name']}`);
                this.props.onClose();
                break;
            default:
                getCurrentTab().REPL.pexec(`search -delete="resource" ${item['name']} ${item['namespace']} ${item['kind']} ${item['cluster']} ${item['selfLink']}`);
                this.props.onClose();
        }
    }
    handleEdit() {
        const { item } = this.props;
        const data = {
            name: this.state.name,
            description: this.state.description,
            searchText: item['searchText'],
            id: item['id'],
        };
        if (!this.state.errors) {
            HTTPClient('post', 'search', SAVE_SEARCH(data))
                .then(() => {
                getCurrentTab().REPL.pexec(`savedsearches`);
                this.props.onClose();
            })
                .catch((err) => {
                this.setState({ errors: err });
            });
        }
    }
    handleSave() {
        const id = Date.now().toString();
        const data = {
            name: this.state.name,
            description: this.state.description,
            searchText: this.props.item['command'].replace(/search|--save/g, '').trim(),
            id,
        };
        const userQuery = convertStringToQuery(data.searchText);
        HTTPClient('post', 'search', SEARCH_RELATED_QUERY(userQuery.keywords, userQuery.filters))
            .then((res) => {
            if (res.data.searchResult[0].items) {
                HTTPClient('post', 'search', SAVE_SEARCH(data))
                    .then(() => {
                    this.setState({ open: false });
                    getCurrentTab().REPL.pexec('savedsearches');
                })
                    .catch((err) => {
                    this.setState({ errors: err });
                });
            }
        })
            .catch((err) => {
            this.setState({ errors: err });
        });
    }
    componentDidUpdate() {
        if (!this.props.modalOpen) {
            this.setState({ name: '', description: '', errors: undefined });
            this.originalInput = false;
        }
        if (this.props.action === 'edit' && this.props.modalOpen && !this.originalInput) {
            HTTPClient('post', 'search', SAVED_SEARCH_QUERY)
                .then((res) => {
                const data = res.data.items.filter((item) => item.name === this.props.item['name']);
                if (data.length === 0) {
                    this.setState({ errors: strings('modal.edit.deleted.search') });
                }
            });
            this.setState({ name: this.props.item['name'], description: this.props.item['description'] });
            this.originalInput = true;
        }
    }
    handleNameChange(event) {
        this.setState({ name: event.target.value });
    }
    handleDescriptionChange(event) {
        this.setState({ description: event.target.value });
    }
    render() {
        const { item, modalOpen } = this.props;
        const bodyLabel = item['kind'];
        let heading = '';
        switch (this.props.action) {
            case 'edit':
                heading = strings('modal.query.save.heading');
                break;
            case 'save':
                heading = strings('modal.save.title');
                break;
            case 'share':
                heading = strings('modal.query.share.heading');
                break;
            case 'remove':
                heading = strings('modal.query.delete.heading');
                break;
            default:
                break;
        }
        return (React.createElement(Modal, { className: this.props.action === 'share' ? 'bx--modal-share' : undefined, danger: this.props.action === 'remove', id: 'remove-resource-modal', open: this.props.action !== 'save' ? modalOpen : this.state.open, primaryButtonText: this.props.action === 'remove' ? strings('modal.remove-kuberesource.heading') : strings('actions.save'), primaryButtonDisabled: (this.props.action === 'edit' && this.state.name === '')
                || (this.props.action === 'edit' && this.state.name === this.props.item['name'] && this.state.description === this.props.item['description'])
                || (this.props.action === 'save' && this.state.name === ''), secondaryButtonText: strings('modal.button.cancel'), modalLabel: bodyLabel ? bodyLabel.toUpperCase() : undefined, modalHeading: heading, onRequestClose: () => this.props.action !== 'save' ? this.props.onClose() : this.setState({ open: false }), onRequestSubmit: () => this.props.action === 'remove' ? this.handleDelete() : this.props.action === 'edit' ? this.handleEdit() : this.handleSave(), "aria-label": heading },
            this.props.action === 'edit' || this.props.action === 'save'
                ? React.createElement("div", { className: 'bx--action-edit' },
                    this.props.action === 'save' ? React.createElement("p", { className: 'save-text' }, strings('modal.save.text')) : null,
                    this.state.errors ? React.createElement("p", { className: 'oops save-text-error' }, strings(this.state.errors)) : null,
                    React.createElement(TextInput, { className: 'bx--action-name', disabled: false, id: 'name', labelText: strings('modal.query.add.name.label'), maxLength: 50, type: 'text', value: this.state.name, onChange: this.handleNameChange, placeholder: strings('modal.query.add.name') }),
                    React.createElement(TextArea, { className: 'bx--action-description', disabled: false, id: 'description', labelText: strings('modal.query.add.desc.label'), maxLength: 140, value: this.state.description, onChange: this.handleDescriptionChange, placeholder: strings('modal.query.add.desc') }))
                : null,
            this.props.action === 'share'
                ? React.createElement("div", { className: 'bx--action-share' },
                    React.createElement("p", { className: 'copy-description' }, strings('modal.query.share.name.label')),
                    React.createElement("div", { className: 'bx--snippet bx--snippet--single' },
                        inBrowser()
                            ? `${window && window.location && window.location.origin}/multicloud/search?filters={"textsearch":"${encodeURIComponent(this.props.item['searchText'])}"}`
                            : `search ${this.props.item['searchText']}`,
                        React.createElement("button", { type: "button", className: "bx--copy-btn", onClick: () => navigator.clipboard.writeText(inBrowser()
                                ? `${window && window.location && window.location.origin}/multicloud/search?filters={"textsearch":"${encodeURIComponent(this.props.item['searchText'])}"}`
                                : `search ${this.props.item['searchText']}`) },
                            React.createElement(Copy24, { className: 'copy-btn' }))))
                : null,
            this.props.action === 'remove'
                ? React.createElement("p", null, strings('modal.remove.confirm', [item['name']]))
                : null));
    }
}
ResourceModal.propTypes = {
    item: PropTypes.object,
    modalOpen: PropTypes.bool,
};
//# sourceMappingURL=Modal.js.map