import { IntrospectionOptions } from 'graphql';
import { SchemaPointerSingle, Source, DocumentLoader, SingleFileOptions } from '@graphql-tools/utils';
import { fetch as crossFetch } from 'cross-fetch';
import { AsyncExecutor, Subscriber } from '@graphql-tools/delegate';
import { w3cwebsocket } from 'websocket';
export declare type FetchFn = typeof import('cross-fetch').fetch;
declare type Headers = Record<string, string> | Array<Record<string, string>>;
/**
 * Additional options for loading from a URL
 */
export interface LoadFromUrlOptions extends SingleFileOptions, Partial<IntrospectionOptions> {
    /**
     * Additional headers to include when querying the original schema
     */
    headers?: Headers;
    /**
     * A custom `fetch` implementation to use when querying the original schema.
     * Defaults to `cross-fetch`
     */
    customFetch?: FetchFn | string;
    /**
     * HTTP method to use when querying the original schema.
     */
    method?: 'GET' | 'POST';
    /**
     * Whether to enable subscriptions on the loaded schema
     */
    enableSubscriptions?: boolean;
    /**
     * Custom WebSocket implementation used by the loaded schema if subscriptions
     * are enabled
     */
    webSocketImpl?: typeof w3cwebsocket | string;
    /**
     * Whether to use the GET HTTP method for queries when querying the original schema
     */
    useGETForQueries?: boolean;
}
/**
 * This loader loads a schema from a URL. The loaded schema is a fully-executable,
 * remote schema since it's created using [@graphql-tools/wrap](/docs/remote-schemas).
 *
 * ```
 * const schema = await loadSchema('http://localhost:3000/graphql', {
 *   loaders: [
 *     new UrlLoader(),
 *   ]
 * });
 * ```
 */
export declare class UrlLoader implements DocumentLoader<LoadFromUrlOptions> {
    loaderId(): string;
    canLoad(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Promise<boolean>;
    canLoadSync(pointer: SchemaPointerSingle, _options: LoadFromUrlOptions): boolean;
    buildAsyncExecutor({ pointer, fetch, extraHeaders, defaultMethod, useGETForQueries, }: {
        pointer: string;
        fetch: typeof crossFetch;
        extraHeaders: any;
        defaultMethod: 'GET' | 'POST';
        useGETForQueries: boolean;
    }): AsyncExecutor;
    buildSubscriber(pointer: string, webSocketImpl: typeof w3cwebsocket): Subscriber;
    getExecutorAndSubscriber(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Promise<{
        executor: AsyncExecutor;
        subscriber: Subscriber;
    }>;
    getSubschemaConfig(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Promise<{
        schema: import("graphql").GraphQLSchema;
        executor: AsyncExecutor;
        subscriber: Subscriber;
    }>;
    load(pointer: SchemaPointerSingle, options: LoadFromUrlOptions): Promise<Source>;
    loadSync(): never;
}
export {};
