import { Kind, GraphQLDirective, valueFromASTUntyped, GraphQLInputObjectType, GraphQLScalarType, GraphQLUnionType, GraphQLEnumType, GraphQLInterfaceType, GraphQLObjectType, DirectiveLocation, TokenKind, getDirectiveValues, GraphQLDeprecatedDirective, isObjectType, isInputObjectType, isInterfaceType, isUnionType, isEnumType, isScalarType, isSchema, isNamedType, getNamedType, isSpecifiedScalarType, GraphQLSchema, isLeafType, print, specifiedDirectives, extendSchema } from 'graphql';
import { createStub, createNamedStub, fieldToFieldConfig, inputFieldToFieldConfig, parseSelectionSet, parseFragmentToInlineFragment, concatInlineFragments, rewireTypes, mergeDeep, SchemaDirectiveVisitor, pruneSchema } from '@graphql-tools/utils/es5';
import { buildDocumentFromTypeDefinitions, extendResolversFromInterfaces, addResolversToSchema, assertResolversPresent, addCatchUndefinedToSchema, addErrorLoggingToSchema, addSchemaLevelResolver, attachDirectiveResolvers } from '@graphql-tools/schema/es5';
import { wrapSchema } from '@graphql-tools/wrap/es5';
import { isSubschemaConfig, delegateToSchema } from '@graphql-tools/delegate/es5';
import { __assign, __spread, __read } from 'tslib';
import { mergeType, mergeInputType, mergeInterface, mergeUnion, mergeEnum } from '@graphql-tools/merge/es5';
import { batchDelegateToSchema } from '@graphql-tools/batch-delegate/es5';

function extractTypeDefinitions(ast) {
    var typeDefs = ast.definitions.filter(function (def) {
        return def.kind === Kind.OBJECT_TYPE_DEFINITION ||
            def.kind === Kind.INTERFACE_TYPE_DEFINITION ||
            def.kind === Kind.INPUT_OBJECT_TYPE_DEFINITION ||
            def.kind === Kind.UNION_TYPE_DEFINITION ||
            def.kind === Kind.ENUM_TYPE_DEFINITION ||
            def.kind === Kind.SCALAR_TYPE_DEFINITION;
    });
    return __assign(__assign({}, ast), { definitions: typeDefs });
}
function extractDirectiveDefinitions(ast) {
    var directiveDefs = ast.definitions.filter(function (def) { return def.kind === Kind.DIRECTIVE_DEFINITION; });
    return __assign(__assign({}, ast), { definitions: directiveDefs });
}
function extractSchemaDefinition(ast) {
    var schemaDefs = ast.definitions.filter(function (def) { return def.kind === Kind.SCHEMA_DEFINITION; });
    return schemaDefs.length ? schemaDefs[schemaDefs.length - 1] : null;
}
function extractSchemaExtensions(ast) {
    var schemaExtensions = ast.definitions.filter(function (def) { return def.kind === Kind.SCHEMA_EXTENSION; });
    return schemaExtensions;
}
function extractTypeExtensionDefinitions(ast) {
    var extensionDefs = ast.definitions.filter(function (def) {
        return def.kind === Kind.OBJECT_TYPE_EXTENSION ||
            def.kind === Kind.INTERFACE_TYPE_EXTENSION ||
            def.kind === Kind.INPUT_OBJECT_TYPE_EXTENSION ||
            def.kind === Kind.UNION_TYPE_EXTENSION ||
            def.kind === Kind.ENUM_TYPE_EXTENSION ||
            def.kind === Kind.SCALAR_TYPE_EXTENSION;
    });
    return __assign(__assign({}, ast), { definitions: extensionDefs });
}

var backcompatOptions = { commentDescriptions: true };
function typeFromAST(node) {
    switch (node.kind) {
        case Kind.OBJECT_TYPE_DEFINITION:
            return makeObjectType(node);
        case Kind.INTERFACE_TYPE_DEFINITION:
            return makeInterfaceType(node);
        case Kind.ENUM_TYPE_DEFINITION:
            return makeEnumType(node);
        case Kind.UNION_TYPE_DEFINITION:
            return makeUnionType(node);
        case Kind.SCALAR_TYPE_DEFINITION:
            return makeScalarType(node);
        case Kind.INPUT_OBJECT_TYPE_DEFINITION:
            return makeInputObjectType(node);
        case Kind.DIRECTIVE_DEFINITION:
            return makeDirective(node);
        default:
            return null;
    }
}
function makeObjectType(node) {
    var config = {
        name: node.name.value,
        description: getDescription(node, backcompatOptions),
        interfaces: function () { return node.interfaces.map(function (iface) { return createNamedStub(iface.name.value, 'interface'); }); },
        fields: function () { return makeFields(node.fields); },
        astNode: node,
    };
    return new GraphQLObjectType(config);
}
function makeInterfaceType(node) {
    var _a;
    var config = {
        name: node.name.value,
        description: getDescription(node, backcompatOptions),
        interfaces: (_a = node.interfaces) === null || _a === void 0 ? void 0 : _a.map(function (iface) {
            return createNamedStub(iface.name.value, 'interface');
        }),
        fields: function () { return makeFields(node.fields); },
        astNode: node,
    };
    return new GraphQLInterfaceType(config);
}
function makeEnumType(node) {
    var values = node.values.reduce(function (prev, value) {
        var _a;
        return (__assign(__assign({}, prev), (_a = {}, _a[value.name.value] = {
            description: getDescription(value, backcompatOptions),
            deprecationReason: getDeprecationReason(value),
            astNode: value,
        }, _a)));
    }, {});
    return new GraphQLEnumType({
        name: node.name.value,
        description: getDescription(node, backcompatOptions),
        values: values,
        astNode: node,
    });
}
function makeUnionType(node) {
    return new GraphQLUnionType({
        name: node.name.value,
        description: getDescription(node, backcompatOptions),
        types: function () { return node.types.map(function (type) { return createNamedStub(type.name.value, 'object'); }); },
        astNode: node,
    });
}
function makeScalarType(node) {
    return new GraphQLScalarType({
        name: node.name.value,
        description: getDescription(node, backcompatOptions),
        astNode: node,
        // TODO: serialize default property setting can be dropped once
        // upstream graphql-js TypeScript typings are updated, likely in v16
        serialize: function (value) { return value; },
    });
}
function makeInputObjectType(node) {
    return new GraphQLInputObjectType({
        name: node.name.value,
        description: getDescription(node, backcompatOptions),
        fields: function () { return makeValues(node.fields); },
        astNode: node,
    });
}
function makeFields(nodes) {
    return nodes.reduce(function (prev, node) {
        var _a;
        return (__assign(__assign({}, prev), (_a = {}, _a[node.name.value] = {
            type: createStub(node.type, 'output'),
            description: getDescription(node, backcompatOptions),
            args: makeValues(node.arguments),
            deprecationReason: getDeprecationReason(node),
            astNode: node,
        }, _a)));
    }, {});
}
function makeValues(nodes) {
    return nodes.reduce(function (prev, node) {
        var _a;
        return (__assign(__assign({}, prev), (_a = {}, _a[node.name.value] = {
            type: createStub(node.type, 'input'),
            defaultValue: node.defaultValue !== undefined ? valueFromASTUntyped(node.defaultValue) : undefined,
            description: getDescription(node, backcompatOptions),
            astNode: node,
        }, _a)));
    }, {});
}
function makeDirective(node) {
    var locations = [];
    node.locations.forEach(function (location) {
        if (location.value in DirectiveLocation) {
            locations.push(location.value);
        }
    });
    return new GraphQLDirective({
        name: node.name.value,
        description: node.description != null ? node.description.value : null,
        locations: locations,
        isRepeatable: node.repeatable,
        args: makeValues(node.arguments),
        astNode: node,
    });
}
// graphql < v13 does not export getDescription
function getDescription(node, options) {
    if (node.description != null) {
        return node.description.value;
    }
    if (options.commentDescriptions) {
        var rawValue = getLeadingCommentBlock(node);
        if (rawValue !== undefined) {
            return dedentBlockStringValue("\n" + rawValue);
        }
    }
}
function getLeadingCommentBlock(node) {
    var loc = node.loc;
    if (!loc) {
        return;
    }
    var comments = [];
    var token = loc.startToken.prev;
    while (token != null &&
        token.kind === TokenKind.COMMENT &&
        token.next != null &&
        token.prev != null &&
        token.line + 1 === token.next.line &&
        token.line !== token.prev.line) {
        var value = String(token.value);
        comments.push(value);
        token = token.prev;
    }
    return comments.length > 0 ? comments.reverse().join('\n') : undefined;
}
function dedentBlockStringValue(rawString) {
    // Expand a block string's raw value into independent lines.
    var lines = rawString.split(/\r\n|[\n\r]/g);
    // Remove common indentation from all lines but first.
    var commonIndent = getBlockStringIndentation(lines);
    if (commonIndent !== 0) {
        for (var i = 1; i < lines.length; i++) {
            lines[i] = lines[i].slice(commonIndent);
        }
    }
    // Remove leading and trailing blank lines.
    while (lines.length > 0 && isBlank(lines[0])) {
        lines.shift();
    }
    while (lines.length > 0 && isBlank(lines[lines.length - 1])) {
        lines.pop();
    }
    // Return a string of the lines joined with U+000A.
    return lines.join('\n');
}
/**
 * @internal
 */
function getBlockStringIndentation(lines) {
    var commonIndent = null;
    for (var i = 1; i < lines.length; i++) {
        var line = lines[i];
        var indent = leadingWhitespace(line);
        if (indent === line.length) {
            continue; // skip empty lines
        }
        if (commonIndent === null || indent < commonIndent) {
            commonIndent = indent;
            if (commonIndent === 0) {
                break;
            }
        }
    }
    return commonIndent === null ? 0 : commonIndent;
}
function leadingWhitespace(str) {
    var i = 0;
    while (i < str.length && (str[i] === ' ' || str[i] === '\t')) {
        i++;
    }
    return i;
}
function isBlank(str) {
    return leadingWhitespace(str) === str.length;
}
function getDeprecationReason(node) {
    var deprecated = getDirectiveValues(GraphQLDeprecatedDirective, node);
    return deprecated === null || deprecated === void 0 ? void 0 : deprecated.reason;
}

function mergeCandidates(typeName, candidates, typeMergingOptions) {
    var initialCandidateType = candidates[0].type;
    if (candidates.some(function (candidate) { return candidate.type.constructor !== initialCandidateType.constructor; })) {
        throw new Error("Cannot merge different type categories into common type " + typeName + ".");
    }
    if (isObjectType(initialCandidateType)) {
        return mergeObjectTypeCandidates(typeName, candidates, typeMergingOptions);
    }
    else if (isInputObjectType(initialCandidateType)) {
        return mergeInputObjectTypeCandidates(typeName, candidates, typeMergingOptions);
    }
    else if (isInterfaceType(initialCandidateType)) {
        return mergeInterfaceTypeCandidates(typeName, candidates, typeMergingOptions);
    }
    else if (isUnionType(initialCandidateType)) {
        return mergeUnionTypeCandidates(typeName, candidates, typeMergingOptions);
    }
    else if (isEnumType(initialCandidateType)) {
        return mergeEnumTypeCandidates(typeName, candidates, typeMergingOptions);
    }
    else if (isScalarType(initialCandidateType)) {
        return mergeScalarTypeCandidates(typeName, candidates, typeMergingOptions);
    }
    else {
        // not reachable.
        throw new Error("Type " + typeName + " has unknown GraphQL type.");
    }
}
function mergeObjectTypeCandidates(typeName, candidates, typeMergingOptions) {
    var description = mergeTypeDescriptions(candidates, typeMergingOptions);
    var fields = fieldConfigMapFromTypeCandidates(candidates, typeMergingOptions);
    var typeConfigs = candidates.map(function (candidate) { return candidate.type.toConfig(); });
    var interfaceMap = typeConfigs
        .map(function (typeConfig) { return typeConfig.interfaces; })
        .reduce(function (acc, interfaces) {
        if (interfaces != null) {
            interfaces.forEach(function (iface) {
                acc[iface.name] = iface;
            });
        }
        return acc;
    }, Object.create(null));
    var interfaces = Object.keys(interfaceMap).map(function (interfaceName) { return interfaceMap[interfaceName]; });
    var astNodes = pluck('astNode', candidates);
    var astNode = astNodes
        .slice(1)
        .reduce(function (acc, astNode) { return mergeType(astNode, acc); }, astNodes[0]);
    var extensionASTNodes = [].concat(pluck('extensionASTNodes', candidates));
    var extensions = Object.assign.apply(Object, __spread([{}], pluck('extensions', candidates)));
    var typeConfig = {
        name: typeName,
        description: description,
        fields: fields,
        interfaces: interfaces,
        astNode: astNode,
        extensionASTNodes: extensionASTNodes,
        extensions: extensions,
    };
    return new GraphQLObjectType(typeConfig);
}
function mergeInputObjectTypeCandidates(typeName, candidates, typeMergingOptions) {
    var description = mergeTypeDescriptions(candidates, typeMergingOptions);
    var fields = inputFieldConfigMapFromTypeCandidates(candidates, typeMergingOptions);
    var astNodes = pluck('astNode', candidates);
    var astNode = astNodes
        .slice(1)
        .reduce(function (acc, astNode) { return mergeInputType(astNode, acc); }, astNodes[0]);
    var extensionASTNodes = [].concat(pluck('extensionASTNodes', candidates));
    var extensions = Object.assign.apply(Object, __spread([{}], pluck('extensions', candidates)));
    var typeConfig = {
        name: typeName,
        description: description,
        fields: fields,
        astNode: astNode,
        extensionASTNodes: extensionASTNodes,
        extensions: extensions,
    };
    return new GraphQLInputObjectType(typeConfig);
}
function pluck(typeProperty, candidates) {
    return candidates.map(function (candidate) { return candidate.type[typeProperty]; }).filter(function (value) { return value != null; });
}
function mergeInterfaceTypeCandidates(typeName, candidates, typeMergingOptions) {
    var description = mergeTypeDescriptions(candidates, typeMergingOptions);
    var fields = fieldConfigMapFromTypeCandidates(candidates, typeMergingOptions);
    var typeConfigs = candidates.map(function (candidate) { return candidate.type.toConfig(); });
    var interfaceMap = typeConfigs
        .map(function (typeConfig) { return typeConfig.interfaces; })
        .reduce(function (acc, interfaces) {
        if (interfaces != null) {
            interfaces.forEach(function (iface) {
                acc[iface.name] = iface;
            });
        }
        return acc;
    }, Object.create(null));
    var interfaces = Object.keys(interfaceMap).map(function (interfaceName) { return interfaceMap[interfaceName]; });
    var astNodes = pluck('astNode', candidates);
    var astNode = astNodes
        .slice(1)
        .reduce(function (acc, astNode) { return mergeInterface(astNode, acc, {}); }, astNodes[0]);
    var extensionASTNodes = [].concat(pluck('extensionASTNodes', candidates));
    var extensions = Object.assign.apply(Object, __spread([{}], pluck('extensions', candidates)));
    var typeConfig = {
        name: typeName,
        description: description,
        fields: fields,
        interfaces: interfaces,
        astNode: astNode,
        extensionASTNodes: extensionASTNodes,
        extensions: extensions,
    };
    return new GraphQLInterfaceType(typeConfig);
}
function mergeUnionTypeCandidates(typeName, candidates, typeMergingOptions) {
    var description = mergeTypeDescriptions(candidates, typeMergingOptions);
    var typeConfigs = candidates.map(function (candidate) { return candidate.type.toConfig(); });
    var typeMap = typeConfigs.reduce(function (acc, typeConfig) {
        typeConfig.types.forEach(function (type) {
            acc[type.name] = type;
        });
        return acc;
    }, Object.create(null));
    var types = Object.keys(typeMap).map(function (typeName) { return typeMap[typeName]; });
    var astNodes = pluck('astNode', candidates);
    var astNode = astNodes
        .slice(1)
        .reduce(function (acc, astNode) { return mergeUnion(astNode, acc); }, astNodes[0]);
    var extensionASTNodes = [].concat(pluck('extensionASTNodes', candidates));
    var extensions = Object.assign.apply(Object, __spread([{}], pluck('extensions', candidates)));
    var typeConfig = {
        name: typeName,
        description: description,
        types: types,
        astNode: astNode,
        extensionASTNodes: extensionASTNodes,
        extensions: extensions,
    };
    return new GraphQLUnionType(typeConfig);
}
function mergeEnumTypeCandidates(typeName, candidates, typeMergingOptions) {
    var description = mergeTypeDescriptions(candidates, typeMergingOptions);
    var typeConfigs = candidates.map(function (candidate) { return candidate.type.toConfig(); });
    var values = typeConfigs.reduce(function (acc, typeConfig) { return (__assign(__assign({}, acc), typeConfig.values)); }, {});
    var astNodes = pluck('astNode', candidates);
    var astNode = astNodes
        .slice(1)
        .reduce(function (acc, astNode) { return mergeEnum(astNode, acc); }, astNodes[0]);
    var extensionASTNodes = [].concat(pluck('extensionASTNodes', candidates));
    var extensions = Object.assign.apply(Object, __spread([{}], pluck('extensions', candidates)));
    var typeConfig = {
        name: typeName,
        description: description,
        values: values,
        astNode: astNode,
        extensionASTNodes: extensionASTNodes,
        extensions: extensions,
    };
    return new GraphQLEnumType(typeConfig);
}
function mergeScalarTypeCandidates(typeName, candidates, typeMergingOptions) {
    var description = mergeTypeDescriptions(candidates, typeMergingOptions);
    var serializeFns = pluck('serialize', candidates);
    var serialize = serializeFns[serializeFns.length - 1];
    var parseValueFns = pluck('parseValue', candidates);
    var parseValue = parseValueFns[parseValueFns.length - 1];
    var parseLiteralFns = pluck('parseLiteral', candidates);
    var parseLiteral = parseLiteralFns[parseLiteralFns.length - 1];
    var astNodes = pluck('astNode', candidates);
    var astNode = astNodes
        .slice(1)
        .reduce(function (acc, astNode) { return mergeScalarTypeDefinitionNodes(acc, astNode); }, astNodes[0]);
    var extensionASTNodes = [].concat(pluck('extensionASTNodes', candidates));
    var extensions = Object.assign.apply(Object, __spread([{}], pluck('extensions', candidates)));
    var typeConfig = {
        name: typeName,
        description: description,
        serialize: serialize,
        parseValue: parseValue,
        parseLiteral: parseLiteral,
        astNode: astNode,
        extensionASTNodes: extensionASTNodes,
        extensions: extensions,
    };
    return new GraphQLScalarType(typeConfig);
}
function mergeTypeDescriptions(candidates, typeMergingOptions) {
    var _a;
    var typeDescriptionsMerger = (_a = typeMergingOptions === null || typeMergingOptions === void 0 ? void 0 : typeMergingOptions.typeDescriptionsMerger) !== null && _a !== void 0 ? _a : defaultTypeDescriptionMerger;
    return typeDescriptionsMerger(candidates);
}
function defaultTypeDescriptionMerger(candidates) {
    return candidates[candidates.length - 1].type.description;
}
function fieldConfigMapFromTypeCandidates(candidates, typeMergingOptions) {
    var fieldConfigCandidatesMap = Object.create(null);
    candidates.forEach(function (candidate) {
        var fieldMap = candidate.type.getFields();
        Object.keys(fieldMap).forEach(function (fieldName) {
            var fieldConfigCandidate = {
                fieldConfig: fieldToFieldConfig(fieldMap[fieldName]),
                fieldName: fieldName,
                type: candidate.type,
                subschema: candidate.subschema,
                transformedSchema: candidate.transformedSchema,
            };
            if (fieldName in fieldConfigCandidatesMap) {
                fieldConfigCandidatesMap[fieldName].push(fieldConfigCandidate);
            }
            else {
                fieldConfigCandidatesMap[fieldName] = [fieldConfigCandidate];
            }
        });
    });
    var fieldConfigMap = Object.create(null);
    Object.keys(fieldConfigCandidatesMap).forEach(function (fieldName) {
        fieldConfigMap[fieldName] = mergeFieldConfigs(fieldConfigCandidatesMap[fieldName], typeMergingOptions);
    });
    return fieldConfigMap;
}
function mergeFieldConfigs(candidates, typeMergingOptions) {
    var _a;
    var fieldConfigMerger = (_a = typeMergingOptions === null || typeMergingOptions === void 0 ? void 0 : typeMergingOptions.fieldConfigMerger) !== null && _a !== void 0 ? _a : defaultFieldConfigMerger;
    return fieldConfigMerger(candidates);
}
function defaultFieldConfigMerger(candidates) {
    return candidates[candidates.length - 1].fieldConfig;
}
function inputFieldConfigMapFromTypeCandidates(candidates, typeMergingOptions) {
    var inputFieldConfigCandidatesMap = Object.create(null);
    candidates.forEach(function (candidate) {
        var inputFieldMap = candidate.type.getFields();
        Object.keys(inputFieldMap).forEach(function (fieldName) {
            var inputFieldConfigCandidate = {
                inputFieldConfig: inputFieldToFieldConfig(inputFieldMap[fieldName]),
                fieldName: fieldName,
                type: candidate.type,
                subschema: candidate.subschema,
                transformedSchema: candidate.transformedSchema,
            };
            if (fieldName in inputFieldConfigCandidatesMap) {
                inputFieldConfigCandidatesMap[fieldName].push(inputFieldConfigCandidate);
            }
            else {
                inputFieldConfigCandidatesMap[fieldName] = [inputFieldConfigCandidate];
            }
        });
    });
    var inputFieldConfigMap = Object.create(null);
    Object.keys(inputFieldConfigCandidatesMap).forEach(function (fieldName) {
        inputFieldConfigMap[fieldName] = mergeInputFieldConfigs(inputFieldConfigCandidatesMap[fieldName], typeMergingOptions);
    });
    return inputFieldConfigMap;
}
function mergeInputFieldConfigs(candidates, typeMergingOptions) {
    var _a;
    var inputFieldConfigMerger = (_a = typeMergingOptions === null || typeMergingOptions === void 0 ? void 0 : typeMergingOptions.inputFieldConfigMerger) !== null && _a !== void 0 ? _a : defaultInputFieldConfigMerger;
    return inputFieldConfigMerger(candidates);
}
function defaultInputFieldConfigMerger(candidates) {
    return candidates[candidates.length - 1].inputFieldConfig;
}
function mergeScalarTypeDefinitionNodes(targetNode, sourceNode) {
    var _a, _b, _c;
    return __assign(__assign({}, targetNode), { description: (_a = sourceNode.description) !== null && _a !== void 0 ? _a : targetNode.description, directives: ((_b = targetNode.directives) !== null && _b !== void 0 ? _b : []).concat((_c = sourceNode.directives) !== null && _c !== void 0 ? _c : []) });
}

function isDocumentNode(schemaLikeObject) {
    return schemaLikeObject.kind !== undefined;
}
function buildTypeCandidates(_a) {
    var schemaLikeObjects = _a.schemaLikeObjects, transformedSchemas = _a.transformedSchemas, extensions = _a.extensions, directiveMap = _a.directiveMap, schemaDefs = _a.schemaDefs, operationTypeNames = _a.operationTypeNames, mergeDirectives = _a.mergeDirectives;
    var typeCandidates = Object.create(null);
    var schemaDef;
    var schemaExtensions = [];
    schemaLikeObjects.forEach(function (schemaLikeObject) {
        if (isDocumentNode(schemaLikeObject)) {
            schemaDef = extractSchemaDefinition(schemaLikeObject);
            schemaExtensions = schemaExtensions.concat(extractSchemaExtensions(schemaLikeObject));
        }
    });
    schemaDefs.schemaDef = schemaDef;
    schemaDefs.schemaExtensions = schemaExtensions;
    setOperationTypeNames(schemaDefs, operationTypeNames);
    schemaLikeObjects.forEach(function (schemaLikeObject) {
        if (isSchema(schemaLikeObject) || isSubschemaConfig(schemaLikeObject)) {
            var schema_1 = wrapSchema(schemaLikeObject);
            transformedSchemas.set(schemaLikeObject, schema_1);
            var operationTypes_1 = {
                query: schema_1.getQueryType(),
                mutation: schema_1.getMutationType(),
                subscription: schema_1.getSubscriptionType(),
            };
            Object.keys(operationTypes_1).forEach(function (operationType) {
                if (operationTypes_1[operationType] != null) {
                    addTypeCandidate(typeCandidates, operationTypeNames[operationType], {
                        type: operationTypes_1[operationType],
                        subschema: schemaLikeObject,
                        transformedSchema: schema_1,
                    });
                }
            });
            if (mergeDirectives) {
                schema_1.getDirectives().forEach(function (directive) {
                    directiveMap[directive.name] = directive;
                });
            }
            var originalTypeMap_1 = schema_1.getTypeMap();
            Object.keys(originalTypeMap_1).forEach(function (typeName) {
                var type = originalTypeMap_1[typeName];
                if (isNamedType(type) &&
                    getNamedType(type).name.slice(0, 2) !== '__' &&
                    type !== operationTypes_1.query &&
                    type !== operationTypes_1.mutation &&
                    type !== operationTypes_1.subscription) {
                    addTypeCandidate(typeCandidates, type.name, {
                        type: type,
                        subschema: schemaLikeObject,
                        transformedSchema: schema_1,
                    });
                }
            });
        }
        else if (isDocumentNode(schemaLikeObject)) {
            var typesDocument = extractTypeDefinitions(schemaLikeObject);
            typesDocument.definitions.forEach(function (def) {
                var type = typeFromAST(def);
                if (type != null) {
                    addTypeCandidate(typeCandidates, type.name, {
                        type: type,
                    });
                }
            });
            var directivesDocument = extractDirectiveDefinitions(schemaLikeObject);
            directivesDocument.definitions.forEach(function (def) {
                var directive = typeFromAST(def);
                directiveMap[directive.name] = directive;
            });
            var extensionsDocument = extractTypeExtensionDefinitions(schemaLikeObject);
            if (extensionsDocument.definitions.length > 0) {
                extensions.push(extensionsDocument);
            }
        }
        else if (isNamedType(schemaLikeObject)) {
            addTypeCandidate(typeCandidates, schemaLikeObject.name, {
                type: schemaLikeObject,
            });
        }
        else {
            throw new Error("Invalid object " + schemaLikeObject);
        }
    });
    return typeCandidates;
}
function setOperationTypeNames(_a, operationTypeNames) {
    var schemaDef = _a.schemaDef, schemaExtensions = _a.schemaExtensions;
    var allNodes = schemaExtensions.slice();
    if (schemaDef != null) {
        allNodes.unshift(schemaDef);
    }
    allNodes.forEach(function (node) {
        if (node.operationTypes != null) {
            node.operationTypes.forEach(function (operationType) {
                operationTypeNames[operationType.operation] = operationType.type.name.value;
            });
        }
    });
}
function addTypeCandidate(typeCandidates, name, typeCandidate) {
    if (!(name in typeCandidates)) {
        typeCandidates[name] = [];
    }
    typeCandidates[name].push(typeCandidate);
}
function buildTypeMap(_a) {
    var typeCandidates = _a.typeCandidates, stitchingInfo = _a.stitchingInfo, operationTypeNames = _a.operationTypeNames, onTypeConflict = _a.onTypeConflict, mergeTypes = _a.mergeTypes, typeMergingOptions = _a.typeMergingOptions;
    var typeMap = Object.create(null);
    Object.keys(typeCandidates).forEach(function (typeName) {
        if (typeName === operationTypeNames.query ||
            typeName === operationTypeNames.mutation ||
            typeName === operationTypeNames.subscription ||
            (mergeTypes === true && !typeCandidates[typeName].some(function (candidate) { return isSpecifiedScalarType(candidate.type); })) ||
            (typeof mergeTypes === 'function' && mergeTypes(typeCandidates[typeName], typeName)) ||
            (Array.isArray(mergeTypes) && mergeTypes.includes(typeName)) ||
            (stitchingInfo != null && typeName in stitchingInfo.mergedTypes)) {
            typeMap[typeName] = mergeCandidates(typeName, typeCandidates[typeName], typeMergingOptions);
        }
        else {
            var candidateSelector = onTypeConflict != null
                ? onTypeConflictToCandidateSelector(onTypeConflict)
                : function (cands) { return cands[cands.length - 1]; };
            typeMap[typeName] = candidateSelector(typeCandidates[typeName]).type;
        }
    });
    return typeMap;
}
function onTypeConflictToCandidateSelector(onTypeConflict) {
    return function (cands) {
        return cands.reduce(function (prev, next) {
            var type = onTypeConflict(prev.type, next.type, {
                left: {
                    schema: prev.transformedSchema,
                },
                right: {
                    schema: next.transformedSchema,
                },
            });
            if (prev.type === type) {
                return prev;
            }
            else if (next.type === type) {
                return next;
            }
            return {
                schemaName: 'unknown',
                type: type,
            };
        });
    };
}

function createStitchingInfo(transformedSchemas, typeCandidates, mergeTypes) {
    var mergedTypes = createMergedTypes(typeCandidates, mergeTypes);
    var selectionSetsByField = Object.create(null);
    Object.entries(mergedTypes).forEach(function (_a) {
        var _b = __read(_a, 2), typeName = _b[0], mergedTypeInfo = _b[1];
        if (mergedTypeInfo.selectionSets == null && mergedTypeInfo.fieldSelectionSets == null) {
            return;
        }
        selectionSetsByField[typeName] = Object.create(null);
        mergedTypeInfo.selectionSets.forEach(function (selectionSet, subschemaConfig) {
            var schema = subschemaConfig.schema;
            var type = schema.getType(typeName);
            var fields = type.getFields();
            Object.keys(fields).forEach(function (fieldName) {
                var field = fields[fieldName];
                var fieldType = getNamedType(field.type);
                if (selectionSet && isLeafType(fieldType) && selectionSetContainsTopLevelField(selectionSet, fieldName)) {
                    return;
                }
                if (selectionSetsByField[typeName][fieldName] == null) {
                    selectionSetsByField[typeName][fieldName] = {
                        kind: Kind.SELECTION_SET,
                        selections: [parseSelectionSet('{ __typename }').selections[0]],
                    };
                }
                selectionSetsByField[typeName][fieldName].selections = selectionSetsByField[typeName][fieldName].selections.concat(selectionSet.selections);
            });
        });
        mergedTypeInfo.fieldSelectionSets.forEach(function (selectionSetFieldMap) {
            Object.keys(selectionSetFieldMap).forEach(function (fieldName) {
                if (selectionSetsByField[typeName][fieldName] == null) {
                    selectionSetsByField[typeName][fieldName] = {
                        kind: Kind.SELECTION_SET,
                        selections: [parseSelectionSet('{ __typename }').selections[0]],
                    };
                }
                selectionSetsByField[typeName][fieldName].selections = selectionSetsByField[typeName][fieldName].selections.concat(selectionSetFieldMap[fieldName].selections);
            });
        });
    });
    return {
        transformedSchemas: transformedSchemas,
        fragmentsByField: undefined,
        selectionSetsByField: selectionSetsByField,
        dynamicSelectionSetsByField: undefined,
        mergedTypes: mergedTypes,
    };
}
function createMergedTypes(typeCandidates, mergeTypes) {
    var mergedTypes = Object.create(null);
    Object.keys(typeCandidates).forEach(function (typeName) {
        if (typeCandidates[typeName].length > 1 &&
            (isObjectType(typeCandidates[typeName][0].type) || isInterfaceType(typeCandidates[typeName][0].type))) {
            var typeCandidatesWithMergedTypeConfig = typeCandidates[typeName].filter(function (typeCandidate) {
                return typeCandidate.subschema != null &&
                    isSubschemaConfig(typeCandidate.subschema) &&
                    typeCandidate.subschema.merge != null &&
                    typeName in typeCandidate.subschema.merge;
            });
            if (mergeTypes === true ||
                (typeof mergeTypes === 'function' && mergeTypes(typeCandidates[typeName], typeName)) ||
                (Array.isArray(mergeTypes) && mergeTypes.includes(typeName)) ||
                typeCandidatesWithMergedTypeConfig.length) {
                var targetSubschemas_1 = [];
                var typeMaps_1 = new Map();
                var supportedBySubschemas_1 = Object.create({});
                var selectionSets_1 = new Map();
                var fieldSelectionSets_1 = new Map();
                typeCandidates[typeName].forEach(function (typeCandidate) {
                    var _a;
                    var subschema = typeCandidate.subschema;
                    if (subschema == null) {
                        return;
                    }
                    typeMaps_1.set(subschema, typeCandidate.transformedSchema.getTypeMap());
                    if (!isSubschemaConfig(subschema)) {
                        return;
                    }
                    var mergedTypeConfig = (_a = subschema === null || subschema === void 0 ? void 0 : subschema.merge) === null || _a === void 0 ? void 0 : _a[typeName];
                    if (mergedTypeConfig == null) {
                        return;
                    }
                    if (mergedTypeConfig.selectionSet) {
                        var selectionSet_1 = parseSelectionSet(mergedTypeConfig.selectionSet);
                        selectionSets_1.set(subschema, selectionSet_1);
                    }
                    if (mergedTypeConfig.fields) {
                        var parsedFieldSelectionSets_1 = Object.create(null);
                        Object.keys(mergedTypeConfig.fields).forEach(function (fieldName) {
                            if (mergedTypeConfig.fields[fieldName].selectionSet) {
                                var rawFieldSelectionSet = mergedTypeConfig.fields[fieldName].selectionSet;
                                parsedFieldSelectionSets_1[fieldName] = parseSelectionSet(rawFieldSelectionSet);
                            }
                        });
                        fieldSelectionSets_1.set(subschema, parsedFieldSelectionSets_1);
                    }
                    if (mergedTypeConfig.resolve != null) {
                        targetSubschemas_1.push(subschema);
                    }
                    else if (mergedTypeConfig.key != null) {
                        mergedTypeConfig.resolve = function (originalResult, context, info, subschema, selectionSet) {
                            var _a;
                            return batchDelegateToSchema({
                                schema: subschema,
                                operation: 'query',
                                fieldName: mergedTypeConfig.fieldName,
                                key: mergedTypeConfig.key(originalResult),
                                argsFromKeys: (_a = mergedTypeConfig.argsFromKeys) !== null && _a !== void 0 ? _a : mergedTypeConfig.args,
                                valuesFromResults: mergedTypeConfig.valuesFromResults,
                                selectionSet: selectionSet,
                                context: context,
                                info: info,
                                skipTypeMerging: true,
                            });
                        };
                        targetSubschemas_1.push(subschema);
                    }
                    else if (mergedTypeConfig.fieldName != null) {
                        mergedTypeConfig.resolve = function (originalResult, context, info, subschema, selectionSet) {
                            return delegateToSchema({
                                schema: subschema,
                                operation: 'query',
                                fieldName: mergedTypeConfig.fieldName,
                                returnType: getNamedType(info.returnType),
                                args: mergedTypeConfig.args(originalResult),
                                selectionSet: selectionSet,
                                context: context,
                                info: info,
                                skipTypeMerging: true,
                            });
                        };
                        targetSubschemas_1.push(subschema);
                    }
                    if (mergedTypeConfig.resolve == null) {
                        return;
                    }
                    var type = typeCandidate.transformedSchema.getType(typeName);
                    var fieldMap = type.getFields();
                    var selectionSet = selectionSets_1.get(subschema);
                    Object.keys(fieldMap).forEach(function (fieldName) {
                        var field = fieldMap[fieldName];
                        var fieldType = getNamedType(field.type);
                        if (selectionSet && isLeafType(fieldType) && selectionSetContainsTopLevelField(selectionSet, fieldName)) {
                            return;
                        }
                        if (!(fieldName in supportedBySubschemas_1)) {
                            supportedBySubschemas_1[fieldName] = [];
                        }
                        supportedBySubschemas_1[fieldName].push(subschema);
                    });
                });
                var sourceSubschemas = typeCandidates[typeName]
                    .filter(function (typeCandidate) { return typeCandidate.subschema != null; })
                    .map(function (typeCandidate) { return typeCandidate.subschema; });
                var targetSubschemasBySubschema_1 = new Map();
                sourceSubschemas.forEach(function (subschema) {
                    var filteredSubschemas = targetSubschemas_1.filter(function (s) { return s !== subschema; });
                    if (filteredSubschemas.length) {
                        targetSubschemasBySubschema_1.set(subschema, filteredSubschemas);
                    }
                });
                mergedTypes[typeName] = {
                    typeName: typeName,
                    targetSubschemas: targetSubschemasBySubschema_1,
                    typeMaps: typeMaps_1,
                    selectionSets: selectionSets_1,
                    fieldSelectionSets: fieldSelectionSets_1,
                    uniqueFields: Object.create({}),
                    nonUniqueFields: Object.create({}),
                };
                Object.keys(supportedBySubschemas_1).forEach(function (fieldName) {
                    if (supportedBySubschemas_1[fieldName].length === 1) {
                        mergedTypes[typeName].uniqueFields[fieldName] = supportedBySubschemas_1[fieldName][0];
                    }
                    else {
                        mergedTypes[typeName].nonUniqueFields[fieldName] = supportedBySubschemas_1[fieldName];
                    }
                });
            }
        }
    });
    return mergedTypes;
}
function completeStitchingInfo(stitchingInfo, resolvers) {
    var selectionSetsByField = stitchingInfo.selectionSetsByField;
    var dynamicSelectionSetsByField = Object.create(null);
    var rawFragments = [];
    Object.keys(resolvers).forEach(function (typeName) {
        var type = resolvers[typeName];
        if (isScalarType(type)) {
            return;
        }
        Object.keys(type).forEach(function (fieldName) {
            var field = type[fieldName];
            if (field.selectionSet) {
                if (typeof field.selectionSet === 'function') {
                    if (!(typeName in dynamicSelectionSetsByField)) {
                        dynamicSelectionSetsByField[typeName] = Object.create(null);
                    }
                    if (!(fieldName in dynamicSelectionSetsByField[typeName])) {
                        dynamicSelectionSetsByField[typeName][fieldName] = [];
                    }
                    dynamicSelectionSetsByField[typeName][fieldName].push(field.selectionSet);
                }
                else {
                    var selectionSet = parseSelectionSet(field.selectionSet);
                    if (!(typeName in selectionSetsByField)) {
                        selectionSetsByField[typeName] = Object.create(null);
                    }
                    if (!(fieldName in selectionSetsByField[typeName])) {
                        selectionSetsByField[typeName][fieldName] = {
                            kind: Kind.SELECTION_SET,
                            selections: [],
                        };
                    }
                    selectionSetsByField[typeName][fieldName].selections = selectionSetsByField[typeName][fieldName].selections.concat(selectionSet.selections);
                }
            }
            if (field.fragment) {
                rawFragments.push({
                    field: fieldName,
                    fragment: field.fragment,
                });
            }
        });
    });
    Object.keys(selectionSetsByField).forEach(function (typeName) {
        var typeSelectionSets = selectionSetsByField[typeName];
        Object.keys(typeSelectionSets).forEach(function (fieldName) {
            var consolidatedSelections = new Map();
            var selectionSet = typeSelectionSets[fieldName];
            selectionSet.selections.forEach(function (selection) {
                consolidatedSelections.set(print(selection), selection);
            });
            selectionSet.selections = Array.from(consolidatedSelections.values());
        });
    });
    var parsedFragments = Object.create(null);
    rawFragments.forEach(function (_a) {
        var field = _a.field, fragment = _a.fragment;
        var parsedFragment = parseFragmentToInlineFragment(fragment);
        var actualTypeName = parsedFragment.typeCondition.name.value;
        if (!(actualTypeName in parsedFragments)) {
            parsedFragments[actualTypeName] = Object.create(null);
        }
        if (!(field in parsedFragments[actualTypeName])) {
            parsedFragments[actualTypeName][field] = [];
        }
        parsedFragments[actualTypeName][field].push(parsedFragment);
    });
    var fragmentsByField = Object.create(null);
    Object.keys(parsedFragments).forEach(function (typeName) {
        Object.keys(parsedFragments[typeName]).forEach(function (field) {
            if (!(typeName in fragmentsByField)) {
                fragmentsByField[typeName] = Object.create(null);
            }
            fragmentsByField[typeName][field] = concatInlineFragments(typeName, parsedFragments[typeName][field]);
        });
    });
    stitchingInfo.selectionSetsByField = selectionSetsByField;
    stitchingInfo.dynamicSelectionSetsByField = dynamicSelectionSetsByField;
    stitchingInfo.fragmentsByField = fragmentsByField;
    return stitchingInfo;
}
function addStitchingInfo(stitchedSchema, stitchingInfo) {
    return new GraphQLSchema(__assign(__assign({}, stitchedSchema.toConfig()), { extensions: __assign(__assign({}, stitchedSchema.extensions), { stitchingInfo: stitchingInfo }) }));
}
function selectionSetContainsTopLevelField(selectionSet, fieldName) {
    return selectionSet.selections.some(function (selection) { return selection.kind === Kind.FIELD && selection.name.value === fieldName; });
}

function stitchSchemas(_a) {
    var _b = _a.subschemas, subschemas = _b === void 0 ? [] : _b, _c = _a.types, types = _c === void 0 ? [] : _c, typeDefs = _a.typeDefs, _d = _a.schemas, schemas = _d === void 0 ? [] : _d, onTypeConflict = _a.onTypeConflict, mergeDirectives = _a.mergeDirectives, _e = _a.mergeTypes, mergeTypes = _e === void 0 ? false : _e, typeMergingOptions = _a.typeMergingOptions, _f = _a.resolvers, resolvers = _f === void 0 ? {} : _f, schemaDirectives = _a.schemaDirectives, _g = _a.inheritResolversFromInterfaces, inheritResolversFromInterfaces = _g === void 0 ? false : _g, logger = _a.logger, _h = _a.allowUndefinedInResolve, allowUndefinedInResolve = _h === void 0 ? true : _h, _j = _a.resolverValidationOptions, resolverValidationOptions = _j === void 0 ? {} : _j, directiveResolvers = _a.directiveResolvers, _k = _a.schemaTransforms, schemaTransforms = _k === void 0 ? [] : _k, _l = _a.parseOptions, parseOptions = _l === void 0 ? {} : _l, pruningOptions = _a.pruningOptions;
    if (typeof resolverValidationOptions !== 'object') {
        throw new Error('Expected `resolverValidationOptions` to be an object');
    }
    var schemaLikeObjects = [];
    subschemas.forEach(function (subschemaOrSubschemaArray) {
        if (Array.isArray(subschemaOrSubschemaArray)) {
            schemaLikeObjects = schemaLikeObjects.concat(subschemaOrSubschemaArray);
        }
        else {
            schemaLikeObjects.push(subschemaOrSubschemaArray);
        }
    });
    schemas.forEach(function (schemaLikeObject) {
        if (!isSchema(schemaLikeObject) &&
            !isSubschemaConfig(schemaLikeObject) &&
            typeof schemaLikeObject !== 'string' &&
            !isDocumentNode$1(schemaLikeObject) &&
            !Array.isArray(schemaLikeObject)) {
            throw new Error('Invalid schema passed');
        }
    });
    schemas.forEach(function (schemaLikeObject) {
        if (isSchema(schemaLikeObject) || isSubschemaConfig(schemaLikeObject)) {
            schemaLikeObjects.push(schemaLikeObject);
        }
    });
    if ((typeDefs && !Array.isArray(typeDefs)) || (Array.isArray(typeDefs) && typeDefs.length)) {
        schemaLikeObjects.push(buildDocumentFromTypeDefinitions(typeDefs, parseOptions));
    }
    schemas.forEach(function (schemaLikeObject) {
        if (typeof schemaLikeObject === 'string' || isDocumentNode$1(schemaLikeObject)) {
            schemaLikeObjects.push(buildDocumentFromTypeDefinitions(schemaLikeObject, parseOptions));
        }
    });
    if (types != null) {
        schemaLikeObjects = schemaLikeObjects.concat(types);
    }
    schemas.forEach(function (schemaLikeObject) {
        if (Array.isArray(schemaLikeObject)) {
            schemaLikeObjects = schemaLikeObjects.concat(schemaLikeObject);
        }
    });
    var transformedSchemas = new Map();
    var extensions = [];
    var directives = [];
    var directiveMap = specifiedDirectives.reduce(function (acc, directive) {
        acc[directive.name] = directive;
        return acc;
    }, Object.create(null));
    var schemaDefs = Object.create(null);
    var operationTypeNames = {
        query: 'Query',
        mutation: 'Mutation',
        subscription: 'Subscription',
    };
    var typeCandidates = buildTypeCandidates({
        schemaLikeObjects: schemaLikeObjects,
        transformedSchemas: transformedSchemas,
        extensions: extensions,
        directiveMap: directiveMap,
        schemaDefs: schemaDefs,
        operationTypeNames: operationTypeNames,
        mergeDirectives: mergeDirectives,
    });
    Object.keys(directiveMap).forEach(function (directiveName) {
        directives.push(directiveMap[directiveName]);
    });
    var stitchingInfo = createStitchingInfo(transformedSchemas, typeCandidates, mergeTypes);
    var typeMap = buildTypeMap({
        typeCandidates: typeCandidates,
        stitchingInfo: stitchingInfo,
        operationTypeNames: operationTypeNames,
        onTypeConflict: onTypeConflict,
        mergeTypes: mergeTypes,
        typeMergingOptions: typeMergingOptions,
    });
    var _m = rewireTypes(typeMap, directives, { skipPruning: true }), newTypeMap = _m.typeMap, newDirectives = _m.directives;
    var schema = new GraphQLSchema({
        query: newTypeMap[operationTypeNames.query],
        mutation: newTypeMap[operationTypeNames.mutation],
        subscription: newTypeMap[operationTypeNames.subscription],
        types: Object.keys(newTypeMap).map(function (key) { return newTypeMap[key]; }),
        directives: newDirectives,
        astNode: schemaDefs.schemaDef,
        extensionASTNodes: schemaDefs.schemaExtensions,
        extensions: null,
    });
    extensions.forEach(function (extension) {
        schema = extendSchema(schema, extension, {
            commentDescriptions: true,
        });
    });
    // We allow passing in an array of resolver maps, in which case we merge them
    var resolverMap = Array.isArray(resolvers) ? resolvers.reduce(mergeDeep, {}) : resolvers;
    var finalResolvers = inheritResolversFromInterfaces
        ? extendResolversFromInterfaces(schema, resolverMap)
        : resolverMap;
    stitchingInfo = completeStitchingInfo(stitchingInfo, finalResolvers);
    schema = addResolversToSchema({
        schema: schema,
        resolvers: finalResolvers,
        resolverValidationOptions: resolverValidationOptions,
        inheritResolversFromInterfaces: false,
    });
    assertResolversPresent(schema, resolverValidationOptions);
    schema = addStitchingInfo(schema, stitchingInfo);
    if (!allowUndefinedInResolve) {
        schema = addCatchUndefinedToSchema(schema);
    }
    if (logger != null) {
        schema = addErrorLoggingToSchema(schema, logger);
    }
    if (typeof finalResolvers['__schema'] === 'function') {
        // TODO a bit of a hack now, better rewrite generateSchema to attach it there.
        // not doing that now, because I'd have to rewrite a lot of tests.
        schema = addSchemaLevelResolver(schema, finalResolvers['__schema']);
    }
    schemaTransforms.forEach(function (schemaTransform) {
        schema = schemaTransform(schema);
    });
    if (directiveResolvers != null) {
        schema = attachDirectiveResolvers(schema, directiveResolvers);
    }
    if (schemaDirectives != null) {
        SchemaDirectiveVisitor.visitSchemaDirectives(schema, schemaDirectives);
    }
    return pruningOptions ? pruneSchema(schema, pruningOptions) : schema;
}
function isDocumentNode$1(object) {
    return object.kind !== undefined;
}

var forwardArgsToSelectionSet = function (selectionSet, mapping) {
    var selectionSetDef = parseSelectionSet(selectionSet);
    return function (field) {
        var selections = selectionSetDef.selections.map(function (selectionNode) {
            if (selectionNode.kind === Kind.FIELD) {
                if (!mapping) {
                    return __assign(__assign({}, selectionNode), { arguments: field.arguments.slice() });
                }
                else if (selectionNode.name.value in mapping) {
                    var selectionArgs_1 = mapping[selectionNode.name.value];
                    return __assign(__assign({}, selectionNode), { arguments: field.arguments.filter(function (arg) { return selectionArgs_1.includes(arg.name.value); }) });
                }
            }
            return selectionNode;
        });
        return __assign(__assign({}, selectionSetDef), { selections: selections });
    };
};

export { forwardArgsToSelectionSet, stitchSchemas };
//# sourceMappingURL=index.esm.js.map
