"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SchemaReporter = exports.reportingLoop = exports.reportServerInfoGql = void 0;
const apollo_server_env_1 = require("apollo-server-env");
exports.reportServerInfoGql = `
  mutation ReportServerInfo($info: EdgeServerInfo!, $executableSchema: String) {
    me {
      __typename
      ... on ServiceMutation {
        reportServerInfo(info: $info, executableSchema: $executableSchema) {
          __typename
          ... on ReportServerInfoError {
            message
            code
          }
          ... on ReportServerInfoResponse {
            inSeconds
            withExecutableSchema
          }
        }
      }
    }
  }
`;
function reportingLoop(schemaReporter, logger, sendNextWithExecutableSchema, fallbackReportingDelayInMs) {
    function inner() {
        if (schemaReporter.stopped())
            return;
        schemaReporter
            .reportServerInfo(sendNextWithExecutableSchema)
            .then((result) => {
            switch (result.kind) {
                case "next": {
                    sendNextWithExecutableSchema = result.withExecutableSchema;
                    setTimeout(inner, result.inSeconds * 1000);
                    return;
                }
                case "stop": {
                    return;
                }
            }
        })
            .catch((error) => {
            logger.error(`Error reporting server info to Apollo Graph Manager during schema reporting: ${error}`);
            sendNextWithExecutableSchema = false;
            setTimeout(inner, fallbackReportingDelayInMs);
        });
    }
    inner();
}
exports.reportingLoop = reportingLoop;
class SchemaReporter {
    constructor(serverInfo, schemaSdl, apiKey, schemaReportingEndpoint, logger) {
        this.headers = new apollo_server_env_1.Headers();
        this.headers.set('Content-Type', 'application/json');
        this.headers.set('x-api-key', apiKey);
        this.headers.set('apollographql-client-name', 'apollo-engine-reporting');
        this.headers.set('apollographql-client-version', require('../package.json').version);
        this.url =
            schemaReportingEndpoint ||
                'https://edge-server-reporting.api.apollographql.com/api/graphql';
        this.serverInfo = serverInfo;
        this.executableSchemaDocument = schemaSdl;
        this.isStopped = false;
        this.logger = logger;
    }
    stopped() {
        return this.isStopped;
    }
    stop() {
        this.isStopped = true;
    }
    reportServerInfo(withExecutableSchema) {
        return __awaiter(this, void 0, void 0, function* () {
            const { data, errors } = yield this.graphManagerQuery({
                info: this.serverInfo,
                executableSchema: withExecutableSchema
                    ? this.executableSchemaDocument
                    : null,
            });
            if (errors) {
                throw new Error((errors || []).map((x) => x.message).join('\n'));
            }
            function msgForUnexpectedResponse(data) {
                return [
                    'Unexpected response shape from Apollo Graph Manager when',
                    'reporting server information for schema reporting. If',
                    'this continues, please reach out to support@apollographql.com.',
                    'Received response:',
                    JSON.stringify(data),
                ].join(' ');
            }
            if (!data || !data.me || !data.me.__typename) {
                throw new Error(msgForUnexpectedResponse(data));
            }
            if (data.me.__typename === 'UserMutation') {
                this.isStopped = true;
                throw new Error([
                    'This server was configured with an API key for a user.',
                    "Only a service's API key may be used for schema reporting.",
                    'Please visit the settings for this graph at',
                    'https://engine.apollographql.com/ to obtain an API key for a service.',
                ].join(' '));
            }
            else if (data.me.__typename === 'ServiceMutation' &&
                data.me.reportServerInfo) {
                if (data.me.reportServerInfo.__typename == 'ReportServerInfoResponse') {
                    return Object.assign(Object.assign({}, data.me.reportServerInfo), { kind: 'next' });
                }
                else {
                    this.logger.error([
                        'Received input validation error from Graph Manager:',
                        data.me.reportServerInfo.message,
                        'Stopping reporting. Please fix the input errors.',
                    ].join(' '));
                    this.stop();
                    return {
                        stopReporting: true,
                        kind: 'stop'
                    };
                }
            }
            throw new Error(msgForUnexpectedResponse(data));
        });
    }
    graphManagerQuery(variables) {
        return __awaiter(this, void 0, void 0, function* () {
            const request = {
                query: exports.reportServerInfoGql,
                operationName: 'ReportServerInfo',
                variables: variables,
            };
            const httpRequest = new apollo_server_env_1.Request(this.url, {
                method: 'POST',
                headers: this.headers,
                body: JSON.stringify(request),
            });
            const httpResponse = yield apollo_server_env_1.fetch(httpRequest);
            if (!httpResponse.ok) {
                throw new Error([
                    `An unexpected HTTP status code (${httpResponse.status}) was`,
                    'encountered during schema reporting.',
                ].join(' '));
            }
            try {
                return yield httpResponse.json();
            }
            catch (error) {
                throw new Error([
                    "Couldn't report server info to Apollo Graph Manager.",
                    'Parsing response as JSON failed.',
                    'If this continues please reach out to support@apollographql.com',
                    error,
                ].join(' '));
            }
        });
    }
}
exports.SchemaReporter = SchemaReporter;
//# sourceMappingURL=schemaReporter.js.map