"use strict";

exports.__esModule = true;
exports.default = void 0;

var _resizeObserverPolyfill = /*#__PURE__*/_interopRequireDefault( /*#__PURE__*/require("resize-observer-polyfill"));

var _passiveLayoutEffect = /*#__PURE__*/_interopRequireDefault( /*#__PURE__*/require("@react-hook/passive-layout-effect"));

var _latest = /*#__PURE__*/_interopRequireDefault( /*#__PURE__*/require("@react-hook/latest"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * A React hook that fires a callback whenever ResizeObserver detects a change to its size
 *
 * @param target A React ref created by `useRef()` or an HTML element
 * @param callback Invoked with a single `ResizeObserverEntry` any time
 *   the `target` resizes
 */
const useResizeObserver = (target, callback) => {
  const resizeObserver = getResizeObserver();
  const storedCallback = (0, _latest.default)(callback);
  (0, _passiveLayoutEffect.default)(() => {
    let didUnsubscribe = false;

    const callback = (entries, observer) => {
      if (didUnsubscribe) return;
      const targetEl = target && 'current' in target ? target.current : target;

      for (let i = 0; i < entries.length; i++) {
        const entry = entries[i];

        if (entry.target === targetEl) {
          storedCallback.current(entry, observer);
        }
      }
    };

    resizeObserver.subscribe(callback);
    return () => {
      didUnsubscribe = true;
      resizeObserver.unsubscribe(callback);
    };
  }, [target, resizeObserver, storedCallback]);
  (0, _passiveLayoutEffect.default)(() => {
    const targetEl = target && 'current' in target ? target.current : target;
    if (!targetEl) return;
    resizeObserver.observer.observe(targetEl);
    return () => resizeObserver.observer.unobserve(targetEl);
  }, [target, resizeObserver.observer]);
  return resizeObserver.observer;
};

const createResizeObserver = () => {
  const callbacks = new Set();
  return {
    observer: new _resizeObserverPolyfill.default((entries, observer) => {
      for (const callback of callbacks) callback(entries, observer);
    }),
    subscribe: callback => callbacks.add(callback),
    unsubscribe: callback => callbacks.delete(callback)
  };
};

let _resizeObserver;

const getResizeObserver = () => !_resizeObserver ? _resizeObserver = createResizeObserver() : _resizeObserver;

var _default = useResizeObserver;
exports.default = _default;