# Cluster API bootstrap provider kubeadm
## What is the Cluster API bootstrap provider kubeadm?

Cluster API bootstrap provider Kubeadm (CABPK) is a component responsible for generating a cloud-init script to
turn a Machine into a Kubernetes Node. This implementation uses [kubeadm](https://github.com/kubernetes/kubeadm)
for Kubernetes bootstrap.

### Resources

* [design doc](https://github.com/kubernetes-sigs/cluster-api/blob/master/docs/proposals/20190610-machine-states-preboot-bootstrapping.md)
* [The Kubebuilder Book](https://book.kubebuilder.io)

## How does CABPK work?
CABPK is integrated into `cluster-api-manager`. Assuming you've set of CAPI and the Docker Manager, create a `Cluster` object and its corresponding `DockerCluster`
infrastructure object.

```yaml
kind: DockerCluster
apiVersion: infrastructure.cluster.x-k8s.io/v1alpha3
metadata:
  name: my-cluster-docker
---
kind: Cluster
apiVersion: cluster.x-k8s.io/v1alpha3
metadata:
  name: my-cluster
spec:
  infrastructureRef:
    kind: DockerCluster
    apiVersion: infrastructure.cluster.x-k8s.io/v1alpha3
    name: my-cluster-docker
```

Now you can start creating machines by defining a `Machine`, its corresponding `DockerMachine` object, and
the `KubeadmConfig` bootstrap object.

```yaml
kind: KubeadmConfig
apiVersion: bootstrap.cluster.x-k8s.io/v1alpha3
metadata:
  name: my-control-plane1-config
spec:
  initConfiguration:
    nodeRegistration:
      kubeletExtraArgs:
        eviction-hard: nodefs.available<0%,nodefs.inodesFree<0%,imagefs.available<0%
  clusterConfiguration:
    controllerManager:
      extraArgs:
        enable-hostpath-provisioner: "true"
---
kind: DockerMachine
apiVersion: infrastructure.cluster.x-k8s.io/v1alpha3
metadata:
  name: my-control-plane1-docker
---
kind: Machine
apiVersion: cluster.x-k8s.io/v1alpha3
metadata:
  name: my-control-plane1
  labels:
    cluster.x-k8s.io/cluster-name: my-cluster
    cluster.x-k8s.io/control-plane: "true"
    set: controlplane
spec:
  bootstrap:
    configRef:
      kind: KubeadmConfig
      apiVersion: bootstrap.cluster.x-k8s.io/v1alpha3
      name: my-control-plane1-config
  infrastructureRef:
    kind: DockerMachine
    apiVersion: infrastructure.cluster.x-k8s.io/v1alpha3
    name: my-control-plane1-docker
  version: "v1.19.1"
```

CABPK's main responsibility is to convert a `KubeadmConfig` bootstrap object into a cloud-init script that is
going to turn a Machine into a Kubernetes Node using `kubeadm`.

The cloud-init script will be saved into a secret `KubeadmConfig.Status.DataSecretName` and then the infrastructure provider
(CAPD in this example) will pick up this value and proceed with the machine creation and the actual bootstrap.

### KubeadmConfig objects
The `KubeadmConfig` object allows full control of Kubeadm init/join operations by exposing raw `InitConfiguration`,
`ClusterConfiguration` and `JoinConfiguration` objects.

CABPK will fill in some values if they are left empty with sensible defaults:

| `KubeadmConfig` field                           | Default                                                      |
| ----------------------------------------------- | ------------------------------------------------------------ |
| `clusterConfiguration.KubernetesVersion`        | `Machine.Spec.Version`                                     |
| `clusterConfiguration.clusterName`              | `Cluster.metadata.name`                                      |
| `clusterConfiguration.controlPlaneEndpoint`     | `Cluster.status.apiEndpoints[0]` |
| `clusterConfiguration.networking.dnsDomain` | `Cluster.spec.clusterNetwork.serviceDomain`              |
| `clusterConfiguration.networking.serviceSubnet` | `Cluster.spec.clusterNetwork.service.cidrBlocks[0]`              |
| `clusterConfiguration.networking.podSubnet` | `Cluster.spec.clusterNetwork.pods.cidrBlocks[0]`              |
| `joinConfiguration.discovery`                   | a short lived BootstrapToken generated by CABPK              |

> IMPORTANT! overriding above defaults could lead to broken Clusters.

#### Examples
Valid combinations of configuration objects are:
- at least one of `InitConfiguration` and `ClusterConfiguration` for the first control plane node only
- `JoinConfiguration` for worker nodes and additional control plane nodes

Bootstrap control plane node:
```yaml
kind: KubeadmConfig
apiVersion: bootstrap.cluster.x-k8s.io/v1alpha3
metadata:
  name: my-control-plane1-config
spec:
  initConfiguration:
    nodeRegistration:
      kubeletExtraArgs:
        eviction-hard: nodefs.available<0%,nodefs.inodesFree<0%,imagefs.available<0%
  clusterConfiguration:
    controllerManager:
      extraArgs:
        enable-hostpath-provisioner: "true"
```

Additional control plane nodes:
```yaml
kind: KubeadmConfig
apiVersion: bootstrap.cluster.x-k8s.io/v1alpha3
metadata:
  name: my-control-plane2-config
spec:
  joinConfiguration:
    nodeRegistration:
      kubeletExtraArgs:
        eviction-hard: nodefs.available<0%,nodefs.inodesFree<0%,imagefs.available<0%
    controlPlane: {}
```

worker nodes:
```yaml
kind: KubeadmConfig
apiVersion: bootstrap.cluster.x-k8s.io/v1alpha3
metadata:
  name: my-worker1-config
spec:
  joinConfiguration:
    nodeRegistration:
      kubeletExtraArgs:
        eviction-hard: nodefs.available<0%,nodefs.inodesFree<0%,imagefs.available<0%
```

### Bootstrap Orchestration
CABPK supports multiple control plane machines initing at the same time.
The generation of cloud-init scripts of different machines is orchestrated in order to ensure a cluster
bootstrap process that will be compliant with the correct Kubeadm init/join sequence. More in detail:
1. cloud-config-data generation starts only after `Cluster.InfrastructureReady` flag is set to `true`.
2. at this stage, cloud-config-data will be generated for the first control plane machine even
if multiple control plane machines are ready (kubeadm init).
3. after `Cluster.metadata.Annotations[cluster.x-k8s.io/control-plane-ready]` is set to true,
the cloud-config-data for all the other machines are generated (kubeadm join/join —control-plane).

### Certificate Management
The user can choose two approaches for certificate management:
1. provide required certificate authorities (CAs) to use for `kubeadm init/kubeadm join --control-plane`; such CAs
should be provided as a `Secrets` objects in the management cluster.
2. let CABPK to generate the necessary `Secrets` objects with a self-signed certificate authority for kubeadm

See [here](ttps://kubernetes.io/docs/tasks/administer-cluster/kubeadm/kubeadm-certs/) for more info about certificate management with kubeadm.

### Additional Features
The `KubeadmConfig` object supports customizing the content of the config-data. The following examples illustrate how to specify these options. They should be adapted to fit your environment and use case.

- `KubeadmConfig.Files` specifies additional files to be created on the machine, either with content inline or by referencing a secret.

    ```yaml
    files:
    - contentFrom:
        secret:
          key: node-cloud.json
          name: ${CLUSTER_NAME}-md-0-cloud-json
      owner: root:root
      path: /etc/kubernetes/cloud.json
      permissions: "0644"
    - path: /etc/kubernetes/cloud.json
      owner: "root:root"
      permissions: "0644"
      content: |
        {
          "cloud": "CustomCloud"
        }
    ```

- `KubeadmConfig.PreKubeadmCommands` specifies a list of commands to be executed before `kubeadm init/join`

    ```yaml
    preKubeadmCommands:
      - hostname "{{ ds.meta_data.hostname }}"
      - echo "{{ ds.meta_data.hostname }}" >/etc/hostname
    ```

- `KubeadmConfig.PostKubeadmCommands` same as above, but after `kubeadm init/join`

    ```yaml
    postKubeadmCommands:
      - echo "success" >/var/log/my-custom-file.log
    ```

- `KubeadmConfig.Users` specifies a list of users to be created on the machine

    ```yaml
    users:
      - name: capiuser
        sshAuthorizedKeys:
        - '${SSH_AUTHORIZED_KEY}'
        sudo: ALL=(ALL) NOPASSWD:ALL
    ```

- `KubeadmConfig.NTP` specifies NTP settings for the machine

  ```yaml
  ntp:
    servers:
      - IP_ADDRESS
    enabled: true
  ```

- `KubeadmConfig.DiskSetup` specifies options for the creation of partition tables and file systems on devices.

  ```yaml
  diskSetup:
    filesystems:
    - device: /dev/disk/azure/scsi1/lun0
      extraOpts:
      - -E
      - lazy_itable_init=1,lazy_journal_init=1
      filesystem: ext4
      label: etcd_disk
    - device: ephemeral0.1
      filesystem: ext4
      label: ephemeral0
      replaceFS: ntfs
    partitions:
    - device: /dev/disk/azure/scsi1/lun0
      layout: true
      overwrite: false
      tableType: gpt
  ```

- `KubeadmConfig.Mounts` specifies a list of mount points to be setup.

    ```yaml
    mounts:
    - - LABEL=etcd_disk
      - /var/lib/etcddisk
    ```

- `KubeadmConfig.Verbosity` specifies the `kubeadm` log level verbosity

    ```yaml
    verbosity: 10
    ```

- `KubeadmConfig.UseExperimentalRetryJoin` replaces a basic kubeadm command with a shell script with retries for joins. This will add about 40KB to userdata.

    ```yaml
    useExperimentalRetryJoin: true
    ```

For more information on cloud-init options, see [cloud config examples](https://cloudinit.readthedocs.io/en/latest/topics/examples.html).
