/* adapted from Eric Youngdale's readelf program */

#ifndef _LIBC
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <linux/elf.h>
#include "../config.h"
#include "readelf.h"
#include <sys/types.h>

void warn(const char *fmt, ...);
char *xstrdup(const char *);

struct needed_tab
{
  const char *soname;
  int type;
};

struct needed_tab needed_tab[] = {
  { "libc.so.5",    LIB_ELF_LIBC5 },
  { "libm.so.5",    LIB_ELF_LIBC5 },
  { "libdl.so.1",   LIB_ELF_LIBC5 },
  { "libc.so.6",    LIB_ELF_LIBC6 },
  { "libm.so.6",    LIB_ELF_LIBC6 },
  { "libdl.so.2",   LIB_ELF_LIBC6 },
  { NULL,           LIB_ELF }
};
#endif

#ifdef _LIBC
static
#endif
char *
readsoname(const char *name, FILE *infile,
	   int expected_type, int *type, const char *e_ident)
{
#ifdef _LIBC
    const ElfW(Ehdr) *epnt;
    const ElfW(Phdr) *ppnt;
#else
    const struct elfhdr *epnt;
    const struct elf_phdr *ppnt;
#endif
    int i, j;
    char *header;
    unsigned int dynamic_addr = 0;
    unsigned int dynamic_size = 0;
    int strtab_val = 0;
    int needed_val;
    int loadaddr = -1;
#ifdef _LIBC
    const ElfW(Dyn) *dpnt;
#else
    const struct dynamic *dpnt;
#endif
    struct stat st;
    char *needed;
    char *soname = NULL;
    int multi_libcs = 0;

    if(expected_type == LIB_DLL)
	{
	    warn("%s does not match type specified for directory!", name);
	    expected_type = LIB_ANY;
	}

    *type = LIB_ELF;

    if (fstat(fileno(infile), &st))
	return NULL;
    header = mmap(0, st.st_size, PROT_READ, MAP_SHARED, fileno(infile), 0);
    if (header == (caddr_t)-1)
	return NULL;

#ifdef _LIBC
    epnt = (const ElfW(Ehdr) *)header;
#else
    epnt = (const struct elfhdr *)header;
#endif
    if ((char *)(epnt+1) > (char *)(header + st.st_size))
	goto skip;

#ifdef _LIBC
    ppnt = (const ElfW(Phdr) *)&header[epnt->e_phoff];
#else
    ppnt = (const struct elf_phdr *)&header[epnt->e_phoff];
#endif
    if ((char *)ppnt < (char *)header ||
	(char *)(ppnt+epnt->e_phnum) > (char *)(header + st.st_size))
	goto skip;

    for(i = 0; i < epnt->e_phnum; i++)
	{
	    if (loadaddr == -1 && ppnt->p_type == PT_LOAD) 
		loadaddr = (ppnt->p_vaddr & 0xfffff000) -
		    (ppnt->p_offset & 0xfffff000);
	    if(ppnt->p_type == 2)
		{
		    dynamic_addr = ppnt->p_offset;
		    dynamic_size = ppnt->p_filesz;
		};
	    ppnt++;
	};
    
#ifdef _LIBC
    dpnt = (const ElfW(Dyn) *) &header[dynamic_addr];
    dynamic_size = dynamic_size / sizeof(const ElfW(Dyn));
#else
    dpnt = (const struct dynamic *) &header[dynamic_addr];
    dynamic_size = dynamic_size / sizeof(const struct dynamic);
#endif
    if ((char *)dpnt < (char *)header ||
	(char *)(dpnt+dynamic_size) > (char *)(header + st.st_size))
	goto skip;
  
    while (dpnt->d_tag != DT_NULL)
	{
	    if (dpnt->d_tag == DT_STRTAB)
		strtab_val = dpnt->d_un.d_val;
	    dpnt++;
	};

    if (!strtab_val)
	goto skip;

#ifdef _LIBC
    dpnt = (const ElfW(Dyn) *) &header[dynamic_addr];
#else
    dpnt = (const struct dynamic *) &header[dynamic_addr];
#endif
    while (dpnt->d_tag != DT_NULL)
	{
	    if (dpnt->d_tag == DT_SONAME || dpnt->d_tag == DT_NEEDED)
		{
		    needed_val = dpnt->d_un.d_val;
		    if (needed_val + strtab_val - loadaddr >= 0 ||
			needed_val + strtab_val - loadaddr < st.st_size)
			{
			    needed = (char *) (header - loadaddr + strtab_val + needed_val);

			    if (dpnt->d_tag == DT_SONAME)
				soname = xstrdup(needed);

			    for (j = 0; needed_tab[j].soname != NULL; j++)
				{
				    if (strcmp(needed, needed_tab[j].soname) == 0)
					{
					    if (*type != LIB_ELF && *type != needed_tab[j].type)
						multi_libcs = 1;
					    *type = needed_tab[j].type;
					}
				}
			}
		}
	    dpnt++;
	};
  
    if (multi_libcs)
	warn("%s appears to be for multiple libc's", name);

    /* If we could not deduce the libc type, and we know what to expect, set the type */
    if(*type == LIB_ELF && expected_type != LIB_ANY) *type = expected_type;

    if(expected_type != LIB_ANY && expected_type != LIB_ELF && 
       expected_type != *type)
	{
	    warn("%s does not match type specified for directory!", name);
	}

 skip:
    munmap(header, st.st_size);

    return soname;
}
