/**
 * \file pappsomspp/processing/cbor/jsonstreamwriter.h
 * \date 19/07/2025
 * \author Olivier Langella
 * \brief PAPPSO JSON stream writer
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of PAPPSOms-tools.
 *
 *     PAPPSOms-tools is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms-tools is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/
#pragma once

#include <QTextStream>
#include <QCborMap>
#include <QCborArray>
#include <QIODevice>


#include "pappsomspp/export-import-config.h"

namespace pappso::cbor
{
/**
 * @brief helper class to write json text using CBOR data structures
 */
class PMSPP_LIB_DECL JsonStreamWriter : public QTextStream
{
  public:
  /** @brief build a QTextStream for JSON writer
   */
  JsonStreamWriter(QIODevice *device);

  /**
   * Destructor
   */
  virtual ~JsonStreamWriter();


  /** @brief starts an array in JSON output
   * simply print "["
   */
  void startArray();


  /** @brief ends an array in JSON output
   * simply print "]"
   */
  void endArray();


  /** @brief start an object in JSON output
   * simply print "{"
   */
  void startMap();


  /** @brief ends an object in JSON output
   * simply print "}"
   */
  void endMap();


  /** @brief converts a CBOR map (object) into JSON text in output
   * @param cbor_map the map to convert
   */
  void writeCborMap(const QCborMap &cbor_map);


  /** @brief converts a CBOR array (list) into JSON text in output
   * @param cbor_array the map to convert
   */
  void writeCborArray(const QCborArray &cbor_array);

  /** @brief writes a key (dictionary entry) to JSON text
   */
  void appendKey(const QString &key);


  /** @brief writes String value to JSON text
   * @warning this function does not escapes text correctly, use with caution
   */
  void appendValue(const QString &value);


  /** @brief writes an integer value to JSON text
   */
  void appendValue(std::size_t integer_size_value);

  /** @brief  convert a string list to a JSON array
   * @warning text is not escaped correctly, use with caution
   */
  void writeArray(QStringList &str_list);

  protected:
  void comma();

  private:
  bool m_isStart     = true;
  bool m_nextIsValue = false;
};
} // namespace pappso::cbor
