/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * ddtd.cc
 *   This is the main file for the server daemon.
 *
 * $Id: ddtd.cc,v 1.40 2001/11/24 21:11:50 benoit Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@debian.org>
 * Copyright (c) 2000 Luca Filipozzi <lfilipoz@debian.org>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */

#include "ddtd.h"

Options *Options::i_ = NULL;
Logger *Server::log = NULL;
Options *Server::opts = NULL;
UDP *Server::udp = NULL;

DdtManager *Server::manager = NULL;
Db *Server::db = NULL;
Dns *Server::dns = NULL;


Mutex *Server::mutex = NULL;

int main(int argc, char** argv)
{ 
    Server *ddtd = new Server;

    // check we are not already running
    ddtd->alreadyRunning();

    if (ddtd->parseArgs(&argc, &argv) == -1)
    {
        exit(1);
    }

    ddtd->becomeDaemon();
    ddtd->run();

    return 0;
}

SendThread::SendThread(Server *ddtd) : Thread((void *)ddtd)
{
    // intentionally empty
}

void *SendThread::run(void *arg)
{
    // every minute, execute the prune routine on each alive hosts
    while (true)
    {
        sleep(60);
        ((Server *)arg)->validateActiveAccounts();
    }
    return NULL;
}

void SendThread::cleanUp() { /* empty */ }


Server::Server ()
{
    opts = Options::Instance();
    sendThread = NULL;

    // exit cleanly at all times
    atexit(atexit_handler);

    // trap terminating signals
    signal(SIGHUP,  signal_handler);
    signal(SIGINT,  signal_handler);
    signal(SIGQUIT, signal_handler);
    signal(SIGABRT, signal_handler);
    signal(SIGTERM, signal_handler);
}

Server::~Server ()
{
    if (log) delete log;
    if (opts) delete opts;
    if (udp) delete udp;

    if (db) delete db;
    if (dns) delete dns;
    if (manager) delete manager;
    
    if (mutex) delete mutex;
    if (sendThread) delete sendThread;
}

/* FIXME: 
 *  this functions is very evil. we look for information in /proc,
 *  this is not very portable. the whole pidfile handling is quite
 *  disgusting as well.
 */
int Server::alreadyRunning()
{
    // check if there's a pidfile .. if yes, check if it is stale
    FILE *pidfp;
    FILE *cmdlinefilefp;
    char cmdline[64];
    char cmdlinefile[64];
    int pid = 0;
    struct stat statbuf;
   
    // pid dir must exist or fopen() would segfault
    if (stat (PID_DIR, &statbuf) == 0)
    {
        if (!S_ISDIR (statbuf.st_mode))
        {
            cout << PID_DIR << " is not a directory.";
            exit (1);
        }
    }
    if (stat (PID_DIR "/ddt", &statbuf) == 0)
    {
        if (!S_ISDIR (statbuf.st_mode))
        {
            cout << PID_DIR "/ddt exists but is not a directory.";
            exit (1);
        }
    }
    else
    {
        cout << "creating " PID_DIR "/ddt ...";
        mkdir (PID_DIR "/ddt", S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
    }
    
    // we check if theres a pidfile
    if ((pidfp = fopen (PID_FILE, "r")) != NULL)
    {
        // what's the pid
        if (fscanf (pidfp, "%d", &pid) == 1)
        {
            // if process exists
            if (kill (pid, 0) == 0)
            {
                // we check if it's a ddtcd or any other random process
                snprintf (cmdlinefile, sizeof (cmdlinefile), "/proc/%d/cmdline", pid);
                
                if (((cmdlinefilefp = fopen (cmdlinefile, "r"))) != NULL)
                {
                    if (fscanf (cmdlinefilefp, "%s", cmdline) == 1)
                    {
                        if (strstr (cmdline, "ddtd") != NULL)
                        {
                            // it was a ddtcd, exit
                            cout << "daemon already running, exiting\n";
                            exit (0);
                        }
                        else
                        {
                            // it was really stale, remove the pid file and
                            // continue
                            cout << "removing stale pid file\n";
                            unlink (PID_FILE);
                        }
                    }
                }
                fclose (cmdlinefilefp);
            }
            else
            {
                cout << "removing stale pid file\n";
                unlink (PID_FILE);
                fclose (pidfp);
            }
        }
    }
    return 0;
}

int Server::parseArgs(int *argc, char ***argv)
{
    optTitle("ddtd - Dynamic Host Information System Daemon\n");
    optrega(&opts->verbose, OPT_FLAG,   'v', "verbose",  " verbose operation");
    optrega(&opts->debug,   OPT_FLAG,   'd', "debug",    " debug operation");
    optrega(&opts->dbname,  OPT_STRING, 'n', "dbname",   "*database name");
    optrega(&opts->dbuser,  OPT_STRING, 'u', "dbuser",   "*database user");
    optrega(&opts->dbpass,  OPT_STRING, 'p', "dbpass",   "*database password");
    optrega(&opts->serverport, OPT_UINT,'S', "serverport", "*server udp port");
    optrega(&opts->clientport, OPT_UINT,'C', "clientport", "*client udp port");
    optrega(&opts->version, OPT_FLAG,   'V', "version",    " version info");
    optAdditionalUsage(opts->additionalUsage);
    optDefaultFile(CONF_FILE);

    // parse command line and/or read config file
    opt(argc, argv);

    // validate other options
    bool exitRequired = false;
    if (opts->version)
    {
        cerr << "ddtd v" << VERSION << endl;
        exit(0);
    }
    if (!optinvoked(&opts->dbuser))
    {
        cerr << "E: no database owner provided." << endl;
        exitRequired = true;
    }
    if (!optinvoked(&opts->dbname))
    {
        cerr << "E: no database name provided." << endl;
        exitRequired = true;
    }
    if (!optinvoked(&opts->dbpass))
    {
        cerr << "E: no database password provided" << endl;
        exitRequired = true;
    }
    if (!optinvoked(&opts->serverport))
    {
        cerr << "E: no server port provided." << endl;
        exitRequired = true;
    }
    if (!optinvoked(&opts->clientport))
    {
        cerr << "E: no client port provided." << endl;
        exitRequired = true;
    }
    if (exitRequired)
    {
        cerr << "missing or bad options. aborting." << endl;
        return -1;
    }

    try
    {
        log = Logger::Instance();
        
        log->setIdent("ddtd");
        log->openSyslog (LOG_USER, LOG_NOTICE);
        log->openFilelog (LOG_FILE, LOG_NOTICE);

        // validate verbosity/debugging options
        if (opts->verbose)
        {
            log->setFilelogPriority(LOG_INFO);
        }
        if (opts->debug)
        {
            log->setFilelogPriority(LOG_DEBUG);
        }

        udp = new UDP(opts->clientport, opts->serverport);

        /* FIXME
         * the database type is hardcoded for now. this will eventually
         * be in the configuration file and will be used to load modules
         * dynamically
         */
        db = new DbPsql(log, opts->dbname, opts->dbuser, opts->dbpass);
        dns = new DnsBind(log);
        manager = new DdtManager(log, db, dns);

        mutex = new Mutex;
        sendThread  = new SendThread(this);
    }
    catch (DbError &e)
    {
        // specialised handler, do the same thing right now. that will change
        cerr << e.message() << endl;
        return -1;
    }
    catch (DdtException &e)
    {
        cerr << e.message() << endl;
        return -1;
    }
    catch (...)
    {
        cerr << "Unknown error" << endl;
        return -1;
    }

    return 0;
}

int Server::becomeDaemon()
{

// become a daemon
#ifndef DDT_DEBUG
#ifdef HAVE_DAEMON
    // XXX: RH6.0's daemon(), undef HAVE_DAEMON and it should work
    daemon(true, false);
#else
    if (fork ()) { exit (0); }
    close (0); close (1); close (2);
#endif // HAVE_DAEMON
#endif // DDT_DEBUG

    log->notice ("daemon starting");

    struct passwd *nobody;
    struct group *nogroup;
    nobody = getpwnam ("nobody");
    nogroup = getgrnam ("nogroup");
    
    // create the dir.
    mkdir(PID_DIR "/ddt", S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);

    // if we are root
    if (getuid() == 0)
    {
        // if nobody and nogroup exists on system
        if (nobody != NULL && nogroup != NULL)
        {
            if (chown(PID_DIR "/ddt", nobody->pw_uid, nogroup->gr_gid) == -1)
            {
                log->debug("failed to give piddir to nobody.nogroup: %s",
                           strerror (errno));
            }
            
            // drop group priviliedges first. :)
            if (setgid(nogroup->gr_gid) != -1)
            {
                log->debug("setgid nogroup; dropping root");
            }
            else
            {
                log->error("failed to setgid nogroup: %s", strerror (errno));
            }

            // drop user priviledges
            if (setuid(nobody->pw_uid) != -1)
            {
                log->debug("setuid nobody; dropping root");
            }
            else
            {
                log->error("failed to setuid nobody: %s", strerror(errno));
            }
        }
        else
        {
            log->error("user nobody and/or group nogroup not found on this "
                       "system; continuing anyway");
        }
    }
    else
    {
        log->error("not root, cannot get nobody uid");
    }

    // write pid file
    int fd;
    if ((fd = creat(PID_FILE, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) != -1)
    {
        char buf[8];
        sprintf(buf, "%d\n", getpid());
        if (write(fd, buf, strlen(buf)) == -1)
        {
            close(fd);
        }
    }
    else
    {
        log->error("failed to create pid file: %s", PID_FILE);
    }

    return 0;
}

int Server::run ()
{
    // start the UDP sender thread
    sendThread->start();
    receiver();

    return 0;
}

int Server::receiver ()
{
    unsigned long from;
    DdtpPacket pkt;
    int bufSize = 2 * sizeof(DdtpPacket);
    char buf[bufSize];
    short md5Length = DdtMD::getAlgoDigestLength(DdtMD::MD_MD5);

    // receive packets and process them
    while (1)
    {
        if (udp->recv(buf, bufSize, &from) == -1)
        {
            log->error("udp receive failed: %d", strerror(errno));
            continue;
        }
        
        if (Packet::decode(&pkt, buf, sizeof(buf)) == -1)
        {
            log->error("failed to decode packet");
            continue;
        }

        int accountId;
        if ((accountId = Packet::getAccountId(&pkt)) == -1)
        {
            log->error("failed to extract account id from packet");
            continue;
        }

        mutex->lock();

        char db_password[2 * md5Length + 1];
        if (manager->fetchAccountInfo(accountId, "updatePassword", db_password,
                                      (2 * md5Length + 1)) == -1)
        {
            log->warning("%d account not found", accountId);

            /* FIXME: we don't reply here even though we should
             * The current libPacket structure renders it hard for us
             * to send the reply right here. I will not attemp to fix
             * that since its not such an inconveniance and we are working
             * on the packets library. expect a lot of things to change
             * there with DDTPv2. */
            
            mutex->unlock();
            continue;
        }
        mutex->unlock();

        if (Packet::decrypt(&pkt, db_password) == -1)
        {
            log->error("failed to decrypt packet");
            continue;
        }

        mutex->lock();
        if (processReceivedPacket(&pkt, accountId, db_password) == -1)
        {
            // note that this is a little hack, in that
            // if we need to send a reply to the server
            // processReceivedPacket returns 0 otherwise
            // we short circuit the loop at this point
            mutex->unlock();
            continue;
        }
        mutex->unlock();

        if (Packet::encrypt(&pkt, db_password) == -1)
        {
            log->error("failed to encrypt packet");
            continue;
        }

        if (Packet::encode(&pkt, buf, bufSize) == -1)
        {
            log->error ("failed to encode packet");
            continue;
        }

        mutex->lock();
        if (udp->sendTo(from, buf, bufSize) == -1)
        {
            log->error("udp sendto failed: %s - (update query reply)",
                       strerror (errno));
            mutex->unlock();
            continue;
        }
        mutex->unlock();
    }
}

void Server::validateActiveAccounts()
{
    mutex->lock();
    manager->pruneActiveAccount(Server::processActiveAccount);
    mutex->unlock();
}

void Server::processActiveAccount(int accountId, unsigned long addr, 
                                  time_t lastAccessTime)
{
    char fqdn[256];
    manager->fetchAccountInfo(accountId, "fqdn", fqdn, sizeof(fqdn));

    // if the entry is "stale", prune it
    if (difftime(time(NULL), lastAccessTime) > (TIMEOUT))
    {
        manager->markOffline(accountId);
        log->notice("%d (%s) timed out; marked offline", accountId, fqdn);
    }
    else // otherwise, send the host an ALIVE_QUERY
    {
        DdtpPacket pkt;
        int bufSize = 2 * sizeof(DdtpPacket);
        char buf[bufSize];

        // generate the DdtpPacket
        Packet::createAliveQuery(&pkt, DDTPv1, accountId);

        if (Packet::encode(&pkt, buf, bufSize) == -1)
        {
            log->error("failed to encode packet");
            return;
        }

        if (udp->sendTo(addr, buf, bufSize) == -1)
        {
            log->error("udp sendto %d (%s) failed: %s - (alive query)",
                       accountId, fqdn, strerror (errno));
            return;
        }

        log->info("%d (%s) alive query sent", accountId, fqdn);
    }
}

int Server::processReceivedPacket(DdtpPacket* pkt, int accountId,
                                  char* db_password)
{
    int rval = -1;

    switch (pkt->protocolVersion)
    {
        case DDTPv1:
        {
            DDTPv1Message* msg=&pkt->DdtpPacket_u.ddtpv1.DDTPv1Packet_u.pt.msg;
            switch (msg->messageType)
            {
                case UPDATE_QUERY:
                    rval = processDDTPv1UpdateQuery (
                        &(msg->DDTPv1Message_u.updateQuery),
                        &(msg->DDTPv1Message_u.updateReply),
                        accountId,
                        db_password);
                    msg->messageType = UPDATE_REPLY;
                    break;
                case ALIVE_REPLY:
                    rval = processDDTPv1AliveReply (
                        &(msg->DDTPv1Message_u.aliveReply),
                        accountId);
                    break;
                default:
                    break;
            }
            pkt->DdtpPacket_u.ddtpv1.encryptionType = PLAINTEXT;
            return rval;
            break;
        }
        case DDTPv2:
        {
            break;
        }
        default:
            break;
    }
    return -1;  // return -1 by default (see hack comment above)
}

int Server::processDDTPv1UpdateQuery (DDTPv1UpdateQuery* updateQuery,
                                      DDTPv1UpdateReply* updateReply,
                                      int accountId,
                                      char* db_password)
{
    short md5Length = DdtMD::getAlgoDigestLength(DdtMD::MD_MD5);
    char fqdn[256];
    manager->fetchAccountInfo (accountId, "fqdn", fqdn, sizeof(fqdn));

    log->debug("received update query from account %d (%s)", accountId, fqdn);

    // retreive the password that's in the message and MD5 hash it
    char          msg_password[md5Length * 2 + 1];
    unsigned char       digest[md5Length];

    char updatePassword[9];
    strncpy  (updatePassword, updateQuery->updatePassword, 9);

    DdtCipher cipher(DdtCipher::ALGO_ROT13, DdtCipher::MODE_NONE);
    cipher.encrypt((unsigned char*)updatePassword, strlen(updatePassword), 
                   (unsigned char*)updatePassword, strlen(updatePassword));

    DdtMD::digest(DdtMD::MD_MD5, (unsigned char*)updatePassword, strlen(updatePassword), digest, md5Length);

    for (int i = 0; i < md5Length; i++)
    {
        sprintf(&(msg_password[i*2]), "%02x", digest[i]);
    }

    // validate the message password against the database password
    if (strncasecmp (msg_password, db_password, 2 * MD5_DIGEST_LENGTH))
    {
        log->warning ("%d (%s) password invalid", accountId, fqdn);
        updateReply->status = INVALID_PASSWORD;
        return 0;
    }

    // process the query
    switch (updateQuery->opCode)
    {
        // FIXME: here we should set the hoststatus to WAITING_ONLINE
        //        in the database and wait a first keepalive reply
        //        before actually changing the IP to avoid bad IPs.
        //        this will be in protocol DDTPv2 most likely

        case MARK_ONLINE:
            if (manager->markOnline(accountId,
                               ntohl (updateQuery->ipAddress)) != -1)
            {
                struct in_addr addr;
                addr.s_addr = ntohl (updateQuery->ipAddress);
                log->notice ("%d (%s) marked online at %s",
                              accountId, fqdn, inet_ntoa (addr));
                updateReply->status = UPDATE_SUCCEEDED;
            }
            else
            {
                log->error ("%d (%s) could not be marked online", accountId, fqdn);
                updateReply->status = UPDATE_FAILED;
            }
            break;
            
        case MARK_OFFLINE:
            if (manager->markOffline(accountId) != -1)
            {
                log->notice ("%d (%s) marked offline", accountId, fqdn);
                updateReply->status = UPDATE_SUCCEEDED;
            }
            else
            {
                log->error ("%d (%s) could not be marked offline", accountId, fqdn);
                updateReply->status = UPDATE_FAILED;
            }
            break;
            
        default:
            updateReply->status = INVALID_OPCODE;
            log->error ("%d (%s) invalid opcode", accountId, fqdn);
            break;
    }
    return 0; // reply hack
};

int Server::processDDTPv1AliveReply (DDTPv1AliveReply *aliveReply,
                                     int accountId)
{
    char fqdn[256];
    manager->fetchAccountInfo (accountId, "fqdn", fqdn, sizeof(fqdn));

    log->debug ("received alive reply from account %d (%s)", accountId, fqdn);
    if (manager->markAlive (accountId) == -1)
    {
        log->error ("%d (%s) could not be marked alive", accountId, fqdn);
        return -1; // no reply hack
    }
    log->info ("%d (%s) is alive", accountId, fqdn);
    return -1; // no reply hack
}

void Server::atexit_handler ()
{
    FILE *fp;
    int pid;

    if (log) log->notice("daemon stopping");

    // remove pid file
    if ((fp = fopen (PID_FILE, "r")) == NULL)
    {
        log->error ("failed to open pid file: %s", strerror (errno));
    }
    else
    {
        if ((fscanf (fp, "%d", &pid) == 1) && (pid == getpid()))
        {
            if (unlink (PID_FILE) == -1)
            {
                log->debug ("failed to unlink pid file: %s", strerror (errno));
            }
        }
    }
}

void Server::signal_handler (int signum)
{
    switch (signum)
    {
        case SIGHUP:
        case SIGINT:
        case SIGQUIT:
        case SIGABRT:
        case SIGTERM:
            log->debug ("received signal (%d)", signum);
            exit (0);
            break;
    }
}
