/*
 * Decompiled with CFR 0.152.
 */
package org.apache.skywalking.apm.dependencies.io.grpc.internal;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.apache.skywalking.apm.dependencies.com.google.common.base.Optional;
import org.apache.skywalking.apm.dependencies.com.google.common.base.Preconditions;
import org.apache.skywalking.apm.dependencies.com.google.common.base.Splitter;
import org.apache.skywalking.apm.dependencies.com.google.common.collect.ImmutableList;
import org.apache.skywalking.apm.dependencies.com.google.common.collect.ImmutableMap;
import org.apache.skywalking.apm.dependencies.com.google.common.io.Files;
import org.apache.skywalking.apm.dependencies.io.grpc.internal.JsonParser;
import org.apache.skywalking.apm.dependencies.io.grpc.internal.JsonUtil;

public final class SpiffeUtil {
    private static final Integer URI_SAN_TYPE = 6;
    private static final String USE_PARAMETER_VALUE = "x509-svid";
    private static final String KTY_PARAMETER_VALUE = "RSA";
    private static final String CERTIFICATE_PREFIX = "-----BEGIN CERTIFICATE-----\n";
    private static final String CERTIFICATE_SUFFIX = "-----END CERTIFICATE-----";
    private static final String PREFIX = "spiffe://";

    private SpiffeUtil() {
    }

    public static SpiffeId parse(String uri) {
        String path;
        String trustDomain;
        SpiffeUtil.doInitialUriValidation(uri);
        Preconditions.checkArgument(uri.toLowerCase(Locale.US).startsWith(PREFIX), "Spiffe Id must start with spiffe://");
        String domainAndPath = uri.substring(PREFIX.length());
        if (!domainAndPath.contains("/")) {
            trustDomain = domainAndPath;
            path = "";
        } else {
            String[] parts = domainAndPath.split("/", 2);
            trustDomain = parts[0];
            path = parts[1];
            Preconditions.checkArgument(!path.isEmpty(), "Path must not include a trailing '/'");
        }
        SpiffeUtil.validateTrustDomain(trustDomain);
        SpiffeUtil.validatePath(path);
        if (!path.isEmpty()) {
            path = "/" + path;
        }
        return new SpiffeId(trustDomain, path);
    }

    private static void doInitialUriValidation(String uri) {
        Preconditions.checkArgument(Preconditions.checkNotNull(uri, "uri").length() > 0, "Spiffe Id can't be empty");
        Preconditions.checkArgument(uri.length() <= 2048, "Spiffe Id maximum length is 2048 characters");
        Preconditions.checkArgument(!uri.contains("#"), "Spiffe Id must not contain query fragments");
        Preconditions.checkArgument(!uri.contains("?"), "Spiffe Id must not contain query parameters");
    }

    private static void validateTrustDomain(String trustDomain) {
        Preconditions.checkArgument(!trustDomain.isEmpty(), "Trust Domain can't be empty");
        Preconditions.checkArgument(trustDomain.length() < 256, "Trust Domain maximum length is 255 characters");
        Preconditions.checkArgument(trustDomain.matches("[a-z0-9._-]+"), "Trust Domain must contain only letters, numbers, dots, dashes, and underscores ([a-z0-9.-_])");
    }

    private static void validatePath(String path) {
        if (path.isEmpty()) {
            return;
        }
        Preconditions.checkArgument(!path.endsWith("/"), "Path must not include a trailing '/'");
        for (String segment : Splitter.on("/").split(path)) {
            SpiffeUtil.validatePathSegment(segment);
        }
    }

    private static void validatePathSegment(String pathSegment) {
        Preconditions.checkArgument(!pathSegment.isEmpty(), "Individual path segments must not be empty");
        Preconditions.checkArgument(!pathSegment.equals(".") && !pathSegment.equals(".."), "Individual path segments must not be relative path modifiers (i.e. ., ..)");
        Preconditions.checkArgument(pathSegment.matches("[a-zA-Z0-9._-]+"), "Individual path segments must contain only letters, numbers, dots, dashes, and underscores ([a-zA-Z0-9.-_])");
    }

    public static Optional<SpiffeId> extractSpiffeId(X509Certificate[] certChain) throws CertificateParsingException {
        Preconditions.checkArgument(Preconditions.checkNotNull(certChain, "certChain").length > 0, "certChain can't be empty");
        Collection<List<?>> subjectAltNames = certChain[0].getSubjectAlternativeNames();
        if (subjectAltNames == null) {
            return Optional.absent();
        }
        String uri = null;
        for (List<?> altName : subjectAltNames) {
            if (altName.size() < 2 || !URI_SAN_TYPE.equals(altName.get(0))) continue;
            if (uri != null) {
                throw new IllegalArgumentException("Multiple URI SAN values found in the leaf cert.");
            }
            uri = (String)altName.get(1);
        }
        if (uri == null) {
            return Optional.absent();
        }
        return Optional.of(SpiffeUtil.parse(uri));
    }

    public static SpiffeBundle loadTrustBundleFromFile(String trustBundleFile) throws IOException {
        Map<String, ?> trustDomainsNode = SpiffeUtil.readTrustDomainsFromFile(trustBundleFile);
        HashMap<String, List<Object>> trustBundleMap = new HashMap<String, List<Object>>();
        HashMap<String, Long> sequenceNumbers = new HashMap<String, Long>();
        for (String trustDomainName : trustDomainsNode.keySet()) {
            Map<String, ?> domainNode = JsonUtil.getObject(trustDomainsNode, trustDomainName);
            if (domainNode.size() == 0) {
                trustBundleMap.put(trustDomainName, Collections.emptyList());
                continue;
            }
            Long sequenceNumber = JsonUtil.getNumberAsLong(domainNode, "spiffe_sequence");
            sequenceNumbers.put(trustDomainName, sequenceNumber == null ? -1L : sequenceNumber);
            List<Map<String, ?>> keysNode = JsonUtil.getListOfObjects(domainNode, "keys");
            if (keysNode == null || keysNode.size() == 0) {
                trustBundleMap.put(trustDomainName, Collections.emptyList());
                continue;
            }
            trustBundleMap.put(trustDomainName, SpiffeUtil.extractCert(keysNode, trustDomainName));
        }
        return new SpiffeBundle(sequenceNumbers, trustBundleMap);
    }

    private static Map<String, ?> readTrustDomainsFromFile(String filePath) throws IOException {
        File file = new File(Preconditions.checkNotNull(filePath, "trustBundleFile"));
        String json = new String(Files.toByteArray(file), StandardCharsets.UTF_8);
        Object jsonObject = JsonParser.parse(json);
        if (!(jsonObject instanceof Map)) {
            throw new IllegalArgumentException("SPIFFE Trust Bundle should be a JSON object. Found: " + (jsonObject == null ? null : jsonObject.getClass()));
        }
        Map root = (Map)jsonObject;
        Map<String, ?> trustDomainsNode = JsonUtil.getObject(root, "trust_domains");
        Preconditions.checkNotNull(trustDomainsNode, "Mandatory trust_domains element is missing");
        Preconditions.checkArgument(trustDomainsNode.size() > 0, "Mandatory trust_domains element is missing");
        return trustDomainsNode;
    }

    private static void checkJwkEntry(Map<String, ?> jwkNode, String trustDomainName) {
        String kty = JsonUtil.getString(jwkNode, "kty");
        if (kty == null || !kty.equals(KTY_PARAMETER_VALUE)) {
            throw new IllegalArgumentException(String.format("'kty' parameter must be '%s' but '%s' found. Certificate loading for trust domain '%s' failed.", KTY_PARAMETER_VALUE, kty, trustDomainName));
        }
        if (jwkNode.containsKey("kid")) {
            throw new IllegalArgumentException(String.format("'kid' parameter must not be set. Certificate loading for trust domain '%s' failed.", trustDomainName));
        }
        String use = JsonUtil.getString(jwkNode, "use");
        if (use == null || !use.equals(USE_PARAMETER_VALUE)) {
            throw new IllegalArgumentException(String.format("'use' parameter must be '%s' but '%s' found. Certificate loading for trust domain '%s' failed.", USE_PARAMETER_VALUE, use, trustDomainName));
        }
    }

    private static List<X509Certificate> extractCert(List<Map<String, ?>> keysNode, String trustDomainName) {
        ArrayList<X509Certificate> result = new ArrayList<X509Certificate>();
        for (Map<String, ?> keyNode : keysNode) {
            SpiffeUtil.checkJwkEntry(keyNode, trustDomainName);
            List<String> rawCerts = JsonUtil.getListOfStrings(keyNode, "x5c");
            if (rawCerts == null) break;
            if (rawCerts.size() != 1) {
                throw new IllegalArgumentException(String.format("Exactly 1 certificate is expected, but %s found. Certificate loading for trust domain '%s' failed.", rawCerts.size(), trustDomainName));
            }
            ByteArrayInputStream stream = new ByteArrayInputStream((CERTIFICATE_PREFIX + rawCerts.get(0) + "\n" + CERTIFICATE_SUFFIX).getBytes(StandardCharsets.UTF_8));
            try {
                Collection<? extends Certificate> certs = CertificateFactory.getInstance("X509").generateCertificates(stream);
                X509Certificate[] certsArray = certs.toArray(new X509Certificate[0]);
                assert (certsArray.length == 1);
                result.add(certsArray[0]);
            }
            catch (CertificateException e) {
                throw new IllegalArgumentException(String.format("Certificate can't be parsed. Certificate loading for trust domain '%s' failed.", trustDomainName), e);
            }
        }
        return result;
    }

    public static final class SpiffeBundle {
        private final ImmutableMap<String, Long> sequenceNumbers;
        private final ImmutableMap<String, ImmutableList<X509Certificate>> bundleMap;

        private SpiffeBundle(Map<String, Long> sequenceNumbers, Map<String, List<X509Certificate>> trustDomainMap) {
            this.sequenceNumbers = ImmutableMap.copyOf(sequenceNumbers);
            ImmutableMap.Builder builder = ImmutableMap.builder();
            for (Map.Entry<String, List<X509Certificate>> entry : trustDomainMap.entrySet()) {
                builder.put(entry.getKey(), ImmutableList.copyOf((Collection)entry.getValue()));
            }
            this.bundleMap = builder.build();
        }

        public ImmutableMap<String, Long> getSequenceNumbers() {
            return this.sequenceNumbers;
        }

        public ImmutableMap<String, ImmutableList<X509Certificate>> getBundleMap() {
            return this.bundleMap;
        }
    }

    public static class SpiffeId {
        private final String trustDomain;
        private final String path;

        private SpiffeId(String trustDomain, String path) {
            this.trustDomain = trustDomain;
            this.path = path;
        }

        public String getTrustDomain() {
            return this.trustDomain;
        }

        public String getPath() {
            return this.path;
        }
    }
}

