/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hudi.common.util;

import java.util.function.Supplier;

/**
 * Simple utility to test validation conditions (to replace Guava's PreConditions)
 */
public class ValidationUtils {

  /**
   * Ensures the truth of an expression.
   */
  public static void checkArgument(final boolean expression) {
    if (!expression) {
      throw new IllegalArgumentException();
    }
  }

  /**
   * Ensures the truth of an expression, throwing the custom errorMessage otherwise.
   */
  public static void checkArgument(final boolean expression, final String errorMessage) {
    if (!expression) {
      throw new IllegalArgumentException(errorMessage);
    }
  }

  /**
   * Ensures the truth of an expression, throwing the custom errorMessage otherwise.
   */
  public static void checkArgument(final boolean expression, final Supplier<String> errorMessageSupplier) {
    if (!expression) {
      throw new IllegalArgumentException(errorMessageSupplier.get());
    }
  }

  /**
   * Ensures the truth of an expression involving the state of the calling instance, but not
   * involving any parameters to the calling method.
   *
   * @param expression a boolean expression
   * @throws IllegalStateException if {@code expression} is false
   */
  public static void checkState(final boolean expression) {
    if (!expression) {
      throw new IllegalStateException();
    }
  }

  /**
   * Ensures the truth of an expression involving the state of the calling instance, but not
   * involving any parameters to the calling method.
   *
   * @param expression a boolean expression
   * @param errorMessage - error message
   * @throws IllegalStateException if {@code expression} is false
   */
  public static void checkState(final boolean expression, String errorMessage) {
    if (!expression) {
      throw new IllegalStateException(errorMessage);
    }
  }

  /**
   * Ensures the truth of an expression, throwing the custom errorMessage otherwise.
   */
  public static void checkState(final boolean expression, final Supplier<String> errorMessageSupplier) {
    if (!expression) {
      throw new IllegalArgumentException(errorMessageSupplier.get());
    }
  }
}
