/*******************************************************************************
 * Copyright (c) 2007, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.model.value;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Iterator;
import java.util.ListIterator;
import org.eclipse.persistence.tools.utility.ArrayTools;
import org.eclipse.persistence.tools.utility.iterator.ArrayIterator;
import org.eclipse.persistence.tools.utility.iterator.ArrayListIterator;
import org.eclipse.persistence.tools.utility.model.AbstractModel;


/**
 * Implementation of {@link ListValueModel} that can be used for
 * returning a list iterator on a static list, but still allows listeners to be added.
 * Listeners will <em>never</em> be notified of any changes, because there should be none.
 */
public class StaticListValueModel<E>
	extends AbstractModel
	implements ListValueModel<E>, Serializable
{
	/** The elements. */
	protected final Object[] elements;

	private static final long serialVersionUID = 1L;


	/**
	 * Construct a static list value model for the specified elements.
	 */
	public StaticListValueModel(E... elements) {
		super();
		this.elements = elements.clone();
	}

	/**
	 * Construct a static list value model for the specified elements.
	 */
	public StaticListValueModel(Iterable<? extends E> elements) {
		super();
		this.elements = ArrayTools.array(elements);
	}


	// ********** ListValueModel implementation **********

	@Override
	@SuppressWarnings("unchecked")
	public Iterator<E> iterator() {
		// we can cast here since our constructors require the elements to be
		// of type E and ArrayIterator is read-only
		return (Iterator<E>) new ArrayIterator<Object>(this.elements);
	}

	@Override
	@SuppressWarnings("unchecked")
	public ListIterator<E> listIterator() {
		// we can cast here since our constructors require the elements to be
		// of type E and ArrayListIterator is read-only
		return (ListIterator<E>) new ArrayListIterator<Object>(this.elements);
	}

	@Override
	public int size() {
		return this.elements.length;
	}

	@Override
	@SuppressWarnings("unchecked")
	public E get(int index) {
		// we can cast here since our constructors require the elements to be
		// of type E
		return (E) this.elements[index];
	}

	@Override
	public Object[] toArray() {
		return this.elements.clone();
	}


	// ********** Object overrides **********

	@Override
	public void toString(StringBuilder sb) {
		sb.append(Arrays.toString(this.elements));
	}
}