/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalProperty;
import org.w3c.dom.Element;

/**
 * The external form of a property, which is a child of an entity.
 *
 * @see Property
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class Property extends AbstractExternalForm
                     implements ExternalProperty {

	/**
	 * The index of this external property.
	 */
	private int index;

	/**
	 * The property name used to store and retrieve the name property.
	 */
	static final String NAME = "name";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static final String PROPERTY = "property";

	/**
	 * The property name used to store and retrieve the value property.
	 */
	static final String VALUE = "value";

	/**
	 * The property name used to store and retrieve the value-type property.
	 */
	static final String VALUE_TYPE = "value-type";

	/**
	 * Creates a new <code>Property</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	Property(AbstractExternalForm parent, int index) {
		super(parent);
		this.index = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName, List<String> elementNamesOrder) {
		return addChild(getParent(), elementName, index, elementNamesOrder);
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(VALUE);
		names.add(VALUE_TYPE);
		return names;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PROPERTY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getValue() {
		return getAttribute(VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getValueType() {
		return getAttribute(VALUE_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeSelf() {
		removeChild(getParent(), getElementName(), index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		setAttribute(NAME, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValue(String value) {
		setAttribute(VALUE, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValueType(String valueType) {
		setAttribute(VALUE_TYPE, valueType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return getName();
	}
}