/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.EnumType;
import javax.persistence.FetchType;
import javax.persistence.TemporalType;
import org.eclipse.persistence.annotations.JoinFetchType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalBatchFetch;
import org.eclipse.persistence.tools.mapping.orm.ExternalCollectionTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalElementCollectionMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlField;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectTypeConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalOrderColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalStructConverter;
import org.eclipse.persistence.tools.mapping.orm.ExternalTypeConverter;
import org.eclipse.persistence.tools.utility.ClassNameTools;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form for a element collection mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class ElementCollectionMapping extends ConvertibleMapping
                                     implements ExternalElementCollectionMapping {

	/**
	 * The attribute name used to store and retrieve the class property.
	 */
	static String CLASS = "class";

	/**
	 * The element name used to store and retrieve the convert-key child text node.
	 */
	static String CONVERT_KEY = "convert-key";

	/**
	 * The node name used to store and retrieve the {@link Element} encapsulated by this external form.
	 */
	static String ELEMENT_COLLECTION = "element-collection";

	/**
	 * The element name used to store and retrieve the expiry child text node.
	 */
	static String ENUMERATED = "enumerated";

	/**
	 * The attribute name used to store and retrieve the fetch property.
	 */
	static String FETCH = "fetch";

	/**
	 * The element name used to store and retrieve the join-fetch child text node.
	 */
	static String JOIN_FETCH = "join-fetch";

	/**
	 * The element name used to store and retrieve the lob child text node.
	 */
	static String LOB = "lob";

	/**
	 * The element name used to store and retrieve the map-key child node.
	 */
	static String MAP_KEY = "map-key";

	/**
	 * The element name used to store and retrieve the map-key-association child nodes.
	 */
	static String MAP_KEY_ASSOCIATION_OVERRIDE = "map-key-association-override";

	/**
	 * The element name used to store and retrieve the map-key-attribute child nodes.
	 */
	static String MAP_KEY_ATTRIBUTE_OVERRIDE = "map-key-attribute-override";

	/**
	 * The element name used to store and retrieve the map-key-class child node.
	 */
	static String MAP_KEY_CLASS = "map-key-class";

	/**
	 * The element name used to store and retrieve the map-key-column child node.
	 */
	static String MAP_KEY_COLUMN = "map-key-column";

	/**
	 * The element name used to store and retrieve the map-key-convert child node.
	 */
	static String MAP_KEY_CONVERT = "map-key-convert";

	/**
	 * The element name used to store and retrieve the map-key-enumerated child node.
	 */
	static String MAP_KEY_ENUMERATED = "map-key-enumerated";

	/**
	 * The element name used to store and retrieve the map-key-join-column child node.
	 */
	static String MAP_KEY_JOIN_COLUMN = "map-key-join-column";

	/**
	 * The element name used to store and retrieve the map-key-temporal child node.
	 */
	static String MAP_KEY_TEMPORAL = "map-key-temporal";

	/**
	 * The element name used to store and retrieve the order-by child text node.
	 */
	static String ORDER_BY = "order-by";

	/**
	 * The attribute name used to store and retrieve the target-class property.
	 */
	static String TARGET_CLASS = "target-class";

	/**
	 * The element name used to store and retrieve the temporal child text node.
	 */
	static String TEMPORAL = "temporal";

	/**
	 * Creates a new <code>ElementCollectionMapping</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned
	 * by the parent
	 */
	ElementCollectionMapping(EmbeddableEntity parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addAssociationOverride(int index, String name) {
		AssociationOverride associationOverride = buildAssociationOverride(index);
		associationOverride.addSelf();
		associationOverride.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addAttributeOverride(int index, String name) {
		AttributeOverride attributeOverride = buildAttributeOverride(index);
		attributeOverride.addSelf();
		attributeOverride.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addBatchFetch() {
		BatchFetch batchFetch = buildBatchFetch();
		batchFetch.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable addCollectionTable() {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.addSelf();
		return collectionTable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride addMapKeyAssociationOverride(int index) {
		AssociationOverride associationOverride = buildMapKeyAssociationOverride(mapKeyAssociationOverridesSize());
		associationOverride.addSelf();
		return associationOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride addMapKeyAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildMapKeyAttributeOverride(mapKeyAttributeOverridesSize());
		attributeOverride.addSelf();
		return attributeOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addMapKeyColumn() {
		EntityColumn column = buildColumn(MAP_KEY_COLUMN);
		column.addSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addMapKeyJoinColumn(int index, String name) {
		JoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
		joinColumn.addSelf();
		joinColumn.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addOrderColumn() {
		OrderColumn orderColumn = buildOrderColumn();
		orderColumn.addSelf();
	}

	private AssociationOverride buildAssociationOverride(int index) {
		return new AssociationOverride(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TARGET_CLASS);
		names.add(FETCH);
		names.add(ACCESS);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	private AttributeOverride buildAttributeOverride(int index) {
		return new AttributeOverride(this, index);
	}

	private BatchFetch buildBatchFetch() {
		return new BatchFetch(this);
	}

	private CollectionTable buildCollectionTable() {
		return new CollectionTable(this);
	}

	private EntityColumn buildColumn(String elementName) {
		return new EntityColumn(this, elementName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EntityColumn.COLUMN);
		names.add(NoSqlField.FIELD);
		names.add(AttributeOverride.ATTRIBUTE_OVERRIDE);
		names.add(AssociationOverride.ASSOCIATION_OVERRIDE);
		names.add(ORDER_BY);
		names.add(OrderColumn.ORDER_COLUMN);
		names.add(MAP_KEY);
		names.add(MAP_KEY_CLASS);
		names.add(MAP_KEY_TEMPORAL);
		names.add(MAP_KEY_ENUMERATED);
		names.add(MAP_KEY_CONVERT);
		names.add(MAP_KEY_ATTRIBUTE_OVERRIDE);
		names.add(MAP_KEY_ASSOCIATION_OVERRIDE);
		names.add(MAP_KEY_COLUMN);
		names.add(MAP_KEY_JOIN_COLUMN);
		names.add(CollectionTable.COLLECTION_TABLE);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(CONVERT);
		names.add(Converter.CONVERTER);
		names.add(TypeConverter.TYPE_CONVERTER);
		names.add(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		names.add(StructConverter.STRUCT_CONVERTER);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private JoinColumn buildJoinColumn(String elementName, int index) {
		return new JoinColumn(this, elementName, index);
	}

	private Converter buildKeyConverter() {
		return new Converter(this);
	}

	private ObjectTypeConverter buildKeyObjectTypeConverter() {
		return new ObjectTypeConverter(this);
	}

	private StructConverter buildKeyStructConverter() {
		return new StructConverter(this);
	}

	private TypeConverter buildKeyTypeConverter() {
		return new TypeConverter(this);
	}

	private AssociationOverride buildMapKeyAssociationOverride(int index) {
		return new AssociationOverride(this, index);
	}

	private AttributeOverride buildMapKeyAttributeOverride(int index) {
		return new AttributeOverride(this, index);
	}

	private ExternalNoSqlField buildNoSqlField() {
		return new NoSqlField(this);
	}

	private OrderColumn buildOrderColumn() {
		return new OrderColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalBatchFetch getBatchFetch() {

		if (hasChild(BatchFetch.BATCH_FETCH)) {
			return buildBatchFetch();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalCollectionTable getCollectionTable() {

		if (hasChild(CollectionTable.COLLECTION_TABLE)) {
			return buildCollectionTable();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getColumn() {

		if (hasChild(EntityColumn.COLUMN)) {
			return buildColumn(EntityColumn.COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getConvertKey() {
		return getChildTextNode(CONVERT_KEY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ELEMENT_COLLECTION;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EnumType getEnumeratedType() {
		return getChildEnumNode(ENUMERATED, EnumType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FetchType getFetchType() {
		return getEnumAttribute(FETCH, FetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public JoinFetchType getJoinFetchType() {
		return getChildEnumNode(JOIN_FETCH, JoinFetchType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalConverter getKeyConverter() {

		if (hasChild(Converter.CONVERTER)) {
			return buildKeyConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalObjectTypeConverter getKeyObjectTypeConverter() {

		if (hasChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER)) {
			return buildKeyObjectTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalStructConverter getKeyStructConverter() {

		if (hasChild(StructConverter.STRUCT_CONVERTER)) {
			return buildKeyStructConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTypeConverter getKeyTypeConverter() {

		if (hasChild(TypeConverter.TYPE_CONVERTER)) {
			return buildKeyTypeConverter();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKey() {

		Element element = getChild(MAP_KEY);

		if (element == null) {
			return null;
		}

		return getAttribute(element, NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride getMapKeyAssociationOverride(int index) {

		Element element = getChild(AssociationOverride.ASSOCIATION_OVERRIDE, index);

		if (element == null) {
			return null;
		}

		return buildAssociationOverride(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride getMapKeyAttributeOverride(int index) {

		Element element = getChild(AttributeOverride.ATTRIBUTE_OVERRIDE, index);

		if (element == null) {
			return null;
		}

		return buildAttributeOverride(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKeyClassName() {

		Element element = getChild(MAP_KEY_CLASS);

		if (element == null) {
			return null;
		}

		return getAttribute(element, CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getMapKeyColumn() {

		if (hasChild(MAP_KEY_COLUMN)) {
			return buildColumn(MAP_KEY_COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKeyConverterName() {
		return getChildTextNode(MAP_KEY_CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EnumType getMapKeyEnumeratedType() {
		return getChildEnumNode(MAP_KEY_ENUMERATED, EnumType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getMapKeyJoinColumn(int index) {

		Element element = getChild(MAP_KEY_JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getMapKeyTemporalType() {
		return getChildEnumNode(MAP_KEY_TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getNoSqlField() {
		ExternalNoSqlField field = buildNoSqlField();
		return field.getName();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getOrderByFieldName() {
		return getChildTextNode(ORDER_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalOrderColumn getOrderColumn() {

		if (hasChild(OrderColumn.ORDER_COLUMN)) {
			return buildOrderColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTargetClassName() {
		return getAttribute(TARGET_CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTargetClassShortName() {
		return ClassNameTools.simpleName(getTargetClassName());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getTemporalType() {
		return getChildEnumNode(TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isElementCollectionMapping() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isLob() {
		return getChildBooleanNode(LOB);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalAssociationOverride> mapKeyAssociationOverrides() {

		int count = mapKeyAssociationOverridesSize();
		List<ExternalAssociationOverride> associationOverrides = new ArrayList<ExternalAssociationOverride>(count);

		for (int index = count; --index >= 0;) {
			ExternalAssociationOverride associationOverride = buildAssociationOverride(index);
			associationOverrides.add(0, associationOverride);
		}

		return new ListListIterable<ExternalAssociationOverride>(associationOverrides);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyAssociationOverridesSize() {
		return getChildrenSize(AssociationOverride.ASSOCIATION_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalAttributeOverride> mapKeyAttributeOverrides() {

		int count = mapKeyAttributeOverridesSize();
		List<ExternalAttributeOverride> attributeOverrides = new ArrayList<ExternalAttributeOverride>(count);

		for (int index = count; --index >= 0;) {
			ExternalAttributeOverride attributeOverride = buildAttributeOverride(index);
			attributeOverrides.add(0, attributeOverride);
		}

		return new ListListIterable<ExternalAttributeOverride>(attributeOverrides);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyAttributeOverridesSize() {
		return getChildrenSize(AttributeOverride.ATTRIBUTE_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalJoinColumn> mapKeyJoinColumns() {

		int count = mapKeyJoinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalJoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
			joinColumns.add(0, joinColumn);
		}

		return new ListListIterable<ExternalJoinColumn>(joinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyJoinColumnsSize() {
		return getChildrenSize(MAP_KEY_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeBatchFetch() {
		removeChild(BatchFetch.BATCH_FETCH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeCollectionTable() {
		CollectionTable collectionTable = buildCollectionTable();
		collectionTable.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeColumn() {
		EntityColumn column = buildColumn(EntityColumn.COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyConverter() {
		removeChild(Converter.CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyObjectTypeConverter() {
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyStructConverter() {
		removeChild(StructConverter.STRUCT_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeKeyTypeConverter() {
		removeChild(TypeConverter.TYPE_CONVERTER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyAssociationOverride(int index) {
		AssociationOverride associationOverride = buildAssociationOverride(index);
		associationOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildMapKeyAttributeOverride(index);
		attributeOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyColumn() {
		EntityColumn column = buildColumn(MAP_KEY_COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeOrderColumn() {
		OrderColumn orderColumn = buildOrderColumn();
		orderColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyConverter(String name, String className) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		Converter valueConverter = buildKeyConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
		valueConverter.setConverterClassName(className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyObjectTypeConverter(String name) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(Converter.CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		ObjectTypeConverter valueConverter = buildKeyObjectTypeConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyStructConverter(String name, String converter) {

		removeChild(TypeConverter.TYPE_CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(Converter.CONVERTER);

		StructConverter valueConverter = buildKeyStructConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
		valueConverter.setConverterClassName(converter);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAsKeyTypeConverter(String name) {

		removeChild(Converter.CONVERTER);
		removeChild(ObjectTypeConverter.OBJECT_TYPE_CONVERTER);
		removeChild(StructConverter.STRUCT_CONVERTER);

		TypeConverter valueConverter = buildKeyTypeConverter();
		valueConverter.addSelf();
		valueConverter.setName(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAttributeType(String attributeType) {
		setAttribute(ATTRIBUTE_TYPE, attributeType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setColumn(String columnName) {
		EntityColumn column = buildColumn(EntityColumn.COLUMN);
		column.addSelf();
		column.setName(columnName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setConvertKey(String value) {
		updateChildTextNode(CONVERT_KEY, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setEnumeratedType(EnumType type) {
		updateChildTextNode(ENUMERATED, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFetchType(FetchType fetchType) {
		setAttribute(FETCH, fetchType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setJoinFetchType(JoinFetchType type) {
		updateChildTextNode(JOIN_FETCH, type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLob(Boolean lob) {
		updateChildTextNode(LOB, lob);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKey(String mapKey) {

		if (mapKey == null) {
			removeChild(MAP_KEY);
		}
		else {
			Element element = getChild(MAP_KEY);

			if (element == null) {
				element = addChild(MAP_KEY);
			}

			setAttribute(element, NAME, mapKey);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyClassName(String className) {

		if (className == null) {
			removeChild(MAP_KEY_CLASS);
		}
		else {
			Element element = getChild(MAP_KEY_CLASS);

			if (element == null) {
				element = addChild(MAP_KEY_CLASS);
			}

			setAttribute(element, CLASS, className);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyConverterName(String converterName) {
		updateChildTextNode(MAP_KEY_CONVERT, converterName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyEnumeratedType(EnumType enumType) {
		updateChildTextNode(MAP_KEY_ENUMERATED, enumType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyTemporalType(TemporalType temporalType) {
		updateChildTextNode(MAP_KEY_TEMPORAL, temporalType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setNoSqlField(String field) {
		ExternalNoSqlField noSqlField = buildNoSqlField();
		noSqlField.setName(field);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOrderByFieldName(String orderBy) {
		updateChildTextNode(ORDER_BY, orderBy);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTargetClassName(String className) {
		setAttribute(TARGET_CLASS, className);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTemporalType(TemporalType type) {
		updateChildTextNode(TEMPORAL, type);
	}
}