/*******************************************************************************
 * Copyright (c) 2006, 2012 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalCollectionTable;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;
import org.eclipse.persistence.tools.utility.iterable.ListListIterable;
import org.w3c.dom.Element;

/**
 * The external form for a collection table, which is a child of an element collection mapping.
 *
 * @see BasicCollectionMapping
 * @see BasicMapMapping
 * @see ElementCollectionMapping
 *
 * @version 2.5
 * @author Les Davis
 */
@SuppressWarnings("nls")
final class CollectionTable extends Table
                            implements ExternalCollectionTable {

	/**
	 * The node name used to store and retrieve the <code>Element</code>
	 * encapsulated by this external form.
	 */
	static final String COLLECTION_TABLE = "collection-table";

	/**
	 * Creates a new <code>CollectionTable</code>.
	 *
	 * @param parent The parent of this external form
	 */
	CollectionTable(BasicCollectionMapping parent) {
		super(parent);
	}

	/**
	 * Creates a new <code>CollectionTable</code>.
	 *
	 * @param parent The parent of this external form
	 */
	CollectionTable(BasicMapMapping parent) {
		super(parent);
	}

	/**
	 * Creates a new <code>CollectionTable</code>.
	 *
	 * @param parent The parent of this external form
	 */
	CollectionTable(ElementCollectionMapping parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addJoinColumn(int index, String name) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.addSelf();
		joinColumn.setName(name);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(CATALOG);
		names.add(SCHEMA);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(JoinColumn.JOIN_COLUMN);
		names.add(UniqueConstraint.UNIQUE_CONSTRAINT);
		return names;
	}

	private JoinColumn buildJoinColumn(int index) {
		return new JoinColumn(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return COLLECTION_TABLE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getJoinColumn(int index) {

		Element element = getChild(JoinColumn.JOIN_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildJoinColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isDefault() {
		return !(hasChildren() || hasAttributes());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListIterable<ExternalJoinColumn> joinColumns() {

		int count = joinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = count; --index >= 0;) {
			ExternalJoinColumn joinColumn = buildJoinColumn(index);
			joinColumns.add(0, joinColumn);
		}

		return new ListListIterable<ExternalJoinColumn>(joinColumns);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinColumnsSize() {
		return getChildrenSize(JoinColumn.JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(index);
		joinColumn.removeSelf();
	}
}