/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* axes.c: Stuff to draw axes at the edges of the graphic.
 *
 * There is a numeric Hertz frequency axis on the left, a note name
 * frequency axis on the right (A0, B0, C1, D1 etc)
 * along the bottom the time axis in seconds (actually just the times at the
 * current playing position, the left/rightmost edges  and maybe bar lines)
 * and along the top various status information (FFT freq, window etc).
 *
 * The corners belong to the frequency axes: if the topmost pixel row has a
 * numbered tick, the text will overflow into the gap, while the leftmost and
 * rightmost times are left and right aligned so do not overflow the corners.
 */

#include "spettro.h"
#include "axes.h"

#include "audio.h"
#include "barlines.h"
#include "convert.h"
#include "gui.h"
#include "paint.h"
#include "text.h"
#include "ui.h"
#include "window.h"

#include <string.h>	/* for strlen() */

/* Array of numbers to print, or NO_NUMBER */
/* Where to put each tick/label on the y-axis */
typedef struct {
    freq_t value;	/* Number to show, NO_NUMBER if just a tick */
    unsigned short y;	/* screen coordinate of this tick */
} tick_t;
static tick_t *ticks = NULL;
static int number_of_ticks = 0;	/* Number of labels allocated in this array */

/* The digit that changes from label to label.
 * This ensures that a range from 999 to 1001 prints 999.5 and 1000.5
 * instead of 999 1000 1000 1000 1001.
 */
static int decimal_places_to_print;

/* Value to store in the ticks[k].value field to mean
 * "Put a tick here, but don't print a number."
 */
#define NO_NUMBER (0.0)

/* Is this entry in "ticks" one of the numberless ticks? */
#define JUST_A_TICK(ticks, k)	(ticks[k].value == NO_NUMBER)

/* Forward declarations */

static int calculate_ticks(int log_scale);
static int calculate_log_ticks(void);
static void draw_frequency_axis(void);
static void draw_note_names(void);

void
draw_axes(void)
{
    if (show_frequency_axes)
	draw_frequency_axes();

    if (show_time_axes) {
	draw_status_line();
	draw_time_axis();
    }
}

void
draw_frequency_axes(void)
{
    draw_frequency_axis();
    draw_note_names();
}

void
draw_time_axes(void)
{
    draw_status_line();
    draw_time_axis();
}

/* Put ticks on Frequency axis. Only called when show_axes is TRUE. */
static void
draw_frequency_axis()
{
    int tick_count = calculate_ticks(1);
    int i;

    if (min_x - 1 != frequency_axis_width - 1)
    fprintf(stdout, "frequency axis with does not match min_x\n");

    gui_paint_rect(0, 0, min_x - 1, disp_height - 1, black);

    for (i=0; i < tick_count; i++) {
	gui_lock();
	gui_putpixel(min_x - 1, ticks[i].y, green);
	gui_putpixel(min_x - 2, ticks[i].y, green);
	gui_unlock();
	if (ticks[i].value != NO_NUMBER) {
	    draw_text(freq_to_string(ticks[i].value),
		      min_x - 4, ticks[i].y,
		      RIGHT, CENTER);
	}
    }

    gui_update_rect(0, 0, frequency_axis_width - 1, disp_height - 1);
}

/*
 * Note names are sequenced A0 B0 C1 D1 E1 F1 G1 A1 B1 C3 ... F8 G8 A8
 */
static void
draw_note_names()
{
    /* Note letters within each numeric octave */
    char note_names[] = { 'C', 'D', 'E', 'F', 'G', 'A', 'B' };
    char note_name[3]; /* "A0" etc */
    int octave;	/* numeric: 0 to 8 */
    int note;	/* index into note_names */

    gui_paint_rect(max_x + 1, 0, disp_width - 1, disp_height - 1, black);

    note_name[2] = '\0';
    for (octave = 0; octave <= 9; octave++) {
	note_name[1] = octave + '0';
	for (note = ((octave == 0) ? 5 : 0);	/* from A0 */
	     note <= ((octave == 9) ? 5 : 6);	/*   to A9 */
	     note++) {
	    freq_t freq, half_a_pixel;

	    note_name[0] = note_names[note];
	    freq = note_name_to_freq(note_name);
	    if (freq == 0.0) {
		/* Bad note name? Can't happen. */
		fprintf(stdout, "Internal error in draw_note_names()\n");
		continue;
	    }
	    half_a_pixel = sqrt(v_pixel_freq_ratio());
	    /* If the tick is within the axis range, draw it.
	     * The half pixel slop ensures that the top and bottom pixel rows
	     * receive a tick if the vertical pixel position of that label
	     * would round to the pixel row in question. Otherwise half the
	     * time at random the top or bottom pixel rows are not labeled.
	     */
	    if (DELTA_GE(freq, min_freq / half_a_pixel) &&
	        DELTA_LE(freq, max_freq * half_a_pixel)) {
		int y = min_y + freq_to_magindex(freq);
		gui_lock();
		gui_putpixel(max_x + 1, y, green);
		gui_putpixel(max_x + 2, y, green);
		gui_unlock();
		draw_text(note_name, max_x + 4, y, LEFT, CENTER);
	    }
	}
    }
    gui_update_rect(max_x + 1, 0, disp_width - 1, disp_height - 1);
}

/* Decide where to put ticks and numbers on an axis.
 *
 * Graph-labeling convention is that the least significant digit that changes
 * from one label to the next should change by 1, 2 or 5, so we step by the
 * largest suitable value of 10^n * {1, 2 or 5} that gives us the required
 * number of divisions / numeric labels.
 */

/* The old code used to make 6 to 14 divisions and number every other tick.
 * What we now mean by "division" is one of the gaps between numbered segments
 * so we ask for a minimum of 3 to give the same effect as the old minimum of
 * 6 half-divisions.
 * This results in the same axis labeling for all maximum values
 * from 0 to 12000 in steps of 1000 and gives sensible results from 13000 on,
 * to a maximum of 7 divisions and 8 labels from 0 to 14000.
 */
#define TARGET_DIVISIONS 3

/* Add a tick to the frequency axis
 * k: which element of tick[] should we writing to?
 * log_scale: Whether we're doing a linear or logarithmic section of the scale
 * val: The frequency that this tick represents
 * just_a_tick: Whether to display just a tick with no numeric frequency label
 */
static int
add_tick(int k, int log_scale, freq_t val, bool just_a_tick)
{
    freq_t range = max_freq - min_freq;

    if (k >= number_of_ticks) {
	ticks = Realloc(ticks, (k+1) * sizeof(*ticks));
	number_of_ticks = k+1;
    }

    if (DELTA_GE(val, min_freq) && DELTA_LE(val, max_freq)) {
	ticks[k].value = just_a_tick ? NO_NUMBER : val;
	ticks[k].y = min_y + round((max_y - min_y) *
				   (log_scale == 2
				    ? (log(val / min_freq)) /
				      (log(max_freq / min_freq))
				    : (val - min_freq) / range));
	k++;
    }
    return k;
}

/* log_scale is pseudo-boolean:
 * 0 means use a linear scale,
 * 1 means use a log scale and
 * 2 is an internal value used when calling back from calculate_log_ticks() to
 *   label the range with linear numbering but logarithmic spacing.
 */
static int
calculate_ticks(int log_scale)
{
    freq_t stride;	/* Put numbered ticks at multiples of this */
    freq_t range = max_freq - min_freq;
    int k;
    freq_t value;	/* Temporary */

    if (log_scale == 1)
	    return calculate_log_ticks();

    /* Linear version */

    /* Choose a step between successive axis labels so that one digit
     * changes by 1, 2 or 5 and that gives us at least the number of
     * divisions (and numeric labels) that we would like to have.
     *
     * We do this by starting "stride" at the lowest power of ten <= max_freq,
     * which can give us at most 9 divisions (e.g. from 0 to 9999, step 1000)
     * Then try 5*this, 2*this and 1*this.
     */
    stride = pow(10.0, floor(log10(max_freq)));
    do {
	if (range / (stride * 5) >= TARGET_DIVISIONS) {
	    stride *= 5;
	    break;
	}
	if (range / (stride * 2) >= TARGET_DIVISIONS) {
	    stride *= 2;
	    break;
	}
	if (range / stride >= TARGET_DIVISIONS) break;
	stride /= 10;
    } while (1);	/* This is an odd loop! */

    /* Ensure that the least significant digit that changes gets printed, */
    decimal_places_to_print = round(-floor(log10(stride)));
    if (decimal_places_to_print < 0)
	    decimal_places_to_print = 0;

    /* Now go from the first multiple of stride that's >= min_freq to
     * the last one that's <= max_freq. */
    k = 0;
    value = ceil(min_freq / stride) * stride;

    /* Add the half-way tick before the first number if it's in range */
    k = add_tick(k, log_scale, value - stride / 2, TRUE);

    while (DELTA_LE(value, max_freq)) {
	/* Add a tick next to each printed number */
	k = add_tick(k, log_scale, value, FALSE);

	/* and at the half-way tick after the number if it's in range */
	k = add_tick(k, log_scale, value + stride/2, TRUE);

	value += stride;
    }

    return k;
}

/* Number/tick placer for logarithmic scales.
 *
 * Some say we should number 1, 10, 100, 1000, 1000 and place ticks at
 * 2,3,4,5,6,7,8,9, 20,30,40,50,60,70,80,90, 200,300,400,500,600,700,800,900
 * Others suggest numbering 1,2,5, 10,20,50, 100,200,500.
 *
 * Ticking 1-9 is visually distinctive and emphasizes that we are using
 * a log scale, as well as mimicking log graph paper.
 * Numbering the powers of ten and, if that doesn't give enough labels,
 * numbering also the 2 and 5 multiples might work.
 *
 * Apart from our [number] and tick styles:
 * [1] 2 5 [10] 20 50 [100]  and
 * [1] [2] 3 4 [5] 6 7 8 9 [10]
 * the following are also seen in use:
 * [1] [2] 3 4 [5] 6 7 [8] 9 [10]  and
 * [1] [2] [3] [4] [5] [6] 7 [8] 9 [10]
 * in https://www.lhup.edu/~dsimanek/scenario/errorman/graphs2.htm
 *
 * This works fine for wide ranges, not so well for narrow ranges like
 * 5000-6000, so for ranges less than a decade we apply the above
 * linear numbering style 0.2 0.4 0.6 0.8 or whatever, but calculating
 * the positions of the legends logarithmically.
 *
 * Alternatives could be:
 * - by powers or two from some starting frequency defaulting to
 *   the Nyquist frequency (22050, 11025, 5512.5 ...) or from some
 *   musical pitch (220, 440, 880, 1760)
 * - with a musical note scale  C0 ' D0 ' E0 F0 ' G0 ' A0 ' B0 C1
 * - with manuscript staff lines, piano note or guitar string overlay.
 */

/* Helper functions: add ticks and labels at start_value and all powers of ten
 * times it that are in the min_freq..max_freq range.
 * This is used to plonk ticks at 1, 10, 100, 1000 then at 2, 20, 200, 2000
 * then at 5, 50, 500, 5000 and so on.
 */
static int
add_log_ticks(int k, freq_t start_value, bool include_number)
{
    freq_t value;

    /* Include the top and bottom rows if the tick for a frequency label
     * would round to that pixel row */
    double half_a_pixel = sqrt(v_pixel_freq_ratio());

    for (value = start_value; DELTA_LE(value, max_freq * half_a_pixel);
         value *= 10.0) {
	if (k >= number_of_ticks) {
	    ticks = Realloc(ticks, (k+1) * sizeof(*ticks));
	    number_of_ticks = k+1;
	}
	if (DELTA_GE(value, min_freq / half_a_pixel)) {
	    ticks[k].value = include_number ? value : NO_NUMBER;
	    ticks[k].y = min_y + round((max_y - min_y)
	    			       * (log(value) - log(min_freq))
				       / (log(max_freq) - log(min_freq)));
	    k++;
	}
    }
    return k;
}

static int
calculate_log_ticks(void)
{
    int k = 0;	/* Number of ticks we have placed in "ticks" array */
    double underpinning;	/* Largest power of ten that is <= min_freq */

    /* If the interval is less than a decade, just apply the same
     * numbering-choosing scheme as used with linear axis, with the
     * ticks positioned logarithmically.
     */
    if (max_freq / min_freq < 10.0)
	    return calculate_ticks(2);

    /* First hack: label the powers of ten. */

    /* Find largest power of ten that is <= minimum value */
    underpinning = pow(10.0, floor(log10(min_freq)));

    /* Go powering up by 10 from there, numbering as we go. */
    k = add_log_ticks(k, underpinning, TRUE);

    /* Do we have enough numbers? If so, add numberless ticks at 2 and 5 */
    /* No of labels is n. of divisions + 1 */
    if (k >= TARGET_DIVISIONS + 1) {
	k = add_log_ticks(k, underpinning * 2.0, FALSE);
	k = add_log_ticks(k, underpinning * 5.0, FALSE);
    } else {
	int i;
	/* Not enough numbers: add numbered ticks at 2 and 5 and
	 * unnumbered ticks at all the rest */
	for (i = 2; i <= 9; i++)
	    k = add_log_ticks(k, underpinning * (1.0 * i),
			      i == 2 || i == 5);
    }

    /* Greatest possible number of ticks calculation:
     * The worst case is when the else clause adds 8 ticks with the maximal
     * number of divisions, which is when k == TARGET_DIVISIONS, 3,
     * for example 100, 1000, 10000.
     * The else clause adds another 8 ticks inside each division as well as
     * up to 8 ticks after the last number (from 20000 to 90000)
     * and 8 before to the first (from 20 to 90 in the example).
     * Maximum possible ticks is 3+8+8+8+8=35
     */

    return k;
}

/* Stuff for axes along the top and bottom edges of the screen */

/* Draw the status line along the top of the graph */
void
draw_status_line(void)
{
    char s[128];

    /* First, blank it */
    gui_paint_rect(min_x, max_y+1, max_x, disp_height-1, black);

    /* max_y + 2, BOTTOM leaves one blank pixel row between the graph
     * and the bottom of the text.
     * TOP and (disp_height - 1) - 2 would also work.
     */

    /* For frequency ranges, 1% precision (3 significant digits) is OK,
     * but to keep the labels short and avoid %g saying 1.42E+04 for 14200,
     * we show the whole number for anything over 99.
     */

#define hz_prec(freq)  (/* 100000 up */   (freq) >= 99999.5 ? 6 : \
			/* 10000-99999 */ (freq) >= 9999.5  ? 5 : \
			/* 1000-9999 */   (freq) >= 999.5   ? 4 : \
			/* 100-999, 52.1, 1.25 */             3)
    /* Left-justified with one blank column to its left */
    sprintf(s, "%.*g-", hz_prec(min_freq), min_freq);
    sprintf(s+strlen(s), "%.*gHz, ", hz_prec(max_freq), max_freq);
    sprintf(s+strlen(s), "%.3g octaves", log(max_freq / min_freq) / log(2.0));
    draw_text(s, min_x + 1, max_y + 2, LEFT, BOTTOM);

    /* Centered */
    sprintf(s, "%g pix/sec, %g pix/oct",
	    ppsec, round(log(2.0) / log(v_pixel_freq_ratio())));
    draw_text(s, disp_width / 2, max_y + 2, CENTER, BOTTOM);

    /* Right-justified with one blank column to its right */
    sprintf(s, "%.3gdB, %s window ",
	    (double)dyn_range, window_name(window_function));
    if (fft_freq > 999)
	/* Avoid it saying 1.25+E03 for values >= 1000 */
	sprintf(s+strlen(s), "@ %ldHz", (long)round(fft_freq));
    else
	sprintf(s+strlen(s), "@ %.3gHz", fft_freq);
    draw_text(s, max_x - 1, max_y + 2, RIGHT, BOTTOM);

    gui_update_rect(min_x, max_y + 1, max_x, disp_height - 1);
}

static void draw_time_axis_progress(void);

/* Draw the times along the bottom of the graph */
void
draw_time_axis(void)
{
    /* Time of centre of left- and rightmost pixel columns */
    secs_t min_time = screen_column_to_start_time(min_x) + secpp/2;
    secs_t max_time = screen_column_to_start_time(max_x) + secpp/2;

    /* First, blank it */
    gui_paint_rect(min_x, 0, max_x, min_y-1, black);

    draw_time_axis_progress();

    /* Current playing time; if at end of file, print only end-of-file time */
    if (playing != STOPPED)
	draw_text(seconds_to_string(disp_time), disp_offset, 1, CENTER, BOTTOM);

    /* From */
    if (DELTA_GE(min_time, 0.0)) {
	draw_text(seconds_to_string(min_time), min_x+1, 1, LEFT, BOTTOM);
    } else {
	/* Draw 0.0 at wherever it is on-screen */
	int x = time_to_screen_column(0.0);
	char *s = seconds_to_string(0.0);

	/* We center it on the left edge but stop it overflowing into
	 * the frequency axis.
	 */
	if (x - text_width(s)/2 > min_x+1)
	    draw_text(s, x, 1, CENTER, BOTTOM);
	else
	    draw_text(s, min_x+1, 1, LEFT, BOTTOM);
    }

    /* To */
    if (DELTA_LE(max_time, audio_file_length())) {
	draw_text(seconds_to_string(max_time), max_x-1, 1, RIGHT, BOTTOM);
    } else {
	/* Draw max_time wherever it is on-screen.
	 * We mark the start time of each column we label,
	 * so truncate end time to the stride.
	 */
	int x = time_to_screen_column(audio_file_length());
	char *s = seconds_to_string(audio_file_length());

	/*
	 * Center it on the rightmost column or, if it would
	 * overflow the right edge into the note name axis,
	 * right-adjusted. The "<=" makes it behave correctly
	 * when the text width is an odd number (which it always is).
	 */
	if (x + text_width(s)/2 <= max_x-1)
	    draw_text(s, x, 1, CENTER, BOTTOM);
	else
	    draw_text(s, max_x-1, 1, RIGHT, BOTTOM);
    }

    /* Bar lines */
    if (left_bar_time != UNDEFINED &&
	DELTA_GE(left_bar_time, screen_column_to_start_time(min_x) + secpp/2) &&
	DELTA_LE(left_bar_time, screen_column_to_start_time(max_x) + secpp/2)) {
	draw_text(seconds_to_string(left_bar_time),
		  time_to_screen_column(left_bar_time), 1, CENTER, BOTTOM);
    }
    if (right_bar_time != UNDEFINED &&
	DELTA_GE(right_bar_time, screen_column_to_start_time(min_x) + secpp/2) &&
	DELTA_LE(right_bar_time, screen_column_to_start_time(max_x) + secpp/2)) {
	draw_text(seconds_to_string(right_bar_time),
		  time_to_screen_column(right_bar_time), 1, CENTER, BOTTOM);
    }

    gui_update_rect(min_x, 0, max_x, min_y - 1);
}

/* Show on the time axis which part of the audio file is displayed.
 *
 * The width of the progress bar represents the whole piece from time 0
 * to time audio_file_length() and the edges of the cursor represent the
 * edges of what you can see on the screen.
 */
static void
draw_time_axis_progress(void)
{
    int x1, x2;	 /* Left and right box pixel positions */
    int y1, y2;  /* bottom and top box pixe lpositions */

    /* How much time does the graphic show? */
    unsigned graph_width = max_x - min_x + 1;
    secs_t graph_time = graph_width / ppsec;

    /* Cursor size measured from left edge of left pixel to right of right */
    unsigned box_width = graph_width * (graph_time / audio_file_length());

    /* It's no good calculating the left and right positions separately
     * because that makes the left and right edges update independently
     * making the box's width change and the box crawl like a caterpillar.
     * We obtain constant box width as well as start/end alignment
     * by aligning a fixed-width box's left edge if it's on the left half of
     * the screen, and aligning the right edge if it's on the right.
     * No one will notice a possible single-pixel position irregularity
     * as the box slides past the date line.
     */
    if (disp_time < audio_file_length() / 2) {
	x1 = min_x + floor(graph_width *
		     screen_column_to_start_time(min_x) / audio_file_length());
	x2 = x1 + (box_width - 1);
    } else {
	x2 = min_x + floor(graph_width *
		     screen_column_to_start_time(max_x) / audio_file_length());
        x1 = x2 - (box_width - 1);
    }

    y1 = 0; y2 = time_axis_height - 1;

    /* Don't spill over into the frequency axes,
     * and if the left/right edge is outside the graph, don't show it.
     */
#ifndef min
# define min(a,b) ((a)<=(b)?(a):(b))
#endif
#ifndef max
# define max(a,b) ((a)>=(b)?(a):(b))
#endif
    gui_lock();
    /* top */    gui_draw_line(max(x1, min_x), y2, min(x2, max_x), y2, white);
    /* bottom */ gui_draw_line(max(x1, min_x), y1, min(x2, max_x), y1, white);
    /* left */   if (x1 >= min_x) gui_draw_line(x1, y1, x1, y2, white);
    /* right */  if (x2 <= max_x) gui_draw_line(x2, y1, x2, y2, white);
    gui_unlock();
}

void
show_title(bool onoff)
{
  if (!show_songbar && onoff) {
     /* Add the song's name bar under the time axis */
    show_songbar = 1;
  } else {
    /* Remove the songbar */
    show_songbar = 0;
  }
}
