﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/eventbridge/EventBridge_EXPORTS.h>
#include <aws/eventbridge/model/RuleState.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace EventBridge {
namespace Model {

/**
 * <p>Contains information about a rule in Amazon EventBridge.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/eventbridge-2015-10-07/Rule">AWS
 * API Reference</a></p>
 */
class Rule {
 public:
  AWS_EVENTBRIDGE_API Rule() = default;
  AWS_EVENTBRIDGE_API Rule(Aws::Utils::Json::JsonView jsonValue);
  AWS_EVENTBRIDGE_API Rule& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_EVENTBRIDGE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the rule.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Rule& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the rule.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  Rule& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The event pattern of the rule. For more information, see <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-and-event-patterns.html">Events
   * and Event Patterns</a> in the <i> <i>Amazon EventBridge User Guide</i> </i>.</p>
   */
  inline const Aws::String& GetEventPattern() const { return m_eventPattern; }
  inline bool EventPatternHasBeenSet() const { return m_eventPatternHasBeenSet; }
  template <typename EventPatternT = Aws::String>
  void SetEventPattern(EventPatternT&& value) {
    m_eventPatternHasBeenSet = true;
    m_eventPattern = std::forward<EventPatternT>(value);
  }
  template <typename EventPatternT = Aws::String>
  Rule& WithEventPattern(EventPatternT&& value) {
    SetEventPattern(std::forward<EventPatternT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state of the rule.</p> <p>Valid values include:</p> <ul> <li> <p>
   * <code>DISABLED</code>: The rule is disabled. EventBridge does not match any
   * events against the rule.</p> </li> <li> <p> <code>ENABLED</code>: The rule is
   * enabled. EventBridge matches events against the rule, <i>except</i> for Amazon
   * Web Services management events delivered through CloudTrail.</p> </li> <li> <p>
   * <code>ENABLED_WITH_ALL_CLOUDTRAIL_MANAGEMENT_EVENTS</code>: The rule is enabled
   * for all events, including Amazon Web Services management events delivered
   * through CloudTrail.</p> <p>Management events provide visibility into management
   * operations that are performed on resources in your Amazon Web Services account.
   * These are also known as control plane operations. For more information, see <a
   * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html#logging-management-events">Logging
   * management events</a> in the <i>CloudTrail User Guide</i>, and <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-service-event.html#eb-service-event-cloudtrail">Filtering
   * management events from Amazon Web Services services</a> in the <i> <i>Amazon
   * EventBridge User Guide</i> </i>.</p> <p>This value is only valid for rules on
   * the <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-what-is-how-it-works-concepts.html#eb-bus-concepts-buses">default</a>
   * event bus or <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-event-bus.html">custom
   * event buses</a>. It does not apply to <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-saas.html">partner
   * event buses</a>.</p> </li> </ul>
   */
  inline RuleState GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(RuleState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline Rule& WithState(RuleState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the rule.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Rule& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The scheduling expression. For example, "cron(0 20 * * ? *)", "rate(5
   * minutes)". For more information, see <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-create-rule-schedule.html">Creating
   * an Amazon EventBridge rule that runs on a schedule</a>.</p>
   */
  inline const Aws::String& GetScheduleExpression() const { return m_scheduleExpression; }
  inline bool ScheduleExpressionHasBeenSet() const { return m_scheduleExpressionHasBeenSet; }
  template <typename ScheduleExpressionT = Aws::String>
  void SetScheduleExpression(ScheduleExpressionT&& value) {
    m_scheduleExpressionHasBeenSet = true;
    m_scheduleExpression = std::forward<ScheduleExpressionT>(value);
  }
  template <typename ScheduleExpressionT = Aws::String>
  Rule& WithScheduleExpression(ScheduleExpressionT&& value) {
    SetScheduleExpression(std::forward<ScheduleExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the role that is used for target
   * invocation.</p> <p>If you're setting an event bus in another account as the
   * target and that account granted permission to your account through an
   * organization instead of directly by the account ID, you must specify a
   * <code>RoleArn</code> with proper permissions in the <code>Target</code>
   * structure, instead of here in this parameter.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  Rule& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the rule was created on behalf of your account by an Amazon Web Services
   * service, this field displays the principal name of the service that created the
   * rule.</p>
   */
  inline const Aws::String& GetManagedBy() const { return m_managedBy; }
  inline bool ManagedByHasBeenSet() const { return m_managedByHasBeenSet; }
  template <typename ManagedByT = Aws::String>
  void SetManagedBy(ManagedByT&& value) {
    m_managedByHasBeenSet = true;
    m_managedBy = std::forward<ManagedByT>(value);
  }
  template <typename ManagedByT = Aws::String>
  Rule& WithManagedBy(ManagedByT&& value) {
    SetManagedBy(std::forward<ManagedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the event bus associated with the rule. If you omit this,
   * the default event bus is used.</p>
   */
  inline const Aws::String& GetEventBusName() const { return m_eventBusName; }
  inline bool EventBusNameHasBeenSet() const { return m_eventBusNameHasBeenSet; }
  template <typename EventBusNameT = Aws::String>
  void SetEventBusName(EventBusNameT&& value) {
    m_eventBusNameHasBeenSet = true;
    m_eventBusName = std::forward<EventBusNameT>(value);
  }
  template <typename EventBusNameT = Aws::String>
  Rule& WithEventBusName(EventBusNameT&& value) {
    SetEventBusName(std::forward<EventBusNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  Aws::String m_eventPattern;
  bool m_eventPatternHasBeenSet = false;

  RuleState m_state{RuleState::NOT_SET};
  bool m_stateHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_scheduleExpression;
  bool m_scheduleExpressionHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::String m_managedBy;
  bool m_managedByHasBeenSet = false;

  Aws::String m_eventBusName;
  bool m_eventBusNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace EventBridge
}  // namespace Aws
