/*
 * @(#)MarkedInstruction.java
 *
 * Copyright (C) 2002,2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.compiler;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.bcel.generic.Instruction;
import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.InstructionList;

/**
 * Contains one original method instruction, along with all the analysis
 * module marks for this instruction.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:25 $
 * @since     December 17, 2002
 */
public class MarkedInstruction
{
    private static final org.apache.log4j.Logger LOG =
        org.apache.log4j.Logger.getLogger( MarkedInstruction.class );
    private final InstructionHandle origInstr;
    private final List marks = new LinkedList();
    private boolean closed = false;
    
    
    /**
     * constant pool index for the name of the class's signature.
     */
    private final int classSigPoolIndex;
    
    /**
     * constant pool index for the method-ref for invoking the logger.
     */
    private final int staticMethodPoolIndex;
    
    /**
     * Reference to the owning method's index
     */
    private final short methodIndex;
    
    /**
     * @throws IllegalStateException if the class file has already been
     *         modified (identified by a class name field).
     */
    MarkedInstruction( short methodIndex, int classSigPoolIndex,
            int staticMethodPoolIndex, InstructionHandle instr )
    {
        if (instr == null)
        {
            throw new IllegalArgumentException("no null args");
        }
        this.origInstr = instr;
        this.methodIndex = methodIndex;
        this.classSigPoolIndex = classSigPoolIndex;
        this.staticMethodPoolIndex = staticMethodPoolIndex;
    }
    
    
    /**
     * Retrieve the list of all the marks (if there are any active ones),
     * but does not add the original instruction (that is for the owner to
     * know where it goes - before or after).  If there are no active
     * marks, then <tt>null</tt> will be returned.  After this has been
     * called, all the active marks will be removed (assuming that they
     * have been added to the modified method instruction list).
     */
    InstructionList getMarkedList()
    {
        InstructionList markedList = null;
        if (!this.marks.isEmpty())
        {
            markedList = new InstructionList();
            // mark the instruction first, then add the instruction
            Iterator iter = this.marks.iterator();
            while (iter.hasNext())
            {
                LOG.debug( "Adding mark to list." );
                ((MeasureMark)iter.next()).addToInstructionList( markedList );
            }
            markedList.setPositions( true );
            // since we just added the marks, we shouldn't add them
            // again, so remove all current marks.
            this.marks.clear();
        }
        return markedList;
    }
    
    
    public int getInstructionPosition()
    {
        return this.origInstr.getPosition();
    }
    
    
    /**
     * Retrieve the instruction represented by this mark.
     */
    public Instruction getInstruction()
    {
        return this.origInstr.getInstruction();
    }
    
    
    /**
     * Mark this instruction with the given measure and mark indicies; the
     * method index is pre-defined by the particular method this instruction
     * resides in.
     */
    public void addMark( short measureIndex, short markIndex )
    {
        MeasureMark mm = new MeasureMark( this.classSigPoolIndex,
            this.staticMethodPoolIndex, this.methodIndex,
            measureIndex, markIndex );
        this.marks.add( mm );
    }
    
    
    /**
     * Retrieve the handle this instruction represents.
     */
    InstructionHandle getHandle()
    {
        return this.origInstr;
    }
}

