/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf.wal;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import java.util.zip.CRC32;
import java.util.zip.CRC32C;
import java.util.zip.GZIPInputStream;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWAL;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalConfig;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalRecord;
import org.eclipse.rdf4j.sail.nativerdf.wal.ValueStoreWalValueKind;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ValueStoreWalReader
implements AutoCloseable {
    private static final Pattern SEGMENT_PATTERN = Pattern.compile("wal-(\\d+)\\.v1(?:\\.gz)?");
    private static final Logger logger = LoggerFactory.getLogger(ValueStoreWalReader.class);
    private final ValueStoreWalConfig config;
    private final JsonFactory jsonFactory = new JsonFactory();
    private final List<SegmentEntry> segments;
    private int segIndex = -1;
    private FileChannel channel;
    private GZIPInputStream gzIn;
    private boolean stop;
    private boolean eos;
    private long lastValidLsn = -1L;
    private final boolean missingSegments;
    private boolean summaryMissing;
    private boolean currentSegmentCompressed;
    private boolean currentSegmentSummarySeen;
    private CRC32 currentSegmentCrc32;

    private ValueStoreWalReader(ValueStoreWalConfig config) {
        List<SegmentEntry> segs;
        this.config = Objects.requireNonNull(config, "config");
        try {
            segs = this.listSegments();
        }
        catch (IOException e) {
            segs = List.of();
        }
        this.segments = segs;
        this.missingSegments = this.hasSequenceGaps(segs);
        this.summaryMissing = false;
        this.currentSegmentCompressed = false;
        this.currentSegmentSummarySeen = false;
    }

    public static ValueStoreWalReader open(ValueStoreWalConfig config) {
        return new ValueStoreWalReader(config);
    }

    public ScanResult scan() throws IOException {
        ArrayList<ValueStoreWalRecord> records = new ArrayList<ValueStoreWalRecord>();
        Iterator<ValueStoreWalRecord> it = this.iterator();
        while (it.hasNext()) {
            records.add(it.next());
        }
        return new ScanResult(records, this.lastValidLsn(), this.isComplete());
    }

    public Iterator<ValueStoreWalRecord> iterator() {
        return new RecordIterator();
    }

    public long lastValidLsn() {
        return this.lastValidLsn;
    }

    private boolean openNextSegment() throws IOException {
        this.closeCurrentSegment();
        ++this.segIndex;
        if (this.segIndex >= this.segments.size()) {
            return false;
        }
        SegmentEntry entry = this.segments.get(this.segIndex);
        Path p = entry.path;
        this.currentSegmentCompressed = entry.compressed;
        this.currentSegmentSummarySeen = false;
        if (this.currentSegmentCompressed) {
            this.gzIn = new GZIPInputStream(Files.newInputStream(p, new OpenOption[0]));
            this.channel = null;
            this.currentSegmentCrc32 = new CRC32();
        } else {
            this.channel = FileChannel.open(p, StandardOpenOption.READ);
            this.gzIn = null;
            this.currentSegmentCrc32 = null;
        }
        return true;
    }

    private void closeCurrentSegment() throws IOException {
        if (this.currentSegmentCompressed && !this.currentSegmentSummarySeen) {
            this.summaryMissing = true;
        }
        this.currentSegmentCrc32 = null;
        if (this.channel != null && this.channel.isOpen()) {
            this.channel.close();
        }
        this.channel = null;
        if (this.gzIn != null) {
            this.gzIn.close();
        }
        this.gzIn = null;
        this.eos = false;
        this.currentSegmentCompressed = false;
        this.currentSegmentSummarySeen = false;
    }

    private static int readIntLE(InputStream in) throws IOException {
        byte[] b = in.readNBytes(4);
        if (b.length < 4) {
            return -1;
        }
        return b[0] & 0xFF | (b[1] & 0xFF) << 8 | (b[2] & 0xFF) << 16 | (b[3] & 0xFF) << 24;
    }

    private List<SegmentEntry> listSegments() throws IOException {
        ArrayList<Item> items = new ArrayList<Item>();
        if (!Files.isDirectory(this.config.walDirectory(), new LinkOption[0])) {
            return List.of();
        }
        try (Stream<Path> stream = Files.list(this.config.walDirectory());){
            stream.forEach(p -> {
                Matcher m = SEGMENT_PATTERN.matcher(p.getFileName().toString());
                if (m.matches()) {
                    long firstId = Long.parseLong(m.group(1));
                    boolean compressed = p.getFileName().toString().endsWith(".gz");
                    int sequence = 0;
                    try {
                        sequence = ValueStoreWAL.readSegmentSequence(p);
                    }
                    catch (IOException e) {
                        logger.warn("Failed to read WAL segment header for {}", (Object)p.getFileName(), (Object)e);
                    }
                    items.add(new Item((Path)p, firstId, sequence, compressed));
                }
            });
        }
        items.sort(Comparator.comparingInt(it -> it.sequence));
        ArrayList<SegmentEntry> segments = new ArrayList<SegmentEntry>(items.size());
        for (Item it2 : items) {
            segments.add(new SegmentEntry(it2.path, it2.firstId, it2.sequence, it2.compressed));
        }
        return segments;
    }

    private boolean hasSequenceGaps(List<SegmentEntry> entries) {
        if (entries.isEmpty()) {
            return false;
        }
        int expected = entries.get((int)0).sequence;
        if (expected > 1) {
            return true;
        }
        for (SegmentEntry entry : entries) {
            if (entry.sequence != expected) {
                return true;
            }
            ++expected;
        }
        return false;
    }

    private ValueStoreWalRecord readOneFromChannel() throws IOException {
        int n;
        ByteBuffer header = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
        header.clear();
        int read = this.channel.read(header);
        if (read == -1) {
            this.eos = true;
            return null;
        }
        if (read < 4) {
            this.stop = true;
            return null;
        }
        header.flip();
        int length = header.getInt();
        if (length <= 0 || (long)length > 0x20000000L) {
            this.stop = true;
            return null;
        }
        byte[] data = new byte[length];
        ByteBuffer dataBuf = ByteBuffer.wrap(data);
        for (int total = 0; total < length; total += n) {
            n = this.channel.read(dataBuf);
            if (n >= 0) continue;
            this.stop = true;
            return null;
        }
        ByteBuffer crcBuf = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
        int crcRead = this.channel.read(crcBuf);
        if (crcRead < 4) {
            this.stop = true;
            return null;
        }
        crcBuf.flip();
        int expectedCrc = crcBuf.getInt();
        CRC32C crc32c = new CRC32C();
        crc32c.update(data, 0, data.length);
        if ((int)crc32c.getValue() != expectedCrc) {
            this.stop = true;
            return null;
        }
        Parsed parsed = this.parseJson(data);
        if (parsed.type == 'M') {
            ValueStoreWalRecord r = new ValueStoreWalRecord(parsed.lsn, parsed.id, parsed.kind, parsed.lex, parsed.dt, parsed.lang, parsed.hash);
            this.lastValidLsn = r.lsn();
            return r;
        }
        if (parsed.lsn > this.lastValidLsn) {
            this.lastValidLsn = parsed.lsn;
        }
        this.eos = false;
        return null;
    }

    private ValueStoreWalRecord readOneFromGzip() throws IOException {
        int length = ValueStoreWalReader.readIntLE(this.gzIn);
        if (length == -1) {
            this.eos = true;
            return null;
        }
        if (length <= 0 || (long)length > 0x20000000L) {
            this.stop = true;
            return null;
        }
        byte[] data = this.gzIn.readNBytes(length);
        if (data.length < length) {
            this.stop = true;
            return null;
        }
        int expectedCrc = ValueStoreWalReader.readIntLE(this.gzIn);
        CRC32C crc32c = new CRC32C();
        crc32c.update(data, 0, data.length);
        if ((int)crc32c.getValue() != expectedCrc) {
            this.stop = true;
            return null;
        }
        Parsed parsed = this.parseJson(data);
        if (this.currentSegmentCrc32 != null && parsed.type != 'S') {
            ByteBuffer lenBuf = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putInt(length);
            lenBuf.flip();
            this.currentSegmentCrc32.update(lenBuf.array(), 0, 4);
            this.currentSegmentCrc32.update(data, 0, data.length);
            ByteBuffer crcBuf = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putInt(expectedCrc);
            crcBuf.flip();
            this.currentSegmentCrc32.update(crcBuf.array(), 0, 4);
        }
        if (parsed.type == 'M') {
            ValueStoreWalRecord r = new ValueStoreWalRecord(parsed.lsn, parsed.id, parsed.kind, parsed.lex, parsed.dt, parsed.lang, parsed.hash);
            this.lastValidLsn = r.lsn();
            return r;
        }
        if (parsed.type == 'S') {
            long computed;
            this.currentSegmentSummarySeen = true;
            if (this.currentSegmentCrc32 != null && parsed.summaryCrc32 != (computed = this.currentSegmentCrc32.getValue() & 0xFFFFFFFFL)) {
                this.stop = true;
            }
        }
        if (parsed.lsn > this.lastValidLsn) {
            this.lastValidLsn = parsed.lsn;
        }
        this.eos = false;
        return null;
    }

    private Parsed parseJson(byte[] jsonBytes) throws IOException {
        Parsed parsed = new Parsed();
        try (JsonParser jp = this.jsonFactory.createParser(jsonBytes);){
            if (jp.nextToken() != JsonToken.START_OBJECT) {
                Parsed parsed2 = parsed;
                return parsed2;
            }
            while (jp.nextToken() != JsonToken.END_OBJECT) {
                String field = jp.getCurrentName();
                jp.nextToken();
                if ("t".equals(field)) {
                    String t = jp.getValueAsString("");
                    parsed.type = (char)(t.isEmpty() ? 63 : (int)t.charAt(0));
                    continue;
                }
                if ("lsn".equals(field)) {
                    parsed.lsn = jp.getValueAsLong(-1L);
                    continue;
                }
                if ("id".equals(field)) {
                    parsed.id = jp.getValueAsInt(0);
                    continue;
                }
                if ("lastId".equals(field)) {
                    parsed.id = jp.getValueAsInt(0);
                    continue;
                }
                if ("vk".equals(field)) {
                    String code = jp.getValueAsString("");
                    parsed.kind = ValueStoreWalValueKind.fromCode(code);
                    continue;
                }
                if ("lex".equals(field)) {
                    parsed.lex = jp.getValueAsString("");
                    continue;
                }
                if ("dt".equals(field)) {
                    parsed.dt = jp.getValueAsString("");
                    continue;
                }
                if ("lang".equals(field)) {
                    parsed.lang = jp.getValueAsString("");
                    continue;
                }
                if ("hash".equals(field)) {
                    parsed.hash = jp.getValueAsInt(0);
                    continue;
                }
                if ("crc32".equals(field)) {
                    parsed.summaryCrc32 = jp.getValueAsLong(0L);
                    continue;
                }
                jp.skipChildren();
            }
        }
        return parsed;
    }

    @Override
    public void close() {
        try {
            this.closeCurrentSegment();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    boolean isComplete() {
        return !this.missingSegments && !this.summaryMissing && !this.stop;
    }

    public static final class ScanResult {
        private final List<ValueStoreWalRecord> records;
        private final long lastValidLsn;
        private final boolean complete;

        public ScanResult(List<ValueStoreWalRecord> records, long lastValidLsn, boolean complete) {
            this.records = List.copyOf(records);
            this.lastValidLsn = lastValidLsn;
            this.complete = complete;
        }

        public List<ValueStoreWalRecord> records() {
            return this.records;
        }

        public long lastValidLsn() {
            return this.lastValidLsn;
        }

        public boolean complete() {
            return this.complete;
        }
    }

    private final class RecordIterator
    implements Iterator<ValueStoreWalRecord> {
        private ValueStoreWalRecord next;
        private boolean prepared;

        private RecordIterator() {
        }

        @Override
        public boolean hasNext() {
            if (this.prepared) {
                return this.next != null;
            }
            try {
                this.prepareNext();
            }
            catch (IOException e) {
                ValueStoreWalReader.this.stop = true;
                this.next = null;
            }
            this.prepared = true;
            return this.next != null;
        }

        @Override
        public ValueStoreWalRecord next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            this.prepared = false;
            ValueStoreWalRecord r = this.next;
            this.next = null;
            return r;
        }

        private void prepareNext() throws IOException {
            this.next = null;
            if (ValueStoreWalReader.this.stop) {
                return;
            }
            while (ValueStoreWalReader.this.channel != null || ValueStoreWalReader.this.gzIn != null || ValueStoreWalReader.this.openNextSegment()) {
                ValueStoreWalRecord r;
                if (ValueStoreWalReader.this.gzIn != null) {
                    r = ValueStoreWalReader.this.readOneFromGzip();
                    if (r != null) {
                        this.next = r;
                        return;
                    }
                    if (ValueStoreWalReader.this.stop) {
                        return;
                    }
                    if (!ValueStoreWalReader.this.eos) continue;
                    ValueStoreWalReader.this.closeCurrentSegment();
                    continue;
                }
                if (ValueStoreWalReader.this.channel == null) continue;
                r = ValueStoreWalReader.this.readOneFromChannel();
                if (r != null) {
                    this.next = r;
                    return;
                }
                if (ValueStoreWalReader.this.stop) {
                    return;
                }
                if (!ValueStoreWalReader.this.eos) continue;
                ValueStoreWalReader.this.closeCurrentSegment();
            }
            return;
        }
    }

    private static final class SegmentEntry {
        final Path path;
        final long firstId;
        final int sequence;
        final boolean compressed;

        SegmentEntry(Path path, long firstId, int sequence, boolean compressed) {
            this.path = path;
            this.firstId = firstId;
            this.sequence = sequence;
            this.compressed = compressed;
        }
    }

    private static class Item {
        final Path path;
        final long firstId;
        final int sequence;
        final boolean compressed;

        Item(Path path, long firstId, int sequence, boolean compressed) {
            this.path = path;
            this.firstId = firstId;
            this.sequence = sequence;
            this.compressed = compressed;
        }
    }

    private static final class Parsed {
        char type = (char)63;
        long lsn = -1L;
        int id = 0;
        ValueStoreWalValueKind kind = ValueStoreWalValueKind.NAMESPACE;
        String lex = "";
        String dt = "";
        String lang = "";
        int hash = 0;
        long summaryCrc32 = 0L;

        private Parsed() {
        }
    }
}

